/***************************************************************************
 *   Copyright (C) 2003 by Sbastien Laot                                 *
 *   sebastien.laout@tuxfamily.org                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef BASKET_H
#define BASKET_H

#include <qwidget.h>
#include <qscrollview.h>
#include <qclipboard.h>
#include <qptrlist.h>
#include <qtimer.h>
#include <kio/job.h>
#include <qcolor.h>

#include "search.h"

class QFrame;
class QVBoxLayout;
class QCheckBox;
class QString;
class QColor;
class QPixmap;
class QAction;
class QStringList;
class QRect;

class QDomElement;

class KDirWatch;

class Basket;
class Item;
class ItemEditorBase;

/** Used to enqueue a file path when the Basket receive a file modification / creation / deletion
  * It associate the file name with an event.
  * All this queue will be treated later.
  * TODO: rename to class WatcherEvent ?
  * @author Sbastien Laot
  */
class FileEvent
{
  public:
	enum Event { Modified = 1, Created, Deleted, Renamed };
	FileEvent(Event evt, const QString &path)
	 : event(evt), filePath(path)
	{ }
  public: // Because it must be fast and theire is no need to be private
	Event   event;
	QString filePath;
};

/** This class handle Basket and add a SearchWidget on top of it.
  * @author Sbastien Laot
  */
class DecoratedBasket : public QWidget
{
  Q_OBJECT
  public:
	DecoratedBasket(QWidget *parent, const QString &folderName, const char *name = 0, WFlags fl = 0);
	~DecoratedBasket();
	void setSearchBarPosition(bool onTop);
	void resetSearch();
	void setSearchBarShown(bool show, bool switchFocus = true);
	bool isSearchBarShown()        { return m_search->isShown();    }
	const SearchData& searchData() { return m_search->searchData(); }
	SearchBar* searchBar()         { return m_search;               }
	Basket*    basket()            { return m_basket;               }
  private:
	QVBoxLayout *m_layout;
	SearchBar   *m_search;
	Basket      *m_basket;
};

/** Basket that contain some Items.
  * @author Sbastien Laot
  */
class Basket : public QScrollView
{
  Q_OBJECT
  public:
	/** Construtor and destructor */
	Basket(QWidget *parent, const QString &folderName, const char *name = "", WFlags fl = 0);
	~Basket();
  public:
	/** Basket properties GET */
	inline QString icon()                   { return m_icon;                   } // Properties ::
	inline bool    showCheckBoxes()         { return m_showCheckBoxes;         }
	inline int     hAlign()                 { return m_hAlign;                 }
	inline int     vAlign()                 { return m_vAlign;                 }
	inline bool    isLocked()               { return m_isLocked;               }
	inline QColor  color()                  { return m_color;                  }
	inline QColor  altColor()               { return m_altColor;               }
	inline bool    insertAtEnd()            { return m_insertAtEnd;            } // Add item policy ::
	inline bool    insertAtCursorPos()      { return m_insertAtCursorPos;      }
	inline bool    isAStack()               { return m_isAStack;               } // Special baskets ::
	inline bool    stackTakeOnSameSide()    { return m_stackTakeOnSameSide;    }
	inline int     stackAfterDrag()         { return m_stackAfterDrag;         }
	inline bool    isAClipboard()           { return m_isAClipboard;           }
	inline int     clipboardMaxItems()      { return m_clipboardMaxItems;      }
	inline int     clipboardWhich()         { return m_clipboardWhich;         }
	inline bool    mirrorOnlyNewFiles()     { return m_mirrorOnlyNewFiles;     }
	inline int     contentOnClickAction()   { return m_contentOnClickAction;   } // On click actions ::
	inline int     fileOnClickAction()      { return m_fileOnClickAction;      }
	inline bool    showSearchBar()          { return m_showSearchBar;          } // Others... ::
	/** Basket properties SET */
	void setName(const char *name); // FIXME: setBasketName(const QString &) && basketName()
	void setIcon(const QString &icon);
	void setShowCheckBoxes(bool show);
	void setAlign(int hAlign, int vAlign);
	void setLocked(bool lock);
	void setColors(const QColor &color, const QColor &altColor);
	void setInsertAtEnd(bool atEnd)          { m_insertAtEnd            = atEnd; resetInsertTo(); }
	void setInsertAtCursorPos(bool yes)      { m_insertAtCursorPos      = yes;                    }
	void setIsAStack(bool enable)            { m_isAStack               = enable && !isAMirror(); }
	void setStackTakeOnSameSide(bool yes)    { m_stackTakeOnSameSide    = yes;                    }
	void setStackAfterDrag(int whatToDo)     { m_stackAfterDrag         = whatToDo;               }
	void setIsAClipboard(bool enable);
	void setClipboardMaxItems(int max)       { m_clipboardMaxItems      = max;
	                                                        if (isAClipboard()) checkClipboard(); }
	void setClipboardWhich(int which)        { m_clipboardWhich         = which;                  }
	void setMirrorOnlyNewFiles(bool only)    { m_mirrorOnlyNewFiles     = only;
	                                           if (isAMirror() && only == false) reloadMirroredFolder(); }
	void setContentOnClickAction(int action) { m_contentOnClickAction   = action;                 }
	void setFileOnClickAction(int action)    { m_fileOnClickAction      = action;                 }
	void setShowSearchBar(bool show)         { m_showSearchBar          = show;                   }
	/** Save work */
	QString folderName();
	QString fullPath();
	bool isAMirror();
	static bool isAMirror(const QString &folderName);
	static QString fullPathForFolderName(const QString &folderName);
	QString fullPathForFileName(const QString &fileName); // Full path of an [existing or not] item
	void deleteFiles();
	void deleteBasketData();
  protected:
	virtual void contextMenuEvent(QContextMenuEvent *event);
	virtual void focusInEvent(QFocusEvent*);
	virtual void focusOutEvent(QFocusEvent*);
	virtual void keyPressEvent(QKeyEvent *event);
	virtual void contentsMousePressEvent(QMouseEvent *event); // For redirected event !!
	virtual void showEvent(QShowEvent *);
	/** Drag and drop functions */
	virtual void dragEnterEvent(QDragEnterEvent*);
	virtual void dragMoveEvent(QDragMoveEvent* event);
	virtual void dragLeaveEvent(QDragLeaveEvent*);
  public:
	virtual void dropEvent(QDropEvent *event);
	static  void acceptDropEvent(QDropEvent *event, bool preCond = true);
	bool  canDragItem()       { return   (isAStack() && !isEmpty()); }
	bool  stackTakeAtEnd()    { return ! (insertAtEnd() ^ stackTakeOnSameSide()); }
	Item* currentStackItem();
	void  consumeStackItem(Item *item);
	void  dragStackItem();
	void  rotateStack();
	void  insertStackItemInCurrentWindow();
	void  stackMoveToOppositeSide();
	Item* lastInsertedItem();
	void  computeInsertPlace(const QPoint &cursorPosition);
	Item* itemAtPosition(const QPoint &pos);
	Item* duplicatedOf(Item *item);
	void  checkClipboard();
	void  processActionAsYouType(QKeyEvent *event);
	void  exportToHTML();
	void  clearDirectory(const QString &path);
  signals:
	void nameChanged(Basket *basket, const QString &name);
	void iconChanged(Basket *basket, const QString &icon);
	void itemsNumberChanged(Basket *basket);
  public slots:
	void linkLookChanged();
	void showItemsToolTipChanged();
	/** Items manipulation */
	void insertItem(Item *item);
	void delItem(Item *item, bool askForMirroredFile = true);
	void changeItemPlace(Item *item);
	void pasteItem(QClipboard::Mode mode = QClipboard::Clipboard);
	void recolorizeItems();
	void clearStack();
	void openMirroredFolder();
	void reloadMirroredFolder();
	void showMirrorOnlyOnceInfo();
	/** Save & load work */
	void save();
	/** Selection of item(s) */
	void selectAll();
	void unselectAll();
	void unselectAllBut(Item *toSelect);
	void invertSelection();
	void selectCheckedItems();
	void selectRange(Item *start, Item *end);
	void clicked(Item *item, bool controlPressed, bool shiftPressed);
	void addSelectedItem();
	void removeSelectedItem();
	void resetSelectedItem();
	inline int countSelecteds() { return m_countSelecteds; };
	void setFocusedItem(Item *item);
	void focusAnItem();
	void ensureVisibleItem(Item *item);
	QRect itemRect(Item *item); // clipped global (desktop as origin) rectangle
	/** Travel the list to find the next shown item, or the previous if step == -1, or the next after 10 if step == 10... */
	Item* nextShownItemFrom(Item *item, int step);
	Item* nextCheckedItem(bool next, bool checked);
	void  gotoNextCheckedItem(bool next, bool checked);
	/** Actions on (selected) items */
	void editItem(Item *item, bool editAnnotations = false);
	void editItem();
	void editItemMetaData();
	void delItem();
	void copyItem();
	void cutItem();
	void openItem();
	void openItemWith();
	void saveItemAs();
	void checkItem();
	void moveOnTop();
	void moveOnBottom();
	void moveItemUp();
	void moveItemDown();
  signals:
	void changedSelectedItems();
	void areSelectedItemsCheckedChanged(bool checked);
  public:
	bool    areSelectedItemsChecked() { return m_areSelectedItemsChecked; }
	void    dontCareOfCreation(const QString &path);
	Item*   itemForFullPath(const QString &path);
	QColor  bgColor();
	QColor  bgAltColor();
	static QString textToHTML(const QString &text);
	static QString textToHTMLWithoutP(const QString &text);
	static QString htmlToParagraph(const QString &html);
	QString copyIcon(const QString &iconName, int size, const QString &destFolder);
	QString copyFile(const QString &srcPath, const QString &destFolder, bool createIt = false);
  protected slots:
	void slotModifiedFile(const QString &path);
	void slotCreatedFile(const QString &path);
	void slotDeletedFile(const QString &path);
	void slotUpdateItems();
	void slotCopyingDone(KIO::Job *, const KURL &, const KURL &to, bool, bool);
	void placeEditor();
	void closeEditor(bool save = true);
	void clipboardChanged(bool selectionMode = false);
	void selectionChanged();
	void newSearch(const SearchData &data);
	void resetSearch();
  public slots:
	void itemSizeChanged(Item *item);
  private:
	QTimer              m_updateTimer;
	QPtrList<FileEvent> m_updateQueue;
	QStringList         m_dontCare;
	static const int    c_updateTime;
  private:
	void load(); // Load is performed only once, during contructor
	void loadItems(const QDomElement &items);
	bool importLauncher(const QString &type, const QDomElement &content, const QString &runCommand,
	                    const QString &annotations, bool checked);
	void relayoutItems();
	void computeShownItems();
	virtual void viewportResizeEvent(QResizeEvent *);

  private:
	/** Basket properties */
	QString  m_icon;
	QColor   m_color;
	QColor   m_altColor;
	bool     m_showCheckBoxes;
	int      m_hAlign;
	int      m_vAlign;
	bool     m_isLocked;
	QString  m_folderName;
	bool     m_showSearchBar;
	// Special baskets ::
	bool     m_isAStack;
	bool     m_stackTakeOnSameSide;
	int      m_stackAfterDrag;
	bool     m_isAClipboard;
	int      m_clipboardMaxItems;
	int      m_clipboardWhich;
	bool     m_mirrorOnlyNewFiles;
	// Add item policy ::
	bool     m_insertAtCursorPos;
	bool    *m_ViewFileContent;
	// On click actions ::
	int      m_contentOnClickAction;
	int      m_fileOnClickAction;

	bool     m_isLoaded;
	int      m_countSelecteds;
	bool     m_isDuringDrag;

	QFrame    *m_frameInsertTo;
	QFrame    *m_framesIT[4];
	Item      *m_lastInsertedItem;
	Item      *m_focusedItem;
	Item      *m_startOfShiftSelectionItem;
	bool       m_areSelectedItemsChecked;
	QWidget   *m_emptyHelp;
	QCheckBox *m_dontShowEmptyHelp;
  public:
	QString  m_mirrorOnlyOnceFileName;

	void showFrameInsertTo();
	void resetInsertTo();
	bool isDuringDrag()         { return m_isDuringDrag;           }
	bool isDuringEdit()         { return m_editor != 0L;           }

	enum ViewFileContent { FileText, FileHTML, FileImage, FileSound, TOTAL };
	bool viewFileContent(ViewFileContent of);
	void setViewFileContent(ViewFileContent of, bool val);
  private:
	KDirWatch      *m_watcher;
	ItemEditorBase *m_editor;
	QKeyEvent      *m_stackedKeyEvent;

  public: // For ItemFactory to save and restore it for multiple URLs drop // FIXME: In a later version, well manage this
	Item *m_insertAtItem;
	bool  m_insertAfter;  // true to insert after m_insertAtItem, false to insert before m_insertAtItem
	bool  m_clickedToExitEdit; // Afterward, I see it could be stored in the item. That's nothing: we win some bytes :-)
	bool  successfulyLoaded() { return m_isLoaded; }
  protected:
	bool  m_insertAtEnd;

	/** Doubly-linked list : */
  private:
	Item *m_firstItem;      // To manage the list
	Item *m_lastItem;       // ...
	Item *m_firstShownItem; // For optimisations (do not browse all the list when only few items are shown)
	Item *m_lastShownItem;  // ... but also to well place insert line
	int   m_count;          // To manage the list too
	int   m_countShown;
  public:
	inline Item* firstItem()             const { return m_firstItem;      }
	inline Item* lastItem()              const { return m_lastItem;       }
	inline Item* firstShownItem()        const { return m_firstShownItem; }
	inline Item* lastShownItem()         const { return m_lastShownItem;  }

	inline bool  isEmpty()               const { return m_count == 0;     }
	inline int   count()                 const { return m_count;          }
	inline int   countShown()            const { return m_countShown;     }

  protected:
	void plugItem(Item *item, Item *atItem, bool after); // The ONLY two methods that are allowed to insert/remove items
	void unplugItem(Item *item);                         //  in the list (because it can cause a lot of bugs)
	inline void  setFirstItem(Item *item)      { m_firstItem      = item; }
	inline void  setLastItem(Item *item)       { m_lastItem       = item; }
	inline void  setFirstShownItem(Item *item) { m_firstShownItem = item; }
	inline void  setLastShownItem(Item *item)  { m_lastShownItem  = item; }

/*
	// How to browse the doubly-linked list :

	// Browse ALL items :
	for (Item *it = firstItem(); it != 0L; it = it->next())
		it->doInterestingThing();

	// Browse all SHOWN items :
	for (Item *it = firstShownItem(); it != 0L; it = it->next())
		if (it->isShown()) {
			it->doInterestingThing();
			if (it == lastShownItem())
				break;
		}
*/
};

#endif // BASKET_H
