#!/bin/sh
# -*- mode: sh; coding: utf-8 -*-
# Test cases to ensure the conflicts recording code works
# Copyright (C) 2004 Canonical Ltd
#           Author: James Blackwell <jblack@gnuarch.org>
#
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.

set -e

arg0="$0"
srcdir=`dirname "$arg0"`

. ${srcdir}/test-framework


# Get ready for the tests
setup_with_somewhat_interesting_explicit_archives

# And here we go
test_class "conflicts"



begin_test "Is baz sane enough to test for conflicts?"
  tla branch jane@example.com--2003/hello-world--mainline--1.0 \
             jane@example.com--2003/hello-world--devo--1.0 \
     || test_fail "Unexpected error with baz branch"

  # First, create a change in main
  tla get jane@example.com--2003/hello-world--mainline--1.0 foo  \
    || test_fail "Unexpected error with baz get"
  cd foo
  echo "/* This comment written by jblack */" > hello-world.c
  echo "/* This comment written by jblack */" > goodbye-world.c
  tla commit -s "I don't feel like coding today" \
    || test_fail "Unexpected Error with baz commit #1"

  cd ..

  # Now, create an incompatable change in devo
  tla get jane@example.com--2003/hello-world--devo--1.0 newfoo \
    || test_fail "Unexpected error with baz get"
  cd newfoo
  echo "/* This comment ALSO written by jblack */" > hello-world.c
  echo "/* This comment ALSO written by jblack */" > goodbye-world.c
  tla commit -s "I don't feel like coding here either" \
    || test_fail "Unexpected Error with baz commit #2"

  # And now we're ready to get our butts kicked with the actual tests

end_test



begin_test "Does baz detect conflicts on merge?"

  tla get jane@example.com--2003/hello-world--devo--1.0  foo
  cd foo

  tla merge jane@example.com--2003/hello-world--mainline--1.0--patch-1 \
    || test $? -gt 0 || test_fail "Expected conflict didn't arrive #1"

end_test 


begin_test "Does baz care?"

  tla get jane@example.com--2003/hello-world--devo--1.0  foo
  cd foo

  tla merge jane@example.com--2003/hello-world--mainline--1.0--patch-1 \
    || test $? -gt 0 || test_fail "Expected conflict didn't arrive #2"

  test -f {arch}/+rejects-list \
    || test_fail "Baz failed to create a conflict list"

end_test 


begin_test "Did baz figure out what conflicted?"

  tla get jane@example.com--2003/hello-world--devo--1.0  foo
  cd foo

  tla merge jane@example.com--2003/hello-world--mainline--1.0--patch-1 \
    || test $? -gt 0 || test_fail "Expected conflict didn't arrive #3"

  file_matches '^\.\/hello-world\.c' {arch}/+rejects-list \
    || test_fail "Baz forgot to record a conflicted file"

  file_matches '^\.\/goodbye-world\.c' {arch}/+rejects-list \
    || test_fail "Baz forgot to record a conflicted file"

end_test 

begin_test "Jane tells baz that she fixes first one, then both of the conflicts"

  tla get jane@example.com--2003/hello-world--devo--1.0  foo 
  cd foo

  tla merge jane@example.com--2003/hello-world--mainline--1.0--patch-1 \
    || test $? -gt 0 || test_fail "Expected conflict didn't arrive #3"

  tla resolved hello-world.c

  test -f {arch}/+rejects-list \
    || test_fail "Baz lied about resolved"

  tla resolved goodbye-world.c 1>&2

  test \! -f {arch}/+rejects-list \
    || test_fail "Baz failed to resolve all of the files"

end_test

begin_test "What if Jane gives a file thats not conflicted?"
  tla get jane@example.com--2003/hello-world--devo--1.0  foo
  cd foo

  tla merge jane@example.com--2003/hello-world--mainline--1.0--patch-1 \
    || test $? -gt 0 || test_fail "Expected conflict didn't arrive #3"

  test -f {arch}/+rejects-list \
    || test_fail "A conflicted tree is lying by saying it isn't"

  file_matches '^\.\/hello-world\.c' {arch}/+rejects-list \
    || test_fail "Baz forgot to record hello world as conflicted"
  file_matches '^\.\/goodbye-world\.c' {arch}/+rejects-list \
    || test_fail "Baz forgot to record goodbye world as conflicted"


  tla resolved not-hello-world.c && test_fail "no error with invalid filename not-hello-world.c"

  cd hierarchy

  echo " ... check wrong dir and file"
  tla resolved ../not-hello-world.c && test_fail "no error with invalid filename ../not-hello-world.c"

  file_matches '^\.\/hello-world\.c' ../{arch}/+rejects-list \
    || test_fail "hello world is not conflicted"
  file_matches '^\.\/goodbye-world\.c' ../{arch}/+rejects-list \
    || test_fail "goodbye world is not conflicted"

  tla resolved hello-world.c && test_fail "no error with invalid filename hello-world.c (from subdir)"
  
  file_matches '^\.\/hello-world\.c' ../{arch}/+rejects-list \
    || test_fail "hello world is not conflicted"
  file_matches '^\.\/goodbye-world\.c' ../{arch}/+rejects-list \
    || test_fail "goodbye world is not conflicted"

  verbose "right file name, right path" 
  tla resolved ../goodbye-world.c
  file_matches '^\.\/hello-world\.c' ../{arch}/+rejects-list \
    || test_fail "hello world is not conflicted"
  file_matches '^\.\/goodbye-world\.c' ../{arch}/+rejects-list \
    && test_fail "goodbye world is still conflicted"
  tla resolved ../hello-world.c 2>&1 \
    | grep -q "No conflicts remain" \
        || test_fail "Baz resolved lies about conflicts!"
  test -e ../{arch}/+rejects-list \
    && test_fail "reject list not cleared"

end_test

clean_workdir
