/***************************************************************************
                         -- menu.cc --
                         -------------------
    begin                : 2001-09-01
    copyright            : (C) 2001 - 2002
    authors              : Jason Kasper (vanRijn)
                         : Ben Jansens (xOr)
    URL                  : http://bbconf.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
 // E_O_H_VR
#include "qmessagebox.h"
#include "vr_filepreview.h"
#include "menu.h"
#include "iostream"

/* 
 *  Constructs a MenuEditor which is a child of 'parent', with the 
 *  name 'name' and widget flags set to 'f' 
 */
MenuEditor::MenuEditor( QWidget* parent,  const char* name, WFlags fl )
    : MenuEditorBase( parent, name, fl )
{
	menuList->setSorting(-1);

	menuList->setColumnWidthMode(0,QListView::Maximum);
	menuList->setColumnWidthMode(1,QListView::Manual);
	menuList->setColumnWidthMode(2,QListView::Manual);
	menuList->setColumnWidthMode(3,QListView::Manual);

	// invisible second and third columns, thank you very much
	menuList->setColumnWidth(1,0);
	menuList->setColumnWidth(2,0);
	menuList->setColumnWidth(3,0);

	setAcceptDrops( TRUE );
	menuList->setAcceptDrops( TRUE );
	menuList->viewport()->setAcceptDrops( TRUE );

	reloading = false;
}

/*  
 *  Destroys the object and frees any allocated resources
 */
MenuEditor::~MenuEditor()
{
    // no need to delete child widgets, Qt does it all for us
}

/* 
 * protected slot
 */
void MenuEditor::modified()
{
	// here's the deal. This will get fired whenever the text in our
	// boxes change or the dropdown changes.  The only problem is that
	// these can and do change whenever we're 1) reloading or 2) the
	// user has changed the selection in the listview.  For these two
	// scenarios, we need to not fire changed()
	if ( reloading || listview_changed )
		return;

	changed();
}

void MenuEditor::loadMenu(TreeNode<menuitem> *menu) {

	reloading = true;
	menuList->clear();
	loadListview(menu);
	reloading = false;

	// now set focus on the first item
	QListViewItem *first = menuList->firstChild();
	if (first) {
		menuList->setCurrentItem( first );
		menuList->setSelected( first, TRUE );
	}

}
void MenuEditor::loadListview(TreeNode<menuitem> *menu, bool submenu) {

	if(!menu->next())
		return;
	while(menu->next()) {
		menu = menu->next();
		menuitem *item = menu->object();
		QString type;
		type.setNum( item->type() );
		QString label = item->description();
		QString data = item->data();

		QListViewItem *newi = 0;
		QListViewItem *current = menuList->currentItem();

		// if we have a begin tag, set the title text
		if ( item->type() == menuitem::begin ) {
			titleText->setText(label);
		// else if we need to end a menu/submenu, set focus on our parent
		} else if ( item->type() == menuitem::end ) {
			if (current) {
				QListViewItem *parent = current->parent();
				if (parent) {
					menuList->setCurrentItem( parent );
					menuList->setSelected( parent, TRUE );
				}
			}
		} else {
			if (!current) {
				newi = new QListViewItem( menuList );
			} else {
				QListViewItem *parent = current->parent();
			
				if ( submenu ) {
					newi = new QListViewItem( current );
					submenu=false;
				} else if ( parent ) {
					newi = new QListViewItem( parent, current );
				} else {
					newi = new QListViewItem( menuList, current );
				}
			}
			
			newi->setText( 0, label );
			newi->setText( 1, type );
			newi->setText( 2, data );
			menuList->setCurrentItem( newi );
			menuList->setSelected( newi, TRUE );

			if(item->type() == menuitem::submenu) {
				loadListview(menu->child(), true);
			}
	
		}
	}

}


void MenuEditor::dataTextChanged(const QString& s)
{
	QListViewItem *i = menuList->currentItem();
	if (!i)
		return;

	i->setText(2,s);
}

void MenuEditor::descriptionTextChanged(const QString& s)
{
	QListViewItem *i = menuList->currentItem();
	if (!i)
		return;

	i->setText(0,s);
}

void MenuEditor::itemDeleteClicked()
{
	QListViewItem *i = menuList->currentItem();
	if ( !i )
		return;
	
	QListViewItem *f = i->itemAbove();

	delete i;
	if ( f ) {
		menuList->setCurrentItem( f );
		menuList->setSelected( f, TRUE );
	}

}

void MenuEditor::itemDownClicked()
{

	QListViewItem *i = menuList->currentItem();
	if ( !i )
		return;

	QListViewItem *below = i->itemBelow();
	if ( !below )
		return;

	// if the item below us has children and is expanded, move
	// ourselves to the top of that sub-menu
	if (below->isOpen()) {
		QListViewItem *child = below->firstChild();
		i->moveItem(child);
		child->moveItem(i);
	// else if we're in the same level, move down 1
	} else if (below->parent() == i->parent()) {
		i->moveItem(below);
	// else, we need to move down below our parent
	} else {
		i->moveItem(i->parent() );
	}

	menuList->setCurrentItem( i );
	menuList->ensureItemVisible( i );
	menuList->setSelected( i, TRUE );
	return;
}

void MenuEditor::itemNewClicked()
{

	QListViewItem *item = 0;

	QListViewItem *current = menuList->currentItem();
	if (!current) {
		item = new QListViewItem( menuList );
	} else {
		QListViewItem *parent = current->parent();
	
		if ( !parent ) {
			item = new QListViewItem( menuList, current );
		} else {
			item = new QListViewItem( parent, current );
		}
	}
	
	item->setText( 0, "New Item - Edit Me!" );
	QString execItem; execItem.setNum(menuitem::exec);
	item->setText( 1, execItem);

	menuList->setCurrentItem( item );
	menuList->ensureItemVisible( item );
	menuList->setSelected( item, TRUE );
}

void MenuEditor::itemNewSubClicked()
{

	QListViewItem *parent = menuList->currentItem();
	QListViewItem *item = 0;
	if ( parent ) {
		bool ok;
		int t = parent->text(1).toInt(&ok);
		if (t != menuitem::submenu ) {
			QMessageBox mb( "Warning! Incompatible menu type!",
				tr("You are changing a '%1'\n"
				"menu item to a 'Submenu' item\n\n"
				"Do you want to continue with this and\n"
				"change this menu item to a Submenu?").arg(typeDropdown->currentText()),
				QMessageBox::Warning,
				QMessageBox::Yes ,
				QMessageBox::No | QMessageBox::Default,
				0);

			if ( mb.exec() == QMessageBox::No)
				return;
		}

		// change current parent into a submenu type
		QString num;
		num.setNum(menuitem::submenu);

		parent->setText( 1, num );
		item = new QListViewItem( parent );
		parent->setOpen( TRUE );

	} else {
		item = new QListViewItem( menuList );
	}

	item->setText( 0, "New Subitem - Edit Me!" );
	QString execItem; execItem.setNum(menuitem::exec);
	item->setText( 1, execItem);

	menuList->setCurrentItem( item );
	menuList->setSelected( item, TRUE );
	menuList->ensureItemVisible( item );

}

void MenuEditor::itemUpClicked()
{
	QListViewItem *i = menuList->currentItem();
	if ( !i )
		return;

	QListViewItem *parent = i->parent();

	QListViewItem *above = i->itemAbove();
	if ( !above )
		return;

	QListViewItem *above2 = above->itemAbove();

	// if the item above me is my parent, then I need to move myself
	// above him
	if ( above == parent ) {
		if ( above2 ) 
			i->moveItem(above2);
	} else if (above->parent() == i->parent() ){
		// if we're on the same level, move him below me
		above->moveItem(i);
	} else {
		// else, move myself below him
		i->moveItem(above);
	}

	menuList->setCurrentItem( i );
	menuList->ensureItemVisible( i );
	menuList->setSelected( i, TRUE );

}

void MenuEditor::dataBrowseClicked()
{
	int iType = typeDropdown->currentItem();
	QString file;
	MyPreview* p = new MyPreview;     
	QFileDialog* fd = new QFileDialog( QString::null, "All (*)", this, "w", true );
	fd->setMode( QFileDialog::ExistingFile );
	// fd->setDir( styledir );
	fd->setCaption( "Select File");
	fd->setShowHiddenFiles(true);
	
	// disable preview pane if we're getting a file to execute or
	// finding a directory
	if ( iType != menuitem::exec && iType != menuitem::stylesdir ) {
		fd->setContentsPreviewEnabled( TRUE );
		fd->setContentsPreview( p, p );
		fd->setPreviewMode( QFileDialog::Contents );
	}

	if ( fd->exec() == QDialog::Accepted ) {
        	file = fd->selectedFile();
		dataText->setText(file);
	}

}

void MenuEditor::descriptionBrowseClicked()
{
	int iType = typeDropdown->currentItem();
	QString file;
	MyPreview* p = new MyPreview;     
	QFileDialog* fd = new QFileDialog( QString::null, "All (*)", this, "w", true );
	if (iType == menuitem::stylesdir) {
		fd->setMode( QFileDialog::DirectoryOnly );
		fd->setCaption( "Select Directory");
	} else {
		fd->setMode( QFileDialog::ExistingFile );
		fd->setCaption( "Select File");
	}
	// fd->setDir( styledir );

	// disable preview pane if we're getting a file to execute or
	// finding a directory
	if ( iType != menuitem::exec && iType != menuitem::stylesdir ) {
		fd->setContentsPreviewEnabled( TRUE );
		fd->setContentsPreview( p, p );
		fd->setPreviewMode( QFileDialog::Contents );
	}

	if ( fd->exec() == QDialog::Accepted ) {
        	file = fd->selectedFile();
		descriptionText->setText(file);
	}

}

void MenuEditor::titleTextChanged(const QString& s)
{
	menuList->setColumnText(0,s);

	if (menuList->columnWidth(0) < ( menuList->width() -10) )
		menuList->setColumnWidth(0, menuList->width() -10);
}

void MenuEditor::customizeWidgets(int iType)
{
	
	// user-friendly-ize our label
	if (iType == menuitem::submenu )  {
		descriptionLabel->setText("Description:");
		dataLabel->setText("Submenu title:");
	} else if (iType == menuitem::exec ){
		descriptionLabel->setText("Description:");
		dataLabel->setText("Command to run:");
	} else if (iType == menuitem::style ){
		descriptionLabel->setText("Description:");
		dataLabel->setText("Style to change to:");
	} else if (iType == menuitem::include ){
		descriptionLabel->setText("File to include:");
		dataLabel->setText("Data:");
	} else if (iType == menuitem::restart ) {
		descriptionLabel->setText("Description:");
		dataLabel->setText("Window manager:");
	} else if (iType == menuitem::stylesdir ) {
		descriptionLabel->setText("Directory of styles:");
		dataLabel->setText("Data:");
	} else {
		descriptionLabel->setText("Description:");
		dataLabel->setText("Data:");
	}
		
	// disable/enable our data line
	if (iType == menuitem::nop || 
	    iType == menuitem::config || 
	    iType == menuitem::reconfig || 
	    iType == menuitem::workspaces || 
	    iType == menuitem::exit )  {
		descriptionText->setEnabled(true);
		descriptionBrowse->setEnabled(false);
		dataText->setEnabled(false);
		dataBrowse->setEnabled(false);
	} else if ( iType == menuitem::submenu ) {
		descriptionText->setEnabled(true);
		descriptionBrowse->setEnabled(false);
		dataText->setEnabled(true);
		dataBrowse->setEnabled(false);
	} else if ( iType == menuitem::include || 
	    iType == menuitem::stylesdir ) {
		descriptionText->setEnabled(true);
		descriptionBrowse->setEnabled(true);
		dataText->setEnabled(false);
		dataBrowse->setEnabled(false);
	} else  {
		descriptionText->setEnabled(true);
		descriptionBrowse->setEnabled(false);
		dataText->setEnabled(true);
		dataBrowse->setEnabled(true);
	}

	/*
	switch ( iType ) {
		case menuitem::submenu:
		case menuitem::nop:
		case menuitem::exec:
		case menuitem::style:
		case menuitem::config:
		case menuitem::include:
		case menuitem::stylesdir:
		case menuitem::stylesmenu:
		case menuitem::restart:
		case menuitem::reconfig:
		case menuitem::workspaces:
		case menuitem::exit:
	}
	*/

}

void MenuEditor::menuSelectionChanged(QListViewItem* item)
{
	if (reloading)
		return;

	listview_changed = true;

	QString desc = item->text(0);
	QString type = item->text(1);
	QString data = item->text(2);

	bool ok;
	int iType = type.toInt(&ok);

	descriptionText->setText(desc);
	dataText->setText(data);
	typeDropdown->setCurrentItem(iType);

	// enable/disable dropdown
	if (iType == menuitem::submenu ) 
		typeDropdown->setEnabled(false);
	else
		typeDropdown->setEnabled(true);

	customizeWidgets(iType);

	listview_changed = false;

}

void MenuEditor::typeDropdownChanged(int i)
{
	QListViewItem *l = menuList->currentItem();
	if (!l)
		return;

	// if our user is changing this item from submenu to something else
	// and this item has children, we tell them nyet
	bool ok;
	int t = l->text(1).toInt(&ok);
	if (t == menuitem::submenu && l->childCount() ) {
		QString str=tr("Sorry, you're trying to change a SubMenu\n"
		"item that has children to '%1'.\n\n"
		"Doing so would destroy the integrity of the menu,\n"
		"so I'm not going to let you do that.").
		arg(typeDropdown->currentText());

		QMessageBox::warning( this, tr("Can't change submenus"), str);
		typeDropdown->setCurrentItem(t);
		return;
	}

	QString num;
	num.setNum(i);
	l->setText(1,num);
	l->setText(3,typeDropdown->currentText() );

	customizeWidgets(i);

}

