// LoggerManager.cpp: implementation of the CLoggerManager class. 
// 
////////////////////////////////////////////////////////////////////// 
 
#include "../../Include/Base/LoggerManager.h" 
#include "../../Include/Base/LoggerManagerModule.h" 
 
#include "../../Include/Base/StringToObjectCollection.h" 
 
#include <time.h> 
#include <sys/types.h> 
#include <sys/timeb.h> 
#include <string.h> 
#include <stdio.h> 
 
////////////////////////////////////////////////////////////////////// 
// Construction/Destruction 
////////////////////////////////////////////////////////////////////// 
 
CLoggerManager::CLoggerManager() : CManager ("Logger") 
{ 
	m_ulLogLevelMask = LOGMASK_FULL; 
	m_ModuleList = new CStringToObjectCollection; 
	m_EnabledModuleList = new CStringToObjectCollection; 

	PASS_INITLOCK (&lock);
} 
 
CLoggerManager::~CLoggerManager() 
{ 
	if (m_ModuleList != NULL) delete m_ModuleList; 
	if (m_EnabledModuleList != NULL) delete m_EnabledModuleList; 

	PASS_DESTROYLOCK (&lock);
} 
 
void CLoggerManager::SetLogLevel (unsigned long mask) 
{ 
	m_ulLogLevelMask = mask; 
} 
 
unsigned long CLoggerManager::GetLogLevel (void) 
{ 
	return m_ulLogLevelMask; 
} 
 
void CLoggerManager::AddModule (char *moduleName, CLoggerManagerModule *module) 
{ 
	m_ModuleList->Add (moduleName,module); 
} 
 
void CLoggerManager::AddModule (CLoggerManagerModule *module) 
{ 
	m_ModuleList->Add (module->GetName(),module); 
} 
 
BOOL CLoggerManager::EnableModule (char *moduleName) 
{ 
	if (m_ModuleList->Lookup (moduleName) == NULL) 
	{ 
		// Can't get module 
		return FALSE; 
	} 
 
	m_EnabledModuleList->Add (moduleName,m_ModuleList->Lookup (moduleName)); 
 
	return TRUE; 
} 
 
BOOL CLoggerManager::DisableModule (char *moduleName) 
{ 
	if (m_EnabledModuleList->Lookup (moduleName) == NULL) 
	{ 
		// Can't get module 
		return FALSE; 
	} 
 
	m_EnabledModuleList->Remove (moduleName); 
 
	return TRUE; 
} 
 
void CLoggerManager::Log (char *file, int line, unsigned long mask, char *text) 
{ 
	PASS_LOCK (&lock);

	// Check mask 
	if (m_ulLogLevelMask & mask) 
	{ 
		// Add to text, the time stamp 
		char* buffer = new char [2048];
		memset (buffer,0,sizeof(buffer)); 
 
#ifdef _WIN32 
		struct _timeb timebuffer; 
#endif // _WIN32 
#ifdef linux 
		struct timeb timebuffer; 
#endif // linux 
#ifdef __SCO__
		struct timeb timebuffer; 
#endif // __SCO__ 
#ifdef __BEOS__
		struct timeb timebuffer; 
#endif // __BEOS__ 
#ifdef __HPUX__ 
		struct timeb timebuffer; 
#endif // __HPUX__ 
		char *timeline; 
 
#ifdef _WIN32 
		_ftime( &timebuffer ); 
#endif // _WIN32 
#ifdef linux 
		ftime( &timebuffer ); 
#endif // linux 
#ifdef __SCO__
		ftime( &timebuffer ); 
#endif // __SCO__ 
#ifdef __BEOS__
		ftime( &timebuffer ); 
#endif // __BEOS__ 
#ifdef __HPUX__ 
		ftime( &timebuffer ); 
#endif // __HPUX__ 

		timeline = ctime( & ( timebuffer.time ) ); 
 
		sprintf (buffer,"%.19s.%hu GMT - File : %s Line : %d - %s",timeline, timebuffer.millitm,file,line,text);	 
 
		// Send text to all modules 
		char key[256]; 
		ULONG32 position; 
 
		CLoggerManagerModule *module = (CLoggerManagerModule *)m_EnabledModuleList->GetFirst(&position,key,256); 
 
		while (module != NULL) 
		{ 
			module->Log(buffer); 
 
			module = (CLoggerManagerModule *)m_EnabledModuleList->GetNext(&position,key,256); 
		} 
		delete [] buffer;
	} 

	PASS_UNLOCK (&lock);
} 
