// MessageDispatcher.cpp: implementation of the CMessageDispatcher class. 
// 
////////////////////////////////////////////////////////////////////// 
 
#include "../../Include/Comm/MessageDispatcher.h" 
 
#include "../../Include/Base/LongToObjectCollection.h" 
#include "../../Include/Base/MasterManager.h" 
#include "../../Include/Comm/ConnectionManager.h" 
#include "../../Include/Comm/Connection.h" 
#include "../../Include/Comm/DispatchMessageHandler.h" 
#include "../../Include/Comm/MessageDispatcher.h" 
#include "../../Include/Comm/DispatchMessage.h" 
 
#include <memory.h> 
 
 
////////////////////////////////////////////////////////////////////// 
// Construction/Destruction 
////////////////////////////////////////////////////////////////////// 
 
CMessageDispatcher::CMessageDispatcher() : CThread ("MessageDispatcher","MessageDispatcher") 
{ 
	memset (m_szDispatcherName,0,sizeof(m_szDispatcherName)); 
	m_HandlerList = new CLongToObjectCollection; 
	m_RunningHandler = new CLongToObjectCollection; 
	m_NextDispatcher = NULL; 
	m_PreviousDispatcher = NULL; 
	m_WaitingMessage = new CLongToObjectCollection;
	m_bHandlerPool = FALSE;
	PASS_INITLOCK (&lock); 
} 
 
CMessageDispatcher::~CMessageDispatcher() 
{ 
	if (m_HandlerList != NULL) delete m_HandlerList; 
	if (m_RunningHandler != NULL) delete m_RunningHandler; 
	if (m_WaitingMessage != NULL) delete m_WaitingMessage; 
	PASS_DESTROYLOCK (&lock); 
} 
 
void CMessageDispatcher::SetDispatcherName (char *name) 
{ 
	if (name == NULL) 
	{ 
		return; 
	} 
 
	strncpy (m_szDispatcherName,name,sizeof(m_szDispatcherName)); 
} 
 
char *CMessageDispatcher::GetDispatcherName (void) 
{ 
	return m_szDispatcherName; 
} 
 
void CMessageDispatcher::AddMessageHandler (CDispatchMessageHandler *handler) 
{ 
	if (handler == NULL) 
	{ 
		return; 
	} 
 
	AddMessageHandler (handler,handler->GetMessageNameHandled(),handler->GetMessageTypeHandled()); 
} 
 
void CMessageDispatcher::AddMessageHandler (CDispatchMessageHandler *handler,char *name, char *type) 
{ 
	PASS_LOCK (&lock); 
 
	if (handler == NULL || name == NULL || type == NULL) 
	{ 
		PASS_UNLOCK (&lock); 
		return; 
	} 
 
	m_HandlerList->Add (m_HandlerList->GetCount(),handler); 
 
	PASS_UNLOCK (&lock); 
} 
 
void CMessageDispatcher::SetNextDispatcher (CMessageDispatcher *next) 
{ 
	m_NextDispatcher = next; 
} 
 
void CMessageDispatcher::SetPreviousDispatcher (CMessageDispatcher *previous) 
{ 
	m_PreviousDispatcher = previous; 
} 
 
void CMessageDispatcher::AddMessage (CDispatchMessage *message) 
{ 
	PASS_LOCK (&lock); 
 
	if (message == NULL) 
	{ 
		PASS_UNLOCK (&lock); 
		return; 
	} 
 
	m_WaitingMessage->Add (m_WaitingMessage->GetCount(),message); 
 
	PASS_UNLOCK (&lock); 
} 
 
BOOL CMessageDispatcher::IsMessageHandled (CDispatchMessage *message) 
{ 
	PASS_LOCK (&lock); 
 
	if (message == NULL) 
	{ 
		PASS_UNLOCK (&lock); 
		return FALSE; 
	} 
 
	ULONG32 position; 
	CDispatchMessageHandler *handler = NULL; 
 
	char *messageName = message->GetMessageName(); 
	char *messageType = message->GetMessageType(); 
 
	if (messageName == NULL || messageType == NULL) 
	{ 
		PASS_UNLOCK (&lock); 
		return FALSE; 
	} 
 
	handler = (CDispatchMessageHandler *)m_HandlerList->GetFirst (&position); 
 
	while (handler != NULL) 
	{ 
		// Get current message name and type 
		char *handledName = handler->GetMessageNameHandled(); 
		char *handledType = handler->GetMessageTypeHandled(); 
 
		if (handledName != NULL && handledType != NULL) 
		{ 
			if (strcmp (handledName,messageName) == 0 && strcmp (handledType,messageType) == 0) 
			{ 
				// Message handled 
				PASS_UNLOCK (&lock); 
				return TRUE; 
			} 
		} 
 
		handler = (CDispatchMessageHandler *)m_HandlerList->GetNext (&position); 
	} 
 
	PASS_UNLOCK (&lock); 
	return FALSE; 
} 
 
CDispatchMessageHandler *CMessageDispatcher::GetHandlerForMessage (CDispatchMessage *message) 
{ 
	if (message == NULL) 
	{ 
		return NULL; 
	} 
 
	ULONG32 position; 
	CDispatchMessageHandler *handler = NULL; 
 
	char *messageName = message->GetMessageName(); 
	char *messageType = message->GetMessageType(); 
 
	if (messageName == NULL || messageType == NULL) 
	{ 
		return NULL; 
	} 
 
	handler = (CDispatchMessageHandler *)m_HandlerList->GetFirst (&position); 
 
	while (handler != NULL) 
	{ 
		// Get current message name and type 
		char *handledName = handler->GetMessageNameHandled(); 
		char *handledType = handler->GetMessageTypeHandled(); 
 
		if (handledName != NULL && handledType != NULL) 
		{ 
			if (strcmp (handledName,messageName) == 0 && strcmp (handledType,messageType) == 0) 
			{ 
				// Message handled, return handler 
				return handler; 
			} 
		} 
 
		handler = (CDispatchMessageHandler *)m_HandlerList->GetNext (&position); 
	} 
 
	return NULL; 
} 
 
void CMessageDispatcher::Tick (void) 
{ 
	PASS_LOCK (&lock); 
 
	// First get the waiting message 
	if (m_WaitingMessage->GetCount() > 0) 
	{ 
		ULONG32 position; 
		 
		CDispatchMessage *message = NULL; 
 
		// Get all the waiting messages 

		message = (CDispatchMessage *)m_WaitingMessage->GetFirst (&position); 
 
		while (message != NULL) 
		{ 
			// Is there a handler for this message ? 
			CDispatchMessageHandler *handler = GetHandlerForMessage (message); 
 
			if (handler != NULL) 
			{ 
					// No handler currently waiting, create the first one 
					CDispatchMessageHandler *newHandler = (CDispatchMessageHandler *)handler->Clone(); 
 
					// Set the message 
					newHandler->SetMessage (message); 
 
					// Start the handler, and put it in the list 
					newHandler->Start(); 
 
					m_RunningHandler->Add (m_RunningHandler->GetCount(),newHandler); 
			} 
 
			// Remove the first message 
			m_WaitingMessage->Remove (message); 
 
			message = (CDispatchMessage *)m_WaitingMessage->GetFirst (&position); 
		} 
 
	} 
 
	// Then check all the running message handler 
	ULONG32 position; 
 
	CDispatchMessageHandler *runningHandler = (CDispatchMessageHandler *)m_RunningHandler->GetFirst (&position); 
 
	while (runningHandler != NULL) 
	{ 
		if (runningHandler->CanBeDeleted() == TRUE) 
		{ 
			// this handler has terminate it's task 
			if (m_NextDispatcher == NULL) 
			{ 
				// No more dispatcher 
 
				// Is this message need to return back to it's sender ? 
				CDispatchMessage *nextMessage = runningHandler->GetMessage(); 
				 
				// Get the '#DispatchKey#Routing#***ReturnBack***' key 
				// Get the '#DispatchKey#Routing#***Sender***' key 
				char returnBack[256]; 
				memset (returnBack,0,sizeof(returnBack)); 
				char senderBack[256]; 
				memset (senderBack,0,sizeof(senderBack)); 
				unsigned long returnConnectionID = 0; 
 
				if (nextMessage->GetValueForKey ("#DispatchKey#Routing#***ReturnBack***",returnBack,sizeof(returnBack)) == TRUE) 
				{ 
					if (strcmp (returnBack,"Yes") == 0) 
					{ 
						if (nextMessage->GetValueForKey ("#DispatchKey#Routing#***Sender***",senderBack,sizeof(senderBack)) == TRUE) 
						{ 
							if (nextMessage->GetValueForKey ("#DispatchKey#Routing#***CurrentConnectionID***",&returnConnectionID) == TRUE) 
							{ 
								// Found return back, the value should contain the connection name 
								// to witch the message has to return back 
 
								// Found this connection 
								CMasterManager *master = CMasterManager::Instance(); 
								CConnectionManager *manager = (CConnectionManager *)master->GetManager("Connection"); 
 
								if (manager == NULL) 
								{ 
									// no connection manager found, delete message 
									// And Log error in logger 
									delete nextMessage; 
								} 
								else 
								{ 
									CConnection *connection = manager->GetConnectionForID (returnConnectionID); 
 
									if (connection == NULL) 
									{ 
										// No connection found, delete message 
										// And log error in logger 
										delete nextMessage; 
									} 
									else 
									{ 
										connection->SendMessage (nextMessage); 
									} 
								} 
							} 
						} 
					} 
				} 
			} 
			else 
			{ 
				// Send the message to the next dispatcher 
				CDispatchMessage *nextMessage = runningHandler->GetMessage(); 
				m_NextDispatcher->AddMessage (nextMessage); 
			} 
 
			// Delete handler and remove it from the list 
			delete runningHandler; 
 
			m_RunningHandler->Remove (runningHandler); 
		} 
 
		runningHandler = (CDispatchMessageHandler *)m_RunningHandler->GetFirst (&position); 
	} 
 
	PASS_UNLOCK (&lock); 
} 
