// SCardHandler.cpp: implementation of the CSCardHandler class.
//
//////////////////////////////////////////////////////////////////////

#include "StdAfx.h"
#include "SCardHandler.h"
#include "PCSCManager.h"
#include "defines.h"
#include "Config.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

typedef  long (CPCSCManager::* S_CardFunction)(CGenericMessage *);

typedef struct
{
    char szFuncName[64];
    S_CardFunction funcPtr;
} SCardFunctionInfo;

static SCardFunctionInfo tFuncMap[] = 
{
    { "SCardEstablishContext", &CPCSCManager::S_CardEstablishContext },
	{ "SCardReleaseContext", &CPCSCManager::S_CardReleaseContext },
	{ "SCardConnect", &CPCSCManager::S_CardConnect },
	{ "SCardDisconnect", &CPCSCManager::S_CardDisconnect },
	{ "SCardStatus", &CPCSCManager::S_CardStatus },
	{ "SCardListReaders", &CPCSCManager::S_CardListReaders },
	{ "SCardGetStatusChange", &CPCSCManager::S_CardGetStatusChange },
	{ "SCardBeginTransaction", &CPCSCManager::S_CardBeginTransaction },
	{ "SCardEndTransaction", &CPCSCManager::S_CardEndTransaction },
	{ "SCardTransmit", &CPCSCManager::S_CardTransmit },
	{ "SCardControl", &CPCSCManager::S_CardControl },
	{ "SCardPinPadControl", &CPCSCManager::S_CardControlPinPad },
	{ 0, 0 }
};
    
CSCardHandler::CSCardHandler() : CMessageHandler ("CSCardHandler")
{
	SetMessageNameHandled ("SCardHandler");
}

CSCardHandler::~CSCardHandler()
{

}

void CSCardHandler::Execute(void)
{
	long lReturn = SCARD_F_UNKNOWN_ERROR;

    CConnection_Mono *pConnection = (CConnection_Mono *)m_Connection;
	if (pConnection == NULL)
	{
		// Error
		Trace (__FILE__, __LINE__, LOGMASK_BELPIC_ERROR, "Can't retrieve Connection");
	    ((CGenericMessage*)m_Message)->AddValueForKey("Return", lReturn);
		return;
	}

    long lConnID = pConnection->GetConnectionID();
   ((CGenericMessage*)m_Message)->AddValueForKey("ConnID", lConnID);

    // Check IP Allowed

   if(SCARD_S_SUCCESS == (lReturn = CheckIPAllowed(pConnection)))
   {
        lReturn = SCardMapFunction((CGenericMessage*)m_Message);
   }

    ((CGenericMessage*)m_Message)->AddValueForKey("Return", lReturn);

	Trace (__FILE__, __LINE__, LOGMASK_BELPIC_INFO, "CSCardHandler::Execute - End");
}

CObject *CSCardHandler::Clone (void)
{
	CSCardHandler *tempObject = new CSCardHandler();
	return (CObject *) tempObject;
}

long CSCardHandler::SCardMapFunction(CGenericMessage *pMessage)
{
    long lReturn = SCARD_E_CARD_UNSUPPORTED;
    CMasterManager *pMaster = CMasterManager::Instance();
	CPCSCManager *pPCSCManager = (CPCSCManager *)pMaster->GetManager ("PCSC");
    
	if (pPCSCManager == NULL)
	{
		// Error
		Trace (__FILE__, __LINE__, LOGMASK_BELPIC_ERROR, "Can't find PCSC Manager");
		return lReturn;
	}

    char szFuncName[64] = {0};
    pMessage->GetValueForKey("FunctionName", szFuncName, sizeof(szFuncName));

    S_CardFunction pFunc = 0;

    int iFuncCount = sizeof(tFuncMap)/sizeof(tFuncMap[0]);
	for (int i = 0; i < iFuncCount - 1; i++)
    {
		if (0 == strcmp(tFuncMap[i].szFuncName, szFuncName)) 
        {
            pFunc = tFuncMap[i].funcPtr; 
            break;
		}
    }
    if(pFunc != NULL)
    {
#ifdef _WIN32
#ifdef _DEBUG
        unsigned long lConnID = 0;
        char szBuffer[256] = {0};
        pMessage->GetValueForKey("ConnID", &lConnID);
        sprintf(szBuffer, "Connection %ld Func=%s\n", lConnID, szFuncName);
        OutputDebugString(szBuffer);
 //		Trace (__FILE__, __LINE__, LOGMASK_BELPIC_ERROR, szFuncName);
#endif
#endif
        lReturn = (pPCSCManager->*pFunc)(pMessage);
    }

    return lReturn;
}

long CSCardHandler::CheckIPAllowed(CConnection_Mono * pConnection)
{
    long lReturn = SCARD_E_NO_ACCESS;
    CConnectionSocket *pSocket = pConnection->GetSocket();
    if(pSocket != NULL)
    {
        struct sockaddr_in *pSockAddr = pSocket->GetSockAddr();
        if(pSockAddr != NULL)
        {
            char *pIPAddress = inet_ntoa(pSockAddr->sin_addr);
            if(pIPAddress && 0 != strcmp(pIPAddress, "127.0.0.1"))
            {
                wxString strMask(pIPAddress);
                char **pszIP = CConfig::GetAllowedIPAddress();
                if(pszIP != NULL)
                {
                    int i = 0;
                    while(pszIP[i] != NULL)
                    {
                        char *pszDummy = pszIP[i++];
                        if(strMask.Matches(pszDummy))
                        {
                            lReturn = SCARD_S_SUCCESS;
                            break;
                        }
                    }
                    free(pszIP[0]);
                    free(pszIP);
                }                
            }
            else
            {
                lReturn = SCARD_S_SUCCESS;
            }
        }
    }
    return lReturn;
}
