/////////////////////////////////////////////////////////////////////////////
// Name:        eidviewerApp.cpp
/////////////////////////////////////////////////////////////////////////////

// ----------------------------------------------------------------------------
// headers
// ----------------------------------------------------------------------------
#include "wx/wxprec.h"

#ifndef WX_PRECOMP
	#include "wx/wx.h"
#endif

#include "wx/image.h"
#include "wx/fontenum.h"
#include "wx/file.h"

#include "eidviewerApp.h"

#include "eidlib.h"
#include "TLVBuffer.h"

#include <opensc/opensc.h>
#include <opensc/pkcs15.h>

#ifndef _WIN32
    #include <winscard.h>
#endif

// #define _TIMING_READ

class MyFontEnumerator : public wxFontEnumerator
{
public:
    bool GotAny() const
        { return !m_facenames.IsEmpty(); }

    const wxArrayString& GetFacenames() const
        { return m_facenames; }

    bool IsInstalled(const wxString & strName) const
    {
        bool bRet = false;
        for ( unsigned int i = 0; i < m_facenames.GetCount(); i++ )
        {
            if (strName == m_facenames.Item(i))
            {
                bRet = true;
                break;
            }
        }
        return bRet;
    }

protected:
    virtual bool OnFacename(const wxString& facename)
    {
        m_facenames.Add(facename);
        return TRUE;
    }

    private:
        wxArrayString m_facenames;
};


char eidviewerApp::hexChars[] = {'0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F'};

// Create a new application object: this macro will allow wxWindows to create
// the application object during program execution (it's better than using a
// static object for many reasons) and also declares the accessor function
// wxGetApp() which will return the reference of the right type (i.e. eidviewerApp and
// not wxApp)
IMPLEMENT_APP(eidviewerApp)

// ============================================================================
// implementation
// ============================================================================

// ----------------------------------------------------------------------------
// the application class
// ----------------------------------------------------------------------------

eidviewerApp::eidviewerApp()
{
    m_pFrame = NULL;
    m_pLocale = NULL;
    m_pFontData = NULL;
}

eidviewerApp::~eidviewerApp()
{
    if(m_pFontData)
    {
        delete m_pFontData;
        m_pFontData = NULL;
    }
    if(m_pLocale)
    {
        delete m_pLocale;
        m_pLocale = NULL;
    }
}

// 'Main program' equivalent: the program execution "starts" here
bool eidviewerApp::OnInit()
{
    LoadConfiguration();

    wxImage::AddHandler( new wxJPEGHandler );

    m_pFontData = ParseFonts();

    // Create the main frame window
    m_pFrame = new eidviewerFrame(_("Identity Card"), 50, 50);

#if defined(__WIN16__) || defined(__WXMOTIF__)
    int width, height;
    m_pFrame->GetSize(& width, & height);
    m_pFrame->SetSize(-1, -1, width, height);
#endif

    SwitchLanguage();
    m_pFrame->SetStatusText(_("Click on the chip to read the card"));
    m_pFrame->Show(TRUE);
    SetTopWindow(m_pFrame);

    // success: wxApp::OnRun() will be called which will enter the main message
    // loop and the application will run. If we returned FALSE here, the
    // application would exit immediately.

    if (wxApp::argc > 1)
    {
         wxString strArg = wxApp::argv[1];    
         ReadFileCard(strArg);
    }
    return TRUE;
}

void eidviewerApp::LoadLocale()
{
    if(m_pLocale == NULL)
    {
        m_pLocale = new wxLocale();
    }
    wxString strCat;
    int iLanguage = wxLANGUAGE_ENGLISH;
    switch(m_Config.m_dwLanguage)
    {
    case 0: // Dutch
        iLanguage = wxLANGUAGE_DUTCH_BELGIAN;
        strCat = wxT("eidgui_nl");
        break;
    case 1: // French
        iLanguage = wxLANGUAGE_FRENCH_BELGIAN;
        strCat = wxT("eidgui_fr");
        break;
    case 2: // German
        iLanguage = wxLANGUAGE_GERMAN;
        strCat = wxT("eidgui_de");
        break;
    }
    m_pLocale->Init(iLanguage);
#ifdef _WIN32
    wxString strValue;
    if(wxGetEnv(wxT("SYSTEMROOT"), &strValue))
    {
        m_pLocale->AddCatalogLookupPathPrefix(strValue);
    }
#endif
    if(!strCat.IsEmpty())
    {
        m_pLocale->AddCatalog(strCat);
    }
}

int eidviewerApp::OnExit()
{
    SaveConfiguration();
    BEID_Exit();	

    return 0;
}

long eidviewerApp::GetLanguage()
{
    return m_Config.m_dwLanguage; 
}

void eidviewerApp::LoadConfiguration()
{
    m_Config.Load();  
}

void eidviewerApp::SaveConfiguration()
{
    m_Config.Save();
}

char *eidviewerApp::Hexify(unsigned char * pData, unsigned long ulLen) 
{
    char *pszHex = new char[ulLen*2 + 1];
    memset(pszHex, 0, ulLen*2 + 1);
    if(pData != NULL)
    {
        int j = 0;
        for(unsigned long i = 0; i < ulLen; i++) 
        {
            pszHex[j++] = hexChars[pData[i]>>4 & 0x0F];
            pszHex[j++] = hexChars[pData[i] & 0x0F];
         }
    }
    return pszHex;
}

void eidviewerApp::SwitchLanguage ()
{
    LoadLocale();

    if(m_pFrame == NULL)
        return;

    m_pFrame->GetNotebook()->LoadStrings(m_Config.m_dwLanguage); 
    m_oErrMap.clear();
    m_oErrMap[BEID_E_BAD_PARAM] = _("Invalid parameter");
    m_oErrMap[BEID_E_INTERNAL] = _("An internal consistency check failed");
    m_oErrMap[BEID_E_INVALID_HANDLE] = _("Invalid handle");
    m_oErrMap[BEID_E_INSUFFICIENT_BUFFER] = _("The data buffer to receive returned data is too small for the returned data");
    m_oErrMap[BEID_E_COMM_ERROR] = _("An internal communications error has been detected");
    m_oErrMap[BEID_E_TIMEOUT] = _("A specified timeout value has expired");
    m_oErrMap[BEID_E_UNKNOWN_CARD] = _("Unknown card in reader");
    m_oErrMap[BEID_E_KEYPAD_CANCELLED] = _("Input cancelled");
    m_oErrMap[BEID_E_KEYPAD_TIMEOUT] = _("Timout returned from pinpad");
    m_oErrMap[BEID_E_KEYPAD_PIN_MISMATCH] = _("The two PINs did not match");
    m_oErrMap[BEID_E_KEYPAD_MSG_TOO_LONG] = _("Message too long on pinpad");
    m_oErrMap[BEID_E_INVALID_PIN_LENGTH] = _("Invalid PIN length");
    m_oErrMap[BEID_E_VERIFICATION] = _("Error in a signature verification or a certificate validation");
    m_oErrMap[BEID_E_NOT_INITIALIZED] = _("Library not initialized");
    m_oErrMap[BEID_E_UNKNOWN] = _("An internal error has been detected, but the source is unknown");
    m_oErrMap[BEID_E_UNSUPPORTED_FUNCTION] = _("Function is not supported");
    m_oErrMap[BEID_E_INCORRECT_VERSION] = _("Incorrect library version");
    m_oErrMap[BEID_E_INVALID_ROOT_CERT] = _("Wrong Root Certificate");    
}

void eidviewerApp::ClearPages()
{
    if(m_pFrame == NULL)
        return;

    m_pFrame->SetStatusText(wxT(""));
    m_pFrame->GetNotebook()->ClearPages(); 
    m_pFrame->GetNotebook()->Refresh();
}

void eidviewerApp::ReadCard()
{
    ClearPages();
    BEID_Exit();

#ifdef _TIMING_READ
    clock_t         t1, t2;        
    t1 = clock();
#endif

    long lHandle = 0;
    // Alround buffer
    BYTE buffer[4096] = {0};
    BEID_Bytes tBytes = {0};
    tBytes.length = 4096;
    tBytes.data = buffer;

    BEID_Status tStatus = {0};
    BEID_ID_Data idData = {0};
    BEID_Address adData = {0};
    BEID_Certif_Check tCheck = {0};

    if(m_pFrame == NULL)
        return;

    wxBusyCursor wait;

    m_pFrame->SetStatusText(_("Connecting to card..."));

    try
    {
#if wxUSE_UNICODE
        tStatus = BEID_Init((char *)(m_Config.m_StrReader.mb_str().data()), m_Config.m_dwOCSP, m_Config.m_dwCRL, &lHandle);
#else
        tStatus = BEID_Init((char *)(m_Config.m_StrReader.c_str()), m_Config.m_dwOCSP, m_Config.m_dwCRL, &lHandle);
#endif
        if(BEID_OK != tStatus.general)
            throw(&tStatus);

        // Read ID Data
        m_pFrame->SetStatusText(_("Reading Identity..."));

        tStatus = BEID_GetID(&idData, &tCheck);
        if(BEID_OK != tStatus.general)
            throw(&tStatus);

        long lSignatureID = tCheck.signatureCheck;
        m_pFrame->GetNotebook()->SetIDData(&idData, tCheck.signatureCheck);
        m_pFrame->GetNotebook()->SetCertificateData(&tCheck); 
        memset(&tCheck, 0, sizeof(BEID_Certif_Check));

        // Read Address Data
        m_pFrame->SetStatusText(_("Reading Address..."));

        tStatus = BEID_GetAddress(&adData, &tCheck);
        if(BEID_OK != tStatus.general)
            throw(&tStatus);

        m_pFrame->GetNotebook()->SetAddressData(&adData, tCheck.signatureCheck);
        memset(&tCheck, 0, sizeof(BEID_Certif_Check));
        // Read Picture Data
        if(BEID_SIGNATURE_VALID == lSignatureID || BEID_SIGNATURE_VALID_WRONG_RRNCERT == lSignatureID)
        {
            m_pFrame->SetStatusText(_("Reading Picture..."));

            tStatus = BEID_GetPicture(&tBytes, &tCheck);
            if(BEID_OK != tStatus.general)
                throw(&tStatus);

            if(BEID_SIGNATURE_VALID == tCheck.signatureCheck && BEID_SIGNATURE_VALID_WRONG_RRNCERT == lSignatureID)
            {
                tCheck.signatureCheck = lSignatureID;
            }
            m_pFrame->GetNotebook()->SetPictureData(&tBytes, tCheck.signatureCheck);
        }
        else
        {
            BEID_Bytes tBytesClear = {0};
            m_pFrame->GetNotebook()->SetPictureData(&tBytesClear, BEID_SIGNATURE_INVALID, TRUE);
        }
        // Read VersionInfo
        BEID_VersionInfo tVersionInfo = {0};
        BEID_Bytes tSignature = {0};
        BYTE bufferSig[256] = {0};
        tSignature.length = 256;
        tSignature.data = bufferSig;

        m_pFrame->SetStatusText(_("Reading VersionInfo..."));

        tStatus = BEID_GetVersionInfo(&tVersionInfo, FALSE, &tSignature);
        if(BEID_OK != tStatus.general)
            throw(&tStatus);

        m_pFrame->GetNotebook()->SetVersionInfoData(&tVersionInfo);

        // Read PinData
        m_pFrame->SetStatusText(_("Reading PIN Data..."));

        std::map<wxString, std::vector<CPin> > PinMap;
        if( BEID_OK == ReadPINs(PinMap))
        {
            GetPINStatus(PinMap);
            m_pFrame->GetNotebook()->SetPINData(PinMap); 
        }
        
        m_pFrame->SetStatusText(_("Done"));
        BEID_Exit();
    }
    catch(BEID_Status *pStatus)
    {
        FillStatusError(pStatus);
        BEID_Exit();
    }
#ifdef _TIMING_READ
    t2 = clock();
    double elapsed = (double)(t2 - t1) / CLOCKS_PER_SEC;
    char buf[256] = {0};
    sprintf(buf, "elapsed = %lf", elapsed);
    MessageBoxA(NULL,buf,"Timer",MB_OK);
#endif
}

wxFont *eidviewerApp::ParseFonts()
{
    wxFont *pFont = NULL;

    MyFontEnumerator fontEnumerator;
    fontEnumerator.EnumerateFacenames();
    if ( fontEnumerator.GotAny() )
    {
        eidviewer::CConfig & oConfig = GetConfig();
        wxString strFonts = oConfig.m_strFonts;
        wxString strFontSel;
        int iIndex = 0;
        int iStart = 0;
        long lFontSize = oConfig.m_lFontSize; 
        bool bFound = false;
        while(-1 != (iIndex = strFonts.Find(',')))
        {
            strFontSel = strFonts.Mid(iStart, iIndex - iStart);
            strFontSel.Trim();
            if(fontEnumerator.IsInstalled(strFontSel))
            {
                bFound = true;
                break;
            }
            iStart = ++iIndex;
            strFonts = strFonts.Mid(iStart);
        }
        if(!bFound && !strFonts.IsEmpty())
        {
            strFontSel = strFonts;
            strFontSel.Trim(); 
            if(fontEnumerator.IsInstalled(strFontSel))
            {
                bFound = true;
            }
        }
        if(bFound)
        {
            pFont = new wxFont(lFontSize, wxDEFAULT, wxNORMAL, wxNORMAL, FALSE, strFontSel);
        }        
    }

    return pFont;
}

void eidviewerApp::FillStatusError(BEID_Status *pStatus)
{
    wxString strTemp;
    switch (pStatus->general)
    {
    case BEID_OK:
    break;
    case BEID_E_SYSTEM:
    case BEID_E_PCSC:
        {
#ifndef _WIN32
            if(pStatus->general == BEID_E_PCSC)
            {
                strTemp = pcsc_stringify_error(pStatus->pcsc);
            }
            else
            {
#endif
            strTemp = wxSysErrorMsg(pStatus->general == BEID_E_PCSC ? pStatus->pcsc : pStatus->system);
#ifndef _WIN32
            }
#endif
            wxString strPrefix(_("System Error : "));
            if(!strTemp.IsEmpty())
            {
                strTemp = strPrefix + strTemp;
            }
            else
            {
                strTemp = strPrefix + _("Unknown errorcode");
            }
        }
        break;
    case BEID_E_CARD:
        {
            strTemp = wxString::Format(_("Card error : SW=%02X%02X"), pStatus->cardSW[0], pStatus->cardSW[1]); 
        }
        break;
    default: // BEID_ERR
        strTemp = _("Error : ");
        strTemp += m_oErrMap[pStatus->general];
        break;
    }
    m_pFrame->SetStatusText(strTemp);
}

long eidviewerApp::ReadPINs(std::map<wxString, std::vector<CPin> > & PinMap)
{
    long lRet = BEID_OK;

    struct sc_context *pCtx = NULL;
    struct sc_reader *pReader = NULL;
    struct sc_card *pCard = NULL;
    wxString strReader(m_Config.m_StrReader);

	if( BEID_OK == sc_establish_context(&pCtx, NULL))
    {
        if(strReader.IsEmpty())
        {
            // Find a reader
            for(int i = 0; i < pCtx->reader_count; i++)
            {
                pReader = pCtx->reader[i];
                if(pReader != NULL)
                {
                    lRet = sc_connect_card(pReader, 0, &pCard);
                    if(BEID_OK == lRet)
                        break;
                    pReader = NULL;
                }
            }
        }
        else
        {
            for(int i = 0; i < pCtx->reader_count; i++)
            {
                pReader = pCtx->reader[i];
                wxString strTemp(pReader->name, *wxConvCurrent);
                if(strReader == strTemp)
                    break;
                pReader = NULL;
            }
            if(pReader != NULL)
            {
                lRet = sc_connect_card(pReader, 0, &pCard);
            }
        }

        if(pCard != NULL)
        {
            struct sc_pkcs15_card *p15Card = NULL;

            if(BEID_OK == (lRet = sc_pkcs15_bind(pCard, &p15Card)))
            {
                // list pins
                int iNumber = 0;
                struct sc_pkcs15_object *pObjs[32];
                struct sc_pkcs15_pin_info *pPinInfo = NULL;
	            iNumber = sc_pkcs15_get_objects(p15Card, SC_PKCS15_TYPE_AUTH_PIN, pObjs, 32);
                std::vector<CPin> VecPin;

                wxString strApp(p15Card->label, *wxConvCurrent);
                for (int i = 0; i < iNumber; i++) 
                {
                    CPin oPin;
                    pPinInfo = (struct sc_pkcs15_pin_info *)pObjs[i]->data;
                    oPin.SetReference(pPinInfo->reference);
                    oPin.SetTriesLeft(pPinInfo->tries_left);
                    oPin.SetFlags(pPinInfo->flags);
                    oPin.SetLabel(wxString(pObjs[i]->label, *wxConvCurrent));
                    oPin.SetType(pPinInfo->type);
                    oPin.SetAuthID(pPinInfo->auth_id.value, pPinInfo->auth_id.len); 
                    oPin.SetApplication(strApp);
                    oPin.SetPinPad(p15Card->card->slot->capabilities & SC_SLOT_CAP_PIN_PAD ? true : false);
                    VecPin.push_back(oPin);
                    pPinInfo = NULL;
                }
                if(VecPin.size() > 0)
                {
                    PinMap[strApp] = VecPin;
                    VecPin.clear();
                }
            }
            if (p15Card != NULL)
            {
		        sc_pkcs15_unbind(p15Card);
            }        
        
            sc_disconnect_card(pCard, 0);
        }

        if (pCtx)
		    sc_release_context(pCtx);
    }
    return lRet;
}

void eidviewerApp::GetPINStatus(std::map<wxString, std::vector<CPin> > & PinMap)
{
    // Get Pin Status
    BEID_Status tStatus = {0};

    std::map<wxString, std::vector<CPin> >::iterator it;
    for (it = PinMap.begin(); it != PinMap.end(); ++it)
    {
        for(unsigned int i = 0; i < (*it).second.size(); ++i)
        {
            CPin & oPin = (*it).second[i];
            BEID_Bytes tSignature = {0};
            BYTE bufferSig[256] = {0};
            tSignature.length = 256;
            tSignature.data = bufferSig;
            BEID_Pin tPin = {0};
            tPin.id = oPin.GetReference();
            tPin.pinType = oPin.GetType();
            char *pszUsage = Hexify(oPin.GetAuthID().GetData(), oPin.GetAuthID().GetSize());
            tPin.usageCode = atol(pszUsage);
            delete [] pszUsage;
            long lLeft = 0;
            tStatus = BEID_GetPINStatus(&tPin, &lLeft, FALSE, &tSignature);
            if(BEID_OK == tStatus.general)
            {
                oPin.SetTriesLeft(lLeft);
            }
            else
            {
                oPin.SetTriesLeft(-1);
            }
        }
    }
}

bool eidviewerApp::ChangePin(CPin *pPin)
{
     m_pFrame->SetStatusText(_("Changing PIN..."));

    long lHandle = 0;
    BEID_Status tStatus = {0};

    BEID_Pin tPin = {0};
    tPin.id = pPin->GetReference();
    tPin.pinType = pPin->GetType();
    char *pszUsage = Hexify(pPin->GetAuthID().GetData(), pPin->GetAuthID().GetSize());
    tPin.usageCode = atol(pszUsage);
    delete [] pszUsage;
    long lLeft = 0;

    wxBusyCursor wait;

    try
    {
#if wxUSE_UNICODE
        tStatus = BEID_Init((char *)(m_Config.m_StrReader.mb_str().data()), -1, -1, &lHandle);
#else
        tStatus = BEID_Init((char *)(m_Config.m_StrReader.c_str()), -1, -1, &lHandle);
#endif
        if(BEID_OK != tStatus.general)
            throw(&tStatus);

        tStatus = BEID_ChangePIN(&tPin, NULL, NULL, &lLeft);
        if(BEID_OK != tStatus.general)
        {
            if(BEID_E_CARD < tStatus.general)
            {
                pPin->SetTriesLeft(-1);
            }
            else
            {
                pPin->SetTriesLeft(lLeft);
            }
            FillStatusError(&tStatus);
            if(BEID_E_PCSC == tStatus.general)
            {
                ClearPages();
            }
        }
        else
        {
             pPin->SetTriesLeft(-1);
             m_pFrame->SetStatusText(_("PIN changed succesful"));
        }
    }
    catch(BEID_Status *pStatus)
    {
        ClearPages();
        FillStatusError(pStatus);
        BEID_Exit();
        return false;
    }
    BEID_Exit();
    return true;
}

void eidviewerApp::SetStatusBarText(wxString strMsg)
{
    if(m_pFrame)
    {
        m_pFrame->SetStatusText(strMsg);
    }
}

void eidviewerApp::ReadRawCard()
{
    BEID_Exit();
    long lHandle = 0;

    BEID_Status tStatus = {0};
    BEID_Raw rawData = {0};

    if(m_pFrame == NULL)
        return;

    m_pFrame->GetToolBar()->SetFocus(); 
    wxBusyCursor wait;

    m_pFrame->SetStatusText(_("Connecting to card..."));

    try
    {
#if wxUSE_UNICODE
        tStatus = BEID_Init((char *)(m_Config.m_StrReader.mb_str().data()), -1, -1, &lHandle);
#else
        tStatus = BEID_Init((char *)(m_Config.m_StrReader.c_str()), -1, -1, &lHandle);
#endif
        
        if(BEID_OK != tStatus.general)
            throw(&tStatus);

        // Fill filename
        BEID_ID_Data idData = {0};
        BEID_Certif_Check tCheck = {0};

        m_pFrame->SetStatusText(_("Reading..."));
           
        tStatus = BEID_GetID(&idData, &tCheck);
        if(BEID_OK != tStatus.general)
            throw(&tStatus);

        wxString strFileName(idData.nationalNumber, wxConvUTF8);
        strFileName += wxT(".eid");
        wxFileDialog oDialog(m_pFrame, _("Save an eID file"), wxT(""), strFileName, wxT("eID files (*.eid)|*.eid"), wxSAVE | wxHIDE_READONLY | wxOVERWRITE_PROMPT);
        if (oDialog.ShowModal() != wxID_OK)
        {
            m_pFrame->SetStatusText(wxT(""));
            BEID_Exit();
            return;
        }
        strFileName = oDialog.GetPath();

        // Read Raw Data
        m_pFrame->SetStatusText(_("Storing Data..."));

        tStatus = BEID_GetRawData(&rawData);
        if(BEID_OK != tStatus.general)
            throw(&tStatus);

        bool bOK = false;
        BEID_Bytes tBytes = {0};
        tBytes.length = 8192;
        tBytes.data = new BYTE[tBytes.length];
        memset(tBytes.data, 0, tBytes.length);
        
        eidviewer::CTLVBuffer oTLV;
        oTLV.MakeTLV(&rawData, &tBytes);
      
        wxFile oEIDFile;
        if(oEIDFile.Open(strFileName, wxFile::write))
        {
            unsigned int iWritten = oEIDFile.Write(tBytes.data, tBytes.length);
            oEIDFile.Close();
            if(iWritten != tBytes.length)
            {
                m_pFrame->SetStatusText(_("Error saving eID file"));
            }
            else
            {
                bOK = true;
            }
        }

        delete [] tBytes.data;
        if(bOK)
        {
            m_pFrame->SetStatusText(_("Done"));
        }
        BEID_Exit();
    }
    catch(BEID_Status *pStatus)
    {
        FillStatusError(pStatus);
        BEID_Exit();
    }
}

void eidviewerApp::ReadFileCard(wxString & strFile)
{
    BYTE ucBuffer[1024] = {0};
    int iRead = -1;
    BEID_Raw rawData = {0};
    CMyByteArray oTempArray;

    if(m_pFrame == NULL)
        return;

    m_pFrame->SetStatusText(_("Opening eID file..."));

    if(strFile.IsEmpty())
    {
        wxFileDialog oDialog(m_pFrame, _("Open an eID file"), wxT(""), wxT(""), wxT("eID files (*.eid)|*.eid"), wxOPEN | wxHIDE_READONLY);
        if (oDialog.ShowModal() != wxID_OK)
        {
            m_pFrame->SetStatusText(wxT(""));
            return;
        }
        strFile = oDialog.GetPath();
    }

    if (strFile.IsEmpty() || !wxFileExists(strFile))
    {
        m_pFrame->SetStatusText( _("File doesn't exist") );
        return;
    }
    wxFile oEIDFile;
    if(oEIDFile.Open(strFile))
    {
        while(!oEIDFile.Eof())
        {
            if(wxInvalidOffset != (iRead = oEIDFile.Read(ucBuffer, sizeof(ucBuffer))))
            {
                oTempArray.Append(ucBuffer, iRead); 
            }
        }
        oEIDFile.Close();
    }
    
    if(oTempArray.GetSize()<= 0)
    {
        m_pFrame->SetStatusText( _("Invalid eID File") );
        return;
    }

    wxBusyCursor wait;

    eidviewer::CTLVBuffer oTLV;
    if(!oTLV.ParseTLV(oTempArray.GetData(), oTempArray.GetSize()))
    {
        m_pFrame->SetStatusText( _("Invalid eID File") );
        return;
    }

    oTLV.FillBinaryData(0x01, rawData.idData, &rawData.idLength);
    oTLV.FillBinaryData(0x02, rawData.idSigData, &rawData.idSigLength);
    oTLV.FillBinaryData(0x03, rawData.addrData, &rawData.addrLength);
    oTLV.FillBinaryData(0x04, rawData.addrSigData, &rawData.addrSigLength);
    oTLV.FillBinaryData(0x05, rawData.pictureData, &rawData.pictureLength);
    oTLV.FillBinaryData(0x06, rawData.cardData, &rawData.cardDataLength);
    oTLV.FillBinaryData(0x07, rawData.tokenInfo, &rawData.tokenInfoLength);
    oTLV.FillBinaryData(0x08, rawData.certRN, &rawData.certRNLength);
    oTLV.FillBinaryData(0x09, rawData.challenge, &rawData.challengeLength);
    oTLV.FillBinaryData(0x0A, rawData.response, &rawData.responseLength);

    ClearPages();
    BEID_Exit();
    long lHandle = 0;

    // Alround buffer
    BYTE buffer[4096] = {0};
    BEID_Bytes tBytes = {0};
    tBytes.length = 4096;
    tBytes.data = buffer;
    BEID_Status tStatus = {0};
    BEID_ID_Data idData = {0};
    BEID_Address adData = {0};
    BEID_Certif_Check tCheck = {0};

    try
    {
#if wxUSE_UNICODE
        tStatus = BEID_Init("VIRTUAL", -1, -1, &lHandle);
#else
        tStatus = BEID_Init("VIRTUAL", -1, -1, &lHandle);
#endif
        
        if(BEID_OK != tStatus.general)
            throw(&tStatus);

        // Read Raw Data
        tStatus = BEID_SetRawData(&rawData);
        if(BEID_OK != tStatus.general)
            throw(&tStatus);

        // Read ID Data
        m_pFrame->SetStatusText(_("Reading Identity..."));

        tStatus = BEID_GetID(&idData, &tCheck);
        if(BEID_OK != tStatus.general)
            throw(&tStatus);

        m_pFrame->GetNotebook()->SetIDData(&idData, tCheck.signatureCheck);
        m_pFrame->GetNotebook()->SetCertificateData(&tCheck); 
        memset(&tCheck, 0, sizeof(BEID_Certif_Check));

        // Read Address Data
        m_pFrame->SetStatusText(_("Reading Address..."));

        tStatus = BEID_GetAddress(&adData, &tCheck);
        if(BEID_OK != tStatus.general)
            throw(&tStatus);

        m_pFrame->GetNotebook()->SetAddressData(&adData, tCheck.signatureCheck);
        memset(&tCheck, 0, sizeof(BEID_Certif_Check));
        
        // Read Picture Data
        m_pFrame->SetStatusText(_("Reading Picture..."));

        tStatus = BEID_GetPicture(&tBytes, &tCheck);
        if(BEID_OK != tStatus.general)
            throw(&tStatus);

        m_pFrame->GetNotebook()->SetPictureData(&tBytes, tCheck.signatureCheck);

        // Read VersionInfo
        BEID_VersionInfo tVersionInfo = {0};
        BEID_Bytes tSignature = {0};
        BYTE bufferSig[256] = {0};
        tSignature.length = 256;
        tSignature.data = bufferSig;

        m_pFrame->SetStatusText(_("Reading VersionInfo..."));

        tStatus = BEID_GetVersionInfo(&tVersionInfo, FALSE, &tSignature);
        if(BEID_OK != tStatus.general)
            throw(&tStatus);

        m_pFrame->GetNotebook()->SetVersionInfoData(&tVersionInfo);

        m_pFrame->SetStatusText(_("Done"));
        BEID_Exit();
    }
    catch(BEID_Status *pStatus)
    {
        FillStatusError(pStatus);
        BEID_Exit();
    }
}
