// Config.cpp: implementation of the CConfig class.
//
//////////////////////////////////////////////////////////////////////

#include "Config.h"
#include "eidlib.h"

#include <openssl/conf.h>
#include <algorithm>
#include <wx/confbase.h>
#include <wx/fileconf.h>
#include <wx/utils.h>
#include <wx/dynload.h>

#ifdef _WIN32
    #include <windows.h>
#endif


#define CONFFILE_BEID "BEID.conf"
#define CONFFILE_BEID_DEFAULT "BEID_default"
#define CONFFILE_BEID_DEFAULT_MAINDIR "maindir"
#define CONFFILE_BEID_DEFAULT_CERTS "certs"
#define CONFFILE_BEID_DEFAULT_CRL "crl"
#define CONFFILE_BEID_DEFAULT_CACHING "caching"
#define CONFFILE_BEID_DEFAULT_PROXYHOST "proxyHost"
#define CONFFILE_BEID_DEFAULT_PROXYPORT "proxyPort"
 
#define CONFFILE_CA_CERTS "CA_certs"
#define CONFFILE_CA_CERTS_HTTPSTORE "httpstore"
#define CONFFILE_CA_CERTS_GOV "government"
#define CONFFILE_CA_CERTS_ROOT "root"
#define CONFFILE_CA_CERTS_ALLOWTESTROOT "allowtestroot"

#define CONFFILE_FIREWALL "Application_Filtering"

#define CONFFILE_AUTOUPDATE "VersionCheck"
#define CONFFILE_AUTOUPDATE_ENABLED "enabled"
#define CONFFILE_AUTOUPDATE_DAYS "delay"
#define CONFFILE_AUTOUPDATE_VERSIONURL "url"
#define CONFFILE_AUTOUPDATE_DOWNLOADURL "downloadURL"
#define CONFFILE_AUTOUPDATE_LASTCHECK "lastCheck"

namespace eidlib
{

class CFindKey
{
private:
    std::string m_strKey;
public:
    explicit CFindKey(std::string strKey)
    {
        m_strKey = strKey;
    }

    bool operator() (std::pair<std::string, std::string> oValue)
    {
#ifdef _WIN32
        return 0 == stricmp(oValue.first.c_str(), m_strKey.c_str());
#else
        return 0 == strcasecmp(oValue.first.c_str(), m_strKey.c_str());
#endif
    }
};


//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////


std::map<std::string, std::vector<std::pair<std::string, std::string> > > CConfig::m_oMapConfig;

CConfig::CConfig()
{

}

CConfig::~CConfig()
{

}

wxString CConfig::GetGlobalDir()
{
  wxString strDir;

#ifdef __VMS__ // Note if __VMS is defined __UNIX is also defined
    strDir = wxT("sys$manager:");
#elif defined(__WXMAC__)
    strDir = wxMacFindFolder(  (short) kOnSystemDisk, kPreferencesFolderType, kDontCreateFolder ) ;
#elif defined( __UNIX__ )
    strDir = wxT("/etc/");
#elif defined(__WXPM__)
    ULONG aulSysInfo[QSV_MAX] = {0};
    UINT drive;
    APIRET rc;

    rc = DosQuerySysInfo( 1L, QSV_MAX, (PVOID)aulSysInfo, sizeof(ULONG)*QSV_MAX);
    if (rc == 0)
    {
        drive = aulSysInfo[QSV_BOOT_DRIVE - 1];
        strDir.Printf(wxT("%c:\\OS2\\"), 'A'+drive-1);
    }
#elif defined(__WXSTUBS__)
    wxASSERT_MSG( FALSE, wxT("TODO") ) ;
#elif defined(__DOS__)
    // There's no such thing as global cfg dir in MS-DOS, let's return
    // current directory (FIXME_MGL?)
    return wxT(".\\");
#else // Windows
    wxChar szWinDir[MAX_PATH];
    ::GetWindowsDirectory(szWinDir, MAX_PATH);
    wxDynamicLibrary pathLoader("Kernel32");
    if(pathLoader.IsLoaded())
    {
        typedef UINT (WINAPI *TGetSystemWindowsDirectoryA)(LPSTR,UINT);
        TGetSystemWindowsDirectoryA pFunc  = (TGetSystemWindowsDirectoryA) ::GetProcAddress( pathLoader.GetLibHandle(), "GetSystemWindowsDirectoryA" );
        if(pFunc != NULL)
        {
            memset(szWinDir, 0, sizeof(szWinDir));
            pFunc(szWinDir, MAX_PATH);
        }
    }
    strDir = szWinDir;
    strDir << wxT('\\');
#endif // Unix/Windows

    return strDir;
}

void CConfig::Load()
{
    int i = 0;
    long lErr = 0;
    STACK_OF(CONF_VALUE) *pSec = NULL;
    CONF_VALUE *pItem = NULL;
    CONF *pConf = NULL;
    wxString strConfPath = GetGlobalDir() + CONFFILE_BEID;

    m_oMapConfig.clear();

    pConf = NCONF_new(NCONF_default());
    if (NCONF_load(pConf, strConfPath.GetData(), &lErr) > 0)
    {
        std::vector<std::pair<std::string, std::string> > oTempVec(8);
        if (NULL != (pSec = NCONF_get_section(pConf, CONFFILE_BEID_DEFAULT)))
        {
            for (i = 0; i < sk_CONF_VALUE_num(pSec); i++)
            {
                pItem = sk_CONF_VALUE_value(pSec, i);
                oTempVec.push_back(std::pair<std::string, std::string>(pItem->name, pItem->value));
            }
            m_oMapConfig[CONFFILE_BEID_DEFAULT] = oTempVec;
            pSec = NULL;
        }
        oTempVec.clear();

        if (NULL != (pSec = NCONF_get_section(pConf, CONFFILE_CA_CERTS)))
        {
            for (i = 0; i < sk_CONF_VALUE_num(pSec); i++)
            {
                pItem = sk_CONF_VALUE_value(pSec, i);
                oTempVec.push_back(std::pair<std::string, std::string>(pItem->name, pItem->value));
            }
            m_oMapConfig[CONFFILE_CA_CERTS] = oTempVec;
            pSec = NULL;
        }
        oTempVec.clear();

        if (NULL != (pSec = NCONF_get_section(pConf, CONFFILE_FIREWALL)))
        {
            for (i = 0; i < sk_CONF_VALUE_num(pSec); i++)
            {
                pItem = sk_CONF_VALUE_value(pSec, i);
                oTempVec.push_back(std::pair<std::string, std::string>(pItem->name, pItem->value));
            }
            m_oMapConfig[CONFFILE_FIREWALL] = oTempVec;
            pSec = NULL;
        }
        oTempVec.clear();

        if (NULL != (pSec = NCONF_get_section(pConf, CONFFILE_AUTOUPDATE)))
        {
            for (i = 0; i < sk_CONF_VALUE_num(pSec); i++)
            {
                pItem = sk_CONF_VALUE_value(pSec, i);
                oTempVec.push_back(std::pair<std::string, std::string>(pItem->name, pItem->value));
            }
            m_oMapConfig[CONFFILE_AUTOUPDATE] = oTempVec;
            pSec = NULL;
        }
        oTempVec.clear();    
    }
    NCONF_free(pConf);
}

void CConfig::LoadFirewall()
{
    int i = 0;
    long lErr = 0;
    STACK_OF(CONF_VALUE) *pSec = NULL;
    CONF_VALUE *pItem = NULL;
    CONF *pConf = NULL;
    wxString strConfPath = wxGetHomeDir();
#ifdef _WIN32
    strConfPath += wxT("\\");
#else
    strConfPath += wxT("/");
#endif
    strConfPath += CONFFILE_BEID;

    pConf = NCONF_new(NCONF_default());
    if (NCONF_load(pConf, strConfPath.GetData(), &lErr) > 0)
    {
        std::vector<std::pair<std::string, std::string> > oTempVec;
        std::vector<std::pair<std::string, std::string> >::iterator itVec;
        if(m_oMapConfig.find(CONFFILE_FIREWALL) != m_oMapConfig.end())
        {
            oTempVec = m_oMapConfig[CONFFILE_FIREWALL];
        }
        if (NULL != (pSec = NCONF_get_section(pConf, CONFFILE_FIREWALL)))
        {
            for (i = 0; i < sk_CONF_VALUE_num(pSec); i++)
            {
                pItem = sk_CONF_VALUE_value(pSec, i);
                itVec = std::find_if(oTempVec.begin(), oTempVec.end(), CFindKey(pItem->name));
                if(itVec != oTempVec.end())
                {
                    (*itVec).second = pItem->value;
                }
                else
                {
                    oTempVec.push_back(std::pair<std::string, std::string>(pItem->name, pItem->value));
                }
            }
            m_oMapConfig[CONFFILE_FIREWALL] = oTempVec;
            pSec = NULL;
        }
        oTempVec.clear();
    }
    NCONF_free(pConf);
}

void CConfig::Unload()
{
    m_oMapConfig.clear();
}

const char *CConfig::FindValue(char *pszSection, char *pszKey)
{
    ConfigMapIT itmap;
    itmap = m_oMapConfig.find(pszSection);
    if(itmap != m_oMapConfig.end())
    {
        ConfigVecIT itvec;
        itvec = std::find_if((*itmap).second.begin(), (*itmap).second.end(), CFindKey(pszKey));
        if(itvec != (*itmap).second.end())
        {
            return (*itvec).second.c_str();
        }
    }
    return NULL;
}

const char *CConfig::GetCertStorePath()
{
    return FindValue(CONFFILE_BEID_DEFAULT, CONFFILE_BEID_DEFAULT_CERTS);
}

const char *CConfig::GetCRLStorePath()
{
    return FindValue(CONFFILE_BEID_DEFAULT, CONFFILE_BEID_DEFAULT_CRL);
}

const char *CConfig::GetHttpStore()
{
    return FindValue(CONFFILE_CA_CERTS, CONFFILE_CA_CERTS_HTTPSTORE);
}

char **CConfig::GetGovCerts()
{
    const char *pValue = FindValue(CONFFILE_CA_CERTS, CONFFILE_CA_CERTS_GOV);
    if(NULL != pValue)
    {
        return SplitString(pValue, strlen(pValue), ',');
    }
    return NULL;
}

char **CConfig::GetRootCerts()
{
    const char *pValue = FindValue(CONFFILE_CA_CERTS, CONFFILE_CA_CERTS_ROOT);
    if(NULL != pValue)
    {
        return SplitString(pValue, strlen(pValue), ',');
    }
    return NULL;
}

int CConfig::AllowTestRoot()
{
    BOOL bRet = FALSE;
    const char *pMode = FindValue(CONFFILE_CA_CERTS, CONFFILE_CA_CERTS_ALLOWTESTROOT);
    if(pMode != NULL && 0 == strcmp(pMode, "1"))
    {
        bRet = TRUE;
    }
    return bRet;
}

int CConfig::AllowCaching()
{
    BOOL bRet = FALSE;
    const char *pCache = FindValue(CONFFILE_BEID_DEFAULT, CONFFILE_BEID_DEFAULT_CACHING);
    if(pCache != NULL && 0 == strcmp(pCache, "1"))
    {
        bRet = TRUE;
    }
    return bRet;
}

int CConfig::GetProgramAccess(char *pszName, char *pszType)
{
    BOOL bRet = FALSE;
    LoadFirewall();
    const char *pValue = FindValue(CONFFILE_FIREWALL, pszName);
    if(NULL != pValue)
    {
        char **pszAccess =  SplitString(pValue, strlen(pValue), ',');
        if(pszAccess != NULL)
        {
            int i = 0;
            while(!bRet && pszAccess[i] != NULL)
            {
                char *pszDummy = pszAccess[i++];
                if(0 == strcmp(pszDummy, "*") || 0 == strcmp(pszDummy, pszType))
                {
                    bRet = TRUE;
                }
            }
            free(pszAccess[0]);
            free(pszAccess);
        }
    }
    return bRet;
}

void CConfig::UpdateProgramAccess(char *pszName, char *pszType)
{
    wxString strConfPath = wxGetHomeDir();
#ifdef _WIN32
    strConfPath += wxT("\\");
#else
    strConfPath += wxT("/");
#endif
    strConfPath += CONFFILE_BEID;

#ifdef _WIN32
    wxFileConfig oConfig(wxT(""), wxT(""), strConfPath, wxT(""), wxCONFIG_USE_GLOBAL_FILE);
#else
    wxFileConfig oConfig(wxT(""), wxT(""), strConfPath, wxT(""), wxCONFIG_USE_GLOBAL_FILE);
#endif

    wxString str;
    wxString strKey;
    strKey = CONFFILE_FIREWALL;
    strKey += wxT("/");
    strKey += pszName;
    if ( oConfig.Read(strKey, &str) ) 
    {
        if(!str.IsEmpty())
        {
            str += wxT(",");
        }
        str += pszType;
        oConfig.Write(strKey, str);
    }
    else 
    {
        oConfig.Write(strKey, pszType);
    }
    oConfig.Flush();
}


const char *CConfig::GetProxyHost()
{
    return FindValue(CONFFILE_BEID_DEFAULT, CONFFILE_BEID_DEFAULT_PROXYHOST);
}

const char *CConfig::GetProxyPort()
{
    return FindValue(CONFFILE_BEID_DEFAULT, CONFFILE_BEID_DEFAULT_PROXYPORT);
}

int CConfig::AllowAutoUpdate()
{
    BOOL bRet = FALSE;
    const char *pMode = FindValue(CONFFILE_AUTOUPDATE, CONFFILE_AUTOUPDATE_ENABLED);
    if(pMode != NULL && 0 == strcmp(pMode, "1"))
    {
        bRet = TRUE;
    }
    return bRet;
}

int CConfig::GetAutoUpdateScheduleDays()
{
    int iRet = 0;
    const char *pMode = FindValue(CONFFILE_AUTOUPDATE, CONFFILE_AUTOUPDATE_DAYS);
    if(pMode != NULL)
    {
        iRet = atoi(pMode);
    }
    return iRet;
}

const char *CConfig::GetAutoUpdateVersionURL()
{
    return FindValue(CONFFILE_AUTOUPDATE, CONFFILE_AUTOUPDATE_VERSIONURL);
}

const char *CConfig::GetAutoUpdateDownloadURL()
{
    return FindValue(CONFFILE_AUTOUPDATE, CONFFILE_AUTOUPDATE_DOWNLOADURL);
}

const char *CConfig::GetAutoUpdateLastCheck()
{
    LoadAutoUpdate();
    return FindValue(CONFFILE_AUTOUPDATE, CONFFILE_AUTOUPDATE_LASTCHECK);
}

void CConfig::LoadAutoUpdate()
{
    int i = 0;
    long lErr = 0;
    STACK_OF(CONF_VALUE) *pSec = NULL;
    CONF_VALUE *pItem = NULL;
    CONF *pConf = NULL;
    wxString strConfPath = wxGetHomeDir();
#ifdef _WIN32
    strConfPath += wxT("\\");
#else
    strConfPath += wxT("/");
#endif
    strConfPath += CONFFILE_BEID;

    pConf = NCONF_new(NCONF_default());
    if (NCONF_load(pConf, strConfPath.GetData(), &lErr) > 0)
    {
        std::vector<std::pair<std::string, std::string> > oTempVec;
        std::vector<std::pair<std::string, std::string> >::iterator itVec;
        if(m_oMapConfig.find(CONFFILE_AUTOUPDATE) != m_oMapConfig.end())
        {
            oTempVec = m_oMapConfig[CONFFILE_AUTOUPDATE];
        }
        if (NULL != (pSec = NCONF_get_section(pConf, CONFFILE_AUTOUPDATE)))
        {
            for (i = 0; i < sk_CONF_VALUE_num(pSec); i++)
            {
                pItem = sk_CONF_VALUE_value(pSec, i);
                itVec = std::find_if(oTempVec.begin(), oTempVec.end(), CFindKey(pItem->name));
                if(itVec != oTempVec.end())
                {
                    (*itVec).second = pItem->value;
                }
                else
                {
                    oTempVec.push_back(std::pair<std::string, std::string>(pItem->name, pItem->value));
                }
            }
            m_oMapConfig[CONFFILE_AUTOUPDATE] = oTempVec;
            pSec = NULL;
        }
        oTempVec.clear();
    }
    NCONF_free(pConf);
}

void CConfig::UpdateLastCheck(char *pszCheck)
{
    wxString strConfPath = wxGetHomeDir();
#ifdef _WIN32
    strConfPath += wxT("\\");
#else
    strConfPath += wxT("/");
#endif
    strConfPath += CONFFILE_BEID;

#ifdef _WIN32
    wxFileConfig oConfig(wxT(""), wxT(""), strConfPath, wxT(""), wxCONFIG_USE_GLOBAL_FILE);
#else
    wxFileConfig oConfig(wxT(""), wxT(""), strConfPath, wxT(""), wxCONFIG_USE_GLOBAL_FILE);
#endif

    wxString str;
    wxString strKey;
    strKey = CONFFILE_AUTOUPDATE;
    strKey += wxT("/");
    strKey += CONFFILE_AUTOUPDATE_LASTCHECK;
    oConfig.Write(strKey, pszCheck);
    oConfig.Flush();
}

char **CConfig::SplitString(const char *psStr, int iLength, char cSep)
{
     const char * psSource;
     char * psTemp;
     char * psDest;
     char ** ppsList;
     int i;
     int iFields;
 
     psTemp = (char *)malloc(iLength + 1);
 
     iFields = 1;
     for (psSource = psStr, psDest = psTemp, i = 0; i < iLength; i++, psSource++, psDest++) 
     {
         *psDest = *psSource;
         if (*psDest == cSep) iFields++;
     }
 
     *psDest = '\0';
 
     ppsList = (char **)malloc(sizeof(char *) * (iFields + 1));
 
     psDest = psTemp;
     ppsList[0] = psDest;
     i = 1;
     while (i < iFields) 
     {
         if (*psDest == cSep) 
         {
             ppsList[i++] = psDest + 1;
             *psDest = 0;
         }
         psDest++;
     }
 
     ppsList[i] = NULL;
 
     return ppsList;
}

}
