/**
 *
 * Beryl gconf settings backend
 *
 * gconf.c
 *
 * Copyright (c) 2006 Robert Carr <racarr@beryl-project.org>
 * Copyright (c) 2007 Dennis Kasprzyk <onestone@beryl-project.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 **/



#define _GNU_SOURCE
#include <stdlib.h>
#include <string.h>
#include <malloc.h>
#include <libintl.h>
#include <string.h>

#include <beryl-settings-backend.h>

#include <gconf/gconf.h>
#include <gconf/gconf-client.h>
#include <gconf/gconf-value.h>

#define METACITY   "/apps/metacity"
#define BERYL	   "/apps/beryl"

#define GROUPKEY    gchar * groupname=setting->parent->name?setting->parent->name:"general"; \
                    gchar * keyname=g_strconcat(setting->is_screen?"s_":"",setting->name,NULL) \

#define PATH        gchar * pathname=g_strconcat(BERYL, "/", currentProfile,\
                            setting->parent->name?"/plugins/":"/",groupname,"/",keyname,NULL)



GConfClient *client = NULL;

guint berylNotifyId = 0;
guint gnomeNotifyId = 0;

typedef enum {
	OptionInt,
	OptionBool,
	OptionKey,
	OptionString,
	OptionSpecial,
} SpecialOptionType;

struct _SpecialOption {
	const char* berylName;
	const char* berylGroup;
	Bool		screen;
	const char* gnomeName;
	SpecialOptionType type;
} const specialOptions[] = {
	{"run", "general", FALSE, METACITY "/global_keybindings/panel_run_dialog", OptionKey},
	{"main_menu", "general", FALSE, METACITY "/global_keybindings/panel_main_menu", OptionKey},
	{"window_menu", "general", FALSE, METACITY "/window_keybindings/activate_window_menu", OptionKey},
	{"run_command_screenshot", "general", FALSE, METACITY "/global_keybindings/run_command_screenshot", OptionKey},

	{"toggle_window_maximized", "general", FALSE, METACITY "/window_keybindings/toggle_maximized", OptionKey},
	{"minimize_window", "general", FALSE, METACITY "/window_keybindings/minimize", OptionKey},
	{"maximize_window", "general", FALSE, METACITY "/window_keybindings/maximize", OptionKey},
	{"unmaximize_window", "general", FALSE, METACITY "/window_keybindings/unmaximize", OptionKey},
	{"toggle_window_maximized_horizontally", "general", FALSE, METACITY "/window_keybindings/maximize_horizontally", OptionKey},
	{"toggle_window_maximized_vertically", "general", FALSE, METACITY "/window_keybindings/maximize_vertically", OptionKey},
	{"raise_window", "general", FALSE, METACITY "/window_keybindings/raise", OptionKey},
	{"lower_window", "general", FALSE, METACITY "/window_keybindings/lower", OptionKey},
	{"close_window", "general", FALSE, METACITY "/window_keybindings/close", OptionKey},
	{"toggle_window_shaded", "general", FALSE, METACITY "/window_keybindings/toggle_shaded", OptionKey},

	{"show_desktop", "general", FALSE, METACITY "/global_keybindings/show_desktop", OptionKey},

	{"initiate", "move", FALSE, METACITY "/window_keybindings/begin_move", OptionKey},
	{"initiate", "resize", FALSE, METACITY "/window_keybindings/begin_resize", OptionKey},

	{"next", "switcher", FALSE, METACITY "/global_keybindings/switch_windows", OptionKey},
	{"prev", "switcher", FALSE, METACITY "/global_keybindings/switch_windows_backward", OptionKey},

	{"command0", "general", FALSE, METACITY "/keybinding_commands/command_1", OptionString},
	{"command1", "general", FALSE, METACITY "/keybinding_commands/command_2", OptionString},
	{"command2", "general", FALSE, METACITY "/keybinding_commands/command_3", OptionString},
	{"command3", "general", FALSE, METACITY "/keybinding_commands/command_4", OptionString},
	{"command4", "general", FALSE, METACITY "/keybinding_commands/command_5", OptionString},
	{"command5", "general", FALSE, METACITY "/keybinding_commands/command_6", OptionString},
	{"command6", "general", FALSE, METACITY "/keybinding_commands/command_7", OptionString},
	{"command7", "general", FALSE, METACITY "/keybinding_commands/command_8", OptionString},
	{"command8", "general", FALSE, METACITY "/keybinding_commands/command_9", OptionString},
	{"command9", "general", FALSE, METACITY "/keybinding_commands/command_10", OptionString},
	{"command10", "general", FALSE, METACITY "/keybinding_commands/command_11", OptionString},
	{"command11", "general", FALSE, METACITY "/keybinding_commands/command_12", OptionString},

	{"run_command0", "general", FALSE, METACITY "/global_keybindings/run_command_1", OptionKey},
	{"run_command1", "general", FALSE, METACITY "/global_keybindings/run_command_2", OptionKey},
	{"run_command2", "general", FALSE, METACITY "/global_keybindings/run_command_3", OptionKey},
	{"run_command3", "general", FALSE, METACITY "/global_keybindings/run_command_4", OptionKey},
	{"run_command4", "general", FALSE, METACITY "/global_keybindings/run_command_5", OptionKey},
	{"run_command5", "general", FALSE, METACITY "/global_keybindings/run_command_6", OptionKey},
	{"run_command6", "general", FALSE, METACITY "/global_keybindings/run_command_7", OptionKey},
	{"run_command7", "general", FALSE, METACITY "/global_keybindings/run_command_8", OptionKey},
	{"run_command8", "general", FALSE, METACITY "/global_keybindings/run_command_9", OptionKey},
	{"run_command9", "general", FALSE, METACITY "/global_keybindings/run_command_10", OptionKey},
	{"run_command10", "general", FALSE, METACITY "/global_keybindings/run_command_11", OptionKey},
	{"run_command11", "general", FALSE, METACITY "/global_keybindings/run_command_12", OptionKey},

	{"rotate_to_1", "rotate", FALSE, METACITY "/global_keybindings/switch_to_workspace_1", OptionKey},
	{"rotate_to_2", "rotate", FALSE, METACITY "/global_keybindings/switch_to_workspace_2", OptionKey},
	{"rotate_to_3", "rotate", FALSE, METACITY "/global_keybindings/switch_to_workspace_3", OptionKey},
	{"rotate_to_4", "rotate", FALSE, METACITY "/global_keybindings/switch_to_workspace_4", OptionKey},
	{"rotate_to_5", "rotate", FALSE, METACITY "/global_keybindings/switch_to_workspace_5", OptionKey},
	{"rotate_to_6", "rotate", FALSE, METACITY "/global_keybindings/switch_to_workspace_6", OptionKey},
	{"rotate_to_7", "rotate", FALSE, METACITY "/global_keybindings/switch_to_workspace_7", OptionKey},
	{"rotate_to_8", "rotate", FALSE, METACITY "/global_keybindings/switch_to_workspace_8", OptionKey},
	{"rotate_to_9", "rotate", FALSE, METACITY "/global_keybindings/switch_to_workspace_9", OptionKey},
	{"rotate_to_10", "rotate", FALSE, METACITY "/global_keybindings/switch_to_workspace_10", OptionKey},
	{"rotate_to_11", "rotate", FALSE, METACITY "/global_keybindings/switch_to_workspace_11", OptionKey},
	{"rotate_to_12", "rotate", FALSE, METACITY "/global_keybindings/switch_to_workspace_12", OptionKey},

	{"rotate_left", "rotate", FALSE, METACITY "/global_keybindings/switch_to_workspace_left", OptionKey},
	{"rotate_right", "rotate", FALSE, METACITY "/global_keybindings/switch_to_workspace_right", OptionKey},

	{"plane_to_1", "plane", FALSE, METACITY "/global_keybindings/switch_to_workspace_1", OptionKey},
	{"plane_to_2", "plane", FALSE, METACITY "/global_keybindings/switch_to_workspace_2", OptionKey},
	{"plane_to_3", "plane", FALSE, METACITY "/global_keybindings/switch_to_workspace_3", OptionKey},
	{"plane_to_4", "plane", FALSE, METACITY "/global_keybindings/switch_to_workspace_4", OptionKey},
	{"plane_to_5", "plane", FALSE, METACITY "/global_keybindings/switch_to_workspace_5", OptionKey},
	{"plane_to_6", "plane", FALSE, METACITY "/global_keybindings/switch_to_workspace_6", OptionKey},
	{"plane_to_7", "plane", FALSE, METACITY "/global_keybindings/switch_to_workspace_7", OptionKey},
	{"plane_to_8", "plane", FALSE, METACITY "/global_keybindings/switch_to_workspace_8", OptionKey},
	{"plane_to_9", "plane", FALSE, METACITY "/global_keybindings/switch_to_workspace_9", OptionKey},
	{"plane_to_10", "plane", FALSE, METACITY "/global_keybindings/switch_to_workspace_10", OptionKey},
	{"plane_to_11", "plane", FALSE, METACITY "/global_keybindings/switch_to_workspace_11", OptionKey},
	{"plane_to_12", "plane", FALSE, METACITY "/global_keybindings/switch_to_workspace_12", OptionKey},

	{"plane_up", "plane", FALSE, METACITY "/global_keybindings/switch_to_workspace_up", OptionKey},
	{"plane_down", "plane", FALSE, METACITY "/global_keybindings/switch_to_workspace_down", OptionKey},
	{"plane_left", "plane", FALSE, METACITY "/global_keybindings/switch_to_workspace_left", OptionKey},
	{"plane_right", "plane", FALSE, METACITY "/global_keybindings/switch_to_workspace_right", OptionKey},

	{"rotate_to_1_window", "rotate", FALSE, METACITY "/window_keybindings/move_to_workspace_1", OptionKey},
	{"rotate_to_2_window", "rotate", FALSE, METACITY "/window_keybindings/move_to_workspace_2", OptionKey},
	{"rotate_to_3_window", "rotate", FALSE, METACITY "/window_keybindings/move_to_workspace_3", OptionKey},
	{"rotate_to_4_window", "rotate", FALSE, METACITY "/window_keybindings/move_to_workspace_4", OptionKey},
	{"rotate_to_5_window", "rotate", FALSE, METACITY "/window_keybindings/move_to_workspace_5", OptionKey},
	{"rotate_to_6_window", "rotate", FALSE, METACITY "/window_keybindings/move_to_workspace_6", OptionKey},
	{"rotate_to_7_window", "rotate", FALSE, METACITY "/window_keybindings/move_to_workspace_7", OptionKey},
	{"rotate_to_8_window", "rotate", FALSE, METACITY "/window_keybindings/move_to_workspace_8", OptionKey},
	{"rotate_to_9_window", "rotate", FALSE, METACITY "/window_keybindings/move_to_workspace_9", OptionKey},
	{"rotate_to_10_window", "rotate", FALSE, METACITY "/window_keybindings/move_to_workspace_10", OptionKey},
	{"rotate_to_11_window", "rotate", FALSE, METACITY "/window_keybindings/move_to_workspace_11", OptionKey},
	{"rotate_to_12_window", "rotate", FALSE, METACITY "/window_keybindings/move_to_workspace_12", OptionKey},

	{"rotate_left_window", "rotate", FALSE, METACITY "/window_keybindings/move_to_workspace_left", OptionKey},
	{"rotate_right_window", "rotate", FALSE, METACITY "/window_keybindings/move_to_workspace_right", OptionKey},

	{"command_screenshot", "general", FALSE, METACITY "/keybinding_commands/command_screenshot", OptionString},
	{"command_window_screenshot", "general", FALSE, METACITY "/keybinding_commands/command_window_screenshot", OptionString},

	{"autoraise", "general", FALSE, METACITY "/general/auto_raise", OptionBool},
	{"autoraise_delay", "general", FALSE, METACITY "/general/auto_raise_delay", OptionInt},
	{"raise_on_click", "general", FALSE, METACITY "/general/raise_on_click", OptionBool},
	{"click_to_focus", "general", FALSE, METACITY "/general/focus_mode", OptionSpecial},
	{"fsp_level", "general", FALSE, METACITY "/general/focus_new_windows", OptionSpecial},
	
	{"audible_bell", "general", FALSE, METACITY "/general/audible_bell", OptionBool},
	{"size", "general", TRUE, METACITY "/general/num_workspaces", OptionInt},
};

#define N_SOPTIONS (sizeof (specialOptions) / sizeof (struct _SpecialOption))

/* Bindings */

static gchar *currentProfile = NULL;

struct _Modifier
{
	gchar *name;
	gint modifier;
}
const static modifiers[] = {
	{"<Shift>", ShiftMask},
	{"<Control>", ControlMask},
	{"<Mod1>", Mod1Mask},
	{"<Mod2>", Mod2Mask},
	{"<Mod3>", Mod3Mask},
	{"<Mod4>", Mod4Mask},
	{"<Mod5>", Mod5Mask},
	{"<Alt>", CompAltMask},
	{"<Meta>", CompMetaMask},
	{"<Super>", CompSuperMask},
	{"<Hyper>", CompHyperMask},
	{"<ModeSwitch>", CompModeSwitchMask},
};

#define N_MODIFIERS (sizeof (modifiers) / sizeof (struct _Modifier))

static Bool stringToBSColor(const char *color, BerylSettingColorValue *rgba)
{
	int c[4];

	if (sscanf(color, "#%2x%2x%2x%2x", &c[0], &c[1], &c[2], &c[3]) == 4)
	{
			rgba->array.array[0] = c[0] << 8 | c[0];
			rgba->array.array[1] = c[1] << 8 | c[1];
			rgba->array.array[2] = c[2] << 8 | c[2];
			rgba->array.array[3] = c[3] << 8 | c[3];

			return TRUE;
	}

	return FALSE;
}

static gchar *BSColorToString(BerylSettingColorValue *rgba)
{
	char tmp[256];

	snprintf(tmp, 256, "#%.2x%.2x%.2x%.2x", rgba->array.array[0] / 256,
			 rgba->array.array[1] / 256, rgba->array.array[2] / 256,
			 rgba->array.array[3] / 256);

	return strdup(tmp);
}


static gchar *mods_to_string(unsigned int mods)
{
	gchar *retstr = g_strdup("");
	gchar *tmpstr = retstr;
	gint i;

	for (i = 0; i < N_MODIFIERS; i++)
	{
		if (mods & modifiers[i].modifier)
		{
			retstr = g_strconcat(retstr, modifiers[i].name, NULL);
			g_free(tmpstr);
			tmpstr = retstr;
		}
	}
	return retstr;
}

static gboolean get_binding_is_enabled(gchar * src)
{
    if (!strlen(src) || strcasecmp(src,"disabled")==0)
        return FALSE;
    if (src[0]=='#')
        return FALSE;
    return TRUE;
}


static unsigned int get_mods_and_endptr(gchar * src, gchar ** ret)
{
    unsigned int mods=0;
    gchar * spos=src;
    while((spos=strchr(spos,'<')) && *src)
    {
        int i;
        for (i=0;i<N_MODIFIERS;i++)
            if (strncasecmp(modifiers[i].name,
                        spos,strlen(modifiers[i].name))==0)
            {
                mods|=modifiers[i].modifier;
                spos+=strlen(modifiers[i].name);
                src=spos;
                break;
            }
        if (i==N_MODIFIERS)
            break;
    }
    *ret=src;
    return mods;
}

static void set_key_binding_from_string(BerylSettingValue * value, gchar * src)
{
    gboolean enabled=get_binding_is_enabled(src);
	beryl_setting_value_set_key_enabled(value,&enabled);

	if (!enabled)
		return;

    int keysym=0;
    int keymods=0;
    gchar * spos;
    keymods=get_mods_and_endptr(src,&spos);
    if (spos && *spos)
    {
        keysym=XStringToKeysym(spos);
    };
    beryl_setting_value_set_keysym(value,&keysym);
    beryl_setting_value_set_keymods(value,&keymods);
}

static void set_button_binding_from_string(BerylSettingValue * value, gchar * src)
{
    gboolean enabled=get_binding_is_enabled(src);
	beryl_setting_value_set_button_enabled(value,&enabled);

	if (!enabled)
		return;

    int button=0;
    int buttonmods=0;
    gchar * spos;
    buttonmods=get_mods_and_endptr(src,&spos);
    if (spos && *spos)
    {
        spos=strcasestr(spos,"Button");
        if (spos && *spos)
        {
            spos+=strlen("Button");
            button=atoi(spos);
        }
    }
    beryl_setting_value_set_button(value,&button);
    beryl_setting_value_set_buttonmods(value,&buttonmods);
}


/* Reading */

static void readBool(BerylSetting * setting)
{
	GROUPKEY;
	PATH;
	GError *e = NULL;
	gboolean val = gconf_client_get_bool(client, pathname, &e);

	if (!e)
		beryl_setting_value_set_bool(&setting->value, &val);
	else
		g_error_free(e);

	g_free(keyname);
	g_free(pathname);
}

static void readInt(BerylSetting * setting)
{
	GROUPKEY;
	PATH;
	GError *e = NULL;
	int val = gconf_client_get_int(client, pathname, &e);

	if (!e)
		beryl_setting_value_set_int(&setting->value, &val);
	else
		g_error_free(e);

	g_free(keyname);
	g_free(pathname);
}

static void readFloat(BerylSetting * setting)
{
	GROUPKEY;
	PATH;
	GError *e = NULL;
	gdouble val = gconf_client_get_float(client, pathname, &e);

	if (!e)
		beryl_setting_value_set_float(&setting->value, &val);
	else
		g_error_free(e);

	g_free(keyname);
	g_free(pathname);
}

static void readString(BerylSetting * setting)
{
	GROUPKEY;
	PATH;
	GError *e = NULL;
	gchar *val = gconf_client_get_string(client, pathname, &e);

	if (!e && val)
	{
		setting->value.value.as_string = val;
		setting->is_default = FALSE;
	}

	if (e)
		g_error_free(e);

	g_free(keyname);
	g_free(pathname);
}

static void readColor(BerylSetting * setting)
{
	GROUPKEY;
	PATH;
	GError *e = NULL;
	gchar *val = gconf_client_get_string(client, pathname, &e);

	BerylSettingColorValue col;

	if (!e && val && stringToBSColor(val,&col))
	{
		beryl_setting_value_set_color(&setting->value, &col);
		setting->is_default = FALSE;
	}

	if (val)
		g_free(val);
	if (e)
		g_error_free(e);

	g_free(keyname);
	g_free(pathname);

}


static void readBinding(BerylSetting * setting)
{
	GROUPKEY;
	PATH;
	setting->is_default=TRUE;
	gchar *bindingpath;
	gchar *str;
	int intv;
	gboolean boolv;
	GError *e = NULL;

	bindingpath = g_strconcat(pathname, "_keyboard", NULL);
	str = gconf_client_get_string(client, bindingpath, &e);

	if (!e && str)
	{
		setting->is_default = FALSE;
		set_key_binding_from_string(&setting->value,str);
		g_free(str);
	}
	if (e)
	{
		g_error_free(e);
		e=NULL;
	}

	g_free(bindingpath);

	bindingpath = g_strconcat(pathname, "_mouse", NULL);
	str = gconf_client_get_string(client, bindingpath, &e);

	if (!e && str)
	{
		setting->is_default=FALSE;
		set_button_binding_from_string(&setting->value,str);
		g_free(str);
	}
	if (e)
	{
		g_error_free(e);
		e=NULL;
	}
	g_free(bindingpath);

	bindingpath = g_strconcat(pathname, "_edge", NULL);
	intv = gconf_client_get_int(client, bindingpath, &e);
	if (!e)
	{
		beryl_setting_value_set_edgemask(&setting->value,&intv);
	}
	g_free(bindingpath);

	if (e)
	{
		g_error_free(e);
		e = NULL;
	}

	bindingpath = g_strconcat(pathname, "_bell", NULL);
	boolv = gconf_client_get_bool(client, bindingpath, &e);
	if (!e)
	{
		setting->is_default=FALSE;
		beryl_setting_value_set_bell(&setting->value,&boolv);
	}
	if (e)
	{
		g_error_free(e);
		e = NULL;
	}
	g_free(bindingpath);
	g_free(pathname);
	g_free(keyname);
}

static void readList(BerylSetting * setting)
{
	GROUPKEY;
	PATH;
	switch (setting->info.for_list.list_of_type)
	{
		case BERYL_SETTING_TYPE_STRING:
		{
			GSList *settingList = gconf_client_get_list(client, pathname, GCONF_VALUE_STRING, NULL);
			gchar * val;

			beryl_setting_list_clear(setting);

			while (settingList && settingList->data)
			{
			    val = settingList->data;
		            BerylSettingValue * value=beryl_setting_list_append(setting);
                            value->value.as_string=g_strdup(val);
                            value->parent->is_default=FALSE;

			    settingList = g_slist_next(settingList);
			}
			break;
		}
		default:
			break;
	}
	g_free(pathname);
	g_free(keyname);
}

static void readSwitch(BerylSetting * setting)
{

	GROUPKEY;
	PATH;

	switch (setting->type)
	{
	case BERYL_SETTING_TYPE_BOOL:
		if (gconf_client_get(client, pathname, NULL))
		{
			setting->is_default = FALSE;
			readBool(setting);
		}
		break;
	case BERYL_SETTING_TYPE_INT:
		if (gconf_client_get(client, pathname, NULL))
		{
			setting->is_default = FALSE;
			readInt(setting);
		}
		break;
	case BERYL_SETTING_TYPE_FLOAT:
		if (gconf_client_get(client, pathname, NULL))
		{
			setting->is_default = FALSE;
			readFloat(setting);
		}
		break;
	case BERYL_SETTING_TYPE_STRING:
		if (gconf_client_get(client, pathname, NULL))
		{
			setting->is_default = FALSE;
			readString(setting);
		}
		break;
	case BERYL_SETTING_TYPE_COLOR:
		readColor(setting);
		break;
	case BERYL_SETTING_TYPE_BINDING:
		readBinding(setting);
		break;
	case BERYL_SETTING_TYPE_LIST:
		if (gconf_client_get(client, pathname, NULL))
		{
			setting->is_default = FALSE;
			readList(setting);
		}
		break;
	default:
		break;
	}


	g_free(keyname);
	g_free(pathname);
}


/* ------ */

/* Writing */


static void writeBool(BerylSetting * setting)
{
	GROUPKEY;
	PATH;
	gboolean val = setting->value.value.as_bool;

	gboolean cur = gconf_client_get_bool(client,pathname, NULL);

	if (!gconf_client_get_without_default(client, pathname, NULL) || val != cur)
	{
		gconf_client_set_bool(client, pathname, val, NULL);
	}

	g_free(keyname);
	g_free(pathname);
}

static void writeInt(BerylSetting * setting)
{
	GROUPKEY;
	PATH;
	int val = setting->value.value.as_int;

	int cur = gconf_client_get_int(client,pathname, NULL);

	if (!gconf_client_get_without_default(client, pathname, NULL) || val != cur)
	{
		gconf_client_set_int(client, pathname, val, NULL);
	}

	g_free(keyname);
	g_free(pathname);
}

static void writeFloat(BerylSetting * setting)
{
	GROUPKEY;
	PATH;
	float val = setting->value.value.as_float;

	float cur =	gconf_client_get_float(client,pathname, NULL);

	if (!gconf_client_get_without_default(client, pathname, NULL) || val != cur)
	{
		gconf_client_set_float(client, pathname, val, NULL);
	}

	g_free(keyname);
	g_free(pathname);
}

static void writeString(BerylSetting * setting)
{
	GROUPKEY;
	PATH;
	gchar *val = setting->value.value.as_string;

	gchar *cur = gconf_client_get_string(client,pathname, NULL);

	if (!gconf_client_get_without_default(client, pathname, NULL)
			|| !(cur && val && strcmp(val,cur) == 0))
	{
		gconf_client_set_string(client, pathname, val, NULL);
	}

	if (cur)
		g_free(cur);

	g_free(keyname);
	g_free(pathname);
}

static void writeColor(BerylSetting * setting)
{

	GROUPKEY;
	PATH;

	BerylSettingColorValue col;
	beryl_setting_value_get_color(&setting->value, &col);

	char *val = BSColorToString(&col);

	gchar *cur = gconf_client_get_string(client,pathname, NULL);

	if (!gconf_client_get_without_default(client, pathname, NULL)
			|| !(cur && val && strcmp(val,cur) == 0))
	{
		gconf_client_set_string(client, pathname, val, NULL);
	}

	if (cur)
		g_free(cur);

	free(val);
	g_free(keyname);
	g_free(pathname);
}

static void setKey(BerylSetting * setting)
{
	GROUPKEY;
	PATH;
	gchar *pre;
	gchar *path;
	gboolean bool_val;

	path = g_strconcat(pathname, "_keyboard", NULL);

	if (beryl_setting_value_get_key_enabled(&setting->value, &bool_val) &&
		bool_val)
	{
		gint ire;
		gchar * smod=g_strdup(""), *ksym="";

		if (beryl_setting_value_get_keymods(&setting->value,&ire))
			{
			g_free(smod);
					smod=mods_to_string(ire);
		}

		if (beryl_setting_value_get_keysym(&setting->value,&ire))
			ksym=XKeysymToString(ire);

		if (!ksym) ksym="None";

		pre = g_strconcat(smod, ksym, NULL);
		g_free(smod);

		gchar *cur = gconf_client_get_string(client,path, NULL);

		if (!gconf_client_get_without_default(client, path, NULL)
				   || !(cur && pre && strcmp(pre,cur) == 0))
		{
			gconf_client_set_string(client, path, pre, NULL);
		}

		if (cur)
			g_free(cur);

		g_free(pre);
	}
	else
	{
		gchar *cur = gconf_client_get_string(client,path, NULL);

		if (!gconf_client_get_without_default(client, path, NULL)
				|| !(cur && strcmp("disabled",cur) == 0))
		{
			gconf_client_set_string(client, path, "disabled", NULL);
		}

		if (cur)
			g_free(cur);
	}

	g_free(keyname);
	g_free(path);
	g_free(pathname);
}

static void setButton(BerylSetting * setting)
{
	GROUPKEY;
	PATH;
	gchar *path;
	gchar *pre;
	gboolean bool_val;

	path = g_strconcat(pathname, "_mouse", NULL);

	if (beryl_setting_value_get_button_enabled(&setting->value, &bool_val) &&
		bool_val)
	{
		gint ire;
		gchar *smod = g_strdup(""), *bnum = g_strdup("Any");

		if (beryl_setting_value_get_buttonmods(&setting->value, &ire))
		{
			g_free(smod);
			smod = mods_to_string(ire);
		}
		if (beryl_setting_value_get_button(&setting->value, &ire))
		{
			g_free(bnum);
			bnum = g_strdup_printf("Button%d", ire);
		}
		pre = g_strconcat(smod, bnum, NULL);
		g_free(smod);
		g_free(bnum);

		gchar *cur = gconf_client_get_string(client,path, NULL);

		if (!gconf_client_get_without_default(client, path, NULL)
				   || !(cur && pre && strcmp(pre,cur) == 0))
		{
			gconf_client_set_string(client, path, pre, NULL);
		}

		if (cur)
			g_free(cur);

		g_free(pre);
	}
	else
	{
		gchar *cur = gconf_client_get_string(client,path, NULL);

		if (!gconf_client_get_without_default(client, path, NULL)
				   || !(cur && strcmp("disabled",cur) == 0))
		{
			gconf_client_set_string(client, path, "disabled", NULL);
		}

		if (cur)
			g_free(cur);
	}


	g_free(path);
	g_free(keyname);
	g_free(pathname);
}

static void setEdgemask(BerylSetting * setting)
{
	GROUPKEY;
	PATH;
	gchar *path;
	int ema;

	if (beryl_setting_value_get_edgemask(&setting->value,&ema))
	{
		path = g_strconcat(pathname, "_edge", NULL);
		int cur =	gconf_client_get_int(client,path, NULL);

		if (!gconf_client_get_without_default(client, path, NULL)
				   || cur != ema)
		{
			gconf_client_set_int(client, path, ema, NULL);
		}
		g_free(path);
	}
	g_free(pathname);
	g_free(keyname);
}

static void setBell(BerylSetting * setting)
{
	GROUPKEY;
	PATH;
	gchar *path;
	gboolean bool_val;

	if (beryl_setting_value_get_bell(&setting->value,&bool_val))
	{
		path = g_strconcat(pathname, "_bell", NULL);

		gboolean cur =	gconf_client_get_bool(client,path, NULL);

		if (!gconf_client_get_without_default(client, path, NULL)
				   || cur != bool_val)
		{
			gconf_client_set_bool(client, path, bool_val, NULL);
		}
		g_free(path);
	}
	g_free(pathname);
	g_free(keyname);

}

static void writeBinding(BerylSetting * setting)
{
	gboolean bool_val;

	if (beryl_setting_get_can_set_key(setting, &bool_val) && bool_val)
	{
		setKey(setting);
	}
	if (beryl_setting_get_can_set_button(setting, &bool_val) && bool_val)
	{
		setButton(setting);
	}
        if (beryl_setting_get_can_set_edgemask(setting,&bool_val) && bool_val)
	{
		setEdgemask(setting);
	}
     	if (beryl_setting_get_can_set_bell(setting,&bool_val) && bool_val)
	{
		setBell(setting);
	}
}

static void copy_string(BerylSettingValue * value, GSList ** list)
{
	*list = g_slist_append(*list,value->value.as_string);
}

static void copy_bool(BerylSettingValue * value, GSList ** list)
{
	*list = g_slist_append(*list,&value->value.as_bool);
}

static void copy_int(BerylSettingValue * value, GSList ** list)
{
	*list = g_slist_append(*list,&value->value.as_int);
}

static void copy_float(BerylSettingValue * value, GSList ** list)
{
	*list = g_slist_append(*list,&value->value.as_float);
}

static Bool compare_string_list(GSList * list1, GSList * list2)
{
	GSList *l1 = list1;
	GSList *l2 = list2;
	for (;l1 && l2; l1 = l1->next, l2 = l2->next)
	{
		if ((l1->data && !l2->data) || (!l1->data && l2->data))
			return FALSE;
		if (l1->data && l2->data && strcmp(l1->data,l2->data) != 0)
			return FALSE;
	}

	if ((l1 && !l2) || (!l1 && l2))
		return FALSE;

	return TRUE;
}

static void writeList(BerylSetting * setting)
{
	GROUPKEY;
	PATH;
	GSList *list = NULL;

	switch(setting->info.for_list.list_of_type)
	{
		/* currently we only support string lists
		case BERYL_SETTING_TYPE_BOOL:
		{
			g_slist_foreach(setting->value.value.as_list,(GFunc)copy_bool,&list);
			gconf_client_set_list(client, pathname,GCONF_VALUE_BOOL,list,NULL);
		}
		break;
		case BERYL_SETTING_TYPE_INT:
		{
			g_slist_foreach(setting->value.value.as_list,(GFunc)copy_int,&list);
			gconf_client_set_list(client, pathname,GCONF_VALUE_INT,list,NULL);
		}
		break;
		case BERYL_SETTING_TYPE_FLOAT:
		{
			g_slist_foreach(setting->value.value.as_list,(GFunc)copy_float,&list);
			gconf_client_set_list(client, pathname,GCONF_VALUE_FLOAT,list,NULL);
		}
		break;
		*/
		case BERYL_SETTING_TYPE_STRING:
		{
			GSList *cur = gconf_client_get_list(client, pathname,
					GCONF_VALUE_STRING, NULL);
			g_slist_foreach(setting->value.value.as_list,(GFunc)copy_string,&list);
			if (!compare_string_list(cur,list))
				gconf_client_set_list(client, pathname,
									  GCONF_VALUE_STRING, list, NULL);
			g_slist_free(cur);
		}
		break;
		default:
			break;
	}

	g_slist_free(list);
	g_free(keyname);
	g_free(pathname);

}

static void writeSwitch(BerylSetting * setting)
{
	switch (setting->type)
	{
	case BERYL_SETTING_TYPE_BOOL:
		writeBool(setting);
		break;
	case BERYL_SETTING_TYPE_INT:
		writeInt(setting);
		break;
	case BERYL_SETTING_TYPE_FLOAT:
		writeFloat(setting);
		break;
	case BERYL_SETTING_TYPE_STRING:
		writeString(setting);
		break;
	case BERYL_SETTING_TYPE_COLOR:
		writeColor(setting);
		break;
	case BERYL_SETTING_TYPE_BINDING:
		writeBinding(setting);
		break;
	case BERYL_SETTING_TYPE_LIST:
		writeList(setting);
		break;
	default:
		break;
	}

}

/* Integration */

static void gnomeIntToBeryl (BerylSetting * setting, int num)
{
	GError *e = NULL;
	int val = gconf_client_get_int(client, specialOptions[num].gnomeName, &e);

	if (!e)
		beryl_setting_value_set_int(&setting->value, &val);
	else
		g_error_free(e);
}

static void gnomeBoolToBeryl (BerylSetting * setting, int num)
{
	GError *e = NULL;
	gboolean val =
			gconf_client_get_bool(client, specialOptions[num].gnomeName, &e);

	if (!e)
		beryl_setting_value_set_bool(&setting->value, &val);
	else
		g_error_free(e);
}

static void gnomeStringToBeryl (BerylSetting * setting, int num)
{
	GError *e = NULL;
	gchar *val =
			gconf_client_get_string(client, specialOptions[num].gnomeName, &e);

	if (!e && val)
	{
		setting->value.value.as_string = val;
	}

	if (e)
		g_error_free(e);

}

static void gnomeKeyToBeryl (BerylSetting * setting, int num)
{
	GROUPKEY;
	PATH;
	gchar *bindingpath;
	gchar *str;
	int intv;
	gboolean boolv;
	GError *e = NULL;

	str = gconf_client_get_string(client, specialOptions[num].gnomeName, &e);

	if (!e && str)
	{
		set_key_binding_from_string(&setting->value,str);
		g_free(str);
	}
	if (e)
	{
		g_error_free(e);
		e = NULL;
	}

	bindingpath = g_strconcat(pathname, "_mouse", NULL);
	str = gconf_client_get_string(client, bindingpath, &e);

	if (!e && str)
	{
		set_button_binding_from_string(&setting->value,str);
		g_free(str);
	}
	if (e)
	{
		g_error_free(e);
		e = NULL;
	}
	g_free(bindingpath);

	bindingpath = g_strconcat(pathname, "_edge", NULL);
	intv = gconf_client_get_int(client, bindingpath, &e);
	if (!e)
	{
		beryl_setting_value_set_edgemask(&setting->value,&intv);
	}
	g_free(bindingpath);

	if (e)
	{
		g_error_free(e);
		e = NULL;
	}

	bindingpath = g_strconcat(pathname, "_bell", NULL);
	boolv = gconf_client_get_bool(client, bindingpath, &e);
	if (!e)
	{
		beryl_setting_value_set_bell(&setting->value,&boolv);
	}
	if (e)
	{
		g_error_free(e);
		e = NULL;
	}
	g_free(bindingpath);
	g_free(pathname);
	g_free(keyname);
}


static void read_integrated_option(BerylSetting * setting)
{
	GROUPKEY;

	int num = -1;
	unsigned int i;
	for (i = 0; i < N_SOPTIONS; i++)
	{
		if (strcmp(specialOptions[i].berylName,keyname) == 0 &&
			strcmp(specialOptions[i].berylGroup,groupname) == 0 &&
		    specialOptions[i].screen == setting->is_screen)
		{
			num = i;
			break;
		}
	}

	if (i < 0)
		return;

	setting->is_default=FALSE;

	switch (specialOptions[num].type)
	{
		case OptionInt:
			gnomeIntToBeryl (setting, num);
			break;
		case OptionBool:
			gnomeBoolToBeryl (setting, num);
			break;
		case OptionKey:
			gnomeKeyToBeryl (setting, num);
			break;
		case OptionString:
			gnomeStringToBeryl (setting, num);
			break;
		case OptionSpecial:
			if (strcmp(specialOptions[num].berylName, "fsp_level") == 0)
			{
				GError *e = NULL;
				gchar *fsp = gconf_client_get_string(client, 
						specialOptions[num].gnomeName, &e);

				if (e)
					g_error_free(e);

				if (fsp)
				{
					gchar *fsp_level;

					if (strcmp(fsp, "strict") == 0)
						fsp_level = "Normal";
					else
						fsp_level = "None";

					setting->value.value.as_string = g_strdup(fsp_level);
					g_free(fsp);
				}
			} 
			else if (strcmp(specialOptions[num].berylName, "click_to_focus") == 0)
			{
				GError *e = NULL;
				gchar *focus_mode = gconf_client_get_string(client, 
						specialOptions[num].gnomeName, &e);

				if (e)
					g_error_free(e);

				if (focus_mode)
				{
					gboolean click_to_focus = (strcmp(focus_mode, "click") == 0);
					setting->value.value.as_bool = click_to_focus;
					g_free(focus_mode);
				}
			}
			break;
		default:
			break;
	}
	g_free(keyname);
}

static void berylIntToGnome (BerylSetting * setting, int num)
{
	int val = setting->value.value.as_int;
	GError *e = NULL;
	int cur =
			gconf_client_get_int(client, specialOptions[num].gnomeName, &e);

	if (!e && cur != val)
	{
		gconf_client_set_int(client, specialOptions[num].gnomeName, val, NULL);
	}
}

static void berylBoolToGnome (BerylSetting * setting, int num)
{
	gboolean val = setting->value.value.as_bool;
	GError *e = NULL;
	gboolean cur =
			gconf_client_get_bool(client, specialOptions[num].gnomeName, &e);

	if (!e && cur != val)
	{
		gconf_client_set_bool(client,
								 specialOptions[num].gnomeName, val, NULL);
	}
}

static void berylStringToGnome (BerylSetting * setting, int num)
{
	gchar *val = setting->value.value.as_string;
	GError *e = NULL;
	gchar * cur = gconf_client_get_string(client, specialOptions[num].gnomeName, &e);

	if (!e && val && cur && strcmp(cur,val) != 0)
	{
		gconf_client_set_string(client,
								specialOptions[num].gnomeName, val, NULL);
	}

	if (cur)
		g_free(cur);

	if (e)
		g_error_free(e);
}

static void berylKeyToGnome (BerylSetting * setting, int num)
{
	gboolean bool_val;

	if (beryl_setting_get_can_set_key(setting, &bool_val) && bool_val)
	{

		gchar *pre;
		gboolean bool_val;

		if (beryl_setting_value_get_key_enabled(&setting->value, &bool_val) &&
			bool_val)
		{
			gint ire;
			gchar * smod=g_strdup(""), *ksym="";

			if (beryl_setting_value_get_keymods(&setting->value,&ire))
				{
				g_free(smod);
						smod=mods_to_string(ire);
			}

			if (beryl_setting_value_get_keysym(&setting->value,&ire))
				ksym=XKeysymToString(ire);

			if (!ksym) ksym="None";

			pre = g_strconcat(smod, ksym, NULL);
			g_free(smod);

			GError *e = NULL;
			gchar * cur = gconf_client_get_string(client,
					specialOptions[num].gnomeName, &e);

			if (!e && pre && cur && strcmp(cur,pre) != 0)
			{
				gconf_client_set_string(client,	specialOptions[num].gnomeName,
										pre, NULL);
			}

			if (cur)
				g_free(cur);

			if (e)
				g_error_free(e);

			g_free(pre);
		}
		else
		{
			GError *e = NULL;
			gchar * cur = gconf_client_get_string(client,
					specialOptions[num].gnomeName, &e);

			if (!e && cur && strcmp(cur,"disabled") != 0)
			{
				gconf_client_set_string(client,	specialOptions[num].gnomeName,
										"disabled", NULL);
			}

			if (cur)
				g_free(cur);

			if (e)
				g_error_free(e);
		}
	}
	if (beryl_setting_get_can_set_button(setting, &bool_val) && bool_val)
	{
		setButton(setting);
	}
        if (beryl_setting_get_can_set_edgemask(setting,&bool_val) && bool_val)
	{
		setEdgemask(setting);
	}
     	if (beryl_setting_get_can_set_bell(setting,&bool_val) && bool_val)
	{
		setBell(setting);
	}
}

static void write_integrated_option(BerylSetting * setting)
{
	GROUPKEY;

	int num = -1;
	unsigned int i;
	for (i = 0; i < N_SOPTIONS; i++)
	{
		if (strcmp(specialOptions[i].berylName,keyname) == 0 &&
			strcmp(specialOptions[i].berylGroup,groupname) == 0 &&
		    specialOptions[i].screen == setting->is_screen)
		{
			num = i;
			break;
		}
	}

	if (i < 0)
		return;

	setting->is_default=FALSE;

	switch (specialOptions[num].type)
	{
		case OptionInt:
			berylIntToGnome (setting, num);
			break;
		case OptionBool:
			berylBoolToGnome (setting, num);
			break;
		case OptionKey:
			berylKeyToGnome (setting, num);
			break;
		case OptionString:
			berylStringToGnome (setting, num);
		case OptionSpecial:
			if (strcmp(specialOptions[num].berylName, "click_to_focus") == 0)
			{
				gchar *val = setting->value.value.as_bool ? "click" : "mouse";
				GError *e = NULL;
				gchar * cur = gconf_client_get_string(client, specialOptions[num].gnomeName, &e);

				if (!e && val && cur && strcmp(cur,val) != 0)
				{
					gconf_client_set_string(client,
								specialOptions[num].gnomeName, val, NULL);
				}
				if (cur)
					g_free(cur);
				if (e)
					g_error_free(e);
			}
			else if (strcmp(specialOptions[num].berylName, "fsp_level") == 0)
			{
				gchar *val;
				GError *e = NULL;
				gchar * cur = gconf_client_get_string(client, specialOptions[num].gnomeName, &e);

				if ((strcmp(setting->value.value.as_string, "None") == 0) ||
					(strcmp(setting->value.value.as_string, "Low") == 0))
				{
					val = "smart";
				}
				else
				{
					val = "strict";
				}

				if (!e && val && cur && strcmp(cur,val) != 0)
				{
					gconf_client_set_string(client,
								specialOptions[num].gnomeName, val, NULL);
				}

				if (cur)
					g_free(cur);
				if (e)
					g_error_free(e);
			}
		default:
			break;
	}
	g_free(keyname);
}

void static valueChanged(GConfClient *client, guint cnxn_id, GConfEntry *entry,
					gpointer user_data)
{
	BerylSettingsContext *c = (BerylSettingsContext *)user_data;
	gchar **path = g_strsplit(gconf_entry_get_key(entry),"/",-1);
	int i = 0;
	for (i = 0;path[i];i++);
	if (i >= 4)
	{
		BerylSettingsPlugin * plugin = NULL;

		if (strcmp(path[i-2],"general") == 0)
		{
			if (strcmp(currentProfile,path[i-3]) != 0)
				return;
			plugin = beryl_settings_context_find_plugin(c,NULL);
		}
		else
		{
			if (strcmp(currentProfile,path[i-4]) != 0)
				return;
			plugin = beryl_settings_context_find_plugin(c,path[i-2]);
		}
		if (!plugin)
			return;

		BerylSetting *setting = NULL;
		if (path[i-1][0] == 's' && path[i-1][1] == '_')
			setting = beryl_settings_plugin_find_setting( plugin,
				path[i-1] + 2, TRUE);
		else
			setting = beryl_settings_plugin_find_setting( plugin,
				path[i-1] , FALSE);

		if (!setting)
			return;

		if (beryl_settings_context_get_de_integration_enabled(c)
		   && !get_setting_is_integrated(setting))
		{
			beryl_setting_changed(setting);
		}
	}
	g_strfreev(path);

}

void static gnomeValueChanged(GConfClient *client, guint cnxn_id, GConfEntry *entry,
					gpointer user_data)
{
	BerylSettingsContext *c = (BerylSettingsContext *)user_data;
	int i,num = -1;
	for (i = 0; i < N_SOPTIONS; i++)
	{
		if (strcmp(specialOptions[i].gnomeName,gconf_entry_get_key(entry)) == 0)
		{
			num = i;
			break;
		}
	}

	if (num < 0)
		return;

	BerylSettingsPlugin * plugin = NULL;

	if (strcmp(specialOptions[num].berylGroup, "general") == 0)
	{
		plugin = beryl_settings_context_find_plugin(c, NULL);
	}
	else
	{
		gchar *group = g_strdup(specialOptions[num].berylGroup);
		plugin = beryl_settings_context_find_plugin(c, group);
		g_free(group);
	}

	if (!plugin)
		return;

	gchar *name = g_strdup(specialOptions[num].berylName);
	BerylSetting *setting = beryl_settings_plugin_find_setting( plugin,
			name, specialOptions[num].screen);
	g_free(name);

	if (!setting)
		return;

	beryl_setting_changed(setting);
}

/* ------ */


/* Beryl Settings API */

gboolean backend_init(BerylSettingsContext * context)
{
	g_type_init();

	client = gconf_client_get_default();

	berylNotifyId = gconf_client_notify_add(client, BERYL, valueChanged,
											context, NULL, NULL);

	if (beryl_settings_context_get_de_integration_enabled(context))
	{
		gnomeNotifyId = gconf_client_notify_add(client, METACITY,
				gnomeValueChanged, context, NULL,NULL);
	}

	gconf_client_add_dir(client, BERYL, GCONF_CLIENT_PRELOAD_NONE, NULL);
	gconf_client_add_dir(client, METACITY, GCONF_CLIENT_PRELOAD_NONE, NULL);

	return TRUE;
}

gboolean backend_fini(BerylSettingsContext * context)
{
	if (berylNotifyId)
	{
		gconf_client_notify_remove(client, berylNotifyId);
		berylNotifyId = 0;
	}
	if (gnomeNotifyId)
	{
		gconf_client_notify_remove(client, gnomeNotifyId);
		gnomeNotifyId = 0;
	}

	gconf_client_remove_dir(client, BERYL, NULL);
	gconf_client_remove_dir(client, METACITY, NULL);

	g_object_unref(client);
	client = NULL;

	return TRUE;
}

gboolean read_init(BerylSettingsContext * context)
{
	currentProfile = beryl_settings_context_get_profile(context);
	if (!currentProfile)
		currentProfile = "Default";
	return TRUE;
}

void read_done(BerylSettingsContext * context)
{
}

gboolean write_init(BerylSettingsContext * context)
{
	currentProfile = beryl_settings_context_get_profile(context);
	if (!currentProfile)
		currentProfile = "Default";

	return TRUE;
}

void write_done(BerylSettingsContext * context)
{

}

void read_setting(BerylSettingsContext * context, BerylSetting * setting)
{
	if (beryl_settings_context_get_de_integration_enabled(context)
		   && get_setting_is_integrated(setting))
		read_integrated_option(setting);
	else
		readSwitch(setting);
}

void write_setting(BerylSettingsContext * context, BerylSetting * setting)
{
	if (beryl_settings_context_get_de_integration_enabled(context)
		   && get_setting_is_integrated(setting))
	{
		write_integrated_option(setting);
		return;
	}


	if (setting->is_default)
	{
	//	return;
		beryl_setting_reset_to_default(setting);
	}


	//printf("Writing Setting \"%s\" for plugin \"%s\"\n",setting->name,setting->parent->name);

	writeSwitch(setting);
}

gboolean get_setting_is_integrated(BerylSetting * setting)
{
	if (!beryl_settings_context_get_de_integration_enabled(setting->parent->context))
		return FALSE;
	GROUPKEY;
	unsigned int i;
	for (i = 0; i < N_SOPTIONS; i++)
	{
		if (strcmp(specialOptions[i].berylName,keyname) == 0 &&
			strcmp(specialOptions[i].berylGroup,groupname) == 0 &&
		    specialOptions[i].screen == setting->is_screen)
		{
			g_free(keyname);
			return TRUE;
		}
	}

	g_free(keyname);
	return FALSE;
}

gboolean get_setting_is_read_only(BerylSetting * setting)
{
	return FALSE;
}

GSList *get_existing_profiles(void)
{
	gconf_client_suggest_sync(client,NULL);
	GSList * data = gconf_client_all_dirs(client, BERYL, NULL);
	GSList * tmp = data;
	GSList * ret = NULL;
	for (;tmp;tmp = g_slist_next(tmp))
	{
		if (strcmp(tmp->data,BERYL "/Default") != 0 &&
		   strlen(tmp->data) > 12)
		{
			char *name = tmp->data;
			name += 12;
			ret = g_slist_append(ret,g_strdup(name));
		}
	}
	g_slist_free(data);
	return ret;
}

const gchar * get_short_desc()
{
	return _("GConf Backend");
}

gboolean get_supports_integration()
{
	return TRUE;
}

gboolean delete_profile(gchar * profile)
{
	gchar *path;
	if (profile && strlen(profile))
		path = g_strconcat(BERYL, "/", profile, NULL);
	else
		path = g_strdup(BERYL "/Default");

	gboolean status = FALSE;
	if (gconf_client_dir_exists(client, path, NULL))
	{
		status = gconf_client_recursive_unset(client, path, 1, NULL);
		gconf_client_suggest_sync(client,NULL);
	}

	g_free(path);
	return status;
}
