/*
 * Copyright © Novell, Inc.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * Author: David Reveman <davidr@novell.com>
 * Reorganised by Quinn_Storm <quinnstorm@beryl-project.org>
 *
 */

#ifdef HAVE_CONFIG_H
#  include "../config.h"
#endif

#define _GNU_SOURCE				/* for asprintf */

#include <stdio.h>
#include <malloc.h>
#include <string.h>

#include <beryl.h>
#include <fsp.h>
#include "beryl-private.h"

#define NUM_OPTIONS(d) (sizeof ((d)->opt) / sizeof (CompOption))

#define CLICK_TO_FOCUS_DEFAULT TRUE

#define AUTORAISE_DEFAULT TRUE

#define AUTORAISE_DELAY_DEFAULT 1000
#define AUTORAISE_DELAY_MIN    0
#define AUTORAISE_DELAY_MAX    10000

#define EDGE_TRIGGERING_DELAY_DEFAULT 100
#define EDGE_TRIGGERING_DELAY_MIN    0
#define EDGE_TRIGGERING_DELAY_MAX    10000

#define SLOW_ANIMATIONS_KEY_DEFAULT       "F10"
#define SLOW_ANIMATIONS_MODIFIERS_DEFAULT ShiftMask

#define MAIN_MENU_KEY_DEFAULT       "F1"
#define MAIN_MENU_MODIFIERS_DEFAULT CompAltMask

#define RUN_DIALOG_KEY_DEFAULT       "F2"
#define RUN_DIALOG_MODIFIERS_DEFAULT CompAltMask

#define CLOSE_WINDOW_KEY_DEFAULT       "F4"
#define CLOSE_WINDOW_MODIFIERS_DEFAULT CompAltMask

#define MINIMIZE_WINDOW_KEY_DEFAULT       "F5"
#define MINIMIZE_WINDOW_MODIFIERS_DEFAULT CompAltMask

#define MAXIMIZE_WINDOW_KEY_DEFAULT       "F10"
#define MAXIMIZE_WINDOW_MODIFIERS_DEFAULT CompAltMask

#define LOWER_WINDOW_BUTTON_DEFAULT    6
#define LOWER_WINDOW_MODIFIERS_DEFAULT CompAltMask

#define RAISE_WINDOW_BUTTON_DEFAULT    7
#define RAISE_WINDOW_MODIFIERS_DEFAULT CompAltMask

#define SHOW_DESKTOP_KEY_DEFAULT       "d"
#define SHOW_DESKTOP_MODIFIERS_DEFAULT (CompAltMask | ControlMask)

#define OPACITY_INCREASE_BUTTON_DEFAULT    Button4
#define OPACITY_INCREASE_MODIFIERS_DEFAULT CompAltMask

#define OPACITY_DECREASE_BUTTON_DEFAULT    Button5
#define OPACITY_DECREASE_MODIFIERS_DEFAULT CompAltMask

#define SATURATION_INCREASE_BUTTON_DEFAULT       Button4
#define SATURATION_INCREASE_MODIFIERS_DEFAULT    ControlMask

#define SATURATION_DECREASE_BUTTON_DEFAULT       Button5
#define SATURATION_DECREASE_MODIFIERS_DEFAULT    ControlMask

#define BRIGHTNESS_INCREASE_BUTTON_DEFAULT       Button4
#define BRIGHTNESS_INCREASE_MODIFIERS_DEFAULT    ShiftMask

#define BRIGHTNESS_DECREASE_BUTTON_DEFAULT       Button5
#define BRIGHTNESS_DECREASE_MODIFIERS_DEFAULT    ShiftMask

#define SCREENSHOT_DEFAULT               "gnome-screenshot"
#define RUN_SCREENSHOT_KEY_DEFAULT       "Print"
#define RUN_SCREENSHOT_MODIFIERS_DEFAULT 0

#define WINDOW_SCREENSHOT_DEFAULT               "gnome-screenshot --window"
#define RUN_WINDOW_SCREENSHOT_KEY_DEFAULT       "Print"
#define RUN_WINDOW_SCREENSHOT_MODIFIERS_DEFAULT CompAltMask

#define WINDOW_MENU_BUTTON_DEFAULT    Button3
#define WINDOW_MENU_KEY_DEFAULT       "space"
#define WINDOW_MENU_MODIFIERS_DEFAULT CompAltMask

#define RAISE_ON_CLICK_DEFAULT TRUE

#define AUDIBLE_BELL_DEFAULT FALSE

#define DIM_DEAD_DEFAULT TRUE

#define HIDE_SKIP_TASKBAR_WINDOWS_DEFAULT TRUE

#define TOGGLE_WINDOW_SHADING_KEY_DEFAULT       "s"
#define TOGGLE_WINDOW_SHADING_MODIFIERS_DEFAULT (CompAltMask | ControlMask)

#define TOGGLE_WINDOW_MAXIMIZED_KEY_DEFAULT "F10"
#define TOGGLE_WINDOW_MAXIMIZED_MODIFIERS_DEFAULT CompSuperMask

#define TOGGLE_WINDOW_MAXIMIZED_VERT_KEY_DEFAULT  "v"
#define TOGGLE_WINDOW_MAXIMIZED_VERT_MODIFIERS_DEFAULT  CompSuperMask

#define TOGGLE_WINDOW_MAXIMIZED_HORZ_KEY_DEFAULT  "h"
#define TOGGLE_WINDOW_MAXIMIZED_HORZ_MODIFIERS_DEFAULT  CompSuperMask

#define TOGGLE_WINDOW_FULLSCREEN_KEY_DEFAULT       "Return"
#define TOGGLE_WINDOW_FULLSCREEN_MODIFIERS_DEFAULT (CompAltMask | ControlMask)

#define IGNORE_HINTS_WHEN_MAXIMIZED_DEFAULT TRUE

#define GENERATE_THUMBNAILS_DEFAULT TRUE
#define THUMBNAIL_SIZE_DEFAULT 256
#define THUMBNAIL_SIZE_MIN 32
#define THUMBNAIL_SIZE_MAX 1024
#define THUMBNAIL_SMOOTH_DEFAULT TRUE

#define MAP_ON_SHUTDOWN_DEFAULT FALSE

#define BIG_VIEWPORT_MANAGER_DEFAULT FALSE

static char *textureFilter[] = { N_("Fast"), N_("Good"), N_("Best") };

#define NUM_TEXTURE_FILTER (sizeof (textureFilter) / sizeof (textureFilter[0]))

static char *def_plugin[] = {
	"settings",
	"crashhandler",
	"png",
	"svg",
	"decoration",
	"splash",
	"wobbly",
	"animation",
	"inputzoom",
	"cube",
	"rotate",
	"scale",
	"move",
	"resize",
	"place",
	"switcher",
	"water",
	"neg",
	"put",
	"fade",
	"screenshot",
	"state",
};
static int nDefPlugin = sizeof(def_plugin) / sizeof(char *);

#include <fsp.h>

void compDisplayFreeOptions(CompDisplay * display)
{
	free(display->opt[COMP_DISPLAY_OPTION_COMMAND0].value.s);
	free(display->opt[COMP_DISPLAY_OPTION_COMMAND1].value.s);
	free(display->opt[COMP_DISPLAY_OPTION_COMMAND2].value.s);
	free(display->opt[COMP_DISPLAY_OPTION_COMMAND3].value.s);
	free(display->opt[COMP_DISPLAY_OPTION_COMMAND4].value.s);
	free(display->opt[COMP_DISPLAY_OPTION_COMMAND5].value.s);
	free(display->opt[COMP_DISPLAY_OPTION_COMMAND6].value.s);
	free(display->opt[COMP_DISPLAY_OPTION_COMMAND7].value.s);
	free(display->opt[COMP_DISPLAY_OPTION_COMMAND8].value.s);
	free(display->opt[COMP_DISPLAY_OPTION_COMMAND9].value.s);
	free(display->opt[COMP_DISPLAY_OPTION_COMMAND10].value.s);
	free(display->opt[COMP_DISPLAY_OPTION_COMMAND11].value.s);
	free(display->opt[COMP_DISPLAY_OPTION_TEXTURE_FILTER].value.s);
	free(display->opt[COMP_DISPLAY_OPTION_SCREENSHOT].value.s);
	free(display->opt[COMP_DISPLAY_OPTION_WINDOW_SCREENSHOT].value.s);
	free(display->opt[COMP_DISPLAY_OPTION_FOCUS_STEALING_LEVEL].value.s);
}

void compDisplayInitOptions(CompDisplay * display, char **plugin, int nPlugin)
{
	CompOption *o;
	char *str;

	if (nPlugin == 0)
	{
		nPlugin = nDefPlugin;
		plugin = def_plugin;
	}

	display->dirtyPluginList = TRUE;

	o = &display->opt[COMP_DISPLAY_OPTION_TEXTURE_FILTER];
	o->advanced = False;
	o->name = "texture_filter";
	o->group = N_("Main");
	o->subGroup = N_("");
	o->displayHints = "";
	o->shortDesc = N_("Texture Filter");
	o->longDesc = N_("Texture filtering");
	o->type = CompOptionTypeString;
	o->value.s = strdup("Good");
	o->rest.s.string = textureFilter;
	o->rest.s.nString = NUM_TEXTURE_FILTER;

	o = &display->opt[COMP_DISPLAY_OPTION_CLICK_TO_FOCUS];
	o->advanced = False;
	o->name = "click_to_focus";
	o->group = N_("Main");
	o->subGroup = N_("Focus & Raise");
	o->displayHints = "";
	o->shortDesc = N_("Click To Focus");
	o->longDesc = N_("Click on window moves input focus to it");
	o->type = CompOptionTypeBool;
	o->value.b = CLICK_TO_FOCUS_DEFAULT;

	o = &display->opt[COMP_DISPLAY_OPTION_AUTORAISE];
	o->advanced = False;
	o->name = "autoraise";
	o->group = N_("Main");
	o->subGroup = N_("Focus & Raise");
	o->displayHints = "";
	o->shortDesc = N_("Auto-Raise");
	o->longDesc = N_("Raise selected windows after interval");
	o->type = CompOptionTypeBool;
	o->value.b = AUTORAISE_DEFAULT;

	o = &display->opt[COMP_DISPLAY_OPTION_AUTORAISE_DELAY];
	o->advanced = False;
	o->name = "autoraise_delay";
	o->group = N_("Main");
	o->subGroup = N_("Focus & Raise");
	o->displayHints = "";
	o->shortDesc = N_("Auto-Raise Delay");
	o->longDesc = N_("Interval before raising selected windows");
	o->type = CompOptionTypeInt;
	o->value.i = AUTORAISE_DELAY_DEFAULT;
	o->rest.i.min = AUTORAISE_DELAY_MIN;
	o->rest.i.max = AUTORAISE_DELAY_MAX;

	o = &display->opt[COMP_DISPLAY_OPTION_CLOSE_WINDOW];
	o->advanced = False;
	o->name = "close_window";
	o->group = N_("Bindings");
	o->subGroup = N_("Close Window");
	o->displayHints = "";
	o->shortDesc = N_("Close Window");
	o->longDesc = N_("Close active window");
	o->type = CompOptionTypeAction;
	o->value.action.bell = FALSE;
	o->value.action.edgeMask = 0;
	o->value.action.state = CompActionStateInitKey;
	o->value.action.state |= CompActionStateInitButton;
	o->value.action.type = CompBindingTypeKey;
	o->value.action.key.modifiers = CLOSE_WINDOW_MODIFIERS_DEFAULT;
	o->value.action.key.keysym = XStringToKeysym(CLOSE_WINDOW_KEY_DEFAULT);

	o = &display->opt[COMP_DISPLAY_OPTION_MAIN_MENU];
	o->advanced = False;
	o->name = "main_menu";
	o->group = N_("Bindings");
	o->subGroup = N_("Show Main Menu");
	o->displayHints = "";
	o->shortDesc = N_("Show Main Menu");
	o->longDesc = N_("Show the main menu");
	o->type = CompOptionTypeAction;
	o->value.action.bell = FALSE;
	o->value.action.edgeMask = 0;
	o->value.action.state = CompActionStateInitKey;
	o->value.action.state |= CompActionStateInitButton;
	o->value.action.type = CompBindingTypeKey;
	o->value.action.key.modifiers = MAIN_MENU_MODIFIERS_DEFAULT;
	o->value.action.key.keysym = XStringToKeysym(MAIN_MENU_KEY_DEFAULT);

	o = &display->opt[COMP_DISPLAY_OPTION_RUN_DIALOG];
	o->advanced = False;
	o->name = "run";
	o->group = N_("Bindings");
	o->subGroup = N_("Run Dialog");
	o->displayHints = "";
	o->shortDesc = N_("Run Dialog");
	o->longDesc = N_("Show Run Application dialog");
	o->type = CompOptionTypeAction;
	o->value.action.bell = FALSE;
	o->value.action.edgeMask = 0;
	o->value.action.state = CompActionStateInitKey;
	o->value.action.state |= CompActionStateInitButton;
	o->value.action.type = CompBindingTypeKey;
	o->value.action.key.modifiers = RUN_DIALOG_MODIFIERS_DEFAULT;
	o->value.action.key.keysym = XStringToKeysym(RUN_DIALOG_KEY_DEFAULT);

	o = &display->opt[COMP_DISPLAY_OPTION_UNMAXIMIZE_WINDOW];
	o->advanced = False;
	o->name = "unmaximize_window";
	o->group = N_("Bindings");
	o->subGroup = N_("Unmaximize Window");
	o->displayHints = "";
	o->shortDesc = N_("Unmaximize Window");
	o->longDesc = N_("Unmaximize active window");
	o->type = CompOptionTypeAction;
	o->value.action.bell = FALSE;
	o->value.action.edgeMask = 0;
	o->value.action.state = CompActionStateInitKey;
	o->value.action.state |= CompActionStateInitButton;
	o->value.action.type = CompBindingTypeNone;

	o = &display->opt[COMP_DISPLAY_OPTION_MINIMIZE_WINDOW];
	o->advanced = False;
	o->name = "minimize_window";
	o->group = N_("Bindings");
	o->subGroup = N_("Minimize Window");
	o->displayHints = "";
	o->shortDesc = N_("Minimize Window");
	o->longDesc = N_("Minimize active window");
	o->type = CompOptionTypeAction;
	o->value.action.bell = FALSE;
	o->value.action.edgeMask = 0;
	o->value.action.state = CompActionStateInitKey;
	o->value.action.state |= CompActionStateInitButton;
	o->value.action.type = CompBindingTypeKey;
	o->value.action.key.modifiers = MINIMIZE_WINDOW_MODIFIERS_DEFAULT;
	o->value.action.key.keysym = XStringToKeysym(MINIMIZE_WINDOW_KEY_DEFAULT);

	o = &display->opt[COMP_DISPLAY_OPTION_MAXIMIZE_WINDOW];
	o->advanced = False;
	o->name = "maximize_window";
	o->group = N_("Bindings");
	o->subGroup = N_("Maximize Window");
	o->displayHints = "";
	o->shortDesc = N_("Maximize Window");
	o->longDesc = N_("Maximize active window");
	o->type = CompOptionTypeAction;
	o->value.action.bell = FALSE;
	o->value.action.edgeMask = 0;
	o->value.action.state = CompActionStateInitKey;
	o->value.action.state |= CompActionStateInitButton;
	o->value.action.type = CompBindingTypeKey;
	o->value.action.key.modifiers = MAXIMIZE_WINDOW_MODIFIERS_DEFAULT;
	o->value.action.key.keysym = XStringToKeysym(MAXIMIZE_WINDOW_KEY_DEFAULT);

	o = &display->opt[COMP_DISPLAY_OPTION_MAXIMIZE_WINDOW_HORZ];
	o->advanced = False;
	o->name = "maximize_window_horizontally";
	o->group = N_("Bindings");
	o->subGroup = N_("Maximize Window Horizontally");
	o->displayHints = "";
	o->shortDesc = N_("Maximize Window Horizontally");
	o->longDesc = N_("Maximize active window horizontally");
	o->type = CompOptionTypeAction;
	o->value.action.bell = FALSE;
	o->value.action.edgeMask = 0;
	o->value.action.state = CompActionStateInitKey;
	o->value.action.state |= CompActionStateInitButton;
	o->value.action.type = CompBindingTypeNone;

	o = &display->opt[COMP_DISPLAY_OPTION_MAXIMIZE_WINDOW_VERT];
	o->advanced = False;
	o->name = "maximize_window_vertically";
	o->group = N_("Bindings");
	o->subGroup = N_("Maximize Window Vertically");
	o->displayHints = "";
	o->shortDesc = N_("Maximize Window Vertically");
	o->longDesc = N_("Maximize active window vertically");
	o->type = CompOptionTypeAction;
	o->value.action.bell = FALSE;
	o->value.action.edgeMask = 0;
	o->value.action.state = CompActionStateInitKey;
	o->value.action.state |= CompActionStateInitButton;
	o->value.action.type = CompBindingTypeNone;

	o = &display->opt[COMP_DISPLAY_OPTION_SHOW_DESKTOP];
	o->advanced = False;
	o->name = "show_desktop";
	o->group = N_("Bindings");
	o->subGroup = N_("Hide all windows and focus desktop");
	o->displayHints = "";
	o->shortDesc = N_("Hide all windows and focus desktop");
	o->longDesc = N_("Hide all windows and focus desktop");
	o->type = CompOptionTypeAction;
	o->value.action.bell = FALSE;
	o->value.action.edgeMask = 0;
	o->value.action.state = CompActionStateInitKey;
	o->value.action.state |= CompActionStateInitEdge;
	o->value.action.state |= CompActionStateInitButton;
	o->value.action.type = CompBindingTypeKey;
	o->value.action.key.modifiers = SHOW_DESKTOP_MODIFIERS_DEFAULT;
	o->value.action.key.keysym = XStringToKeysym(SHOW_DESKTOP_KEY_DEFAULT);

#define COMMAND_OPTION_SHORT N_("Command line %d")
#define COMMAND_OPTION_LONG  N_("Command line to be executed in shell when " \
        "run_command%d is invoked")
#define RUN_OPTION_SHORT     N_("Run command %d")
#define RUN_OPTION_LONG      N_("A keybinding that when invoked, will run " \
        "the shell command identified by command%d")

#define COMMAND_OPTION(num, cname, rname)                        \
    o = &display->opt[COMP_DISPLAY_OPTION_COMMAND ## num ];      \
    o->advanced = False;                                         \
    o->name = cname;                                             \
    o->group = N_("Commands");                                   \
    asprintf (&str, "Command %d", num);                          \
    o->subGroup = N_(str);                                       \
    o->displayHints = "command;";                                \
    asprintf (&str, COMMAND_OPTION_SHORT, num);                  \
    o->shortDesc = str;                                          \
    asprintf (&str, COMMAND_OPTION_LONG, num);                   \
    o->longDesc = str;                                           \
    o->type = CompOptionTypeString;                              \
    o->value.s = strdup ("");                                    \
    o->rest.s.string = NULL;                                     \
    o->rest.s.nString = 0;                                       \
                                                                 \
    o = &display->opt[COMP_DISPLAY_OPTION_RUN_COMMAND ## num ];  \
    o->advanced = False;                                         \
    o->name = rname;                                             \
    o->group = N_("Commands");                                   \
    asprintf (&str, "Command %d", num);                          \
    o->subGroup = N_(str);                                       \
    o->displayHints = "";                                        \
    asprintf (&str, RUN_OPTION_SHORT, num);                      \
    o->shortDesc = str;                                          \
    asprintf (&str, RUN_OPTION_LONG, num);                       \
    o->longDesc = str;                                           \
    o->type = CompOptionTypeAction;                              \
    o->value.action.bell = FALSE;                                \
    o->value.action.edgeMask = 0;                                \
    o->value.action.state = CompActionStateInitKey;              \
    o->value.action.state |= CompActionStateInitButton;          \
    o->value.action.type = CompBindingTypeNone

	COMMAND_OPTION(0, "command0", "run_command0");
	COMMAND_OPTION(1, "command1", "run_command1");
	COMMAND_OPTION(2, "command2", "run_command2");
	COMMAND_OPTION(3, "command3", "run_command3");
	COMMAND_OPTION(4, "command4", "run_command4");
	COMMAND_OPTION(5, "command5", "run_command5");
	COMMAND_OPTION(6, "command6", "run_command6");
	COMMAND_OPTION(7, "command7", "run_command7");
	COMMAND_OPTION(8, "command8", "run_command8");
	COMMAND_OPTION(9, "command9", "run_command9");
	COMMAND_OPTION(10, "command10", "run_command10");
	COMMAND_OPTION(11, "command11", "run_command11");

	o = &display->opt[COMP_DISPLAY_OPTION_SLOW_ANIMATIONS];
	o->advanced = False;
	o->name = "slow_animations";
	o->group = N_("Bindings");
	o->subGroup = N_("Slow Animations");
	o->displayHints = "";
	o->shortDesc = N_("Slow Animations");
	o->longDesc = N_("Toggle use of slow animations");
	o->type = CompOptionTypeAction;
	o->value.action.bell = FALSE;
	o->value.action.edgeMask = 0;
	o->value.action.state = CompActionStateInitKey;
	o->value.action.state |= CompActionStateInitButton;
	o->value.action.type = 0;
	//o->value.action.type |= CompBindingTypeKey;
	o->value.action.key.modifiers = SLOW_ANIMATIONS_MODIFIERS_DEFAULT;
	o->value.action.key.keysym = XStringToKeysym(SLOW_ANIMATIONS_KEY_DEFAULT);

	o = &display->opt[COMP_DISPLAY_OPTION_LOWER_WINDOW];
	o->advanced = False;
	o->name = "lower_window";
	o->group = N_("Bindings");
	o->subGroup = N_("Lower Window");
	o->displayHints = "";
	o->shortDesc = N_("Lower Window");
	o->longDesc = N_("Lower window beneath other windows");
	o->type = CompOptionTypeAction;
	o->value.action.bell = FALSE;
	o->value.action.edgeMask = 0;
	o->value.action.state = CompActionStateInitKey;
	o->value.action.state |= CompActionStateInitButton;
	o->value.action.type = CompBindingTypeButton;
	o->value.action.button.modifiers = LOWER_WINDOW_MODIFIERS_DEFAULT;
	o->value.action.button.button = LOWER_WINDOW_BUTTON_DEFAULT;

	o = &display->opt[COMP_DISPLAY_OPTION_RAISE_WINDOW];
	o->advanced = False;
	o->name = "raise_window";
	o->group = N_("Bindings");
	o->subGroup = N_("Raise Window");
	o->displayHints = "";
	o->shortDesc = N_("Raise Window");
	o->longDesc = N_("Raise window above other windows");
	o->type = CompOptionTypeAction;
	o->value.action.bell = FALSE;
	o->value.action.edgeMask = 0;
	o->value.action.state = CompActionStateInitKey;
	o->value.action.state |= CompActionStateInitButton;
	o->value.action.type = CompBindingTypeButton;
	o->value.action.button.modifiers = RAISE_WINDOW_MODIFIERS_DEFAULT;
	o->value.action.button.button = RAISE_WINDOW_BUTTON_DEFAULT;

	o = &display->opt[COMP_DISPLAY_OPTION_ACTIVATE_WINDOW];
	o->advanced = False;
	o->name = "activate_window";
	o->group = N_("Bindings");
	o->subGroup = N_("Activate Window");
	o->displayHints = "";
	o->shortDesc = N_("Activate Window");
	o->longDesc = N_("Activate window");
	o->type = CompOptionTypeAction;
	o->value.action.bell = FALSE;
	o->value.action.edgeMask = 0;
	o->value.action.state = 0;
	o->value.action.type = CompBindingTypeNone;

	o = &display->opt[COMP_DISPLAY_OPTION_OPACITY_INCREASE];
	o->advanced = False;
	o->name = "opacity_increase";
	o->group = N_("Bindings");
	o->subGroup = N_("Increase Opacity");
	o->displayHints = "";
	o->shortDesc = N_("Increase Opacity");
	o->longDesc = N_("Increase window opacity");
	o->type = CompOptionTypeAction;
	o->value.action.bell = FALSE;
	o->value.action.edgeMask = 0;
	o->value.action.state = CompActionStateInitKey;
	o->value.action.state |= CompActionStateInitButton;
	o->value.action.type = CompBindingTypeButton;
	o->value.action.button.modifiers = OPACITY_INCREASE_MODIFIERS_DEFAULT;
	o->value.action.button.button = OPACITY_INCREASE_BUTTON_DEFAULT;

	o = &display->opt[COMP_DISPLAY_OPTION_OPACITY_DECREASE];
	o->advanced = False;
	o->name = "opacity_decrease";
	o->group = N_("Bindings");
	o->subGroup = N_("Decrease Opacity");
	o->displayHints = "";
	o->shortDesc = N_("Decrease Opacity");
	o->longDesc = N_("Decrease window opacity");
	o->type = CompOptionTypeAction;
	o->value.action.bell = FALSE;
	o->value.action.edgeMask = 0;
	o->value.action.state = CompActionStateInitKey;
	o->value.action.state |= CompActionStateInitButton;
	o->value.action.type = CompBindingTypeButton;
	o->value.action.button.modifiers = OPACITY_DECREASE_MODIFIERS_DEFAULT;
	o->value.action.button.button = OPACITY_DECREASE_BUTTON_DEFAULT;

	o = &display->opt[COMP_DISPLAY_OPTION_SATURATION_INCREASE];
	o->advanced = False;
	o->name = "saturation_increase";
	o->group = N_("Bindings");
	o->subGroup = N_("Increase Saturation");
	o->displayHints = "";
	o->shortDesc = N_("Increase Saturation");
	o->longDesc = N_("Increase Saturation");
	o->type = CompOptionTypeAction;
	o->value.action.bell = FALSE;
	o->value.action.edgeMask = 0;
	o->value.action.state = CompActionStateInitKey;
	o->value.action.state |= CompActionStateInitButton;
	o->value.action.type = 0;
	//o->value.action.type |= CompBindingTypeButton;
	o->value.action.button.modifiers = SATURATION_INCREASE_MODIFIERS_DEFAULT;
	o->value.action.button.button = SATURATION_INCREASE_BUTTON_DEFAULT;

	o = &display->opt[COMP_DISPLAY_OPTION_SATURATION_DECREASE];
	o->advanced = False;
	o->name = "saturation_decrease";
	o->group = N_("Bindings");
	o->subGroup = N_("Decrease Saturation");
	o->displayHints = "";
	o->shortDesc = N_("Decrease Saturation");
	o->longDesc = N_("Decrease Saturation");
	o->type = CompOptionTypeAction;
	o->value.action.bell = FALSE;
	o->value.action.edgeMask = 0;
	o->value.action.state = CompActionStateInitKey;
	o->value.action.state |= CompActionStateInitButton;
	o->value.action.type = 0;
	//o->value.action.type |= CompBindingTypeButton;
	o->value.action.button.modifiers = SATURATION_DECREASE_MODIFIERS_DEFAULT;
	o->value.action.button.button = SATURATION_DECREASE_BUTTON_DEFAULT;

	o = &display->opt[COMP_DISPLAY_OPTION_BRIGHTNESS_INCREASE];
	o->advanced = False;
	o->name = "brightness_increase";
	o->group = N_("Bindings");
	o->subGroup = N_("Increase Brightness");
	o->displayHints = "";
	o->shortDesc = N_("Increase Brightness");
	o->longDesc = N_("Increase Brightness");
	o->type = CompOptionTypeAction;
	o->value.action.bell = FALSE;
	o->value.action.edgeMask = 0;
	o->value.action.state = CompActionStateInitKey;
	o->value.action.state |= CompActionStateInitButton;
	o->value.action.type = 0;
	//o->value.action.type |= CompBindingTypeButton;
	o->value.action.button.modifiers = BRIGHTNESS_INCREASE_MODIFIERS_DEFAULT;
	o->value.action.button.button = BRIGHTNESS_INCREASE_BUTTON_DEFAULT;

	o = &display->opt[COMP_DISPLAY_OPTION_BRIGHTNESS_DECREASE];
	o->advanced = False;
	o->name = "brightness_decrease";
	o->group = N_("Bindings");
	o->subGroup = N_("Decrease Brightness");
	o->displayHints = "";
	o->shortDesc = N_("Decrease Brightness");
	o->longDesc = N_("Decrease Brightness");
	o->type = CompOptionTypeAction;
	o->value.action.bell = FALSE;
	o->value.action.edgeMask = 0;
	o->value.action.state = CompActionStateInitKey;
	o->value.action.state |= CompActionStateInitButton;
	o->value.action.type = 0;
	//o->value.action.type |= CompBindingTypeButton;
	o->value.action.button.modifiers = BRIGHTNESS_DECREASE_MODIFIERS_DEFAULT;
	o->value.action.button.button = BRIGHTNESS_DECREASE_BUTTON_DEFAULT;

	o = &display->opt[COMP_DISPLAY_OPTION_RUN_SCREENSHOT];
	o->advanced = False;
	o->name = "run_command_screenshot";
	o->group = N_("Bindings");
	o->subGroup = N_("Take a screenshot");
	o->displayHints = "";
	o->shortDesc = N_("Take a screenshot");
	o->longDesc = N_("Take a screenshot");
	o->type = CompOptionTypeAction;
	o->value.action.bell = FALSE;
	o->value.action.edgeMask = 0;
	o->value.action.state = CompActionStateInitKey;
	o->value.action.state |= CompActionStateInitButton;
	o->value.action.type = CompBindingTypeKey;
	o->value.action.key.modifiers = RUN_SCREENSHOT_MODIFIERS_DEFAULT;
	o->value.action.key.keysym = XStringToKeysym(RUN_SCREENSHOT_KEY_DEFAULT);

	o = &display->opt[COMP_DISPLAY_OPTION_SCREENSHOT];
	o->advanced = False;
	o->name = "command_screenshot";
	o->group = N_("Main");
	o->subGroup = N_("");
	o->displayHints = "command;";
	o->shortDesc = N_("Screenshot command line");
	o->longDesc = N_("Screenshot command line");
	o->type = CompOptionTypeString;
	o->value.s = strdup(SCREENSHOT_DEFAULT);
	o->rest.s.string = NULL;
	o->rest.s.nString = 0;

	o = &display->opt[COMP_DISPLAY_OPTION_RUN_WINDOW_SCREENSHOT];
	o->advanced = False;
	o->name = "run_command_window_screenshot";
	o->group = N_("Bindings");
	o->subGroup = N_("Take a screenshot of a window");
	o->displayHints = "";
	o->shortDesc = N_("Take a screenshot of a window");
	o->longDesc = N_("Take a screenshot of a window");
	o->type = CompOptionTypeAction;
	o->value.action.bell = FALSE;
	o->value.action.edgeMask = 0;
	o->value.action.state = CompActionStateInitKey;
	o->value.action.state |= CompActionStateInitButton;
	o->value.action.type = CompBindingTypeKey;
	o->value.action.key.modifiers = RUN_WINDOW_SCREENSHOT_MODIFIERS_DEFAULT;
	o->value.action.key.keysym =
			XStringToKeysym(RUN_WINDOW_SCREENSHOT_KEY_DEFAULT);

	o = &display->opt[COMP_DISPLAY_OPTION_WINDOW_SCREENSHOT];
	o->advanced = False;
	o->name = "command_window_screenshot";
	o->group = N_("Main");
	o->subGroup = N_("");
	o->displayHints = "command;";
	o->shortDesc = N_("Window screenshot command line");
	o->longDesc = N_("Window screenshot command line");
	o->type = CompOptionTypeString;
	o->value.s = strdup(WINDOW_SCREENSHOT_DEFAULT);
	o->rest.s.string = NULL;
	o->rest.s.nString = 0;

	o = &display->opt[COMP_DISPLAY_OPTION_WINDOW_MENU];
	o->advanced = False;
	o->name = "window_menu";
	o->group = N_("Bindings");
	o->subGroup = N_("Window Menu");
	o->displayHints = "";
	o->shortDesc = N_("Window Menu");
	o->longDesc = N_("Open window menu");
	o->type = CompOptionTypeAction;
	o->value.action.bell = FALSE;
	o->value.action.edgeMask = 0;
	o->value.action.state = CompActionStateInitKey;
	o->value.action.state |= CompActionStateInitButton;
	o->value.action.type = CompBindingTypeButton;
	o->value.action.button.modifiers = WINDOW_MENU_MODIFIERS_DEFAULT;
	o->value.action.button.button = WINDOW_MENU_BUTTON_DEFAULT;
	o->value.action.type |= CompBindingTypeKey;
	o->value.action.key.modifiers = WINDOW_MENU_MODIFIERS_DEFAULT;
	o->value.action.key.keysym = XStringToKeysym(WINDOW_MENU_KEY_DEFAULT);

	o = &display->opt[COMP_DISPLAY_OPTION_RAISE_ON_CLICK];
	o->advanced = False;
	o->name = "raise_on_click";
	o->group = N_("Main");
	o->subGroup = N_("Focus & Raise");
	o->displayHints = "";
	o->shortDesc = N_("Raise On Click");
	o->longDesc = N_("Raise windows when clicked");
	o->type = CompOptionTypeBool;
	o->value.b = RAISE_ON_CLICK_DEFAULT;

	o = &display->opt[COMP_DISPLAY_OPTION_AUDIBLE_BELL];
	o->advanced = False;
	o->name = "audible_bell";
	o->group = N_("Main");
	o->subGroup = N_("");
	o->displayHints = "";
	o->shortDesc = N_("Audible Bell");
	o->longDesc = N_("Audible system beep");
	o->type = CompOptionTypeBool;
	o->value.b = AUDIBLE_BELL_DEFAULT;

	o = &display->opt[COMP_DISPLAY_OPTION_TOGGLE_WINDOW_MAXIMIZED];
	o->advanced = False;
	o->name = "toggle_window_maximized";
	o->group = N_("Bindings");
	o->subGroup = N_("Toggle Window Maximized");
	o->displayHints = "";
	o->shortDesc = N_("Toggle Window Maximized");
	o->longDesc = N_("Toggle active window maximized");
	o->type = CompOptionTypeAction;
	o->value.action.bell = FALSE;
	o->value.action.edgeMask = 0;
	o->value.action.state = CompActionStateInitKey;
	o->value.action.state |= CompActionStateInitButton;
	o->value.action.type = CompBindingTypeKey;
	o->value.action.key.modifiers = TOGGLE_WINDOW_MAXIMIZED_MODIFIERS_DEFAULT;
	o->value.action.key.keysym =
			XStringToKeysym(TOGGLE_WINDOW_MAXIMIZED_KEY_DEFAULT);

	o = &display->opt[COMP_DISPLAY_OPTION_TOGGLE_WINDOW_FULLSCREEN];
	o->advanced = False;
	o->name = "toggle_window_fullscreen";
	o->group = N_("Bindings");
	o->subGroup = N_("Toggle Window Fullscreen");
	o->displayHints = "";
	o->shortDesc = N_("Toggle Window Fullscreen");
	o->longDesc = N_("Toggle active window fullscreen, kiosk mode");
	o->type = CompOptionTypeAction;
	o->value.action.bell = FALSE;
	o->value.action.edgeMask = 0;
	o->value.action.state = CompActionStateInitKey;
	o->value.action.state |= CompActionStateInitButton;
	o->value.action.type = CompBindingTypeKey;
	o->value.action.key.modifiers =
			TOGGLE_WINDOW_FULLSCREEN_MODIFIERS_DEFAULT;
	o->value.action.key.keysym =
			XStringToKeysym(TOGGLE_WINDOW_FULLSCREEN_KEY_DEFAULT);

	o = &display->opt[COMP_DISPLAY_OPTION_TOGGLE_WINDOW_MAXIMIZED_HORZ];
	o->advanced = False;
	o->name = "toggle_window_maximized_horizontally";
	o->group = N_("Bindings");
	o->subGroup = N_("Toggle Window Maximized Horizontally");
	o->displayHints = "";
	o->shortDesc = N_("Toggle Window Maximized Horizontally");
	o->longDesc = N_("Toggle active window maximized horizontally");
	o->type = CompOptionTypeAction;
	o->value.action.bell = FALSE;
	o->value.action.edgeMask = 0;
	o->value.action.state = CompActionStateInitKey;
	o->value.action.state |= CompActionStateInitButton;
	o->value.action.type = CompBindingTypeKey;
	o->value.action.key.modifiers =
			TOGGLE_WINDOW_MAXIMIZED_HORZ_MODIFIERS_DEFAULT;
	o->value.action.key.keysym =
			XStringToKeysym(TOGGLE_WINDOW_MAXIMIZED_HORZ_KEY_DEFAULT);

	o = &display->opt[COMP_DISPLAY_OPTION_TOGGLE_WINDOW_MAXIMIZED_VERT];
	o->advanced = False;
	o->name = "toggle_window_maximized_vertically";
	o->group = N_("Bindings");
	o->subGroup = N_("Toggle Window Maximized Vertically");
	o->displayHints = "";
	o->shortDesc = N_("Toggle Window Maximized Vertically");
	o->longDesc = N_("Toggle active window maximized vertically");
	o->type = CompOptionTypeAction;
	o->value.action.bell = FALSE;
	o->value.action.edgeMask = 0;
	o->value.action.state = CompActionStateInitKey;
	o->value.action.state |= CompActionStateInitButton;
	o->value.action.type = CompBindingTypeKey;
	o->value.action.key.modifiers =
			TOGGLE_WINDOW_MAXIMIZED_VERT_MODIFIERS_DEFAULT;
	o->value.action.key.keysym =
			XStringToKeysym(TOGGLE_WINDOW_MAXIMIZED_VERT_KEY_DEFAULT);

	o = &display->opt[COMP_DISPLAY_OPTION_HIDE_SKIP_TASKBAR_WINDOWS];
	o->advanced = True;
	o->name = "hide_skip_taskbar_windows";
	o->group = N_("Main");
	o->subGroup = N_("");
	o->displayHints = "";
	o->shortDesc = N_("Hide Skip Taskbar Windows");
	o->longDesc = N_("Hide windows not in taskbar when entering show "
					 "desktop mode");
	o->type = CompOptionTypeBool;
	o->value.b = HIDE_SKIP_TASKBAR_WINDOWS_DEFAULT;

	o = &display->opt[COMP_DISPLAY_OPTION_TOGGLE_WINDOW_SHADED];
	o->advanced = False;
	o->name = "toggle_window_shaded";
	o->group = N_("Bindings");
	o->subGroup = N_("Toggle Window Shaded");
	o->displayHints = "";
	o->shortDesc = N_("Toggle Window Shaded");
	o->longDesc = N_("Toggle active window shaded");
	o->type = CompOptionTypeAction;
	o->value.action.bell = FALSE;
	o->value.action.edgeMask = 0;
	o->value.action.state = CompActionStateInitKey;
	o->value.action.state |= CompActionStateInitButton;
	o->value.action.type = CompBindingTypeKey;
	o->value.action.key.modifiers = TOGGLE_WINDOW_SHADING_MODIFIERS_DEFAULT;
	o->value.action.key.keysym =
			XStringToKeysym(TOGGLE_WINDOW_SHADING_KEY_DEFAULT);

	o = &display->opt[COMP_DISPLAY_OPTION_DIM_DEAD];
	o->advanced = True;
	o->name = "dim_dead";
	o->group = N_("Main");
	o->subGroup = N_("");
	o->displayHints = "";
	o->shortDesc = N_("Dim Unresponsive Windows");
	o->longDesc = N_("Desaturate and dim windows that are unresponsive");
	o->type = CompOptionTypeBool;
	o->value.b = DIM_DEAD_DEFAULT;

	o = &display->opt[COMP_DISPLAY_OPTION_IGNORE_HINTS_WHEN_MAXIMIZED];
	o->advanced = False;
	o->name = "ignore_hints_when_maximized";
	o->group = N_("Main");
	o->subGroup = N_("");
	o->displayHints = "";
	o->shortDesc = N_("Ignore Hints When Maximized");
	o->longDesc =
			N_("Ignore size increment and aspect hints when window is "
			   "maximized");
	o->type = CompOptionTypeBool;
	o->value.b = IGNORE_HINTS_WHEN_MAXIMIZED_DEFAULT;

	o = &display->opt[COMP_DISPLAY_OPTION_EDGE_TRIGGERING_DELAY];
	o->advanced = True;
	o->name = "edge_triggering_delay";
	o->group = N_("Main");
	o->subGroup = N_("");
	o->displayHints = "";
	o->shortDesc = N_("Edge Triggering Delay");
	o->longDesc = N_("Interval before triggering actions on edges");
	o->type = CompOptionTypeInt;
	o->value.i = EDGE_TRIGGERING_DELAY_DEFAULT;
	o->rest.i.min = EDGE_TRIGGERING_DELAY_MIN;
	o->rest.i.max = EDGE_TRIGGERING_DELAY_MAX;

	o = &display->opt[COMP_DISPLAY_OPTION_GENERATE_THUMBNAILS];
	o->advanced = False;
	o->name = "generate_thumbnails";
	o->group = N_("Main");
	o->subGroup = N_("Window Thumbnails");
	o->displayHints = "";
	o->shortDesc = N_("Activate thumbnail generation");
	o->longDesc = N_("Generate thumbnails for minimized windows.");
	o->type = CompOptionTypeBool;
	o->value.b = GENERATE_THUMBNAILS_DEFAULT;

	o = &display->opt[COMP_DISPLAY_OPTION_THUMBNAIL_SIZE];
	o->advanced = False;
	o->name = "thumbnail_size";
	o->group = N_("Main");
	o->subGroup = N_("Window Thumbnails");
	o->displayHints = "";
	o->shortDesc = N_("Thumbnail size");
	o->longDesc = N_("Maximum size of the thumbnail image");
	o->type = CompOptionTypeInt;
	o->value.i = THUMBNAIL_SIZE_DEFAULT;
	o->rest.i.min = THUMBNAIL_SIZE_MIN;
	o->rest.i.max = THUMBNAIL_SIZE_MAX;

	o = &display->opt[COMP_DISPLAY_OPTION_THUMBNAIL_SMOOTH];
	o->advanced = False;
	o->name = "smooth_thumbnails";
	o->group = N_("Main");
	o->subGroup = N_("Window Thumbnails");
	o->displayHints = "";
	o->shortDesc = N_("Smooth thumbnails");
	o->longDesc = N_("Generate thumbnails using bilinear filter");
	o->type = CompOptionTypeBool;
	o->value.b = THUMBNAIL_SMOOTH_DEFAULT;

	o = &display->opt[COMP_DISPLAY_OPTION_FOCUS_STEALING_LEVEL];
	o->advanced = False;
	o->name = "fsp_level";
	o->group = N_("Main");
	o->subGroup = N_("");
	o->displayHints = "";
	o->shortDesc = N_("Level of Focus Stealing Prevention");
	o->longDesc =
			N_("How aggressive the focus-stealing-prevention should be");
	o->type = CompOptionTypeString;
	o->value.s = strdup(DEFAULT_FSP);
	o->rest.s.string = fspTypes;
	o->rest.s.nString = FSP_COUNT;

	o = &display->opt[COMP_DISPLAY_OPTION_MAP_ON_SHUTDOWN];
	o->advanced = False;
	o->name = "map_on_shutdown";
	o->group = N_("Main");
	o->subGroup = N_("");
	o->displayHints = "";
	o->shortDesc = N_("Map minimized windows on shutdown");
	o->longDesc = N_("Map minimized windows on shutdown (needed for KWin)");
	o->type = CompOptionTypeBool;
	o->value.b = MAP_ON_SHUTDOWN_DEFAULT;

	o = &display->opt[COMP_DISPLAY_OPTION_BIG_VIEWPORT_MANAGER];
	o->advanced = False;
	o->name = "big_view_manager";
	o->group = N_("Desktop Background");
	o->subGroup = N_("");
	o->displayHints = "";
	o->shortDesc = N_("Desktop manager supports viewports");
	o->longDesc = N_("Check this if you have a desktop manager that supports viewports to allow different wallpapers on varying sides of the cube");
	o->type = CompOptionTypeBool;
	o->value.b = False;
}

CompOption *compGetDisplayOptions(CompDisplay * display, int *count)
{
	if (display)
	{
		*count = NUM_OPTIONS(display);
		return display->opt;
	}
	else
	{
		CompDisplay *d = malloc(sizeof(CompDisplay));

		compDisplayInitOptions(d, NULL, 0);
		*count = NUM_OPTIONS(d);
		return d->opt;
	}
}
