/*
 * Copyright © 2005 Novell, Inc.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * Author: David Reveman <davidr@novell.com>
 */

#ifdef HAVE_CONFIG_H
#  include "../config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <unistd.h>
#include <getopt.h>
#include <string.h>
#include <sys/wait.h>
#include <X11/Xlib.h>
#include <X11/extensions/Xcomposite.h>

#include <beryl.h>
#include <beryl-private.h>

#include "syscheck.h"

char *programName;
char **programArgv;
int programArgc;
REGION infiniteRegion;
REGION emptyRegion;

static Bool disableSm = FALSE;

REGION *getInfiniteRegion(void)
{
	return &infiniteRegion;
}

REGION *getEmptyRegion(void)
{
	return &emptyRegion;
}

char *getProgramName(void)
{
	return programName;
}

char **getProgramArgv(void)
{
	return programArgv;
}

int getProgramArgc(void)
{
	return programArgc;
}

Window currentRoot = 0;

Bool restartSignal = FALSE;

CompWindow *lastFoundWindow = 0;
CompWindow *lastDamagedWindow = 0;

Bool replaceCurrentWm = TRUE;
Bool indirectRendering = FALSE;
Bool strictBinding = FALSE;
Bool useCow = FALSE;
Bool skipGlYield = FALSE;
Bool isNVTFP = FALSE;
Bool copyTexture = FALSE;
Bool noShm = FALSE;
Bool skipTests = FALSE;
Bool nvidia = FALSE;
Bool noContextShare = FALSE;
Bool testOnly = FALSE;

static void usage(void)
{
	printf(_("Usage: %9s "
			 "[--display DISPLAY] "
			 "[--screen n]\n                 "
			 "[--strict-binding | --xgl-binding]\n                 "
			 "[--indirect-rendering | --xgl-rendering]\n                 "
			 "[--force-nvidia | --force-xgl | --force-aiglx]\n                 "
			 "[--no-replace | --replace] "
#if XCOMPOSITE_MINOR >= 3 || XCOMPOSITE_MAJOR > 0
			 "[--use-cow | --no-cow] "
#endif
			 "\n                 "
			 "[--sm-disable] "
			 "[--use-copy | --use-tfp]\n                 "
			 "[--sm-client-id ID]\n                 "
			 "[--skip-tests] [--test-only] "
			 "[--no-context-share]\n                 "
			 "[--version | -v] "
			 "[--help | -h] " "[PLUGIN ...]\n"), programName);
}

static void signalHandler(int sig)
{
	int status;

	switch (sig)
	{
	case SIGCHLD:
		waitpid(-1, &status, WNOHANG | WUNTRACED);
		break;
	case SIGHUP:
		restartSignal = TRUE;
	case SIGTERM:
	case SIGINT:
		releaseDisplay();

		if (!disableSm)
			closeSession();
		exit(1);
		break;
	default:
		break;
	}
}

static inline void init_variables(void)
{
	emptyRegion.rects = &emptyRegion.extents;
	emptyRegion.numRects = 0;
	emptyRegion.extents.x1 = 0;
	emptyRegion.extents.y1 = 0;
	emptyRegion.extents.x2 = 0;
	emptyRegion.extents.y2 = 0;
	emptyRegion.size = 0;

	infiniteRegion.rects = &infiniteRegion.extents;
	infiniteRegion.numRects = 1;
	infiniteRegion.extents.x1 = MINSHORT;
	infiniteRegion.extents.y1 = MINSHORT;
	infiniteRegion.extents.x2 = MAXSHORT;
	infiniteRegion.extents.y2 = MAXSHORT;
}

/* Keep them equal to their sopts version */
#define OPT_HELP 'h'
#define OPT_VERSION 'v'
#define OPT_DISPLAY 1
#define OPT_SCREEN 2
#define OPT_FORCE_NVIDIA 4
#define OPT_FORCE_XGL 5
#define OPT_FORCE_AIGLX 6
#define OPT_USE_COPY 7
#define OPT_SM_CLIENT_ID 8

int main(int argc, char **argv)
{
	char *displayName = 0;
	char *plugin[256];
	int screenNum = -1;
	int nPlugin = 0;
	int result = 0;
	char *clientId = NULL;
	Bool forceNvidia = FALSE;
	Bool forceXgl = FALSE;
	Bool forceAiglx = FALSE;
	Bool checkXglShm = FALSE;
	int optch;
	char sopts[] = "hv";
	struct option lopts[] = {
		{"help", 0, 0, OPT_HELP},
		{"version", 0, 0, OPT_VERSION},
		{"display", 1, 0, OPT_DISPLAY},
		{"screen", 1, 0, OPT_SCREEN},
		{"skip-gl-yield", 0, &skipGlYield, TRUE},
		{"force-nvidia", 0, 0, OPT_FORCE_NVIDIA},
		{"force-xgl", 0, 0, OPT_FORCE_XGL},
		{"force-aiglx", 0, 0, OPT_FORCE_AIGLX},
		{"use-tfp", 0, &copyTexture, FALSE},
		{"use-copy", 0, 0, OPT_USE_COPY},
		{"indirect-rendering", 0, &indirectRendering, TRUE},
		{"xgl-rendering", 0, &indirectRendering, FALSE},
		{"strict-binding", 0, &strictBinding, TRUE},
		{"xgl-binding", 0, &strictBinding, FALSE},
		{"use-cow", 0, &useCow, TRUE},
		{"no-cow", 0, &useCow, FALSE},
		{"no-replace", 0, &replaceCurrentWm, FALSE},
		{"replace", 0, &replaceCurrentWm, TRUE},
		{"sm-disable", 0, &disableSm, TRUE},
		{"skip-tests", 0, &skipTests, TRUE},
		{"test-only", 0, &testOnly, TRUE},
		{"no-context-share", 0, &noContextShare, TRUE},
		{"sm-client-id", 1, 0, OPT_SM_CLIENT_ID},
		{0, 0, 0, 0}
	};

	programName = argv[0];
	programArgc = argc;
	programArgv = argv;

	init_variables();

	/* Process arguments */
	while ((optch = getopt_long(argc, argv, sopts, lopts, NULL)) != EOF)
	{
		switch (optch)
		{
		case OPT_HELP:
			usage();
			return 0;
		case OPT_VERSION:
			printf(PACKAGE_STRING "\n");
			return 0;
		case OPT_DISPLAY:
			if (optarg)
				displayName = optarg;
			break;
		case OPT_SCREEN:
			if (optarg)
				screenNum = atoi(optarg);
			break;
		case OPT_FORCE_NVIDIA:
			forceNvidia = TRUE;
			indirectRendering = FALSE;
			useCow = TRUE;
			strictBinding = FALSE;
			isNVTFP = TRUE;
			break;
		case OPT_FORCE_XGL:
			forceXgl = TRUE;
			indirectRendering = FALSE;
			useCow = FALSE;
			strictBinding = FALSE;
			break;
		case OPT_FORCE_AIGLX:
			forceAiglx = TRUE;
			indirectRendering = TRUE;
			useCow = TRUE;
			strictBinding = TRUE;
			break;
		case OPT_USE_COPY:
			copyTexture = TRUE;
			indirectRendering = FALSE;
			useCow = TRUE;
			strictBinding = FALSE;
			skipGlYield = TRUE;
			checkXglShm = TRUE;
			break;
		case OPT_SM_CLIENT_ID:
			if (optarg)
				clientId = optarg;
			break;
		case 0:				/* Returned when auto-set stuff is in effect */
			break;
		default:
			/* Not recognised option or with missing argument.
			 * getopt_long() prints an error message for us.
			 */
			return 1;

		}
	}

	/* Any remaing args are assumed to be plugins */
	while (argc - optind > 0)
		if (nPlugin < 256)
			plugin[nPlugin++] = argv[optind++];

	if (!skipTests)
	{
		int mode;
		int nScreens;
		result = syscheck(displayName,&mode,&nScreens);

		if (nScreens > 1)
		{
			fprintf(stderr, "%d screens detected.\n",nScreens);
			fprintf(stderr, "Currently we cannot guarantee that Beryl will work"
							" correctly with multiple X screens.\n");
			if (!noContextShare)
				fprintf(stderr, "Using the --no-context-share command line "
								"option can help.\n");
			fprintf(stderr, "Please report any success to the beryl "
							"developer mailing list. The list can be found at "
							"lists.beryl-project.org\n");


		}

		if ((result && result != CHECK_NOTFP) || testOnly)
			return result;

		if (result == CHECK_NOTFP)
		{
			copyTexture = TRUE;
			indirectRendering = FALSE;
			useCow = TRUE;
			strictBinding = FALSE;
			skipGlYield = TRUE;
			checkXglShm = TRUE;
			if (mode == MODE_XGL)
			{
				noShm = TRUE;
				useCow = FALSE;
			}
		} else if (!forceNvidia && !forceXgl && !forceAiglx)
		{
			switch (mode)
			{
				case MODE_NVIDIA:
					indirectRendering = FALSE;
					useCow = TRUE;
					strictBinding = FALSE;
					isNVTFP = TRUE;
					nvidia = TRUE;
					break;
				case MODE_XGL:
					noShm = TRUE;
					useCow = FALSE;
					indirectRendering = FALSE;
					strictBinding = FALSE;
					break;
				default:
					indirectRendering = TRUE;
					useCow = TRUE;
					strictBinding = TRUE;
					break;
			}
		}
		result = 0;
	}

	if (isNVTFP && !skipGlYield)
	{
		char *newargs[argc + 2];
		int i;

		printf(_("Relaunching %s with __GL_YIELD=\"NOTHING\"\n"), argv[0]);
		setenv("__GL_YIELD", "NOTHING", 1);
		for (i = 0; i < argc; i++)
			newargs[i] = argv[i];
		newargs[argc] = "--skip-gl-yield";
		newargs[argc + 1] = NULL;
		execvp(argv[0], newargs);
	}

	signal(SIGHUP, signalHandler);
	signal(SIGCHLD, signalHandler);
	signal(SIGTERM, signalHandler);
	signal(SIGINT, signalHandler);

	if (!disableSm)
		initSession(clientId);

	if (!addDisplay(displayName, screenNum, plugin, nPlugin))
		return 1;

	result = eventLoop();

	releaseDisplay();

	if (!disableSm)
		closeSession();

	exit(result);
}
