/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Conf.h"
#include "Color.h"
#include <ace/config-lite.h>
#include <boost/regex.hpp>
#include <stdexcept>

using namespace std;

Config::Config()
:	m_listenAddrs(),
	m_nextHopProxy(),
	m_isNextHopProxyEnabled(false),
	m_isClientCompressionEnabled(false),
	m_isTrayAnimationEnabled(true),
	m_proxySkipList(),
	m_ptrProxySkipRegex(),
	m_ptrBorderColor(new Color(0x67, 0x67, 0x67)),
	m_pageCleanupLevel(CLEANUP_OFF),
	m_maxScriptNestLevel(6),
	m_maxScriptFetchSize(45),
	m_maxScriptEvalSize(180),
	m_saveTrafficThreshold(15),
	m_reportClientIP(REPORT_IP_OFF),
	m_fixedClientIP(),
	m_allowedTunnelPorts()
{
	// backwards compatibility
	m_nextHopProxy.setType(ProxyDescriptor::HTTP);
	
	// ports allowed by default
	bool success = m_allowedTunnelPorts.fromString("443, 563");
	assert(success);
}

Config::Config(Config const& other)
:	m_listenAddrs(other.m_listenAddrs),
	m_nextHopProxy(other.m_nextHopProxy),
	m_isNextHopProxyEnabled(other.m_isNextHopProxyEnabled),
	m_isClientCompressionEnabled(other.m_isClientCompressionEnabled),
	m_isTrayAnimationEnabled(other.m_isTrayAnimationEnabled),
	m_proxySkipList(other.m_proxySkipList),
	m_ptrProxySkipRegex(other.getProxySkipRegex()),
	m_ptrBorderColor(other.getBorderColor()),
	m_pageCleanupLevel(other.m_pageCleanupLevel),
	m_maxScriptNestLevel(other.m_maxScriptNestLevel),
	m_maxScriptFetchSize(other.m_maxScriptFetchSize),
	m_maxScriptEvalSize(other.m_maxScriptEvalSize),
	m_saveTrafficThreshold(other.m_saveTrafficThreshold),
	m_reportClientIP(other.m_reportClientIP),
	m_fixedClientIP(other.m_fixedClientIP),
	m_allowedTunnelPorts(other.m_allowedTunnelPorts)
{
}

Config::~Config()
{
}

Config&
Config::operator=(Config const& other)
{
	if (&other == this) {
		return *this;
	}
	m_listenAddrs = other.m_listenAddrs;
	m_nextHopProxy = other.m_nextHopProxy;
	m_isNextHopProxyEnabled = other.m_isNextHopProxyEnabled;
	m_isClientCompressionEnabled = other.m_isClientCompressionEnabled;
	m_isTrayAnimationEnabled = other.m_isTrayAnimationEnabled;
	m_proxySkipList = other.m_proxySkipList;
	m_ptrProxySkipRegex = other.getProxySkipRegex();
	m_ptrBorderColor = other.getBorderColor();
	m_pageCleanupLevel = other.m_pageCleanupLevel;
	m_maxScriptNestLevel = other.m_maxScriptNestLevel;
	m_maxScriptFetchSize = other.m_maxScriptFetchSize;
	m_maxScriptEvalSize = other.m_maxScriptEvalSize;
	m_saveTrafficThreshold = other.m_saveTrafficThreshold;
	m_reportClientIP = other.m_reportClientIP;
	m_fixedClientIP = other.m_fixedClientIP;
	m_allowedTunnelPorts = other.m_allowedTunnelPorts;
	return *this;
}

void
Config::setNextHopProxy(ProxyDescriptor const& proxy)
{
	m_nextHopProxy = proxy;
}

void
Config::setProxySkipList(std::list<std::string> const& list)
{
	if (m_proxySkipList == list) {
		return;
	}
	m_proxySkipList = list;
	updateProxySkipRegex();
}

auto_ptr<ProxyDescriptor>
Config::getProxyFor(std::string const& host) const
{
	auto_ptr<ProxyDescriptor> res;
	if (!m_isNextHopProxyEnabled) {
		return res;
	}
	if (m_ptrProxySkipRegex.get()) {
		try {
			if (boost::regex_match(host, *m_ptrProxySkipRegex)) {
				return res;
			}
		} catch (std::exception& e) {}
	}
	return auto_ptr<ProxyDescriptor>(new ProxyDescriptor(m_nextHopProxy));
}

auto_ptr<Color>
Config::getBorderColor() const
{
	if (!m_ptrBorderColor.get()) {
		return auto_ptr<Color>();
	}
	return auto_ptr<Color>(new Color(*m_ptrBorderColor));
}

void
Config::setBorderColor(auto_ptr<Color> color)
{
	m_ptrBorderColor = color;
}

bool
Config::isTunnelPortAllowed(unsigned port) const
{
	return m_allowedTunnelPorts.contains(port);
}

auto_ptr<boost::regex>
Config::getProxySkipRegex() const
{
	auto_ptr<boost::regex> res;
	if (m_ptrProxySkipRegex.get()) {
		res.reset(new boost::regex(*m_ptrProxySkipRegex));
	}
	return res;
}

void
Config::updateProxySkipRegex()
{
	if (m_proxySkipList.empty()) {
		m_ptrProxySkipRegex.reset(0);
		return;
	}

	list<string>::iterator iter = m_proxySkipList.begin();
	list<string>::iterator const end = m_proxySkipList.end();
	string composite_regex;
	composite_regex.reserve(150);
	for (bool first = true; iter != end; ++iter) {
		if (iter->empty()) {
			continue;
		}
		if (!first) {
			composite_regex += '|';
		} else {
			first = false;
		}
		if (iter->c_str()[0] == '.') {
			composite_regex += "(.*";
		} else {
			composite_regex += '(';
		}
		composite_regex += regQuote(*iter);
		if (iter->c_str()[iter->size()-1] == '.') {
			composite_regex += ".*)";
		} else {
			composite_regex += ')';
		}
	}
	
	try {
		using namespace boost::regex_constants;
		m_ptrProxySkipRegex.reset(new boost::regex(composite_regex, icase|nosubs));
	} catch (boost::bad_expression const& e) {
		// should never happen
	} 
}

std::string
Config::regQuote(std::string const& str)
{
	static char const* metachars = ".\\+*?[^]$(){}=!<>|:";
	string quoted;
	string::size_type pos = 0, pos1;
	while ((pos1 = str.find_first_of(metachars, pos)) != string::npos) {
		quoted.append(str, pos, pos1-pos);
		quoted += '\\';
		quoted += str[pos1];
		pos = pos1+1;
	}
	quoted.append(str, pos, string::npos);
	return quoted;
}
