/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "StatusIcon.h"
#include "eggtrayicon.h"
#include <gtkmm/window.h>

using namespace std;

class StatusIcon::Window : public Gtk::Window
{
public:
	Window() {}
	
	Window(GtkWindow* gobj) : Gtk::Window(gobj) {}
	
	virtual ~Window() {}
	
	virtual bool isEmbeddable() const = 0;
	
	using Gtk::Window::realize;
};


class StatusIcon::NormalWindow : public StatusIcon::Window
{
	virtual bool isEmbeddable() const { return false; }
};


class StatusIcon::EmbeddableWindow : public StatusIcon::Window
{
public:
	EmbeddableWindow(Glib::ustring const& name)
	: Window(GTK_WINDOW(egg_tray_icon_new(name.c_str()))) {}
	
	virtual bool isEmbeddable() const { return true; }
};


StatusIcon::StatusIcon(Glib::ustring const& name)
:	m_name(name),
	m_isVisible(false)
{
	auto_ptr<Window> wnd(createEmbeddableWindow());
	if (!wnd.get()) {
		wnd = createNormalWindow();
	}
	applyWindow(wnd);
}

StatusIcon::~StatusIcon()
{
}

void
StatusIcon::setIcon(Glib::RefPtr<Gdk::Pixbuf> const& pixbuf)
{
	m_image.set(pixbuf);
}

void
StatusIcon::setTooltip(Glib::ustring const& tooltip)
{
	m_tooltip = tooltip;
	m_tooltips.set_tip(*m_ptrWindow, m_tooltip);
}

void
StatusIcon::setVisible(bool visible)
{
	if (visible == m_isVisible) {
		return;
	}
	
	m_isVisible = visible;
	if (visible) {
		m_ptrWindow->show();
	} else {
		m_ptrWindow->hide();
	}
}

std::auto_ptr<StatusIcon::Window>
StatusIcon::createEmbeddableWindow()
{
	auto_ptr<Window> wnd(new EmbeddableWindow(m_name));
	wnd->realize();
	if (!EGG_TRAY_ICON(wnd->gobj())->manager_window) {
		// impossible to dock
		wnd.reset(0);
	}
	return wnd;
}

std::auto_ptr<StatusIcon::Window>
StatusIcon::createNormalWindow()
{
	auto_ptr<Window> wnd(new NormalWindow);
	wnd->set_title(m_name);
	wnd->resize(22, 22);
	return wnd;
}

void
StatusIcon::applyWindow(std::auto_ptr<Window> wnd)
{
	m_ptrWindow = wnd;
	m_ptrWindow->add(m_image);
	m_image.show();
	
	if (!m_tooltip.empty()) {
		m_tooltips.set_tip(*m_ptrWindow, m_tooltip);
	}
	if (m_isVisible) {
		m_ptrWindow->show();
	}
	
	m_ptrWindow->signal_delete_event().connect(
		sigc::mem_fun(*this, &StatusIcon::onDeleteEvent)
	);
	m_ptrWindow->signal_window_state_event().connect(
		sigc::mem_fun(*this, &StatusIcon::onWindowStateEvent)
	);
	m_ptrWindow->signal_button_press_event().connect(
		sigc::mem_fun(*this, &StatusIcon::onButtonPressed)
	);
	m_ptrWindow->add_events(Gdk::BUTTON_PRESS_MASK);
}

bool
StatusIcon::onDeleteEvent(GdkEventAny* evt)
{
	/*
	if (m_ptrWindow->isEmbeddable()) {
		applyWindow(createNormalWindow());
		return true;
	}
	*/
	m_closeSignal.emit();
	return false; // event not handled
}

bool
StatusIcon::onWindowStateEvent(GdkEventWindowState* evt)
{
	if (m_ptrWindow->isEmbeddable()) {
		return false; // event not handled
	}
	if (evt->new_window_state & evt->changed_mask & GDK_WINDOW_STATE_ICONIFIED) {
		auto_ptr<Window> wnd(createEmbeddableWindow());
		if (wnd.get()) {
			applyWindow(wnd);
			return true; // event handled
		}
	}
	return false; // event not handled
}

bool
StatusIcon::onButtonPressed(GdkEventButton* evt)
{
	m_mouseClickSignal.emit(evt);
	return true; // event handled
}
