/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ReplacementImage.h"
#include "Color.h"
#include "BString.h"
#include "SBOutStream.h"
#include "CraftedResponse.h"
#include "HttpResponseMetadata.h"
#include "HttpHeadersCollection.h"
#include "HttpHeader.h"
#include "Date.h"
#include "GlobalState.h"
#include "Conf.h"
#include <algorithm>

using namespace std;

char const ReplacementImage::HEADER[6] = { 'G', 'I', 'F', '8', '9', 'a' };
char const ReplacementImage::GRAPHIC_CONTROL_EXTENSION[8] = {
	0x21, // extension introducer
	0xf9, // graphic control label
	0x04, // block size
	0x05, // flags (transparent + disposal_method==1)
	0x00, 0x00, // delay time
	0x00, // transparent color index
	0x00  // block terminator
};
size_t const ReplacementImage::TABLE_SIZE = 4096;
size_t const ReplacementImage::FIRST_AVAILABLE_CODE = 6;
ReplacementImage::LZWCode const ReplacementImage::CLEAR_CODE = 4;
ReplacementImage::LZWCode const ReplacementImage::EOI_CODE = 5;

static char const transgif_1x1[] = {
	0x47, 0x49, 0x46, 0x38, 0x39, 0x61, 0x01, 0x00,
	0x01, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xff, 0xff, 0xff, 0x21, 0xf9, 0x04, 0x01, 0x00,
	0x00, 0x00, 0x00, 0x2c, 0x00, 0x00, 0x00, 0x00,
	0x01, 0x00, 0x01, 0x00, 0x40, 0x02, 0x01, 0x44,
	0x00, 0x3b
};

ReplacementImage::ReplacementImage(int width, int height, Color const* border_color)
{
	if (width < 1 || width > 2000) {
		width = 1;
	}
	if (height < 1 || height > 2000) {
		height = 1;
	}
	
	if (width == 1 && height == 1) {
		m_binaryData.append(
			BString::ChunkPtr(), transgif_1x1,
			transgif_1x1 + sizeof(transgif_1x1)
		);
		return;
	}
	
	SBOutStream strm;
	
	strm.write(HEADER, sizeof(HEADER));
	writeScreenDescriptor(strm, width, height);
	strm.write(GRAPHIC_CONTROL_EXTENSION, sizeof(GRAPHIC_CONTROL_EXTENSION));
	
	if (!border_color || width < 3 || height < 3) {
		writeImageDescriptor(strm, 0, 0, width, height);
		writeLocalColorTable(strm, border_color);
		writeImageData(strm, 0x00, width*height);
	} else {
		/*
		writeImageDescriptor(out, 1, 1, width-2, height-2));
		writeLocalColorTable(out, border_color);
		writeImageData(out, 0x00, (width-2)*(height-2));
		*/
		
		SplittableBuffer temp;
		
		// write two horizontal lines
		writeImageDescriptor(strm, 0, 0, width, 1);
		writeLocalColorTable(strm, border_color);
		strm.swapData(temp); // store what we have so far
		writeImageData(strm, 0x01, width);
		strm.swapData(temp); // now we have line data in temp
		strm.data().append(temp);
		writeImageDescriptor(strm, 0, height - 1, width, 1);
		writeLocalColorTable(strm, border_color);
		strm.data().appendDestructive(temp); // append the other line
		
		// write two vertical lines
		writeImageDescriptor(strm, 0, 1, 1, height - 2);
		writeLocalColorTable(strm, border_color);
		strm.swapData(temp); // store what we have so far
		writeImageData(strm, 0x01, height-2);
		strm.swapData(temp); // now we have line data in temp
		strm.data().append(temp);
		writeImageDescriptor(strm, width-1, 1, 1, height - 2);
		writeLocalColorTable(strm, border_color);
		strm.data().appendDestructive(temp); // append the other line
	}
	strm << uint8_t(0x3b);
	strm.swapData(m_binaryData);
}

auto_ptr<CraftedResponse>
ReplacementImage::createHttpResponse(
	bool is_head_response, int width, int height, int status_code)
{
	auto_ptr<CraftedResponse> response(new CraftedResponse(status_code));
	response->metadata().headers().setHeader(
		HttpHeader(BString("Content-Type"), getContentType())
	);
	response->metadata().headers().setHeader(
		HttpHeader(BString("Date"), Date::formatCurrentTime())
	);
	if (is_head_response) {
		response->metadata().setBodyStatus(HttpResponseMetadata::BODY_FORBIDDEN);
	} else {
		auto_ptr<Color> color(GlobalState::ReadAccessor()->config().getBorderColor());
		ReplacementImage img(width, height, color.get());
		response->body().appendDestructive(img.binaryData());
		response->metadata().setBodyStatus(HttpResponseMetadata::BODY_SIZE_KNOWN);
		response->metadata().setBodySize(response->body().size());
	}
	return response;
}

void
ReplacementImage::writeScreenDescriptor(SBOutStream& out, int width, int height)
{
	char sd[] = {
		(unsigned)width, (unsigned)width >> 8,
		(unsigned)height, (unsigned)height >> 8,
		0x00, // flags
		0x00, // background color index
		0x00  // pixel aspect ratio
	};
	out.write(sd, sizeof(sd));
}

void
ReplacementImage::writeImageDescriptor(SBOutStream& out,
	int xpos, int ypos, int width, int height)
{
	char id[] = {
		0x2c, // image separator
		(unsigned)xpos, ((unsigned)xpos >> 8),
		(unsigned)ypos, ((unsigned)ypos >> 8),
		(unsigned)width, ((unsigned)width >> 8),
		(unsigned)height, ((unsigned)height >> 8),
		0x80 // flags
	};
	out.write(id, sizeof(id));
}

void
ReplacementImage::writeLocalColorTable(SBOutStream& out, Color const* color)
{
	if (!color) {
		static char const coltab[6] = { 0x00 };
		out.write(coltab, sizeof(coltab));
	} else {
		char coltab[] = {
			0x00, 0x00, 0x00, // this one will be transparent
			color->getRed(), color->getGreen(), color->getBlue()
		};
		out.write(coltab, sizeof(coltab));
	};
}

void
ReplacementImage::writeImageData(SBOutStream& out, uint8_t color_idx, size_t length)
{
	LZWCodeStream code_stream;
	writeLZWCodeStream(code_stream, color_idx, length);
	packLZWCodeStream(out, code_stream);
}

/*
We only need to encode a sequence of <length> <color_idx> bytes.
The LZW code size is 2 (minimal possible value).
The imaginary code table would look like this:
0: 0
1: 1
10: 2 // not used
11: 3 // not used
100: clear code
101: end of information
110: col col
111: col col col
1000: col col col col
and so on
*/
void
ReplacementImage::writeLZWCodeStream(LZWCodeStream& strm, uint8_t color_idx, size_t length)
{
	size_t tablepos = 0;
	size_t buflen = 1;
	--length;
	strm.push_back(CLEAR_CODE);
	while (length-- > 0) {
		if (isCodeInTable(buflen + 1, tablepos)) {
			++buflen;
		} else {
			strm.push_back(getCodeFor(buflen, color_idx));
			buflen = 1;
			if (++tablepos + FIRST_AVAILABLE_CODE >= TABLE_SIZE && length > 0) {
				strm.push_back(getCodeFor(buflen, color_idx));
				strm.push_back(CLEAR_CODE);
				tablepos = 0;
			}
		}
	}
	strm.push_back(getCodeFor(buflen, color_idx));
	strm.push_back(EOI_CODE);
}

void
ReplacementImage::packLZWCodeStream(SBOutStream& out, LZWCodeStream const& strm)
{
	char block[255];
	size_t block_size = 0;
	int code_bits = 3;
	unsigned int max_code = (1 << 3) - 1;
	unsigned char cur_byte = 0;
	int cur_byte_bits_occupied = 0;
	out << uint8_t(0x02); // LZW code size
	LZWCodeStream::const_iterator it(strm.begin());
	LZWCodeStream::const_iterator const end(strm.end());
	for (; it != end; ++it) {
		LZWCode code = *it;
		int bits_injected = 0;
		while (bits_injected < code_bits) {
			cur_byte |= ((code >> bits_injected) << cur_byte_bits_occupied);
			int injection = std::min(
				code_bits - bits_injected,
				8 - cur_byte_bits_occupied
			);
			bits_injected += injection;
			cur_byte_bits_occupied += injection;
			if (cur_byte_bits_occupied == 8) {
				block[block_size++] = cur_byte;
				if (block_size == sizeof(block)) {
					out << uint8_t(sizeof(block));
					out.write(block, sizeof(block));
					block_size = 0;	
				}
				cur_byte = 0;
				cur_byte_bits_occupied = 0;
			}
		}
		if (code == CLEAR_CODE) {
			code_bits = 3;
			max_code = (1 << 3) - 1;
		} else if (code >= max_code) {
			max_code = (1 << ++code_bits) - 1;
		}
	}
	if (cur_byte_bits_occupied > 0) {
		block[block_size++] = cur_byte;
	}
	if (block_size != 0) {
		out << uint8_t(block_size);
		out.write(block, block_size);
	}
	out << uint8_t(0x00);
}

bool
ReplacementImage::isCodeInTable(size_t seqlen, size_t tablepos)
{
	if (seqlen == 1) {
		return true;
	} else {
		return (tablepos > seqlen - 2);
	}
}

ReplacementImage::LZWCode
ReplacementImage::getCodeFor(size_t seqlen, uint8_t color_idx)
{
	if (seqlen == 1) {
		return color_idx;
	} else {
		return FIRST_AVAILABLE_CODE + seqlen - 2;
	}
}
