/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ForwardingConfigWindow.h"
#include "Application.h"
#include "AbstractLogView.h"
#include "AutoScrollingWindow.h"
#include "SkinnedButton.h"
#include "Link.h"
#include "Forwarding.h"
#include "ForwardingConfigFile.h"
#include "OperationLog.h"
#include "Log.h"
#include "GlobalState.h"
#include "ProxyDescriptor.h"
#include "SymbolicInetAddr.h"
#include "Verify.h"
#include "ArraySize.h"
#include "ScopedIncDec.h"
#include "CompiledImages.h"
#include "types.h"
#include <ace/config-lite.h>
#include <boost/lexical_cast.hpp>
#include <libview/toolTip.hh>
#include <libview/weakPtr.hh>
#include <glibmm/ustring.h>
#include <glibmm/unicode.h>
#include <glibmm/refptr.h>
#include <gtkmm/window.h>
#include <gtkmm/box.h>
#include <gtkmm/buttonbox.h>
#include <gtkmm/treeview.h>
#include <gtkmm/liststore.h>
#include <gtkmm/layout.h>
#include <gtkmm/entry.h>
#include <gtkmm/comboboxtext.h>
#include <gtkmm/textview.h>
#include <gtkmm/textbuffer.h>
#include <gtkmm/checkbutton.h>
#include <gtkmm/button.h>
#include <gtkmm/statusbar.h>
#include <gtkmm/alignment.h>
#include <gtkmm/frame.h>
#include <gtkmm/scrolledwindow.h>
#include <gtkmm/viewport.h>
#include <gtkmm/messagedialog.h>
#include <gtkmm/menu.h>
#include <gtkmm/stock.h>
#include <gtkmm/image.h>
#include <gtkmm/table.h>
#include <gtkmm/notebook.h>
#include <gtkmm/eventbox.h>
#include <gdkmm/pixbuf.h>
#include <gdkmm/window.h>
#include <gdkmm/cursor.h>
#include <gdkmm/types.h> // for Gdk::ModifierType
#include <gdk/gdk.h> // for GdkEvent
#include <iterator>
#include <sstream>
#include <vector>
#include <memory>
#include <assert.h>
#include <stddef.h>

namespace GtkGUI
{

static ForwardingConfigFile& FCF()
{
	return Application::instance()->forwardingConfigFile();
}


class ForwardingConfigWindow::Impl : public Gtk::Window, private AbstractLogView
{
	DECLARE_NON_COPYABLE(Impl)
public:
	enum ShowEditor { SHOW_NONE, SHOW_BYPASS_EDITOR, SHOW_PROXY_EDITOR };

	Impl();
	
	virtual ~Impl();
	
	static void showWindow();
	
	Forwarding::Config& forwardingConfig() { return m_forwardingConfig; }
	
	Gtk::Statusbar& statusBar() { return m_statusBar; }
	
	void switchToOption(Forwarding::Option const& option, int option_idx);
	
	void switchToNoOption();
	
	void deleteProxyInteractive(int proxy_idx);
	
	ProxyDescriptor const* getPrecedingProxy(int transition_idx);
	
	void startProxyAdding(int transition_idx);

	void startProxyEditing(int proxy_idx);

	void stopProxyOperation();

	void commitProxyOperation();

	void enterBypassEditingMode();

	void leaveBypassEditingMode();
	
	void reloadBypassData();

	void showEditor(ShowEditor editor);
	
	void showError(Glib::ustring const& message);
	
	static void ensureProperSelection(Forwarding::Config& config);
private:
	virtual void on_hide();
	
	virtual bool on_delete_event(GdkEventAny*);
	
	static void setSelection(Forwarding::Config& config, int selected_option);
	
	static bool isSameSelection(
		Forwarding::Config const& config, int selected_option);
	
	bool prepareForWindowDestruction();
	
	bool onFocusChange(GdkEventFocus*);
	
	void onClearLog();

	static Impl* m_spInstance;
	Forwarding::Config m_forwardingConfig;
	OptionsList* m_pOptionsList;
	Gtk::Notebook* m_pEditorNB;
	ProxyChainPanel* m_pProxyChainPanel;
	ProxyEditPanel* m_pProxyEditPanel;
	BypassPanel* m_pBypassPanel;
	Gtk::Statusbar m_statusBar;
};


class ForwardingConfigWindow::MouseOverStatusMsg
{
	DECLARE_NON_COPYABLE(MouseOverStatusMsg)
public:
	MouseOverStatusMsg(Gtk::Statusbar& sb, Glib::ustring const& message);
	
	~MouseOverStatusMsg();
	
	void connectToWidget(Gtk::Widget& widget);
	
	void setMessage(Glib::ustring const& message);
private:
	bool onMouseEnter(GdkEventCrossing*);
	
	bool onMouseLeave(GdkEventCrossing*);
	
	void putMessage();
	
	void removeMessage();
	
	Gtk::Statusbar& m_rStatusBar;
	Glib::ustring m_message;
	guint m_contextId;
	guint m_messageId;
	bool m_isMouseOver;
};


class ForwardingConfigWindow::OptionsList : public Gtk::TreeView
{
	DECLARE_NON_COPYABLE(OptionsList)
public:
	OptionsList(Impl& owner);
	
	virtual ~OptionsList();
	
	void reload(Forwarding::Config const& config);

	int getSelectedItem();
private:
	/**
	 * ITEM_REALIZED means that this item represents an option in
	 * m_rOwner.forwardingConfig().options()
	 */
	enum ItemState { ITEM_NOT_REALIZED = 0, ITEM_REALIZED };
	
	class ModelColumns : public Gtk::TreeModel::ColumnRecord
	{
	public:
		Gtk::TreeModelColumn<Glib::ustring> name;
		Gtk::TreeModelColumn<ItemState> state;
		
		ModelColumns() { add(name); add(state); }
	};
	
	void appendOption(Forwarding::Option const& option);
	
	static int iteratorToIndex(Gtk::TreeIter const& it);
	
	void setSelectedItem(int item);
	
	void createMenu();
	
	void updateMenu();
	
	virtual bool on_button_press_event(GdkEventButton* evt);
	
	void startEditing(Gtk::TreeIter const& it);
	
	void onEditingCanceled();
	
	void onEditingFinished(
		Glib::ustring const& old_text, Glib::ustring const& new_text);
	
	bool handleItemNewName(
		Gtk::TreeIter const& item, Glib::ustring const& new_name);
	
	void onNew();
	
	void onRename();
	
	void onDelete();
	
	void onMoveUp();
	
	void onMoveDown();
	
	void onSelectionChanged();

	static void swapOptions(Forwarding::Config& config, int idx1, int idx2);
	
	virtual bool on_enter_notify_event(GdkEventCrossing* evt);
	
	virtual bool on_leave_notify_event(GdkEventCrossing* evt);
	
	Impl& m_rOwner;
	ModelColumns m_cols;
	Gtk::CellRendererText m_nameCellRenderer;
	Gtk::TreeViewColumn m_nameCol;
	Glib::RefPtr<Gtk::ListStore> m_ptrModel;
	std::auto_ptr<Gtk::Menu> m_ptrMenu;
	Gtk::MenuItem* m_pMenuUp;
	Gtk::MenuItem* m_pMenuDown;
	Gtk::MenuItem* m_pMenuRename;
	Gtk::MenuItem* m_pMenuDelete;
	MouseOverStatusMsg m_statusMsg;
};


class ForwardingConfigWindow::ProxyChainPanel : public Gtk::ScrolledWindow
{
	DECLARE_NON_COPYABLE(ProxyChainPanel)
public:
	/**
	 * If \p prototype is set, creates a dummy panel with a chain
	 * of 2 proxies. The dummy panel is used to keep the minimum width
	 * of the real panel so that a chain of 2 proxies could be displayed
	 * with no scrollbars. This can be achieved by putting both the real
	 * and dummy panels to a Gtk::Notebook as different pages.
	 */
	ProxyChainPanel(Impl& owner, ProxyChainPanel const* prototype = 0);
	
	virtual ~ProxyChainPanel();

	void loadOption(Forwarding::Option const& option);

	void clearChain();

	void switchToNormalMode();
	
	void switchToInsensitiveMode();

	void switchToProxyAddingMode(int transition_idx);

	void switchToProxyEditingMode(int proxy_idx);
private:
	void loadImages();
	
	void copyImages(ProxyChainPanel const& from);
	
	void addDesktopButton(int col);
	
	void addForwardButton(int col, Gdk::Cursor const& cursor);
	
	void addProxyButton(int col, Gdk::Cursor const& cursor);
	
	void addInternetButton(int col);

	void onProxyClicked(Gdk::ModifierType state, int proxy_idx);
	
	void onForwardClicked(Gdk::ModifierType state, int transition_idx);

	Impl& m_rOwner;
	Gtk::Viewport* m_pViewport;
	Gtk::Alignment* m_pAlignment;
	std::auto_ptr<Gtk::Table> m_ptrTable;
	Glib::RefPtr<Gdk::Pixbuf> m_ptrDesktopPixbuf;
	Glib::RefPtr<Gdk::Pixbuf> m_ptrProxyPixbuf;
	Glib::RefPtr<Gdk::Pixbuf> m_ptrProxyGhostPixbuf;
	Glib::RefPtr<Gdk::Pixbuf> m_ptrInternetPixbuf;
	Glib::RefPtr<Gdk::Pixbuf> m_ptrForwardPixbuf;
	std::vector<SkinnedButton*> m_buttons;
	Glib::ustring const m_insertMsg;
	Glib::ustring const m_editDeleteMsg;
	MouseOverStatusMsg m_insertStatusMsg;
	MouseOverStatusMsg m_editDeleteStatusMsg;
};


class ForwardingConfigWindow::ProxyEditPanel : public Gtk::VBox
{
	DECLARE_NON_COPYABLE(ProxyEditPanel)
public:
	ProxyEditPanel(Impl& owner);
	
	virtual ~ProxyEditPanel();
	
	void grabFocus();

	void startProxyAdding(int option_idx, int transition_idx);

	void startProxyEditing(int option_idx, int proxy_idx);

	void cancelCurrentOperation();
private:
	enum Mode { INACTIVE, ADDING, EDITING };

	struct ProxyOption
	{
		ProxyDescriptor::ProxyType type;
		char const* label; // utf-8
	};
	
	ProxyDescriptor::ProxyType getProxyType() const;

	void setProxyType(ProxyDescriptor::ProxyType type);

	void onProxyTypeChanged();

	void onProxyTypeChangedTo(ProxyDescriptor::ProxyType type);
	
	bool validateInteractive(ProxyDescriptor& target) const;
	
	bool haveNextHopProxy() const;
	
	static Glib::ustring trim(Glib::ustring const& str);
	
	void onOK();

	void onCancel();

	static ProxyOption const m_sProxyOptions[];
	Impl& m_rOwner;
	Mode m_mode;
	int m_optionIdx;
	int m_proxyIdx; // index of proxy being edited or insert position for a new proxy
	Gtk::ComboBoxText* m_pProxyType;
	Gtk::Entry* m_pProxyHost;
	Gtk::Entry* m_pProxyPort;
	Gtk::Entry* m_pProxyUser;
	Gtk::Entry* m_pProxyPass;
};


class ForwardingConfigWindow::BypassPanel :
	public Gtk::VBox,
	private Forwarding::BypassVisitor
{
	DECLARE_NON_COPYABLE(BypassPanel)
public:
	BypassPanel(Impl& owner);
	
	virtual ~BypassPanel();

	void loadFrom(Forwarding::Option const& opt, int option_idx);
	
	void removeTooltips();
private:
	virtual void visit(Forwarding::HostMaskMatchPattern const& p);

	virtual void visit(Forwarding::SimpleHostnamesMatchPattern const& p);
	
	void onBypassHostsHelp();

	void onBypassLocalHelp();

	void onBypassHostsModified();
	
	void onBypassLocalModified();
	
	void onModified(bool modified);

	void onOK();

	void onCancel();
	
	static Glib::ustring nextPattern(Glib::ustring& patterns);

	Impl& m_rOwner;
	Glib::RefPtr<Gtk::TextBuffer> m_ptrBypassHosts;
	Gtk::TextView* m_pBypassHostsView;
	Gtk::CheckButton* m_pBypassLocal;
	Gtk::Button* m_pOKBtn;
	Gtk::Button* m_pCancelBtn;
	view::WeakPtr<view::ToolTip> m_ptrBypassHostsTip;
	view::WeakPtr<view::ToolTip> m_ptrBypassLocalTip;
	int m_optionIdx;
	unsigned char m_ignoreModifications;
	bool m_bypassLocalInitial;
	bool m_wasModified;
};

void
ForwardingConfigWindow::showWindow()
{
	Impl::showWindow();
}


/*===================== ForwardingConfigWindow::Impl =====================*/

ForwardingConfigWindow::Impl* ForwardingConfigWindow::Impl::m_spInstance = 0;

ForwardingConfigWindow::Impl::Impl()
:	AbstractLogView(*OperationLog::instance()),
	m_forwardingConfig(FCF().getForwardingConfig()),
	m_pOptionsList(0),
	m_pEditorNB(0),
	m_pProxyChainPanel(0),
	m_pProxyEditPanel(0),
	m_pBypassPanel(0)
{
	set_title("Forwarding Configuration");
	set_icon(CompiledImages::window_icon_png.getPixbuf());
	
	signal_focus_in_event().connect(
		sigc::mem_fun(*this, &Impl::onFocusChange)
	);
	signal_focus_out_event().connect(
		sigc::mem_fun(*this, &Impl::onFocusChange)
	);
	
	Gtk::VBox* top_vbox = manage(new Gtk::VBox);
	add(*top_vbox);
	
	Gtk::Alignment* proxy_chain_padding = manage(new Gtk::Alignment());
	top_vbox->pack_start(*proxy_chain_padding, Gtk::PACK_SHRINK);
	proxy_chain_padding->set_padding(5, 0, 5, 5); // except bottom
	
	Gtk::Notebook* proxy_chain_nb = manage(new Gtk::Notebook);
	proxy_chain_padding->add(*proxy_chain_nb);
	proxy_chain_nb->set_show_tabs(false);
	proxy_chain_nb->set_show_border(false);
	proxy_chain_nb->set_scrollable(false);
	
	m_pProxyChainPanel = manage(new ProxyChainPanel(*this));
	proxy_chain_nb->append_page(*m_pProxyChainPanel);
	proxy_chain_nb->append_page(*manage(new ProxyChainPanel(*this, m_pProxyChainPanel)));
	
	Gtk::Alignment* opt_row_align = manage(new Gtk::Alignment(0.0f, 0.5f));
	top_vbox->pack_start(*opt_row_align);
	opt_row_align->set_padding(3, 0, 3, 3); // except bottom
	
	Gtk::HBox* opt_row = manage(new Gtk::HBox);
	opt_row_align->add(*opt_row);
	
	Gtk::VBox* opt_clearlog_vbox = manage(new Gtk::VBox);
	opt_row->pack_start(*opt_clearlog_vbox, Gtk::PACK_SHRINK);
	
	Gtk::ScrolledWindow* list_scroller = manage(new Gtk::ScrolledWindow);
	opt_clearlog_vbox->pack_start(*list_scroller);
	list_scroller->set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
	list_scroller->set_shadow_type(Gtk::SHADOW_IN);
	list_scroller->set_size_request(150, 90);
	list_scroller->set_border_width(1);
	
	m_pOptionsList = manage(new OptionsList(*this));
	list_scroller->add(*m_pOptionsList);
	
	Gtk::Alignment* clearlog_align = manage(new Gtk::Alignment(0.0f, 0.5f, 0.0f, 1.0f));
	opt_clearlog_vbox->pack_start(*clearlog_align, Gtk::PACK_SHRINK);
	clearlog_align->set_padding(5, 0, 2, 0); // top and left padding
	
	Gtk::Button* clearlog_btn = manage(new Gtk::Button("Clear Log"));
	clearlog_align->add(*clearlog_btn);
	clearlog_btn->signal_clicked().connect(
		sigc::mem_fun(*this, &Impl::onClearLog)
	);
	
	Gtk::Alignment* nb_padding = manage(new Gtk::Alignment);
	nb_padding->set_padding(0, 0, 6, 4);
	opt_row->pack_start(*nb_padding);
	
	m_pEditorNB = manage(new Gtk::Notebook);
	nb_padding->add(*m_pEditorNB);
	m_pEditorNB->set_show_tabs(false);
	m_pEditorNB->set_show_border(false);
	m_pEditorNB->set_scrollable(false);
	m_pEditorNB->append_page(*manage(new Gtk::Alignment)); // empty page
	
	Gtk::Alignment* proxy_editor_align = manage(new Gtk::Alignment(0.0f, 0.5f, 0.0f, 1.0f));
	m_pEditorNB->append_page(*proxy_editor_align);
	
	m_pProxyEditPanel = manage(new ProxyEditPanel(*this));
	proxy_editor_align->add(*m_pProxyEditPanel);
	
	m_pBypassPanel = manage(new BypassPanel(*this));
	m_pEditorNB->append_page(*m_pBypassPanel);
	
	Gtk::Frame* log_frame = manage(new Gtk::Frame(" Log "));
	top_vbox->pack_start(*log_frame, Gtk::PACK_SHRINK);
	log_frame->set_border_width(1);
	
	AutoScrollingWindow* log_scroll_window = manage(new AutoScrollingWindow);
	log_frame->add(*log_scroll_window);
	log_scroll_window->set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
	log_scroll_window->set_shadow_type(Gtk::SHADOW_IN);
	log_scroll_window->set_size_request(-1, 100);
	log_scroll_window->set_border_width(2);
	
	Gtk::TextView* log = manage(
		new Gtk::TextView(OperationLog::instance()->getTextBuffer())
	);
	log_scroll_window->add(*log);
	log->set_editable(false);
	log->set_cursor_visible(false);
	log->set_wrap_mode(Gtk::WRAP_WORD);
	
	top_vbox->pack_start(m_statusBar, Gtk::PACK_SHRINK);
	m_statusBar.set_has_resize_grip(true);
	
	show_all_children();
	
	m_pOptionsList->reload(m_forwardingConfig);
	
	Application::instance()->destroySignal().connect(
		sigc::mem_fun(*this, &Impl::hide)
	);
}

ForwardingConfigWindow::Impl::~Impl()
{
	assert(m_spInstance);
	m_spInstance = 0;
	
	// prevent slots being called during destruction
	sigc::trackable::notify_callbacks();
}

void
ForwardingConfigWindow::Impl::showWindow()
{
	if (m_spInstance) {
		m_spInstance->present();
	} else {
		(m_spInstance = new Impl)->show();
	}
}

void
ForwardingConfigWindow::Impl::on_hide()
{
	delete this;
}

bool
ForwardingConfigWindow::Impl::on_delete_event(GdkEventAny*)
{
	if (prepareForWindowDestruction()) {
		delete this;
	}
	return true; // stop event propagation
}

bool
ForwardingConfigWindow::Impl::prepareForWindowDestruction()
{
	if (!m_pOptionsList->is_sensitive()) {
		Gtk::MessageDialog dialog(
			*this,
			"Are you sure you want to close this window?\n"
			"Any <b>unsaved changes</b> will be lost.",
			true, /* use_markup */
			Gtk::MESSAGE_QUESTION, Gtk::BUTTONS_YES_NO,
			true /* modal */
		);
		int const response = dialog.run();
		return (response == Gtk::RESPONSE_YES);
	}
	
	// If our selection differs from the global one, update the
	// global one and write a new forwarding.xml
	int const selected_option = m_pOptionsList->getSelectedItem();
	if (selected_option != -1
	    && !isSameSelection(FCF().getForwardingConfig(), selected_option)) {
		setSelection(m_forwardingConfig, selected_option);
		FCF().applyAndSave(m_forwardingConfig);
	}
	return true;
}

bool
ForwardingConfigWindow::Impl::onFocusChange(GdkEventFocus*)
{
	bool const visible = property_is_active();
	AbstractLogView::reportVisibility(visible);
	if (!visible) {
		m_pBypassPanel->removeTooltips();
	}
	return false; // propagate event further
}

void
ForwardingConfigWindow::Impl::onClearLog()
{
	OperationLog::instance()->clear();
}

void
ForwardingConfigWindow::Impl::showEditor(ShowEditor what)
{
	int page_idx = 0;
	if (what == SHOW_PROXY_EDITOR) {
		page_idx = 1;
	} else if (what == SHOW_BYPASS_EDITOR) {
		page_idx = 2;
	}
	m_pEditorNB->set_current_page(page_idx);
}

void
ForwardingConfigWindow::Impl::switchToOption(
	Forwarding::Option const& opt, int const option_idx)
{
	m_pProxyChainPanel->loadOption(opt);
	m_pBypassPanel->loadFrom(opt, option_idx);
}

void
ForwardingConfigWindow::Impl::switchToNoOption()
{
	Forwarding::Utf8String opt_name;
	Forwarding::Option dummy_option(opt_name);
	m_pProxyChainPanel->loadOption(dummy_option);
}

void
ForwardingConfigWindow::Impl::deleteProxyInteractive(int const proxy_idx)
{
	int const option_idx = m_pOptionsList->getSelectedItem();
	if (option_idx == -1) {
		return;
	}
	
	Gtk::MessageDialog dialog(
		*this,
		"Really delete this proxy?\n",
		false, /* use_markup */
		Gtk::MESSAGE_QUESTION, Gtk::BUTTONS_YES_NO,
		true /* modal */
	);
	int const res = dialog.run();
	if (res != Gtk::RESPONSE_YES) {
		return;
	}
	
	Forwarding::Config new_config(m_forwardingConfig);

	assert(option_idx >= 0);
	assert(option_idx < (int)new_config.options().size());

	Forwarding::OptionList::iterator opt(new_config.options().begin());
	std::advance(opt, option_idx);

	assert(proxy_idx >= 0);
	assert(proxy_idx < (int)opt->proxyChain().size());

	Forwarding::ProxyDescriptorList::iterator proxy(opt->proxyChain().begin());
	std::advance(proxy, proxy_idx);

	opt->proxyChain().erase(proxy);
	ensureProperSelection(new_config);
	if (!FCF().applyAndSave(new_config)) {
		return;
	} else {
		m_forwardingConfig.swap(new_config);
	}

	m_pProxyChainPanel->loadOption(*opt);
}

ProxyDescriptor const*
ForwardingConfigWindow::Impl::getPrecedingProxy(int const transition_idx)
{
	assert(transition_idx >= 0);
	
	if (transition_idx == 0) {
		return 0;
	}
	
	int const option_idx = m_pOptionsList->getSelectedItem();
	if (option_idx == -1) {
		return 0;
	}
	assert(option_idx < (int)m_forwardingConfig.options().size());

	Forwarding::OptionList::const_iterator opt(m_forwardingConfig.options().begin());
	std::advance(opt, option_idx);
	
	Forwarding::ProxyDescriptorList::const_iterator proxy(opt->proxyChain().begin());
	std::advance(proxy, transition_idx - 1);
	
	return &*proxy;
}

void
ForwardingConfigWindow::Impl::startProxyAdding(int const transition_idx)
{
	assert(transition_idx >= 0);

	int const option_idx = m_pOptionsList->getSelectedItem();
	if (option_idx == -1) {
		return;
	}
	assert(option_idx < (int)m_forwardingConfig.options().size());
	
	m_pOptionsList->set_sensitive(false);
	m_pProxyChainPanel->switchToProxyAddingMode(transition_idx);
	m_pProxyEditPanel->startProxyAdding(option_idx, transition_idx);
	showEditor(SHOW_PROXY_EDITOR);
	m_pProxyEditPanel->grabFocus();
}

void
ForwardingConfigWindow::Impl::startProxyEditing(int const proxy_idx)
{
	assert(proxy_idx >= 0);

	int const option_idx = m_pOptionsList->getSelectedItem();
	if (option_idx == -1) {
		return;
	}
	assert(option_idx < (int)m_forwardingConfig.options().size());
	
	m_pOptionsList->set_sensitive(false);
	m_pProxyChainPanel->switchToProxyEditingMode(proxy_idx);
	m_pProxyEditPanel->startProxyEditing(option_idx, proxy_idx);
	showEditor(SHOW_PROXY_EDITOR);
	m_pProxyEditPanel->grabFocus();
}

void
ForwardingConfigWindow::Impl::stopProxyOperation()
{
	m_pProxyEditPanel->cancelCurrentOperation();
	showEditor(SHOW_BYPASS_EDITOR);
	m_pOptionsList->set_sensitive(true);
	m_pProxyChainPanel->switchToNormalMode();
}

void
ForwardingConfigWindow::Impl::commitProxyOperation()
{
	stopProxyOperation();

	long const option_idx = m_pOptionsList->getSelectedItem();
	if (option_idx == -1) {
		return;
	}
	assert(option_idx >= 0);
	assert(option_idx < (int)m_forwardingConfig.options().size());

	Forwarding::OptionList::iterator opt(m_forwardingConfig.options().begin());
	std::advance(opt, option_idx);
	m_pProxyChainPanel->loadOption(*opt);
}

/**
 * \brief Enter bypass editing mode and block other operations.
 *
 * This gets called when user modifies bypass options.
 */
void
ForwardingConfigWindow::Impl::enterBypassEditingMode()
{
	m_pOptionsList->set_sensitive(false);
	m_pProxyChainPanel->switchToInsensitiveMode();
}

void
ForwardingConfigWindow::Impl::leaveBypassEditingMode()
{
	m_pOptionsList->set_sensitive(true);
	m_pProxyChainPanel->switchToNormalMode();
}

void
ForwardingConfigWindow::Impl::reloadBypassData()
{
	int const option_idx = m_pOptionsList->getSelectedItem();
	if (option_idx == -1) {
		return;
	}
	assert(option_idx >= 0);
	assert(option_idx < (int)m_forwardingConfig.options().size());
	
	Forwarding::OptionList::iterator opt(m_forwardingConfig.options().begin());
	std::advance(opt, option_idx);

	m_pBypassPanel->loadFrom(*opt, option_idx);
}

void
ForwardingConfigWindow::Impl::showError(Glib::ustring const& message)
{
	Gtk::MessageDialog dialog(
		*this, message,
		false, /* use_markup */
		Gtk::MESSAGE_ERROR, Gtk::BUTTONS_OK,
		true /* modal */
	);
	dialog.run();
}

/**
 * \brief Ensure that one and only one option is selected.
 */
void
ForwardingConfigWindow::Impl::ensureProperSelection(Forwarding::Config& config)
{
	Forwarding::OptionList::iterator it(config.options().begin());
	Forwarding::OptionList::iterator const end(config.options().end());
	bool found = false;
	for (; it != end; ++it) {
		if (found) {
			it->setSelected(false);
		} else {
			found = it->isSelected();
		}
	}
	
	if (!found && !config.options().empty()) {
		// if none is selected, select the first one
		config.options().front().setSelected(true);
	}
}

void
ForwardingConfigWindow::Impl::setSelection(
	Forwarding::Config& config, int const selected_option)
{
	Forwarding::OptionList::iterator it(config.options().begin());
	Forwarding::OptionList::iterator const end(config.options().end());
	for (int i = 0; it != end; ++it, ++i) {
		it->setSelected(i == selected_option);
	}
}

bool
ForwardingConfigWindow::Impl::isSameSelection(
	Forwarding::Config const& config, int const selected_option)
{
	Forwarding::OptionList::const_iterator it(config.options().begin());
	Forwarding::OptionList::const_iterator const end(config.options().end());
	for (int i = 0; it != end; ++it, ++i) {
		bool const selected = (i == selected_option);
		if (it->isSelected() != selected) {
			return false;
		}
	}
	return true;
}


/*============= ForwardingConfigWindow::MouseOverStatusMsg ==============*/

ForwardingConfigWindow::MouseOverStatusMsg::MouseOverStatusMsg(
	Gtk::Statusbar& sb, Glib::ustring const& message)
:	m_rStatusBar(sb),
	m_message(message),
	m_contextId(sb.get_context_id("MouseOverStatusMsg")),
	m_messageId(0),
	m_isMouseOver(false)
{
}

ForwardingConfigWindow::MouseOverStatusMsg::~MouseOverStatusMsg()
{
	if (m_isMouseOver) {
		removeMessage();
	}
}

void
ForwardingConfigWindow::MouseOverStatusMsg::connectToWidget(Gtk::Widget& widget)
{
	widget.signal_enter_notify_event().connect(
		sigc::mem_fun(*this, &MouseOverStatusMsg::onMouseEnter)
	);
	widget.signal_leave_notify_event().connect(
		sigc::mem_fun(*this, &MouseOverStatusMsg::onMouseLeave)
	);
}

void
ForwardingConfigWindow::MouseOverStatusMsg::setMessage(
	Glib::ustring const& message)
{
	m_message = message;
	if (m_isMouseOver) {
		removeMessage();
		putMessage();
	}
}

bool
ForwardingConfigWindow::MouseOverStatusMsg::onMouseEnter(GdkEventCrossing*)
{
	if (!m_isMouseOver) {
		m_isMouseOver = true;
		putMessage();
	}
	return false; // propagate event further
}

bool
ForwardingConfigWindow::MouseOverStatusMsg::onMouseLeave(GdkEventCrossing*)
{
	if (m_isMouseOver) {
		m_isMouseOver = false;
		removeMessage();
	}
	return false; // propagate event further
}

void
ForwardingConfigWindow::MouseOverStatusMsg::putMessage()
{
	m_messageId = m_rStatusBar.push(m_message, m_contextId);
}

void
ForwardingConfigWindow::MouseOverStatusMsg::removeMessage()
{
	m_rStatusBar.remove_message(m_messageId, m_contextId);
	m_messageId = 0;
}


/*================= ForwardingConfigWindow::OptionsList ==================*/

ForwardingConfigWindow::OptionsList::OptionsList(Impl& owner)
:	m_rOwner(owner),
	m_ptrModel(Gtk::ListStore::create(m_cols)),
	m_pMenuUp(0),
	m_pMenuDown(0),
	m_pMenuRename(0),
	m_pMenuDelete(0),
	m_statusMsg(owner.statusBar(), "Use context menu")
{
	add_events(Gdk::ENTER_NOTIFY_MASK|Gdk::LEAVE_NOTIFY_MASK);
	m_statusMsg.connectToWidget(*this);
	
	set_model(m_ptrModel);
	set_headers_visible(false);
	set_enable_search(false);
	
	createMenu();
	
	m_nameCol.pack_start(m_nameCellRenderer, true);
	m_nameCol.add_attribute(m_nameCellRenderer.property_text(), m_cols.name);
	append_column(m_nameCol);
	
	m_nameCellRenderer.signal_editing_canceled().connect(
		sigc::mem_fun(*this, &OptionsList::onEditingCanceled)
	);
	m_nameCellRenderer.signal_edited().connect(
		sigc::mem_fun(*this, &OptionsList::onEditingFinished)
	);
	
	get_selection()->signal_changed().connect(
		sigc::mem_fun(*this, &OptionsList::onSelectionChanged)
	);
}

ForwardingConfigWindow::OptionsList::~OptionsList()
{
	// prevent slots being called during destruction
	sigc::trackable::notify_callbacks();
}

void
ForwardingConfigWindow::OptionsList::reload(Forwarding::Config const& config)
{
	m_ptrModel->clear();

	Forwarding::OptionList const& options = config.options();
	Forwarding::OptionList::const_iterator it(options.begin());
	Forwarding::OptionList::const_iterator const end(options.end());
	for (; it != end; ++it) {
		appendOption(*it);
	}
	if (getSelectedItem() == -1 && !options.empty()) {
		setSelectedItem(0);
	}
}

void
ForwardingConfigWindow::OptionsList::appendOption(
	Forwarding::Option const& option)
{
	Gtk::TreeRow row(*m_ptrModel->append());
	row[m_cols.name] = Glib::ustring(option.getName().raw());
	row[m_cols.state] = ITEM_REALIZED;
	if (option.isSelected()) {
		get_selection()->select(row);
	}
}

/**
 * \return the row number (zero-based) corrsponding to the iterator,
 *         or -1 if iterator is invalid.
 */
int
ForwardingConfigWindow::OptionsList::iteratorToIndex(Gtk::TreeIter const& it)
{
	if (it) {
		return Gtk::TreePath(it).back();
	} else {
		return -1;
	}
}

/**
 * \return selected item index or -1 if no item is selected.
 */
int
ForwardingConfigWindow::OptionsList::getSelectedItem()
{
	return iteratorToIndex(get_selection()->get_selected());
}

void
ForwardingConfigWindow::OptionsList::setSelectedItem(int const item)
{
	Gtk::TreePath path;
	path.append_index(item);
	get_selection()->select(path);
}

void
ForwardingConfigWindow::OptionsList::createMenu()
{
	m_ptrMenu.reset(new Gtk::Menu);
	
	m_ptrMenu->items().push_back(
		Gtk::Menu_Helpers::ImageMenuElem(
			"New",
			*manage(new Gtk::Image(Gtk::Stock::NEW, Gtk::ICON_SIZE_MENU)),
			sigc::mem_fun(*this, &OptionsList::onNew)
		)
	);
	
	m_ptrMenu->items().push_back(
		Gtk::Menu_Helpers::MenuElem(
			"Rename",
			sigc::mem_fun(*this, &OptionsList::onRename)
		)
	);
	m_pMenuRename = &m_ptrMenu->items().back();
	
	m_ptrMenu->items().push_back(Gtk::Menu_Helpers::SeparatorElem());
	
	m_ptrMenu->items().push_back(
		Gtk::Menu_Helpers::ImageMenuElem(
			"Move Up",
			*manage(new Gtk::Image(Gtk::Stock::GO_UP, Gtk::ICON_SIZE_MENU)),
			sigc::mem_fun(*this, &OptionsList::onMoveUp)
		)
	);
	m_pMenuUp = &m_ptrMenu->items().back();
	
	m_ptrMenu->items().push_back(
		Gtk::Menu_Helpers::ImageMenuElem(
			"Move Down",
			*manage(new Gtk::Image(Gtk::Stock::GO_DOWN, Gtk::ICON_SIZE_MENU)),
			sigc::mem_fun(*this, &OptionsList::onMoveDown)
		)
	);
	m_pMenuDown = &m_ptrMenu->items().back();
	
	m_ptrMenu->items().push_back(Gtk::Menu_Helpers::SeparatorElem());
	
	m_ptrMenu->items().push_back(
		Gtk::Menu_Helpers::ImageMenuElem(
			"Delete",
			*manage(new Gtk::Image(Gtk::Stock::DELETE, Gtk::ICON_SIZE_MENU)),
			sigc::mem_fun(*this, &OptionsList::onDelete)
		)
	);
	m_pMenuDelete = &m_ptrMenu->items().back();
}

void
ForwardingConfigWindow::OptionsList::updateMenu()
{
	int const item = getSelectedItem();
	int const num_items = m_ptrModel->children().size();
	m_pMenuUp->set_sensitive(item > 0);
	m_pMenuDown->set_sensitive(item < num_items - 1 && item != -1);
	m_pMenuRename->set_sensitive(item != -1);
	m_pMenuDelete->set_sensitive(item != -1);
}

bool
ForwardingConfigWindow::OptionsList::on_button_press_event(GdkEventButton* evt)
{
	bool const rval = Gtk::TreeView::on_button_press_event(evt);
	
	if (evt->type == GDK_BUTTON_PRESS && evt->button == 3) {
		updateMenu();
		m_ptrMenu->popup(evt->button, evt->time);
	}
	
	return rval;
}

void
ForwardingConfigWindow::OptionsList::startEditing(Gtk::TreeIter const& it)
{
	m_nameCellRenderer.property_editable() = true;
	set_cursor(
		Gtk::TreePath(it), m_nameCol, m_nameCellRenderer,
		true // start_editing
	);
}

void
ForwardingConfigWindow::OptionsList::onEditingCanceled()
{
	m_nameCellRenderer.property_editable() = false;
	Gtk::TreeIter const sel(get_selection()->get_selected());
	assert(sel);
	Gtk::TreeRow const row(*sel);
	if (row[m_cols.state] == ITEM_NOT_REALIZED) {
		// creating
		m_ptrModel->erase(sel);
	}
}

void
ForwardingConfigWindow::OptionsList::onEditingFinished(
	Glib::ustring const& old_text, Glib::ustring const& new_text)
{
	m_nameCellRenderer.property_editable() = false;
	Gtk::TreeIter const sel(get_selection()->get_selected());
	assert(sel);
	Gtk::TreeRow row(*sel);
	
	if (handleItemNewName(sel, new_text)) {
		row[m_cols.name] = new_text;
		row[m_cols.state] = ITEM_REALIZED;
	} else {
		if (row[m_cols.state] == ITEM_NOT_REALIZED) {
			// creating
			m_ptrModel->erase(sel);
		}
	}
}

bool
ForwardingConfigWindow::OptionsList::handleItemNewName(
	Gtk::TreeIter const& item, Glib::ustring const& new_name)
{
	if (new_name.empty()) {
		return false;
	}
	
	int const option_idx = iteratorToIndex(item);
	assert(option_idx >= 0);
	
	Gtk::TreeRow row(*item);
	bool const renaming = (row[m_cols.state] == ITEM_REALIZED);
	
	Forwarding::Utf8String const option_name(new_name.raw());
	Forwarding::Config new_config(m_rOwner.forwardingConfig());
	Forwarding::OptionList::iterator pos(new_config.options().begin());
	std::advance(pos, option_idx);
	Forwarding::OptionList::iterator new_pos(pos);
	if (renaming) {
		pos->setName(option_name);
	} else { // creating
		new_pos = new_config.options().insert(pos, Forwarding::Option(option_name));
	}
	
	Impl::ensureProperSelection(new_config);
	if (!FCF().applyAndSave(new_config)) {
		return false;
	} else {
		m_rOwner.forwardingConfig().swap(new_config);
		if (!renaming) {
			m_rOwner.switchToOption(*new_pos, option_idx);
		}
		return true;
	}
}

void
ForwardingConfigWindow::OptionsList::onNew()
{
	Gtk::TreeRow row(*m_ptrModel->append());
	row[m_cols.state] = ITEM_NOT_REALIZED;
	startEditing(row);
}

void
ForwardingConfigWindow::OptionsList::onRename()
{
	Gtk::TreeIter const sel(get_selection()->get_selected());
	if (sel) {
		startEditing(sel);
	}
}

void
ForwardingConfigWindow::OptionsList::onDelete()
{
	Gtk::TreeIter const sel(get_selection()->get_selected());
	if (!sel) {
		return;
	}
	int const option_idx = iteratorToIndex(sel);
	
	Gtk::MessageDialog dialog(
		m_rOwner,
		"Really delete this profile?",
		false, /* use_markup */
		Gtk::MESSAGE_QUESTION, Gtk::BUTTONS_YES_NO,
		true /* modal */
	);
	int const res = dialog.run();
	if (res != Gtk::RESPONSE_YES) {
		return;
	}
	
	Gtk::TreeRow row(*sel);
	
	if (row[m_cols.state] == ITEM_REALIZED) {
		Forwarding::Config new_config(m_rOwner.forwardingConfig());
		Forwarding::OptionList::iterator it(new_config.options().begin());
		std::advance(it, option_idx);
		new_config.options().erase(it);
		Impl::ensureProperSelection(new_config);
		if (!FCF().applyAndSave(new_config)) {
			return;
		} else {
			m_rOwner.forwardingConfig().swap(new_config);
		}
	}
	m_ptrModel->erase(sel);
}

void
ForwardingConfigWindow::OptionsList::onMoveUp()
{
	Gtk::TreeIter const it(get_selection()->get_selected());
	int const item = iteratorToIndex(it);
	int const prev = item - 1;
	if (prev < 0 /*|| item >= (int)m_ptrModel->children().size()*/) {
		return;
	}
	
	Forwarding::Config new_config(m_rOwner.forwardingConfig());
	swapOptions(new_config, item, prev);
	
	Impl::ensureProperSelection(new_config);
	if (!FCF().applyAndSave(new_config)) {
		return;
	} else {
		m_rOwner.forwardingConfig().swap(new_config);
	}
	
	Gtk::TreeIter target(it);
	--target;
	m_ptrModel->iter_swap(it, target);
}

void
ForwardingConfigWindow::OptionsList::onMoveDown()
{
	Gtk::TreeIter const it(get_selection()->get_selected());
	int const item = iteratorToIndex(it);
	int next = item + 1;
	if (item < 0 || next >= (int)m_ptrModel->children().size()) {
		return;
	}
	
	Forwarding::Config new_config(m_rOwner.forwardingConfig());
	swapOptions(new_config, item, next);
	
	Impl::ensureProperSelection(new_config);
	if (!FCF().applyAndSave(new_config)) {
		return;
	} else {
		m_rOwner.forwardingConfig().swap(new_config);
	}

	Gtk::TreeIter target(it);
	++target;
	m_ptrModel->iter_swap(it, target);
}

void
ForwardingConfigWindow::OptionsList::onSelectionChanged()
{
	Gtk::TreeIter const sel(get_selection()->get_selected());
	if (!sel) {
		m_rOwner.switchToNoOption();
		m_rOwner.showEditor(m_rOwner.SHOW_NONE);
		return;
	}
	
	Gtk::TreeRow row(*sel);
	
	if (row[m_cols.state] == ITEM_NOT_REALIZED) {
		m_rOwner.switchToNoOption();
		m_rOwner.showEditor(m_rOwner.SHOW_NONE);
		return;
	}
	
	int const option_idx = iteratorToIndex(sel);
	assert(option_idx >= 0);
	
	Forwarding::Config const& config = m_rOwner.forwardingConfig();
	Forwarding::OptionList::const_iterator opt(config.options().begin());
	std::advance(opt, option_idx);

	m_rOwner.switchToOption(*opt, option_idx);
	if (opt->proxyChain().empty()) {
		m_rOwner.showEditor(m_rOwner.SHOW_NONE);
	} else {
		m_rOwner.showEditor(m_rOwner.SHOW_BYPASS_EDITOR);
	}
}

void
ForwardingConfigWindow::OptionsList::swapOptions(
	Forwarding::Config& config, int const idx1, int const idx2)
{
	Forwarding::OptionList::iterator it1(config.options().begin());
	Forwarding::OptionList::iterator it2(it1);
	std::advance(it1, idx1);
	std::advance(it2, idx2);
	it1->swap(*it2);
}

bool
ForwardingConfigWindow::OptionsList::on_enter_notify_event(GdkEventCrossing* evt)
{
	// The base-class version returns true, which prevents
	// MouseOverStatusMsg from working.
	Gtk::TreeView::on_enter_notify_event(evt);
	return false; // propagate event further
}

bool
ForwardingConfigWindow::OptionsList::on_leave_notify_event(GdkEventCrossing* evt)
{
	// Same as above.
	Gtk::TreeView::on_enter_notify_event(evt);
	return false; // propagate event further
}


/*================ ForwardingConfigWindow::ProxyChainPanel ===============*/

ForwardingConfigWindow::ProxyChainPanel::ProxyChainPanel(
	Impl& owner, ProxyChainPanel const* prototype /* = 0*/)
:	m_rOwner(owner),
	m_pViewport(0),
	m_pAlignment(0),
	m_insertMsg("Insert a proxy"),
	m_editDeleteMsg("Click to edit / Ctrl + click to delete"),
	m_insertStatusMsg(owner.statusBar(), m_insertMsg),
	m_editDeleteStatusMsg(owner.statusBar(), m_editDeleteMsg)
{
	if (prototype) {
		copyImages(*prototype);
	} else {
		loadImages();
	}
	
	if (prototype) {
		set_policy(Gtk::POLICY_NEVER, Gtk::POLICY_NEVER);
	} else {
		set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_NEVER);
	}
	set_shadow_type(Gtk::SHADOW_NONE);
	
	m_pViewport = manage(
		new Gtk::Viewport(*get_hadjustment(), *get_vadjustment())
	);
	add(*m_pViewport);
	m_pViewport->set_shadow_type(Gtk::SHADOW_OUT);
	
	m_pAlignment = manage(new Gtk::Alignment(0.5f, 0.5f, 0.0f, 0.0f));
	m_pViewport->add(*m_pAlignment);
	m_pAlignment->set_padding(4, 0, 4, 0); // top and left padding
	
	// create a dummy option with two proxies
	Forwarding::Utf8String const option_name;
	Forwarding::Option dummy_option(option_name);
	for (int i = 0; i < 2; ++i) {
		dummy_option.proxyChain().push_back(ProxyDescriptor());
	}
	
	loadOption(dummy_option);
}

ForwardingConfigWindow::ProxyChainPanel::~ProxyChainPanel()
{
	// prevent slots being called during destruction
	sigc::trackable::notify_callbacks();
}

void
ForwardingConfigWindow::ProxyChainPanel::switchToNormalMode()
{
	int const num_buttons = m_buttons.size();

	for (int i = 1; i < num_buttons; i += 2) {
		SkinnedButton* forward_btn = m_buttons[i];
		forward_btn->setEnabled(true);
	}

	for (int i = 2; i < num_buttons - 1; i += 2) {
		SkinnedButton* proxy_btn = m_buttons[i];
		proxy_btn->setEnabled(true);
	}
	
	m_insertStatusMsg.setMessage(m_insertMsg);
	m_editDeleteStatusMsg.setMessage(m_editDeleteMsg);
}

void
ForwardingConfigWindow::ProxyChainPanel::switchToInsensitiveMode()
{
	int const num_buttons = m_buttons.size();

	for (int i = 1; i < num_buttons; i += 2) {
		SkinnedButton* forward_btn = m_buttons[i];
		forward_btn->setDisabledSkin(m_ptrForwardPixbuf);
		forward_btn->setEnabled(false);
	}

	for (int i = 2; i < num_buttons - 1; i += 2) {
		SkinnedButton* proxy_btn = m_buttons[i];
		proxy_btn->setDisabledSkin(m_ptrProxyPixbuf);
		proxy_btn->setEnabled(false);
	}
	
	m_insertStatusMsg.setMessage(Glib::ustring());
	m_editDeleteStatusMsg.setMessage(Glib::ustring());
}

void
ForwardingConfigWindow::ProxyChainPanel::switchToProxyAddingMode(
	int const transition_idx)
{
	int const num_buttons = m_buttons.size();
	int const selected_idx = transition_idx * 2 + 1;

	for (int i = 1; i < num_buttons; i += 2) {
		SkinnedButton* forward_btn = m_buttons[i];
		if (i == selected_idx) {
			forward_btn->setDisabledSkin(m_ptrProxyPixbuf);
		} else {
			forward_btn->setDisabledSkin(m_ptrForwardPixbuf);
		}
		forward_btn->setEnabled(false);
	}

	for (int i = 2; i < num_buttons - 1; i += 2) {
		SkinnedButton* proxy_btn = m_buttons[i];
		proxy_btn->setDisabledSkin(m_ptrProxyGhostPixbuf);
		proxy_btn->setEnabled(false);
	}
	
	m_insertStatusMsg.setMessage(Glib::ustring());
	m_editDeleteStatusMsg.setMessage(Glib::ustring());
}

void
ForwardingConfigWindow::ProxyChainPanel::switchToProxyEditingMode(
	int const proxy_idx)
{
	int const num_buttons = m_buttons.size();
	int const selected_idx = proxy_idx * 2 + 2;

	for (int i = 1; i < num_buttons; i += 2) {
		SkinnedButton* forward_btn = m_buttons[i];
		forward_btn->setDisabledSkin(m_ptrForwardPixbuf);
		forward_btn->setEnabled(false);
	}

	for (int i = 2; i < num_buttons - 1; i += 2) {
		SkinnedButton* proxy_btn = m_buttons[i];
		if (i == selected_idx) {
			proxy_btn->setDisabledSkin(m_ptrProxyPixbuf);
		} else {
			proxy_btn->setDisabledSkin(m_ptrProxyGhostPixbuf);
		}
		proxy_btn->setEnabled(false);
	}
	
	m_insertStatusMsg.setMessage(Glib::ustring());
	m_editDeleteStatusMsg.setMessage(Glib::ustring());
}

void
ForwardingConfigWindow::ProxyChainPanel::loadImages()
{
	m_ptrDesktopPixbuf = CompiledImages::desktop_png.getPixbuf();
	m_ptrProxyPixbuf = CompiledImages::proxy_png.getPixbuf();
	m_ptrProxyGhostPixbuf = CompiledImages::proxy_ghost_png.getPixbuf();
	m_ptrInternetPixbuf = CompiledImages::internet_png.getPixbuf();
	m_ptrForwardPixbuf = CompiledImages::forward_png.getPixbuf();
}

void
ForwardingConfigWindow::ProxyChainPanel::copyImages(ProxyChainPanel const& from)
{
	m_ptrDesktopPixbuf = from.m_ptrDesktopPixbuf;
	m_ptrProxyPixbuf = from.m_ptrProxyPixbuf;
	m_ptrProxyGhostPixbuf = from.m_ptrProxyGhostPixbuf;
	m_ptrInternetPixbuf = from.m_ptrInternetPixbuf;
	m_ptrForwardPixbuf = from.m_ptrForwardPixbuf;
}

void
ForwardingConfigWindow::ProxyChainPanel::clearChain()
{
	m_pAlignment->remove();
	m_ptrTable.reset();
	m_buttons.clear();
}

void
ForwardingConfigWindow::ProxyChainPanel::loadOption(Forwarding::Option const& opt)
{
	Gdk::Cursor const hand_cursor(Gdk::HAND2);
	
	clearChain();
	
	/*
	Table layout:
	+-------+---+---+------+---+---+--------+
	| +---+ |       | +--+ |       |  ****  |
	| |   | |  ==>  | |  | |  ==>  | ****** |
	| +---+ |       | +--+ |       |  ****  |
	+-------+---+---+------+---+---+--------+
	|This Comput|   host:port  |   |Internet|
	+-------+---+---+------+---+---+--------+
	|       | # | # |      | # | # |        +
	+-------+---+---+------+---+---+--------+
	The last row is invisible, and its only purpose is to specify
	minimum sizes for columns marked with #.  Without them, Gtk::Table
	will add unnecessary spaces here and there.
	*/
	
	int const num_proxies = opt.proxyChain().size();
	int const num_cols = num_proxies + (num_proxies + 1) * 2 + 2;
	m_ptrTable.reset(new Gtk::Table(/* rows = */3, /* cols = */num_cols));
	m_pAlignment->add(*m_ptrTable);
	m_ptrTable->set_row_spacings(3);
	
	addDesktopButton(0);
	addForwardButton(1, hand_cursor); // takes 2 cols
	
	Gtk::Label* this_computer = manage(new Gtk::Label("This Computer"));
	m_ptrTable->attach(
		*this_computer,
		0, 2, 1, 2, // left, right, top, bottom
		Gtk::SHRINK|Gtk::FILL
	);
	this_computer->set_alignment(0.0f, 0.5f);
	
	int col = 3;
	
	typedef Forwarding::ProxyDescriptorList::const_iterator Iter;
	Iter it(opt.proxyChain().begin());
	Iter const end(opt.proxyChain().end());
	for (; it != end; ++it, col += 3) {
		ProxyDescriptor const& proxy = *it;
		
		addProxyButton(col, hand_cursor);
		addForwardButton(col + 1, hand_cursor); // takes 2 cols
		
		Gtk::Label* label = manage(new Gtk::Label());
		m_ptrTable->attach(
			*manage(new Gtk::Label(proxy.getAddr().toString())),
			col - 1, col + 2, 1, 2, // left, right, top, bottom
			Gtk::SHRINK
		);
	}

	addInternetButton(col);
	
	m_ptrTable->attach(
		*manage(new Gtk::Label("Internet")),
		col, col + 1, 1, 2, // left, right, top, bottom
		Gtk::SHRINK
	);
	
	m_ptrTable->show_all();
}

void
ForwardingConfigWindow::ProxyChainPanel::addDesktopButton(int const col)
{
	SkinnedButton* desktop_btn = manage(new SkinnedButton(m_ptrDesktopPixbuf));
	m_ptrTable->attach(*desktop_btn, col, col + 1, 0, 1, Gtk::SHRINK);
	m_buttons.push_back(desktop_btn);
	desktop_btn->setEnabled(false);
	
	// extra row
	Gtk::Alignment* spacer = manage(new Gtk::Alignment);
	m_ptrTable->attach(*spacer, col, col + 1, 2, 3, Gtk::SHRINK);
}

void
ForwardingConfigWindow::ProxyChainPanel::addForwardButton(
	int const col, Gdk::Cursor const& cursor)
{
	SkinnedButton* forward_btn = manage(new SkinnedButton(m_ptrForwardPixbuf));
	m_ptrTable->attach(*forward_btn, col, col + 2, 0, 1, Gtk::SHRINK);
	forward_btn->setHoverSkin(m_ptrProxyGhostPixbuf);
	forward_btn->setHoverCursor(cursor);
	
	int const transition_idx = m_buttons.size() / 2;
	forward_btn->signalClicked().connect(
		sigc::bind(
			sigc::mem_fun(*this, &ProxyChainPanel::onForwardClicked),
			transition_idx
		)
	);
	
	m_buttons.push_back(forward_btn);
	m_insertStatusMsg.connectToWidget(*forward_btn);
	
	// extra row
	int const width = m_ptrForwardPixbuf->get_width() / 2;
	
	Gtk::Alignment* spacer1 = manage(new Gtk::Alignment);
	m_ptrTable->attach(*spacer1, col, col + 1, 2, 3, Gtk::EXPAND);
	spacer1->set_size_request(width, 0);
	
	Gtk::Alignment* spacer2 = manage(new Gtk::Alignment);
	m_ptrTable->attach(*spacer2, col + 1, col + 2, 2, 3, Gtk::EXPAND);
	spacer2->set_size_request(width, 0);
}

void
ForwardingConfigWindow::ProxyChainPanel::addProxyButton(
	int const col, Gdk::Cursor const& cursor)
{
	SkinnedButton* proxy_btn = manage(new SkinnedButton(m_ptrProxyPixbuf));
	m_ptrTable->attach(*proxy_btn, col, col + 1, 0, 1, Gtk::SHRINK);
	proxy_btn->setHoverCursor(cursor);
	
	int const proxy_idx = m_buttons.size() / 2 - 1;
	proxy_btn->signalClicked().connect(
		sigc::bind(
			sigc::mem_fun(*this, &ProxyChainPanel::onProxyClicked),
			proxy_idx
		)
	);
	
	m_buttons.push_back(proxy_btn);
	m_editDeleteStatusMsg.connectToWidget(*proxy_btn);
	
	// extra row
	Gtk::Alignment* spacer = manage(new Gtk::Alignment);
	m_ptrTable->attach(*spacer, col, col + 1, 2, 3, Gtk::SHRINK);
}

void
ForwardingConfigWindow::ProxyChainPanel::addInternetButton(int const col)
{
	SkinnedButton* internet_btn = manage(new SkinnedButton(m_ptrInternetPixbuf));
	m_ptrTable->attach(*internet_btn, col, col + 1, 0, 1, Gtk::SHRINK);
	m_buttons.push_back(internet_btn);
	internet_btn->setEnabled(false);
	
	// extra row
	Gtk::Alignment* spacer = manage(new Gtk::Alignment);
	m_ptrTable->attach(*spacer, col, col + 1, 2, 3, Gtk::SHRINK);
}

void
ForwardingConfigWindow::ProxyChainPanel::onProxyClicked(
	Gdk::ModifierType const state, int const proxy_idx)
{
	if (state & Gdk::CONTROL_MASK) {
		m_rOwner.deleteProxyInteractive(proxy_idx);
	} else {
		m_rOwner.startProxyEditing(proxy_idx);
	}
}

void
ForwardingConfigWindow::ProxyChainPanel::onForwardClicked(
	Gdk::ModifierType, int const transition_idx)
{
	ProxyDescriptor const* proxy = m_rOwner.getPrecedingProxy(transition_idx);
	if (proxy && proxy->getType() == ProxyDescriptor::HTTP) {
		m_rOwner.showError("No proxy can follow an HTTP proxy.");
	} else {
		m_rOwner.startProxyAdding(transition_idx);
	}
}


/*================ ForwardingConfigWindow::ProxyEditPanel ================*/

ForwardingConfigWindow::ProxyEditPanel::ProxyOption const
ForwardingConfigWindow::ProxyEditPanel::m_sProxyOptions[] = {
	{ ProxyDescriptor::HTTP, "HTTP proxy" },
	{ ProxyDescriptor::SOCKS4, "SOCKS4 proxy" },
	{ ProxyDescriptor::SOCKS4A, "SOCKS4A proxy" },
	{ ProxyDescriptor::SOCKS5, "SOCKS5 proxy" }
};

ForwardingConfigWindow::ProxyEditPanel::ProxyEditPanel(Impl& owner)
:	m_rOwner(owner),
	m_mode(INACTIVE),
	m_optionIdx(0),
	m_proxyIdx(0),
	m_pProxyType(0),
	m_pProxyHost(0),
	m_pProxyPort(0),
	m_pProxyUser(0),
	m_pProxyPass(0)
{
	set_spacing(3);
	
	m_pProxyType = manage(new Gtk::ComboBoxText);
	pack_start(*m_pProxyType, Gtk::PACK_SHRINK);
	for (int i = 0; i < (int)ARRAY_SIZE(m_sProxyOptions); ++i) {
		m_pProxyType->append_text(m_sProxyOptions[i].label);
	}
	m_pProxyType->signal_changed().connect(
		sigc::mem_fun(*this, &ProxyEditPanel::onProxyTypeChanged)
	);
	
	Gtk::HBox* hp_hbox = manage(new Gtk::HBox);
	pack_start(*hp_hbox, Gtk::PACK_SHRINK);
	hp_hbox->set_spacing(5);
	
	m_pProxyHost = manage(new Gtk::Entry);
	hp_hbox->pack_start(*m_pProxyHost);
	m_pProxyHost->set_size_request(130);
	hp_hbox->pack_start(*manage(new Gtk::Label(":")), Gtk::PACK_SHRINK);
	m_pProxyPort = manage(new Gtk::Entry);
	hp_hbox->pack_start(*m_pProxyPort);
	m_pProxyPort->set_size_request(50);
	
	Gtk::Table* user_pass_table = manage(new Gtk::Table(2, 2));
	pack_start(*user_pass_table, Gtk::PACK_SHRINK);
	user_pass_table->set_row_spacings(3);
	user_pass_table->set_col_spacings(3);
	
	user_pass_table->attach(
		*manage(new Gtk::Label("Username:", Gtk::ALIGN_RIGHT)),
		0, 1, 0, 1, Gtk::SHRINK
	);
	
	m_pProxyUser = manage(new Gtk::Entry);
	user_pass_table->attach(*m_pProxyUser, 1, 2, 0, 1);
	m_pProxyUser->set_size_request(100);
	
	user_pass_table->attach(
		*manage(new Gtk::Label("Password:", Gtk::ALIGN_RIGHT)),
		0, 1, 1, 2, Gtk::SHRINK
	);
	
	m_pProxyPass = manage(new Gtk::Entry);
	user_pass_table->attach(*m_pProxyPass, 1, 2, 1, 2);
	m_pProxyPass->set_size_request(100);
	
	Gtk::Alignment* stretch_spacer = manage(new Gtk::Alignment);
	pack_start(*stretch_spacer);
	
	Gtk::Alignment* btn_box_align = manage(new Gtk::Alignment(0.5f, 0.5f, 0.0f, 0.0f));
	pack_start(*btn_box_align, Gtk::PACK_SHRINK);
	
	Gtk::HButtonBox* btn_box = manage(new Gtk::HButtonBox);
	btn_box_align->add(*btn_box);
	btn_box->set_spacing(10);
	
	Gtk::Button* cancel_btn = manage(new Gtk::Button("Cancel"));
	btn_box->pack_start(*cancel_btn, Gtk::PACK_SHRINK);
	cancel_btn->signal_clicked().connect(
		sigc::mem_fun(*this, &ProxyEditPanel::onCancel)
	);
	
	Gtk::Button* ok_btn = manage(new Gtk::Button("OK"));
	btn_box->pack_start(*ok_btn, Gtk::PACK_SHRINK);
	ok_btn->signal_clicked().connect(
		sigc::mem_fun(*this, &ProxyEditPanel::onOK)
	);
}

ForwardingConfigWindow::ProxyEditPanel::~ProxyEditPanel()
{
}

void
ForwardingConfigWindow::ProxyEditPanel::grabFocus()
{
	m_pProxyType->grab_focus();
}

void
ForwardingConfigWindow::ProxyEditPanel::startProxyAdding(
	int const option_idx, int const transition_idx)
{
	m_mode = ADDING;
	m_optionIdx = option_idx;
	m_proxyIdx = transition_idx;

	m_pProxyType->set_active(-1);
	
	Glib::ustring const empty;
	
	m_pProxyHost->set_text(empty);
	m_pProxyHost->set_sensitive();

	m_pProxyPort->set_text(empty);
	m_pProxyPort->set_sensitive();
	
	m_pProxyUser->set_text(empty);
	m_pProxyUser->set_sensitive();
	
	m_pProxyPass->set_text(empty);
	m_pProxyPass->set_sensitive();
}

void
ForwardingConfigWindow::ProxyEditPanel::startProxyEditing(
	int const option_idx, int const proxy_idx)
{
	Forwarding::OptionList& options = m_rOwner.forwardingConfig().options();

	assert(option_idx >= 0);
	assert(option_idx < (int)options.size());
	
	Forwarding::OptionList::iterator opt(options.begin());
	std::advance(opt, option_idx);
	
	Forwarding::ProxyDescriptorList& proxies = opt->proxyChain();

	assert(proxy_idx >= 0);
	assert(proxy_idx < (int)proxies.size());
	
	Forwarding::ProxyDescriptorList::iterator proxy(proxies.begin());
	std::advance(proxy, proxy_idx);

	m_mode = EDITING;
	m_optionIdx = option_idx;
	m_proxyIdx = proxy_idx;

	setProxyType(proxy->getType());
	
	// Strings in ProxyDescriptor are actually UTF-8, as they came either
	// from forwarding.xml or from the GUI.

	m_pProxyHost->set_text(proxy->getAddr().getHost());

	std::ostringstream port;
	port << proxy->getAddr().getPort();
	m_pProxyPort->set_text(port.str());

	m_pProxyUser->set_text(proxy->getUserName());
	m_pProxyPass->set_text(proxy->getPassword());
}

void
ForwardingConfigWindow::ProxyEditPanel::cancelCurrentOperation()
{
	m_mode = INACTIVE;
	m_optionIdx = 0;
	m_proxyIdx = 0;
}

ProxyDescriptor::ProxyType
ForwardingConfigWindow::ProxyEditPanel::getProxyType() const
{
	int const selection = m_pProxyType->get_active_row_number();
	if (selection == -1) {
		return ProxyDescriptor::INVALID;
	} else {
		return m_sProxyOptions[selection].type;
	}
	return ProxyDescriptor::INVALID;
}

void
ForwardingConfigWindow::ProxyEditPanel::setProxyType(
	ProxyDescriptor::ProxyType type)
{
	for (unsigned i = 0; i < ARRAY_SIZE(m_sProxyOptions); ++i) {
		if (m_sProxyOptions[i].type == type) {
			m_pProxyType->set_active(i);
			onProxyTypeChangedTo(type);
			return;
		}
	}
	m_pProxyType->set_active(-1);
	onProxyTypeChangedTo(ProxyDescriptor::INVALID);
}

void
ForwardingConfigWindow::ProxyEditPanel::onProxyTypeChanged()
{
	onProxyTypeChangedTo(getProxyType());
}

void
ForwardingConfigWindow::ProxyEditPanel::onProxyTypeChangedTo(
	ProxyDescriptor::ProxyType type)
{
	switch (type) {
		case ProxyDescriptor::INVALID:
		m_pProxyHost->set_sensitive(false);
		m_pProxyPort->set_sensitive(false);
		m_pProxyUser->set_sensitive(false);
		m_pProxyPass->set_sensitive(false);
		break;
		case ProxyDescriptor::HTTP:
		m_pProxyHost->set_sensitive(true);
		m_pProxyPort->set_sensitive(true);
		m_pProxyUser->set_sensitive(false);
		m_pProxyPass->set_sensitive(false);
		break;
		case ProxyDescriptor::SOCKS4:
		case ProxyDescriptor::SOCKS4A:
		m_pProxyHost->set_sensitive(true);
		m_pProxyPort->set_sensitive(true);
		m_pProxyUser->set_sensitive(true);
		m_pProxyPass->set_sensitive(false);
		break;
		case ProxyDescriptor::SOCKS5:
		m_pProxyHost->set_sensitive(true);
		m_pProxyPort->set_sensitive(true);
		m_pProxyUser->set_sensitive(true);
		m_pProxyPass->set_sensitive(true);
		break;
		default:
		assert(0);
	}
}

bool
ForwardingConfigWindow::ProxyEditPanel::validateInteractive(
	ProxyDescriptor& target) const
{
	ProxyDescriptor::ProxyType const type = getProxyType();
	if (type == ProxyDescriptor::INVALID) {
		m_pProxyType->grab_focus();
		m_rOwner.showError("Proxy type is not selected.");
		return false;
	} else if (type == ProxyDescriptor::HTTP && haveNextHopProxy()) {
		m_pProxyType->grab_focus();
		m_rOwner.showError("HTTP proxies can't forward to another proxy.");
		return false;
	}
	
	Glib::ustring const host(trim(m_pProxyHost->get_text()));
	if (host.empty()) {
		m_pProxyHost->grab_focus();
		m_rOwner.showError("Proxy host is required.");
		return false;
	}
	
	uint16_t port = 0;
	try {
		Glib::ustring const prt(trim(m_pProxyPort->get_text()));
		port = boost::lexical_cast<uint16_t>(prt.raw());
	} catch (boost::bad_lexical_cast&) {}
	if (port == 0) {
		m_pProxyPort->grab_focus();
		m_rOwner.showError("Proxy port must be an integer in range of [1, 65535]");
		return false;
	}
	
	Glib::ustring user(m_pProxyUser->get_text());
	Glib::ustring pass(m_pProxyPass->get_text());
	if (type == ProxyDescriptor::HTTP) {
		user.clear();
	}
	if (type != ProxyDescriptor::SOCKS5) {
		pass.clear();
	}

	ProxyDescriptor proxy;
	proxy.setType(type);
	SymbolicInetAddr const addr(host.raw(), port);
	proxy.setAddr(addr);
	proxy.setUserName(user.raw());
	proxy.setPassword(pass.raw());
	proxy.swap(target);
	return true;
}

bool
ForwardingConfigWindow::ProxyEditPanel::haveNextHopProxy() const
{
	Forwarding::Config const& config = m_rOwner.forwardingConfig();
	
	assert(m_optionIdx >= 0);
	assert(m_optionIdx < (int)config.options().size());

	Forwarding::OptionList::const_iterator opt(config.options().begin());
	std::advance(opt, m_optionIdx);
	
	int const next = (m_mode == ADDING ? m_proxyIdx : m_proxyIdx + 1);
	return (next < (int)opt->proxyChain().size());
}

Glib::ustring
ForwardingConfigWindow::ProxyEditPanel::trim(Glib::ustring const& str)
{
	Glib::ustring::const_iterator begin(str.begin());
	Glib::ustring::const_iterator end(str.end());
	for (; begin != end && Glib::Unicode::isspace(*begin); ++begin) {
		// skip leading spaces
	}
	if (begin == end) {
		return Glib::ustring(begin, end);
	}
	
	assert(!Glib::Unicode::isspace(*begin));
	
	--end;
	for (; end != begin && Glib::Unicode::isspace(*end); --end) {
		// skip trailing spaces
	}
	++end;
	return Glib::ustring(begin, end);
}

void
ForwardingConfigWindow::ProxyEditPanel::onOK()
{
	ProxyDescriptor pdesc;
	if (!validateInteractive(pdesc)) {
		return;
	}

	Forwarding::Config new_config(m_rOwner.forwardingConfig());
	
	assert(m_optionIdx >= 0);
	assert(m_optionIdx < (int)new_config.options().size());

	Forwarding::OptionList::iterator opt(new_config.options().begin());
	std::advance(opt, m_optionIdx);

	assert(m_proxyIdx >= 0);
	assert(m_proxyIdx <= (int)opt->proxyChain().size());
	// transition_idx may point past the last proxy

	Forwarding::ProxyDescriptorList::iterator proxy(opt->proxyChain().begin());
	std::advance(proxy, m_proxyIdx);

	if (m_mode == ADDING) {
		opt->proxyChain().insert(proxy, pdesc);
	} else if (m_mode == EDITING) {
		proxy->swap(pdesc);
	} else {
		assert(0);
	}
	
	Impl::ensureProperSelection(new_config);
	if (!FCF().applyAndSave(new_config)) {
		return;
	} else {
		m_rOwner.forwardingConfig().swap(new_config);
	}
	
	m_rOwner.commitProxyOperation();
}

void
ForwardingConfigWindow::ProxyEditPanel::onCancel()
{
	m_rOwner.stopProxyOperation();
}


/*================= ForwardingConfigWindow::BypasssPanel =================*/

ForwardingConfigWindow::BypassPanel::BypassPanel(Impl& owner)
:	m_rOwner(owner),
	m_ptrBypassHosts(Gtk::TextBuffer::create()),
	m_pBypassHostsView(0),
	m_pBypassLocal(0),
	m_pOKBtn(0),
	m_pCancelBtn(0),
	m_optionIdx(-1),
	m_ignoreModifications(0),
	m_bypassLocalInitial(false),
	m_wasModified(false)
{
	{
		Gtk::Alignment* spacer = manage(new Gtk::Alignment);
		spacer->set_size_request(5, 5);
		pack_start(*spacer, Gtk::PACK_SHRINK);
	}
	
	{
		Gtk::HBox* hbox = manage(new Gtk::HBox);
		pack_start(*hbox, Gtk::PACK_SHRINK);
		
		Gtk::Label* label = manage(
			new Gtk::Label("Direct connection to certain hosts:")
		);
		hbox->pack_start(*label, Gtk::PACK_SHRINK);
		
		Link* help_link = manage(new Link(" (?)"));
		hbox->pack_start(*help_link, Gtk::PACK_SHRINK);
		help_link->signalClicked().connect(
			sigc::mem_fun(*this, &BypassPanel::onBypassHostsHelp)
		);
	}
	
	{
		Gtk::Alignment* spacer = manage(new Gtk::Alignment);
		spacer->set_size_request(3, 3);
		pack_start(*spacer, Gtk::PACK_SHRINK);
	}
	
	Gtk::ScrolledWindow* text_scroller = manage(new Gtk::ScrolledWindow);
	pack_start(*text_scroller);
	
	text_scroller->set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
	text_scroller->set_shadow_type(Gtk::SHADOW_IN);
	text_scroller->set_size_request(-1, 40);
	
	m_pBypassHostsView = manage(new Gtk::TextView(m_ptrBypassHosts));
	text_scroller->add(*m_pBypassHostsView);
	m_pBypassHostsView->set_wrap_mode(Gtk::WRAP_WORD);
	m_ptrBypassHosts->signal_modified_changed().connect(
		sigc::mem_fun(*this, &BypassPanel::onBypassHostsModified)
	);
	
	{
		Gtk::Alignment* spacer = manage(new Gtk::Alignment);
		spacer->set_size_request(3, 3);
		pack_start(*spacer, Gtk::PACK_SHRINK);
	}
	
	{
		Gtk::HBox* hbox = manage(new Gtk::HBox);
		pack_start(*hbox, Gtk::PACK_SHRINK);
		
		m_pBypassLocal = manage(
			new Gtk::CheckButton("Direct connection to local addresses")
		);
		hbox->pack_start(*m_pBypassLocal, Gtk::PACK_SHRINK);
		m_pBypassLocal->signal_toggled().connect(
			sigc::mem_fun(*this, &BypassPanel::onBypassLocalModified)
		);
		
		Link* help_link = manage(new Link(" (?)"));
		hbox->pack_start(*help_link, Gtk::PACK_SHRINK);
		help_link->signalClicked().connect(
			sigc::mem_fun(*this, &BypassPanel::onBypassLocalHelp)
		);
	}
	
	{
		Gtk::Alignment* spacer = manage(new Gtk::Alignment);
		spacer->set_size_request(5, 5);
		pack_start(*spacer, Gtk::PACK_SHRINK);
	}
	
	Gtk::Alignment* btn_box_align = manage(new Gtk::Alignment(0.5f, 0.5f, 0.0f, 0.0f));
	pack_start(*btn_box_align, Gtk::PACK_SHRINK);
	
	Gtk::HButtonBox* btn_box = manage(new Gtk::HButtonBox);
	btn_box_align->add(*btn_box);
	btn_box->set_spacing(10);
	
	m_pCancelBtn = manage(new Gtk::Button("Cancel"));
	btn_box->pack_start(*m_pCancelBtn, Gtk::PACK_SHRINK);
	m_pCancelBtn->set_sensitive(false);
	m_pCancelBtn->signal_clicked().connect(
		sigc::mem_fun(*this, &BypassPanel::onCancel)
	);
	
	m_pOKBtn = manage(new Gtk::Button("OK"));
	btn_box->pack_start(*m_pOKBtn, Gtk::PACK_SHRINK);
	m_pOKBtn->set_sensitive(false);
	m_pOKBtn->signal_clicked().connect(
		sigc::mem_fun(*this, &BypassPanel::onOK)
	);
}

ForwardingConfigWindow::BypassPanel::~BypassPanel()
{
	removeTooltips();
}

void
ForwardingConfigWindow::BypassPanel::loadFrom(
	Forwarding::Option const& opt, int const option_idx)
{
	ScopedIncDec<unsigned char> guard(m_ignoreModifications);
	m_optionIdx = option_idx;
	m_ptrBypassHosts->set_text(Glib::ustring());
	m_pBypassLocal->set_active(false);
	opt.bypassList().accept(*this);
	m_ptrBypassHosts->set_modified(false);
	m_bypassLocalInitial = m_pBypassLocal->get_active();
	m_wasModified = false;
	m_pOKBtn->set_sensitive(false);
	m_pCancelBtn->set_sensitive(false);
}

void
ForwardingConfigWindow::BypassPanel::removeTooltips()
{
	// Deleting a NULL pointer is not an error.
	delete &*m_ptrBypassHostsTip;
	delete &*m_ptrBypassLocalTip;
}

void
ForwardingConfigWindow::BypassPanel::visit(
	Forwarding::HostMaskMatchPattern const& p)
{
	if (m_ptrBypassHosts->begin() != m_ptrBypassHosts->end()) {
		m_ptrBypassHosts->insert(m_ptrBypassHosts->end(), Glib::ustring(", "));
	}
	m_ptrBypassHosts->insert(m_ptrBypassHosts->end(), p.getPattern().raw());
}

void
ForwardingConfigWindow::BypassPanel::visit(
	Forwarding::SimpleHostnamesMatchPattern const& p)
{
	m_pBypassLocal->set_active(true);
}

void
ForwardingConfigWindow::BypassPanel::onBypassHostsHelp()
{
	Glib::ustring const markup(
		"Examples:\n"
		"host.com, *.host.com, 192.168.*.*"
	);
	
	removeTooltips();
	
	// view::ToolTip deletes itself when closed
	m_ptrBypassHostsTip = new view::ToolTip(*m_pBypassHostsView, markup);
	m_ptrBypassHostsTip->show();
}

void
ForwardingConfigWindow::BypassPanel::onBypassLocalHelp()
{
	Glib::ustring const markup(
		"Examples of local addresses:\n"
		"http://files/upload/\n"
		"http://webmail/login.php"
	);
	
	removeTooltips();
	
	// view::ToolTip deletes itself when closed
	m_ptrBypassHostsTip = new view::ToolTip(*m_pBypassLocal, markup);
	m_ptrBypassHostsTip->show();
}

void
ForwardingConfigWindow::BypassPanel::onBypassHostsModified()
{
	if (m_ignoreModifications) {
		return;
	}

	if (m_ptrBypassHosts->get_modified()) {
		if (!m_wasModified) {
			onModified(true);
		}
	} else if (m_wasModified) {
		if (m_pBypassLocal->get_active() == m_bypassLocalInitial) {
			onModified(false);
		}
	}
}

void
ForwardingConfigWindow::BypassPanel::onBypassLocalModified()
{
	if (m_ignoreModifications) {
		return;
	}
	
	if (m_pBypassLocal->get_active() != m_bypassLocalInitial) {
		if (!m_wasModified) {
			onModified(true);
			
		}
	} else if (m_wasModified) {
		if (!m_ptrBypassHosts->get_modified()) {
			onModified(false);
		}
	}
}

void
ForwardingConfigWindow::BypassPanel::onModified(bool const modified)
{
	if (modified == m_wasModified) {
		return;
	}

	if (modified) {
		m_rOwner.enterBypassEditingMode();
	} else {
		m_rOwner.leaveBypassEditingMode();
	}

	m_pOKBtn->set_sensitive(modified);
	m_pCancelBtn->set_sensitive(modified);

	m_wasModified = modified;
}

void
ForwardingConfigWindow::BypassPanel::onOK()
{
	Forwarding::Config new_config(m_rOwner.forwardingConfig());
	Forwarding::OptionList::iterator opt(new_config.options().begin());
	assert(m_optionIdx >= 0);
	assert(m_optionIdx < (int)new_config.options().size());
	std::advance(opt, m_optionIdx);
	opt->bypassList().clear();

	Glib::ustring hosts(m_ptrBypassHosts->get_text());
	for (;;) {
		Glib::ustring const pattern(nextPattern(hosts));
		if (pattern.empty()) {
			break;
		}
		opt->bypassList().push_back(
			Forwarding::BypassPatternConstPtr(
				new Forwarding::HostMaskMatchPattern(
					Forwarding::Utf8String(pattern.raw())
				)
			)
		);
	}

	if (m_pBypassLocal->get_active()) {
		opt->bypassList().push_back(
			Forwarding::BypassPatternConstPtr(
				new Forwarding::SimpleHostnamesMatchPattern()
			)
		);
	}

	Impl::ensureProperSelection(new_config);
	if (!FCF().applyAndSave(new_config)) {
		return;
	} else {
		m_rOwner.forwardingConfig().swap(new_config);
	}
	
	onModified(false);
}

void
ForwardingConfigWindow::BypassPanel::onCancel()
{
	onModified(false);
	m_rOwner.reloadBypassData();
}

Glib::ustring
ForwardingConfigWindow::BypassPanel::nextPattern(Glib::ustring& patterns)
{
	static char const sep[] = ",; \t\r\n";

	size_t const begin = patterns.find_first_not_of(sep);
	if (begin == patterns.npos) {
		Glib::ustring().swap(patterns);
		return Glib::ustring();
	}

	size_t end = patterns.find_first_of(sep, begin + 1);
	if (end == patterns.npos) {
		end = patterns.size();
	}

	Glib::ustring const pattern(patterns, begin, end - begin);
	patterns.erase(0, end);
	return pattern;
}

} // namespace GtkGUI
