/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Application.h"
#include "TrayIcon.h"
#include "TrayMenu.h"
#include "AboutDialog.h"
#include "LogDialog.h"
#include "RequestLogWindow.h"
#include "BasicConfigDialog.h"
#include "AdvancedConfigWindow.h"
#include "FilterConfigWindow.h"
#include "ForwardingConfigWindow.h"
#include "Conf.h"
#include "ConfigFile.h"
#include "ForwardingConfigFile.h"
#include "UrlsFile.h"
#include "ContentFilters.h"
#include "GlobalState.h"
#include "WorkerThreadPoolExSingleton.h"
#include "ScopedIncDec.h"
#include "AtomicOps.h"
#include <wx/wx.h>
#include <wx/event.h>
#include <wx/image.h>

using namespace std;

namespace wxGUI
{

IMPLEMENT_APP_NO_MAIN(Application)

BEGIN_EVENT_TABLE(Application, wxApp)
	EVT_IDLE(onIdle)
END_EVENT_TABLE()


Application::Application()
:	m_isExiting(false),
	m_commandNestLevel(0),
	m_havePendingCommands(0),
	m_commandQueue(COMMAND_QUEUE_CAPACITY, *this),
	m_networkActivityHandler(m_commandQueue),
	m_filterJsLogHandler(m_commandQueue),
	m_requestLogHandler(m_commandQueue)
{
}

Application::~Application()
{
}

Application*
Application::instance()
{
	return &wxGetApp();
}

bool
Application::OnInit()
{
	wxImage::AddHandler(new wxPNGHandler);
	
	m_appDir.Assign(argv[0]);
	m_appDir.AssignDir(m_appDir.GetPath(
		wxPATH_GET_VOLUME|wxPATH_GET_SEPARATOR
	));
	m_confDir.Assign(m_appDir);
	m_confDir.AppendDir(_T("conf"));
	
	{
		wxFileName config_fname(m_confDir);
		config_fname.SetName(_T("config"));
		m_ptrConfigFile.reset(new ConfigFile(config_fname));
	}
	
	{
		wxFileName fwd_config_fname(m_confDir);
		fwd_config_fname.SetName(_T("forwarding.xml"));
		m_ptrFwdConfigFile.reset(new ForwardingConfigFile(fwd_config_fname));
	}
	
	{
		wxFileName urls_fname(m_confDir);
		urls_fname.SetName(_T("urls"));
		m_ptrUrlsFile.reset(new UrlsFile(UrlsFile::URLS, urls_fname));
	}
	
	{
		wxFileName urls_local_fname(m_confDir);
		urls_local_fname.SetName(_T("urls.local"));
		m_ptrUrlsLocalFile.reset(
			new UrlsFile(UrlsFile::URLS_LOCAL, urls_local_fname)
		);
	}
	
	{
		wxFileName filters_dir(m_confDir);
		filters_dir.AppendDir(_T("filters"));
		m_ptrContentFilters.reset(new ContentFilters(filters_dir));
	}
	
	m_ptrTrayIcon.reset(new TrayIcon);
	m_ptrTrayMenu.reset(new TrayMenu);
	
	bool errors = !m_ptrConfigFile->loadAndForceApply();
	
	{
		// !!! assumes config is already loaded !!!
		ObsoleteForwardingInfo const fwd_fallback(
			GlobalState::ReadAccessor()->config()
			.getObsoleteForwardingInfo()
		);
		m_ptrFwdConfigFile->loadAndApply(fwd_fallback);
	}
	
	m_ptrUrlsFile->loadAndForceApply();
	m_ptrUrlsLocalFile->loadAndForceApply();
	m_ptrContentFilters->loadAndApply();
	if (errors) {
		showLogDialog();
	}
	
	WorkerThreadPoolExSingleton::instance()->activate();
	return true;
}

int
Application::OnExit()
{
	// This will finish the worker threads.
	WorkerThreadPoolExSingleton::instance()->deactivate();
	
	/*
	Initialyzing these things in the constructor
	or deleting them in the destructor makes WxWidgets crash.
	That's why we initialize them in OnInit() and delete them in OnExit().
	*/
	m_ptrTrayIcon.reset(0);
	m_ptrTrayMenu.reset(0);
	return 0;
}

void
Application::showTrayMenu()
{
	m_ptrTrayMenu->popup(*m_ptrTrayIcon);
}

void
Application::showAboutDialog()
{
	AboutDialog::show();
}

void
Application::showLogDialog()
{
	LogDialog::show();
}

void
Application::showRequestLogWindow()
{
	RequestLogWindow::show();
}

void
Application::showBasicConfigDialog()
{
	BasicConfigDialog::show();
}

void
Application::showAdvancedConfigWindow()
{
	AdvancedConfigWindow::show();
}

void
Application::showFilterConfigWindow()
{
	FilterConfigWindow::show();
}

void
Application::showForwardingConfigWindow()
{
	ForwardingConfigWindow::show();
}

void
Application::requestAppExit()
{
	// If we just call ExitMainLoop(), we may get a deadlock
	// while waiting for worker threads to finish. This happens if
	// a worker thread is blocked while sending a command through
	// m_commandQueue.
	if (m_commandQueue.close() == 0) {
		// command queue is empty
		ExitMainLoop();
	} else {
		// Wait for the pending commands to be processed.
		// The actual quit will be done in notifyQueueEmpty().
		m_isExiting = true;
	}
}

wxFileName
Application::getConfigFileName() const
{
	wxFileName fname(m_confDir);
	fname.SetName(_T("config"));
	return fname;
}

wxFileName
Application::getConfigDefaultFileName() const
{
	wxFileName fname(m_confDir);
	fname.SetName(_T("config.default"));
	return fname;
}

wxFileName
Application::getStandardUrlPatternsFileName() const
{
	wxFileName fname(m_confDir);
	fname.SetName(_T("urls"));
	return fname;
}

wxFileName
Application::getLocalUrlPatternsFileName() const
{
	wxFileName fname(m_confDir);
	fname.SetName(_T("urls.local"));
	return fname;
}

wxFileName
Application::getFiltersDir() const
{
	wxFileName dir(m_confDir);
	dir.AppendDir(_T("filters"));
	return dir;
}

wxFileName
Application::getResourcesDir() const
{
	wxFileName dir(m_confDir);
	dir.RemoveLastDir();
	dir.AppendDir(_T("resources"));
	return dir;
}

void
Application::handleNetworkActivity()
{
	m_ptrTrayIcon->displayNetworkActivity();
}

void
Application::notifyCommandsPending()
{
	AtomicOps::set(&m_havePendingCommands, 1);
	wxWakeUpIdle();
}

// Note: this function is called from the main thread only.
void
Application::notifyQueueEmpty()
{
	if (m_isExiting) {
		ExitMainLoop();
	}
}

void
Application::onIdle(wxIdleEvent& evt)
{
	processCommands();
}

void
Application::processCommands()
{
	if (m_commandNestLevel) {
		// If a command does things that invoke a recursive main loop,
		// we can end up here.
		return;
	}
	
	if (AtomicOps::set(&m_havePendingCommands, 0) == 0) {
		return;
	}
	
	ScopedIncDec<int> nest_level_manager(m_commandNestLevel);
	
	unsigned max_commands = m_commandQueue.getCapacity();
	/*
	We limit the number of iterations to give the GUI thread
	a chance to draw something if other threads keep posting commands.
	*/
	for (; max_commands > 0; --max_commands) {
		InterthreadCommandQueue::CommandPtr command = m_commandQueue.pop();
		if (!command) {
			return;
		}
		(*command)();
	}
	notifyCommandsPending(); // there could be more commands in the queue
}

} // namespace wxGUI
