/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ForwardingConfigWindow.h"
#include "Application.h"
#include "AbstractLogView.h"
#include "Forwarding.h"
#include "ForwardingConfigFile.h"
#include "OperationLog.h"
#include "Log.h"
#include "LogWidget.h"
#include "GlobalState.h"
#include "ProxyDescriptor.h"
#include "SymbolicInetAddr.h"
#include "Verify.h"
#include "ArraySize.h"
#include "ScopedIncDec.h"
#include "types.h"
#include <ace/config-lite.h>
#include <boost/lexical_cast.hpp>
#include <wx/string.h>
#include <wx/frame.h>
#include <wx/event.h>
#include <wx/icon.h>
#include <wx/sizer.h>
#include <wx/gbsizer.h>
#include <wx/panel.h>
#include <wx/stattext.h>
#include <wx/hyperlink.h>
#include <wx/listctrl.h>
#include <wx/textctrl.h>
#include <wx/button.h>
#include <wx/checkbox.h>
#include <wx/choice.h>
#include <wx/strconv.h>
#include <wx/msgdlg.h>
#include <wx/scrolwin.h>
#include <wx/bmpbuttn.h>
#include <wx/bitmap.h>
#include <wx/filename.h>
#include <wx/font.h>
#include <wx/cursor.h>
#include <wx/gdicmn.h>
#include <wx/statusbr.h>
#include <wx/statbox.h>
#include <wx/menu.h>
#include <wx/utils.h>
#include <wx/cshelp.h>
#include <iterator>
#include <sstream>
#include <vector>
#include <assert.h>
#include <stddef.h>

namespace wxGUI
{

static ForwardingConfigFile& FCF()
{
	return wxGetApp().forwardingConfigFile();
}

static wxString utf8ToWxString(Forwarding::Utf8String const& utf8)
{
	return wxString(wxConvUTF8.cMB2WC(utf8.raw().c_str()), *wxConvCurrent);
}

static Forwarding::Utf8String wxStringToUtf8(wxString const& wxs)
{
	std::string const utf8(wxConvUTF8.cWC2MB(wxs.wc_str(*wxConvCurrent)));
	return Forwarding::Utf8String(utf8);
}


class ForwardingConfigWindow::Impl : public wxFrame, private AbstractLogView
{
	DECLARE_NON_COPYABLE(Impl)
public:
	enum ShowEditor { SHOW_NONE, SHOW_BYPASS_EDITOR, SHOW_PROXY_EDITOR };

	Impl();
	
	virtual ~Impl();
	
	static void show();
	
	Forwarding::Config& forwardingConfig() { return m_forwardingConfig; }

	void switchToOption(Forwarding::Option const& option, int option_idx);
	
	void switchToNoOption();

	void deleteProxyInteractive(int proxy_idx);

	ProxyDescriptor const* getPrecedingProxy(int transition_idx);

	void startProxyAdding(int transition_idx);

	void startProxyEditing(int proxy_idx);

	void stopProxyOperation();

	void commitProxyOperation();

	void enterBypassEditingMode();

	void leaveBypassEditingMode();
	
	void reloadBypassData();

	void showEditor(ShowEditor editor);

	void showError(wxString const& message);

	static void ensureProperSelection(Forwarding::Config& config);
private:
	static void setSelection(Forwarding::Config& config, int selected_option);
	
	static bool isSameSelection(
			Forwarding::Config const& config, int selected_option);

	bool prepareForWindowDestruction();

	void onWindowClose(wxCloseEvent& evt);
	
	void onWindowActivate(wxActivateEvent& evt);
	
	void onClearLog(wxCommandEvent& evt);

	static Impl* m_spInstance;
	Forwarding::Config m_forwardingConfig;
	wxBoxSizer* m_pOptionsSizer;
	OptionsList* m_pOptionsList;
	ProxyChainPanel* m_pProxyChainPanel;
	ProxyEditPanel* m_pProxyEditPanel;
	BypassPanel* m_pBypassPanel;
	
	DECLARE_EVENT_TABLE()
};


class ForwardingConfigWindow::MouseOverEvtHandler : public wxEvtHandler
{
public:
	MouseOverEvtHandler(wxStatusBar* sb, wxString text);

	virtual ~MouseOverEvtHandler();
private:
	void onMouseOver(wxMouseEvent& evt);

	void onMouseOut(wxMouseEvent& evt);

	wxStatusBar* m_pStatusBar;
	wxString m_text;

	DECLARE_EVENT_TABLE()
};


class ForwardingConfigWindow::OptionsList : public wxListCtrl
{
	DECLARE_NON_COPYABLE(OptionsList)
public:
	OptionsList(wxWindow* parent, Impl& owner);
	
	virtual ~OptionsList();
	
	void reload(Forwarding::Config const& config);

	long getSelectedItem() const;
private:
	/**
	 * ITEM_REALIZED means that this item represents an option in
	 * m_rOwner.forwardingConfig().options()
	 */
	enum ItemState { ITEM_NOT_REALIZED = 0, ITEM_REALIZED };
	
	void appendOption(Forwarding::Option const& option);

	void setSelectedItem(long item);
	
	void onContextMenu(wxContextMenuEvent& evt);

	void onNew(wxCommandEvent& evt);
	
	void onRename(wxCommandEvent& evt);
	
	void onDelete(wxCommandEvent& evt);
	
	void onMoveUp(wxCommandEvent& evt);
	
	void onMoveDown(wxCommandEvent& evt);

	void onLabelEditEnd(wxListEvent& evt);
	
	void onItemSelected(wxListEvent& evt);

	void onItemUnselected(wxListEvent& evt);

	static void swapOptions(Forwarding::Config& config, long idx1, long idx2);

	Impl& m_rOwner;

	DECLARE_EVENT_TABLE()
};


class ForwardingConfigWindow::ProxyChainPanel : public wxScrolledWindow
{
	DECLARE_NON_COPYABLE(ProxyChainPanel)
public:
	ProxyChainPanel(wxWindow* parent, Impl& owner);
	
	virtual ~ProxyChainPanel();

	void loadOption(Forwarding::Option const& option);

	void clearChain();

	void switchToNormalMode();
	
	void switchToInsensitiveMode();

	void switchToProxyAddingMode(int transition_idx);

	void switchToProxyEditingMode(int proxy_idx);
private:
	void loadBitmaps();
	
	void addDesktopButton(int id, int col);

	void addForwardButton(int id, int col, wxCursor const& cursor);
	
	void addProxyButton(int id, int col, wxCursor const& cursor);
	
	void addInternetButton(int id, int col);

	void setPreferredSize();

	void onButtonPressed(wxCommandEvent& evt);

	Impl& m_rOwner;
	wxGridBagSizer* m_pSizer;
	wxBitmap m_desktopBitmap;
	wxBitmap m_proxyBitmap;
	wxBitmap m_proxyGhostBitmap;
	wxBitmap m_internetBitmap;
	wxBitmap m_forwardBitmap;
	std::vector<wxBitmapButton*> m_buttons;

	DECLARE_EVENT_TABLE()
};


class ForwardingConfigWindow::ProxyEditPanel : public wxPanel
{
	DECLARE_NON_COPYABLE(ProxyEditPanel)
public:
	ProxyEditPanel(wxWindow* parent, Impl& owner);
	
	virtual ~ProxyEditPanel();
	
	void grabFocus();

	void startProxyAdding(int option_idx, int transition_idx);

	void startProxyEditing(int option_idx, int proxy_idx);

	void cancelCurrentOperation();
private:
	enum Mode { INACTIVE, ADDING, EDITING };

	struct ProxyOption
	{
		ProxyDescriptor::ProxyType type;
		wxChar const* label;
	};
	
	void setText(wxTextCtrl* ctrl, Forwarding::Utf8String const& text);
	
	ProxyDescriptor::ProxyType getProxyType() const;

	void setProxyType(ProxyDescriptor::ProxyType type);

	void onProxyTypeChanged(wxCommandEvent& evt);

	void onProxyTypeChanged(ProxyDescriptor::ProxyType type);
	
	bool validateInteractive(ProxyDescriptor& target) const;
	
	bool haveNextHopProxy() const;

	void onOK(wxCommandEvent& evt);

	void onCancel(wxCommandEvent& evt);

	static ProxyOption const m_sProxyOptions[];
	Impl& m_rOwner;
	Mode m_mode;
	int m_optionIdx;
	int m_proxyIdx; // index of proxy being edited or insert position for a new proxy
	wxChoice* m_pProxyType;
	wxTextCtrl* m_pProxyHost;
	wxTextCtrl* m_pProxyPort;
	wxTextCtrl* m_pProxyUser;
	wxTextCtrl* m_pProxyPass;

	DECLARE_EVENT_TABLE()
};


class ForwardingConfigWindow::BypassPanel :
	public wxPanel,
	private Forwarding::BypassVisitor
{
	DECLARE_NON_COPYABLE(BypassPanel)
public:
	BypassPanel(wxWindow* parent, Impl& owner);
	
	virtual ~BypassPanel();

	void loadFrom(Forwarding::Option const& opt, int option_idx);
private:
	virtual void visit(Forwarding::HostMaskMatchPattern const& p);

	virtual void visit(Forwarding::SimpleHostnamesMatchPattern const& p);

	void onBypassHostsHelp(wxHyperlinkEvent& evt);

	void onBypassLocalHelp(wxHyperlinkEvent& evt);

	void onBypassHostsModified(wxCommandEvent& evt);
	
	void onBypassLocalModified(wxCommandEvent& evt);
	
	void onModified(bool modified);

	void onOK(wxCommandEvent& evt);

	void onCancel(wxCommandEvent& evt);
	
	static wxString nextPattern(wxString& patterns);

	Impl& m_rOwner;
	wxTextAttr m_normalTextStyle;
	wxTextAttr m_helpTextStyle;
	wxTextCtrl* m_pBypassHosts;
	wxCheckBox* m_pBypassLocal;
	wxButton* m_pOKBtn;
	wxButton* m_pCancelBtn;
	wxSimpleHelpProvider m_helpProvider;
	int m_optionIdx;
	unsigned char m_ignoreModifications;
	bool m_bypassLocalInitial;
	bool m_wasModified;

	DECLARE_EVENT_TABLE()
};


enum {
	ID_CLEARLOG_BTN,
	ID_OPTIONS_LIST,
	ID_MENU_NEW,
	ID_MENU_RENAME,
	ID_MENU_DELETE,
	ID_MENU_UP,
	ID_MENU_DOWN,
	ID_PROXY_TYPE,
	ID_BYPASS_HOSTS,
	ID_BYPASS_LOCAL,
	ID_BYPASS_HOSTS_HELP,
	ID_BYPASS_LOCAL_HELP,
	ID_CHAIN_BUTTON_FIRST = wxID_HIGHEST + 1 // must be the last one
};

BEGIN_EVENT_TABLE(ForwardingConfigWindow::Impl, wxFrame)
	EVT_ACTIVATE(onWindowActivate)
	EVT_CLOSE(onWindowClose)
	EVT_BUTTON(ID_CLEARLOG_BTN, onClearLog)
END_EVENT_TABLE()

BEGIN_EVENT_TABLE(ForwardingConfigWindow::MouseOverEvtHandler, wxEvtHandler)
	EVT_ENTER_WINDOW(onMouseOver)
	EVT_LEAVE_WINDOW(onMouseOut)
END_EVENT_TABLE()

BEGIN_EVENT_TABLE(ForwardingConfigWindow::OptionsList, wxListCtrl)
	EVT_MENU(ID_MENU_NEW, onNew)
	EVT_MENU(ID_MENU_RENAME, onRename)
	EVT_MENU(ID_MENU_DELETE, onDelete)
	EVT_MENU(ID_MENU_UP, onMoveUp)
	EVT_MENU(ID_MENU_DOWN, onMoveDown)
	EVT_CONTEXT_MENU(onContextMenu)
	EVT_LIST_END_LABEL_EDIT(wxID_ANY, onLabelEditEnd)
	EVT_LIST_ITEM_SELECTED(wxID_ANY, onItemSelected)
	EVT_LIST_ITEM_DESELECTED(wxID_ANY, onItemUnselected)
END_EVENT_TABLE()

BEGIN_EVENT_TABLE(ForwardingConfigWindow::ProxyChainPanel, wxScrolledWindow)
	EVT_BUTTON(wxID_ANY, onButtonPressed)
END_EVENT_TABLE()

BEGIN_EVENT_TABLE(ForwardingConfigWindow::ProxyEditPanel, wxPanel)
	EVT_CHOICE(ID_PROXY_TYPE, onProxyTypeChanged)
	EVT_BUTTON(wxID_OK, onOK)
	EVT_BUTTON(wxID_CANCEL, onCancel)
END_EVENT_TABLE()

BEGIN_EVENT_TABLE(ForwardingConfigWindow::BypassPanel, wxPanel)
	EVT_HYPERLINK(ID_BYPASS_HOSTS_HELP, onBypassHostsHelp)
	EVT_HYPERLINK(ID_BYPASS_LOCAL_HELP, onBypassLocalHelp)
	EVT_TEXT(ID_BYPASS_HOSTS, onBypassHostsModified)
	EVT_CHECKBOX(ID_BYPASS_LOCAL, onBypassLocalModified)
	EVT_BUTTON(wxID_OK, onOK)
	EVT_BUTTON(wxID_CANCEL, onCancel)
END_EVENT_TABLE()


void
ForwardingConfigWindow::show()
{
	Impl::show();
}


/*===================== ForwardingConfigWindow::Impl =====================*/

ForwardingConfigWindow::Impl* ForwardingConfigWindow::Impl::m_spInstance = 0;

ForwardingConfigWindow::Impl::Impl()
:	wxFrame(wxGetApp().GetTopWindow(), -1, _T("Forwarding Configuration")),
	AbstractLogView(*OperationLog::instance()),
	m_forwardingConfig(FCF().getForwardingConfig()),
	m_pOptionsSizer(0),
	m_pOptionsList(0),
	m_pProxyChainPanel(0),
	m_pProxyEditPanel(0),
	m_pBypassPanel(0)
{	
	SetExtraStyle(wxWS_EX_BLOCK_EVENTS);
	SetIcon(wxIcon(_T("AppIcon"), wxBITMAP_TYPE_ICO_RESOURCE, 16, 16));
	SetStatusBar(new wxStatusBar(this, -1));

	wxBoxSizer* topsizer = new wxBoxSizer(wxVERTICAL);
	SetSizer(topsizer);
	
	wxPanel* top_panel = new wxPanel(this);
	topsizer->Add(top_panel, 1, wxGROW);
	
	wxBoxSizer* panel_sizer = new wxBoxSizer(wxVERTICAL);
	top_panel->SetSizer(panel_sizer);

	m_pProxyChainPanel = new ProxyChainPanel(top_panel, *this);
	panel_sizer->Add(m_pProxyChainPanel, 0, wxGROW|wxTOP|wxLEFT|wxRIGHT, 5);
	
	m_pOptionsSizer = new wxBoxSizer(wxHORIZONTAL);
	panel_sizer->Add(m_pOptionsSizer, 0, wxGROW|wxTOP|wxLEFT|wxRIGHT, 3);
	
	wxBoxSizer* opt_clearlog_sizer = new wxBoxSizer(wxVERTICAL);
	m_pOptionsSizer->Add(opt_clearlog_sizer, 0, wxEXPAND|wxRIGHT, 6);

	m_pOptionsList = new OptionsList(top_panel, *this);
	opt_clearlog_sizer->Add(m_pOptionsList, 1); 
	
	opt_clearlog_sizer->Add(
		new wxButton(top_panel, ID_CLEARLOG_BTN, _T("Clear Log")),
		0, wxTOP, 5
	);

	m_pProxyEditPanel = new ProxyEditPanel(top_panel, *this);
	m_pOptionsSizer->Add(m_pProxyEditPanel, 0, wxEXPAND);
	
	m_pBypassPanel = new BypassPanel(top_panel, *this);
	m_pOptionsSizer->Add(m_pBypassPanel, 1, wxEXPAND); 
	m_pOptionsSizer->Show(m_pOptionsSizer, false);

	wxStaticBoxSizer* logsizer = new wxStaticBoxSizer(
		new wxStaticBox(top_panel, -1, _T("Log")), wxVERTICAL
	);
	panel_sizer->Add(logsizer, 0, wxEXPAND);
	LogWidget* log_widget = new LogWidget(
		OperationLog::instance(),
		top_panel, -1, wxEmptyString,
		wxDefaultPosition, wxSize(-1, 85)
	);
	logsizer->Add(log_widget, 1, wxEXPAND);
	
	m_pOptionsList->reload(m_forwardingConfig);

	panel_sizer->SetSizeHints(top_panel);
	topsizer->SetSizeHints(this);

	m_pOptionsSizer->Show(m_pProxyEditPanel, false);
}

ForwardingConfigWindow::Impl::~Impl()
{
}

void
ForwardingConfigWindow::Impl::show()
{
	if (!m_spInstance) {
		m_spInstance = new Impl;
		m_spInstance->Show();
	} else {
		m_spInstance->Show();
		m_spInstance->Raise();
	}
}

void
ForwardingConfigWindow::Impl::onWindowActivate(wxActivateEvent& evt)
{
	AbstractLogView::reportVisibility(evt.GetActive());
}

bool
ForwardingConfigWindow::Impl::prepareForWindowDestruction()
{
	if (!m_pOptionsList->IsEnabled()) {
		int const response = wxMessageBox(
			_T("Are you sure you want to close this window?\n")
			_T("Any unsaved changes will be lost."),
			_T("Question"),
			wxYES_NO, this
		);
		return (response == wxYES);
	}
	
	// If our selection differs from the global one, update the
	// global one and write a new forwarding.xml
	int const selected_option = m_pOptionsList->getSelectedItem();
	if (!isSameSelection(FCF().getForwardingConfig(), selected_option)) {
		if (selected_option != -1) {
			setSelection(m_forwardingConfig, selected_option);
			FCF().applyAndSave(m_forwardingConfig);
		}
	}
	return true;
}

void
ForwardingConfigWindow::Impl::onWindowClose(wxCloseEvent& evt)
{
	if (evt.CanVeto()) {
		if (!prepareForWindowDestruction()) {
			evt.Veto();
			return;
		}
	}
	
	if (m_spInstance == this) {
		m_spInstance = 0;
	}
	Destroy();
}

void
ForwardingConfigWindow::Impl::onClearLog(wxCommandEvent&)
{
	OperationLog::instance()->clear();
}

void
ForwardingConfigWindow::Impl::showEditor(ShowEditor what)
{
	m_pOptionsSizer->Show(m_pProxyEditPanel, (what == SHOW_PROXY_EDITOR));
	m_pOptionsSizer->Show(m_pBypassPanel, (what == SHOW_BYPASS_EDITOR));
	m_pOptionsSizer->Layout();
}

void
ForwardingConfigWindow::Impl::switchToOption(
	Forwarding::Option const& opt, int const option_idx)
{
	m_pProxyChainPanel->loadOption(opt);
	m_pBypassPanel->loadFrom(opt, option_idx);
}

void
ForwardingConfigWindow::Impl::switchToNoOption()
{
	Forwarding::Utf8String opt_name;
	Forwarding::Option dummy_option(opt_name);
	m_pProxyChainPanel->loadOption(dummy_option);
}

void
ForwardingConfigWindow::Impl::deleteProxyInteractive(int proxy_idx)
{
	long const option_idx = m_pOptionsList->getSelectedItem();
	if (option_idx == -1) {
		return;
	}
	
	int res = wxMessageBox(
		_T("Really delete this proxy?\n"),
		_T("Delete Confirmation"),
		wxYES_NO, this
	);
	if (res != wxYES) {
		return;
	}
	
	Forwarding::Config new_config(m_forwardingConfig);

	assert(option_idx >= 0);
	assert(option_idx < (long)new_config.options().size());

	Forwarding::OptionList::iterator opt(new_config.options().begin());
	std::advance(opt, option_idx);

	assert(proxy_idx >= 0);
	assert(proxy_idx < (int)opt->proxyChain().size());

	Forwarding::ProxyDescriptorList::iterator proxy(opt->proxyChain().begin());
	std::advance(proxy, proxy_idx);

	opt->proxyChain().erase(proxy);
	ensureProperSelection(new_config);
	if (!FCF().applyAndSave(new_config)) {
		return;
	} else {
		m_forwardingConfig.swap(new_config);
	}

	m_pProxyChainPanel->loadOption(*opt);
}

ProxyDescriptor const*
ForwardingConfigWindow::Impl::getPrecedingProxy(int const transition_idx)
{
	assert(transition_idx >= 0);
	
	if (transition_idx == 0) {
		return 0;
	}
	
	int const option_idx = m_pOptionsList->getSelectedItem();
	if (option_idx == -1) {
		return 0;
	}
	assert(option_idx < (int)m_forwardingConfig.options().size());

	Forwarding::OptionList::const_iterator opt(m_forwardingConfig.options().begin());
	std::advance(opt, option_idx);
	
	Forwarding::ProxyDescriptorList::const_iterator proxy(opt->proxyChain().begin());
	std::advance(proxy, transition_idx - 1);
	
	return &*proxy;
}

void
ForwardingConfigWindow::Impl::startProxyAdding(int transition_idx)
{
	assert(transition_idx >= 0);

	long const option_idx = m_pOptionsList->getSelectedItem();
	if (option_idx == -1) {
		return;
	}
	assert(option_idx < (long)m_forwardingConfig.options().size());
	
	m_pOptionsList->Enable(false);
	m_pProxyChainPanel->switchToProxyAddingMode(transition_idx);
	m_pProxyEditPanel->startProxyAdding(option_idx, transition_idx);
	showEditor(SHOW_PROXY_EDITOR);
	m_pProxyEditPanel->grabFocus();
}

void
ForwardingConfigWindow::Impl::startProxyEditing(int proxy_idx)
{
	assert(proxy_idx >= 0);

	long const option_idx = m_pOptionsList->getSelectedItem();
	if (option_idx == -1) {
		return;
	}
	assert(option_idx < (long)m_forwardingConfig.options().size());
	
	m_pOptionsList->Enable(false);
	m_pProxyChainPanel->switchToProxyEditingMode(proxy_idx);
	m_pProxyEditPanel->startProxyEditing(option_idx, proxy_idx);
	showEditor(SHOW_PROXY_EDITOR);
	m_pProxyEditPanel->grabFocus();
}

void
ForwardingConfigWindow::Impl::stopProxyOperation()
{
	m_pProxyEditPanel->cancelCurrentOperation();
	showEditor(SHOW_BYPASS_EDITOR);
	m_pOptionsList->Enable();
	m_pProxyChainPanel->switchToNormalMode();
}

void
ForwardingConfigWindow::Impl::commitProxyOperation()
{
	stopProxyOperation();

	long const option_idx = m_pOptionsList->getSelectedItem();
	if (option_idx == -1) {
		return;
	}
	assert(option_idx >= 0);
	assert(option_idx < (long)m_forwardingConfig.options().size());

	Forwarding::OptionList::iterator opt(m_forwardingConfig.options().begin());
	std::advance(opt, option_idx);
	m_pProxyChainPanel->loadOption(*opt);
}

/**
 * \brief Enter bypass editing mode and block other operations.
 *
 * This gets called when user modifies bypass options.
 */
void
ForwardingConfigWindow::Impl::enterBypassEditingMode()
{
	m_pOptionsList->Enable(false);
	m_pProxyChainPanel->switchToInsensitiveMode();
}

void
ForwardingConfigWindow::Impl::leaveBypassEditingMode()
{
	m_pOptionsList->Enable(true);
	m_pProxyChainPanel->switchToNormalMode();
}

void
ForwardingConfigWindow::Impl::reloadBypassData()
{
	long const option_idx = m_pOptionsList->getSelectedItem();
	if (option_idx == -1) {
		return;
	}
	assert(option_idx >= 0);
	assert(option_idx < (long)m_forwardingConfig.options().size());
	
	Forwarding::OptionList::iterator opt(m_forwardingConfig.options().begin());
	std::advance(opt, option_idx);

	m_pBypassPanel->loadFrom(*opt, option_idx);
}

void
ForwardingConfigWindow::Impl::showError(wxString const& message)
{
	wxMessageBox(message, _T("Error"), wxOK|wxICON_ERROR, this);
}

/**
 * \brief Ensure that one and only one option is selected.
 */
void
ForwardingConfigWindow::Impl::ensureProperSelection(Forwarding::Config& config)
{
	Forwarding::OptionList::iterator it(config.options().begin());
	Forwarding::OptionList::iterator const end(config.options().end());
	bool found = false;
	for (; it != end; ++it) {
		if (found) {
			it->setSelected(false);
		} else {
			found = it->isSelected();
		}
	}
	
	if (!found && !config.options().empty()) {
		// if none is selected, select the first one
		config.options().front().setSelected(true);
	}
}

void
ForwardingConfigWindow::Impl::setSelection(
	Forwarding::Config& config, int const selected_option)
{
	Forwarding::OptionList::iterator it(config.options().begin());
	Forwarding::OptionList::iterator const end(config.options().end());
	for (int i = 0; it != end; ++it, ++i) {
		it->setSelected(i == selected_option);
	}
}

bool
ForwardingConfigWindow::Impl::isSameSelection(
	Forwarding::Config const& config, int const selected_option)
{
	Forwarding::OptionList::const_iterator it(config.options().begin());
	Forwarding::OptionList::const_iterator const end(config.options().end());
	for (int i = 0; it != end; ++it, ++i) {
		bool const selected = (i == selected_option);
		if (it->isSelected() != selected) {
			return false;
		}
	}
	return true;
}


/*============== ForwardingConfigWindow::MouseOverEvtHandler =============*/

ForwardingConfigWindow::MouseOverEvtHandler::MouseOverEvtHandler(
	wxStatusBar* sb, wxString text)
:	m_pStatusBar(sb),
	m_text(text)
{
	assert(m_pStatusBar);
}

ForwardingConfigWindow::MouseOverEvtHandler::~MouseOverEvtHandler()
{
}

void
ForwardingConfigWindow::MouseOverEvtHandler::onMouseOver(wxMouseEvent& evt)
{
	evt.Skip();
	m_pStatusBar->SetStatusText(m_text);
}

void
ForwardingConfigWindow::MouseOverEvtHandler::onMouseOut(wxMouseEvent& evt)
{
	evt.Skip();
	
	// When moving cursor from one control to another, it's possible
	// to get mouse-out and mouse-over events in any order.
	if (m_pStatusBar->GetStatusText() == m_text) {
		m_pStatusBar->SetStatusText(wxEmptyString);
	}
}


/*================= ForwardingConfigWindow::OptionsList ==================*/

ForwardingConfigWindow::OptionsList::OptionsList(
	wxWindow* parent, Impl& owner)
:	wxListCtrl(
			   parent, ID_OPTIONS_LIST, wxDefaultPosition, wxSize(140, 90),
			   wxLC_REPORT|wxLC_EDIT_LABELS|wxLC_NO_HEADER|wxLC_SINGLE_SEL
	),
	m_rOwner(owner)
{
	int width = 0;
	int height = 0;
	GetClientSize(&width, &height);
	wxListItem header;
	header.SetWidth(width);
	InsertColumn(0, header);
	PushEventHandler(
		new MouseOverEvtHandler(
			m_rOwner.GetStatusBar(), _T("Use context menu")
		)
	);
}

ForwardingConfigWindow::OptionsList::~OptionsList()
{
}

void
ForwardingConfigWindow::OptionsList::reload(Forwarding::Config const& config)
{
	DeleteAllItems();

	Forwarding::OptionList const& options = config.options();
	Forwarding::OptionList::const_iterator it(options.begin());
	Forwarding::OptionList::const_iterator const end(options.end());
	for (; it != end; ++it) {
		appendOption(*it);
	}
	if (getSelectedItem() == -1 && !options.empty()) {
		setSelectedItem(0);
	}
}

void
ForwardingConfigWindow::OptionsList::appendOption(
	Forwarding::Option const& option)
{
	long const row = GetItemCount();
	InsertItem(row, utf8ToWxString(option.getName()));
	SetItemData(row, ITEM_REALIZED);
	if (option.isSelected()) {
		setSelectedItem(row);
	}
}

/**
 * \return selected item index or -1 if no item is selected.
 */
long
ForwardingConfigWindow::OptionsList::getSelectedItem() const
{
	return GetNextItem(
		-1, wxLIST_NEXT_ALL, wxLIST_STATE_SELECTED
	);
}

void
ForwardingConfigWindow::OptionsList::setSelectedItem(long const item)
{
	SetItemState(
		item,
		wxLIST_STATE_FOCUSED|wxLIST_STATE_SELECTED,
		wxLIST_STATE_FOCUSED|wxLIST_STATE_SELECTED
	);
}

void
ForwardingConfigWindow::OptionsList::onContextMenu(wxContextMenuEvent& evt)
{
	wxMenu menu;
	menu.Append(ID_MENU_NEW, _T("New"));
	menu.Append(ID_MENU_RENAME, _T("Rename"));
	menu.AppendSeparator();
	menu.Append(ID_MENU_UP, _T("Move Up"));
	menu.Append(ID_MENU_DOWN, _T("Move Down"));
	menu.AppendSeparator();
	menu.Append(ID_MENU_DELETE, _T("Delete"));
	
	long const selected_item = getSelectedItem();
	if (selected_item == -1) {
		menu.Enable(ID_MENU_UP, false);
		menu.Enable(ID_MENU_DOWN, false);
		menu.Enable(ID_MENU_RENAME, false);
		menu.Enable(ID_MENU_DELETE, false);
	} else if (selected_item == 0) {
		menu.Enable(ID_MENU_UP, false);
	} else if (selected_item == GetItemCount() - 1) {
		menu.Enable(ID_MENU_DOWN, false);
	}
	
	PopupMenu(&menu, ScreenToClient(evt.GetPosition()));
}

void
ForwardingConfigWindow::OptionsList::onNew(wxCommandEvent& evt)
{
	long const row = GetItemCount();
	InsertItem(row, wxEmptyString);
	SetItemData(row, ITEM_NOT_REALIZED);
	EditLabel(row);
}

void
ForwardingConfigWindow::OptionsList::onRename(wxCommandEvent& evt)
{
	long const row = getSelectedItem();
	if (row >= 0) {
		EditLabel(row);
	}
}

void
ForwardingConfigWindow::OptionsList::onDelete(wxCommandEvent& evt)
{
	long const row = getSelectedItem();
	if (row < 0) {
		return;
	}

	int const res = wxMessageBox(
		_T("Really delete this profile?"),
		_T("Delete Confirmation"),
		wxYES_NO, this
	);
	if (res != wxYES) {
		return;
	}
	
	if (GetItemData(row) == ITEM_REALIZED) {
		Forwarding::Config new_config(m_rOwner.forwardingConfig());
		Forwarding::OptionList::iterator it(new_config.options().begin());
		std::advance(it, row);
		new_config.options().erase(it);
		Impl::ensureProperSelection(new_config);
		if (!FCF().applyAndSave(new_config)) {
			return;
		} else {
			m_rOwner.forwardingConfig().swap(new_config);
		}
	}
	
	DeleteItem(row);
}

void
ForwardingConfigWindow::OptionsList::onMoveUp(wxCommandEvent& evt)
{
	long const item = getSelectedItem();
	long const prev = item - 1;
	if (prev < 0 /*|| item >= m_pOptionsCtrl->GetItemCount()*/) {
		return;
	}
	
	Forwarding::Config new_config(m_rOwner.forwardingConfig());
	swapOptions(new_config, item, prev);
	
	Impl::ensureProperSelection(new_config);
	if (!FCF().applyAndSave(new_config)) {
		return;
	} else {
		m_rOwner.forwardingConfig().swap(new_config);
	}

	wxString const text(GetItemText(prev));
	long const data(GetItemData(prev));
	InsertItem(item + 1, text);
	SetItemData(item + 1, data);
	DeleteItem(prev);
}

void
ForwardingConfigWindow::OptionsList::onMoveDown(wxCommandEvent& evt)
{
	long const item = getSelectedItem();
	long next = item + 1;
	if (item < 0 || next >= GetItemCount()) {
		return;
	}
	
	Forwarding::Config new_config(m_rOwner.forwardingConfig());
	swapOptions(new_config, item, next);
	
	Impl::ensureProperSelection(new_config);
	if (!FCF().applyAndSave(new_config)) {
		return;
	} else {
		m_rOwner.forwardingConfig().swap(new_config);
	}

	wxString const text(GetItemText(next));
	long const data(GetItemData(next));
	InsertItem(item, text); // moves item and next downwards
	SetItemData(item, data);
	DeleteItem(next + 1);
}

void
ForwardingConfigWindow::OptionsList::onLabelEditEnd(wxListEvent& evt)
{
	long const row = evt.GetIndex();
	assert(row >= 0);
	
	bool const renaming = (GetItemData(row) == ITEM_REALIZED);

	if (evt.IsEditCancelled() && !renaming) {
		DeleteItem(row);
		return;
	}

	if (evt.GetLabel().empty()) {
		evt.Veto();
		return;
	}
	
	Forwarding::Utf8String option_name(wxStringToUtf8(evt.GetLabel()));
	Forwarding::Config new_config(m_rOwner.forwardingConfig());
	Forwarding::OptionList::iterator pos(new_config.options().begin());
	std::advance(pos, row);
	Forwarding::OptionList::iterator new_pos(pos);
	if (renaming) {
		pos->setName(option_name);
	} else { // creating
		new_pos = new_config.options().insert(pos, Forwarding::Option(option_name));
	}
	
	Impl::ensureProperSelection(new_config);
	if (!FCF().applyAndSave(new_config)) {
		DeleteItem(row);
		return;
	} else {
		m_rOwner.forwardingConfig().swap(new_config);
		if (!renaming) {
			SetItemData(row, ITEM_REALIZED);
			m_rOwner.switchToOption(*new_pos, row);
		}
	}
}

void
ForwardingConfigWindow::OptionsList::onItemSelected(wxListEvent& evt)
{
	long const row = evt.GetIndex();
	assert(row >= 0);
	
	if (GetItemData(row) == ITEM_NOT_REALIZED) {
		return;
	}

	Forwarding::Config const& config = m_rOwner.forwardingConfig();
	Forwarding::OptionList::const_iterator opt(config.options().begin());
	std::advance(opt, row);

	m_rOwner.switchToOption(*opt, row);
	if (opt->proxyChain().empty()) {
		m_rOwner.showEditor(m_rOwner.SHOW_NONE);
	} else {
		m_rOwner.showEditor(m_rOwner.SHOW_BYPASS_EDITOR);
	}
}

void
ForwardingConfigWindow::OptionsList::onItemUnselected(wxListEvent&)
{
	m_rOwner.switchToNoOption();
	m_rOwner.showEditor(m_rOwner.SHOW_NONE);
}

void
ForwardingConfigWindow::OptionsList::swapOptions(
	Forwarding::Config& config, long const idx1, long const idx2)
{
	Forwarding::OptionList::iterator it1(config.options().begin());
	Forwarding::OptionList::iterator it2(it1);
	std::advance(it1, idx1);
	std::advance(it2, idx2);
	it1->swap(*it2);
}


/*================ ForwardingConfigWindow::ProxyChainPanel ===============*/

ForwardingConfigWindow::ProxyChainPanel::ProxyChainPanel(
	wxWindow* parent, Impl& owner)
:	wxScrolledWindow(parent, -1),
	m_rOwner(owner),
	m_pSizer(0)
{
	loadBitmaps();

	SetScrollRate(32, 0);

	setPreferredSize();
}

ForwardingConfigWindow::ProxyChainPanel::~ProxyChainPanel()
{
}

void
ForwardingConfigWindow::ProxyChainPanel::switchToNormalMode()
{
	int const num_buttons = m_buttons.size();

	for (int i = 1; i < num_buttons; i += 2) {
		wxBitmapButton* forward_btn = m_buttons[i];
		forward_btn->SetBitmapLabel(m_forwardBitmap);
		forward_btn->SetBitmapHover(m_proxyGhostBitmap);
		forward_btn->Enable();
	}

	for (int i = 2; i < num_buttons - 1; i += 2) {
		wxBitmapButton* proxy_btn = m_buttons[i];
		proxy_btn->SetBitmapLabel(m_proxyBitmap);
		proxy_btn->Enable();
	}
}

void
ForwardingConfigWindow::ProxyChainPanel::switchToInsensitiveMode()
{
	int const num_buttons = m_buttons.size();

	for (int i = 1; i < num_buttons; i += 2) {
		wxBitmapButton* forward_btn = m_buttons[i];
		forward_btn->SetBitmapLabel(m_forwardBitmap);
		forward_btn->SetBitmapHover(m_proxyGhostBitmap);
		forward_btn->Disable();
	}

	for (int i = 2; i < num_buttons - 1; i += 2) {
		wxBitmapButton* proxy_btn = m_buttons[i];
		proxy_btn->SetBitmapLabel(m_proxyBitmap);
		proxy_btn->Disable();
	}
}

void
ForwardingConfigWindow::ProxyChainPanel::switchToProxyAddingMode(
	int transition_idx)
{
	int const num_buttons = m_buttons.size();
	int const selected_idx = transition_idx * 2 + 1;

	for (int i = 1; i < num_buttons; i += 2) {
		wxBitmapButton* forward_btn = m_buttons[i];
		forward_btn->Disable();
		if (i == selected_idx) {
			forward_btn->SetBitmapLabel(m_proxyBitmap);
		} else {
			forward_btn->SetBitmapLabel(m_forwardBitmap);
		}
	}

	for (int i = 2; i < num_buttons - 1; i += 2) {
		wxBitmapButton* proxy_btn = m_buttons[i];
		proxy_btn->Disable();
		proxy_btn->SetBitmapLabel(m_proxyGhostBitmap);
	}
}

void
ForwardingConfigWindow::ProxyChainPanel::switchToProxyEditingMode(
	int proxy_idx)
{
	int const num_buttons = m_buttons.size();
	int const selected_idx = proxy_idx * 2 + 2;

	for (int i = 1; i < num_buttons; i += 2) {
		wxBitmapButton* forward_btn = m_buttons[i];
		forward_btn->Disable();
		forward_btn->SetBitmapLabel(m_forwardBitmap);
	}

	for (int i = 2; i < num_buttons - 1; i += 2) {
		wxBitmapButton* proxy_btn = m_buttons[i];
		proxy_btn->Disable();
		if (i == selected_idx) {
			proxy_btn->SetBitmapLabel(m_proxyBitmap);
		} else {
			proxy_btn->SetBitmapLabel(m_proxyGhostBitmap);
		}
	}
}

void
ForwardingConfigWindow::ProxyChainPanel::loadBitmaps()
{
	wxFileName fname(wxGetApp().getResourcesDir());

	fname.SetName(_T("desktop.png"));
	m_desktopBitmap.LoadFile(fname.GetFullPath(), wxBITMAP_TYPE_PNG);
	
	fname.SetName(_T("proxy.png"));
	m_proxyBitmap.LoadFile(fname.GetFullPath(), wxBITMAP_TYPE_PNG);
	
	fname.SetName(_T("proxy-ghost.png"));
	m_proxyGhostBitmap.LoadFile(fname.GetFullPath(), wxBITMAP_TYPE_PNG);

	fname.SetName(_T("internet.png"));
	m_internetBitmap.LoadFile(fname.GetFullPath(), wxBITMAP_TYPE_PNG);

	fname.SetName(_T("forward.png"));
	m_forwardBitmap.LoadFile(fname.GetFullPath(), wxBITMAP_TYPE_PNG);
}

void
ForwardingConfigWindow::ProxyChainPanel::setPreferredSize()
{
	// create a dummy option with two proxies
	Forwarding::Utf8String const option_name;
	Forwarding::Option dummy_option(option_name);
	for (int i = 0; i < 2; ++i) {
		dummy_option.proxyChain().push_back(ProxyDescriptor());
	}
	
	loadOption(dummy_option);
	
	// get the size to fit a chain with two proxies
	int client_w = 0, client_h = 0, tmp = 0;
	GetVirtualSize(&client_w, &client_h);
	
	// force horizontal scroll bar to appear
	SetClientSize(client_w / 2, client_h);
	int min_h = 0;
	GetSize(&tmp, &min_h);
	
	// force scroll bar to disappear
	SetClientSize(client_w, client_h);
	int min_w = 0;
	GetSize(&min_w, &tmp);

	SetMinSize(wxSize(min_w, min_h)); 

	clearChain();
}

void
ForwardingConfigWindow::ProxyChainPanel::clearChain()
{
	SetSizer(0);
	m_pSizer = 0;
	DestroyChildren();
	m_buttons.clear();
}

void
ForwardingConfigWindow::ProxyChainPanel::loadOption(Forwarding::Option const& opt)
{
	wxCursor const hand_cursor(wxCURSOR_HAND);
	
	wxGridSizer* topsizer = new wxGridSizer(1, 1, 0);
	SetSizer(topsizer);
	DestroyChildren();
	m_buttons.clear();
	
	/*
	Grid layout:
	+-------+---+---+------+---+---+--------+
	| +---+ |       | +--+ |       |  ****  |
	| |   | |  ==>  | |  | |  ==>  | ****** |
	| +---+ |       | +--+ |       |  ****  |
	+-------+---+---+------+---+---+--------+
	|This Comput|   host:port  |   |Internet|
	+-------+---+---+------+---+---+--------+
	*/
	
	m_pSizer = new wxGridBagSizer(/* vgap = */3, /* hgap = */0);
	topsizer->Add(m_pSizer, 0, wxALIGN_CENTER|wxALIGN_CENTER_VERTICAL);
	m_pSizer->SetFlexibleDirection(wxBOTH);
	
	int id = ID_CHAIN_BUTTON_FIRST;
	
	addDesktopButton(id, 0);	
	addForwardButton(++id, 1, hand_cursor); // takes 2 cols
	
	m_pSizer->Add(
		new wxStaticText(this, -1, _T("This Computer")),
		wxGBPosition(1, 0), wxGBSpan(1, 2),
		wxALIGN_LEFT|wxALIGN_CENTER_VERTICAL
	);
	
	int col = 3;
	
	typedef Forwarding::ProxyDescriptorList::const_iterator Iter;
	Iter it(opt.proxyChain().begin());
	Iter const end(opt.proxyChain().end());
	for (; it != end; ++it, col += 3) {
		ProxyDescriptor const& proxy = *it;
		
		addProxyButton(++id, col, hand_cursor);
		addForwardButton(++id, col + 1, hand_cursor); // takes 2 cols

		m_pSizer->Add(
			new wxStaticText(this, -1, proxy.getAddr().toString().c_str()),
			wxGBPosition(1, col - 1), wxGBSpan(1, 3),
			wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL
		);
	}

	addInternetButton(++id, col);

	m_pSizer->Add(
		new wxStaticText(this, -1, _T("Internet")),
		wxGBPosition(1, col), wxGBSpan(1, 1),
		wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL
	);
	
	topsizer->FitInside(this);
}

void
ForwardingConfigWindow::ProxyChainPanel::addDesktopButton(int id, int col)
{
	wxBitmapButton* desktop_btn = new wxBitmapButton(
		this, id, m_desktopBitmap, wxDefaultPosition, wxDefaultSize, 0
	);
	m_pSizer->Add(
		desktop_btn, wxGBPosition(0, 0), wxDefaultSpan,
		wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL
	);
	m_buttons.push_back(desktop_btn);
	desktop_btn->Disable();
}

void
ForwardingConfigWindow::ProxyChainPanel::addForwardButton(
	int id, int col, wxCursor const& cursor)
{
	wxBitmapButton* forward_btn = new wxBitmapButton(
		this, id, m_forwardBitmap, wxDefaultPosition, wxDefaultSize, 0
	);
	m_pSizer->Add(
		forward_btn, wxGBPosition(0, col), wxGBSpan(1, 2),
		wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL
	);
	m_buttons.push_back(forward_btn);
	forward_btn->SetBitmapHover(m_proxyGhostBitmap);
	forward_btn->SetCursor(cursor);
	forward_btn->PushEventHandler(
		new MouseOverEvtHandler(
			m_rOwner.GetStatusBar(), _T("Insert a proxy")
		)
	);
}

void
ForwardingConfigWindow::ProxyChainPanel::addProxyButton(
	int id, int col, wxCursor const& cursor)
{
	wxBitmapButton* proxy_btn = new wxBitmapButton(
		this, id, m_proxyBitmap, wxDefaultPosition, wxDefaultSize, 0
	);
	m_pSizer->Add(
		proxy_btn, wxGBPosition(0, col), wxDefaultSpan,
		wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL
	);
	m_buttons.push_back(proxy_btn);
	proxy_btn->SetCursor(cursor);
	proxy_btn->PushEventHandler(
		new MouseOverEvtHandler(
			m_rOwner.GetStatusBar(), _T("Click to edit / Ctrl + click to delete")
		)
	);
}

void
ForwardingConfigWindow::ProxyChainPanel::addInternetButton(int id, int col)
{
	wxBitmapButton* internet_btn = new wxBitmapButton(
		this, id, m_internetBitmap, wxDefaultPosition, wxDefaultSize, 0
	);
	m_pSizer->Add(
		internet_btn, wxGBPosition(0, col), wxDefaultSpan,
		wxALIGN_LEFT|wxALIGN_CENTER_VERTICAL
	);
	m_buttons.push_back(internet_btn);
	internet_btn->Disable();
}

void
ForwardingConfigWindow::ProxyChainPanel::onButtonPressed(wxCommandEvent& evt)
{
	Refresh(); // workaround a bug in wxWidgets

	int const button_idx = evt.GetId() - ID_CHAIN_BUTTON_FIRST;
	assert(button_idx >= 0);
	
	/*
	Buttons are indexed like this:
	------------------------------
	[0] => This Comuter
	[1] => Arrow
	for (int i = 0; i < NUM_PROXIES; ++i) {
		[i*2]   => Proxy
		[i*2+1] => Arrow
	}
	[2+NUM_PROXIES*2] => Internet
	------------------------------
	The first one and the last one are unclickable, so this leaves us with
	Proxy buttons at even positions and Arrow buttons at odd positions.
	*/
	
	if (button_idx & 1) {
		// arrow
		int const transition_idx = button_idx / 2;
		ProxyDescriptor const* proxy = m_rOwner.getPrecedingProxy(transition_idx);
		if (proxy && proxy->getType() == ProxyDescriptor::HTTP) {
			m_rOwner.showError("No proxy can follow an HTTP proxy.");
		} else {
			m_rOwner.startProxyAdding(transition_idx);
		}
	} else {
		// proxy
		int const proxy_idx = button_idx / 2 - 1;
		if (wxGetKeyState(WXK_CONTROL)) {
			m_rOwner.deleteProxyInteractive(proxy_idx);
		} else {
			m_rOwner.startProxyEditing(proxy_idx);
		}
	}
}


/*================ ForwardingConfigWindow::ProxyEditPanel ================*/

ForwardingConfigWindow::ProxyEditPanel::ProxyOption const
ForwardingConfigWindow::ProxyEditPanel::m_sProxyOptions[] = {
	{ ProxyDescriptor::HTTP, _T("HTTP proxy") },
	{ ProxyDescriptor::SOCKS4, _T("SOCKS4 proxy") },
	{ ProxyDescriptor::SOCKS4A, _T("SOCKS4A proxy") },
	{ ProxyDescriptor::SOCKS5, _T("SOCKS5 proxy") }
};

ForwardingConfigWindow::ProxyEditPanel::ProxyEditPanel(
	wxWindow* parent, Impl& owner)
:	wxPanel(parent, -1),
	m_rOwner(owner),
	m_mode(INACTIVE),
	m_optionIdx(0),
	m_proxyIdx(0),
	m_pProxyType(0),
	m_pProxyHost(0),
	m_pProxyPort(0),
	m_pProxyUser(0),
	m_pProxyPass(0)
{
	wxBoxSizer *topsizer = new wxBoxSizer(wxVERTICAL);
	SetSizer(topsizer);
	
	m_pProxyType = new wxChoice(this, ID_PROXY_TYPE);
	topsizer->Add(m_pProxyType, 0, wxGROW|wxALIGN_CENTER_VERTICAL);
	for (int i = 0; i < (int)ARRAY_SIZE(m_sProxyOptions); ++i) {
		m_pProxyType->Append(m_sProxyOptions[i].label);
	}
	
	topsizer->AddSpacer(3);
	
	wxBoxSizer *hp_sizer = new wxBoxSizer(wxHORIZONTAL);
	topsizer->Add(hp_sizer, 0, wxALIGN_CENTRE);
	m_pProxyHost = new wxTextCtrl(
		this, -1, wxEmptyString, wxDefaultPosition, wxSize(120, -1)
	);
	hp_sizer->Add(m_pProxyHost, 0, wxALIGN_CENTRE|wxRIGHT, 5);
	hp_sizer->Add(new wxStaticText(this, -1, _T(":")), 0, wxALIGN_CENTRE);
	m_pProxyPort = new wxTextCtrl(
		this, -1, wxEmptyString, wxDefaultPosition, wxSize(45, -1)
	);
	hp_sizer->Add(m_pProxyPort, 0, wxALIGN_CENTRE|wxLEFT, 5);
	
	topsizer->AddSpacer(3);
	
	wxFlexGridSizer* user_pass_sizer = new wxFlexGridSizer(2, 3, 3);
	topsizer->Add(user_pass_sizer, 0, wxGROW);
	user_pass_sizer->AddGrowableCol(1, 1);
	
	user_pass_sizer->Add(
		new wxStaticText(this, -1, _T("Username:")),
		0, wxALIGN_RIGHT|wxALIGN_CENTER_VERTICAL
	);
	m_pProxyUser = new wxTextCtrl(
		this, -1, wxEmptyString, wxDefaultPosition, wxSize(100, -1)
	);
	user_pass_sizer->Add(m_pProxyUser, 0, wxGROW);
	
	user_pass_sizer->Add(
		new wxStaticText(this, -1, _T("Password:")),
		0, wxALIGN_RIGHT|wxALIGN_CENTER_VERTICAL
	);
	m_pProxyPass = new wxTextCtrl(
		this, -1, wxEmptyString, wxDefaultPosition, wxSize(100, -1)
	);
	user_pass_sizer->Add(m_pProxyPass, 0, wxGROW);
	
	topsizer->Add(5, 5, 1);

	wxBoxSizer *buttonsizer = new wxBoxSizer(wxHORIZONTAL);
	topsizer->Add(buttonsizer, 0, wxALIGN_CENTRE);
	buttonsizer->Add(new wxButton(this, wxID_OK, _T("OK")), 0, wxLEFT|wxRIGHT, 5);
	wxButton* cancel_btn = new wxButton(this, wxID_CANCEL, _T("Cancel"));
	buttonsizer->Add(cancel_btn, 0, wxLEFT|wxRIGHT, 5);

	topsizer->SetSizeHints(this);
}

ForwardingConfigWindow::ProxyEditPanel::~ProxyEditPanel()
{
}

void
ForwardingConfigWindow::ProxyEditPanel::grabFocus()
{
	m_pProxyType->SetFocus();
}

void
ForwardingConfigWindow::ProxyEditPanel::startProxyAdding(
	int const option_idx, int const transition_idx)
{
	m_mode = ADDING;
	m_optionIdx = option_idx;
	m_proxyIdx = transition_idx;

	m_pProxyType->SetSelection(-1);

	m_pProxyHost->Clear();
	m_pProxyHost->Enable();

	m_pProxyPort->Clear();
	m_pProxyPort->Enable();
	
	m_pProxyUser->Clear();
	m_pProxyUser->Enable();
	
	m_pProxyPass->Clear();
	m_pProxyPass->Enable();
}

void
ForwardingConfigWindow::ProxyEditPanel::startProxyEditing(
	int const option_idx, int const proxy_idx)
{
	Forwarding::OptionList& options = m_rOwner.forwardingConfig().options();

	assert(option_idx >= 0);
	assert(option_idx < (int)options.size());
	
	Forwarding::OptionList::iterator opt(options.begin());
	std::advance(opt, option_idx);
	
	Forwarding::ProxyDescriptorList& proxies = opt->proxyChain();

	assert(proxy_idx >= 0);
	assert(proxy_idx < (int)proxies.size());
	
	Forwarding::ProxyDescriptorList::iterator proxy(proxies.begin());
	std::advance(proxy, proxy_idx);

	m_mode = EDITING;
	m_optionIdx = option_idx;
	m_proxyIdx = proxy_idx;

	setProxyType(proxy->getType());
	
	// Strings in ProxyDescriptor are actually UTF-8, as they came either
	// from forwarding.xml or from the GUI.

	setText(m_pProxyHost, Forwarding::Utf8String(proxy->getAddr().getHost()));

	wxString port;
	port << proxy->getAddr().getPort();
	m_pProxyPort->SetValue(port);

	setText(m_pProxyUser, Forwarding::Utf8String(proxy->getUserName()));
	setText(m_pProxyPass, Forwarding::Utf8String(proxy->getPassword()));
}

void
ForwardingConfigWindow::ProxyEditPanel::cancelCurrentOperation()
{
	m_mode = INACTIVE;
	m_optionIdx = 0;
	m_proxyIdx = 0;
}

void
ForwardingConfigWindow::ProxyEditPanel::setText(
	wxTextCtrl* ctrl, Forwarding::Utf8String const& text)
{
	ctrl->SetValue(utf8ToWxString(text));
}

ProxyDescriptor::ProxyType
ForwardingConfigWindow::ProxyEditPanel::getProxyType() const
{
	long const selection = m_pProxyType->GetSelection();
	if (selection == -1) {
		return ProxyDescriptor::INVALID;
	} else {
		return m_sProxyOptions[selection].type;
	}
}

void
ForwardingConfigWindow::ProxyEditPanel::setProxyType(
	ProxyDescriptor::ProxyType type)
{
	for (unsigned i = 0; i < ARRAY_SIZE(m_sProxyOptions); ++i) {
		if (m_sProxyOptions[i].type == type) {
			m_pProxyType->SetSelection(i);
			onProxyTypeChanged(type);
			return;
		}
	}
	m_pProxyType->SetSelection(-1);
	onProxyTypeChanged(ProxyDescriptor::INVALID);
}

void
ForwardingConfigWindow::ProxyEditPanel::onProxyTypeChanged(wxCommandEvent& evt)
{
	onProxyTypeChanged(m_sProxyOptions[evt.GetSelection()].type);
}

void
ForwardingConfigWindow::ProxyEditPanel::onProxyTypeChanged(
	ProxyDescriptor::ProxyType type)
{
	switch (type) {
		case ProxyDescriptor::INVALID:
		m_pProxyHost->Enable(false);
		m_pProxyPort->Enable(false);
		m_pProxyUser->Enable(false);
		m_pProxyPass->Enable(false);
		break;
		case ProxyDescriptor::HTTP:
		m_pProxyHost->Enable(true);
		m_pProxyPort->Enable(true);
		m_pProxyUser->Enable(false);
		m_pProxyPass->Enable(false);
		break;
		case ProxyDescriptor::SOCKS4:
		case ProxyDescriptor::SOCKS4A:
		m_pProxyHost->Enable(true);
		m_pProxyPort->Enable(true);
		m_pProxyUser->Enable(true);
		m_pProxyPass->Enable(false);
		break;
		case ProxyDescriptor::SOCKS5:
		m_pProxyHost->Enable(true);
		m_pProxyPort->Enable(true);
		m_pProxyUser->Enable(true);
		m_pProxyPass->Enable(true);
		break;
		default:
		assert(0);
	}
}

bool
ForwardingConfigWindow::ProxyEditPanel::validateInteractive(
	ProxyDescriptor& target) const
{
	ProxyDescriptor::ProxyType const type = getProxyType();
	if (type == ProxyDescriptor::INVALID) {
		m_pProxyType->SetFocus();
		m_rOwner.showError(_T("Proxy type not selected."));
		return false;
	} else if (type == ProxyDescriptor::HTTP && haveNextHopProxy()) {
		m_pProxyType->SetFocus();
		m_rOwner.showError(_T("HTTP proxies can't forward to another proxy."));
		return false;
	}
	
	wxString host(m_pProxyHost->GetValue());
	host.Trim(false); // from left
	host.Trim(true); // from right
	if (host.IsEmpty()) {
		m_pProxyHost->SetFocus();
		m_rOwner.showError(_T("Proxy host is required."));
		return false;
	}
	
	uint16_t port = 0;
	try {
		wxString prt(m_pProxyPort->GetValue());
		host.Trim(false); // from left
		host.Trim(true); // from right
		port = boost::lexical_cast<uint16_t>(prt.mb_str());
	} catch (boost::bad_lexical_cast&) {}
	if (port == 0) {
		m_pProxyPort->SetFocus();
		m_rOwner.showError(_T("Proxy port must be an integer in range of [1, 65535]"));
		return false;
	}
	
	wxString user(m_pProxyUser->GetValue());
	wxString pass(m_pProxyPass->GetValue());
	if (type == ProxyDescriptor::HTTP) {
		user.Clear();
	}
	if (type != ProxyDescriptor::SOCKS5) {
		pass.Clear();
	}

	ProxyDescriptor proxy;
	proxy.setType(type);
	SymbolicInetAddr const addr(wxStringToUtf8(host).raw(), port);
	proxy.setAddr(addr);
	proxy.setUserName(wxStringToUtf8(user).raw());
	proxy.setPassword(wxStringToUtf8(pass).raw());
	proxy.swap(target);
	return true;
}

bool
ForwardingConfigWindow::ProxyEditPanel::haveNextHopProxy() const
{
	Forwarding::Config const& config = m_rOwner.forwardingConfig();
	
	assert(m_optionIdx >= 0);
	assert(m_optionIdx < (int)config.options().size());

	Forwarding::OptionList::const_iterator opt(config.options().begin());
	std::advance(opt, m_optionIdx);
	
	int const next = (m_mode == ADDING ? m_proxyIdx : m_proxyIdx + 1);
	return (next < (int)opt->proxyChain().size());
}

void
ForwardingConfigWindow::ProxyEditPanel::onOK(wxCommandEvent&)
{
	ProxyDescriptor pdesc;
	if (!validateInteractive(pdesc)) {
		return;
	}

	Forwarding::Config new_config(m_rOwner.forwardingConfig());
	
	assert(m_optionIdx >= 0);
	assert(m_optionIdx < (int)new_config.options().size());

	Forwarding::OptionList::iterator opt(new_config.options().begin());
	std::advance(opt, m_optionIdx);

	assert(m_proxyIdx >= 0);
	assert(m_proxyIdx <= (int)opt->proxyChain().size());
	// transition_idx may point past the last proxy

	Forwarding::ProxyDescriptorList::iterator proxy(opt->proxyChain().begin());
	std::advance(proxy, m_proxyIdx);

	if (m_mode == ADDING) {
		opt->proxyChain().insert(proxy, pdesc);
	} else if (m_mode == EDITING) {
		proxy->swap(pdesc);
	} else {
		assert(0);
	}
	
	Impl::ensureProperSelection(new_config);
	if (!FCF().applyAndSave(new_config)) {
		return;
	} else {
		m_rOwner.forwardingConfig().swap(new_config);
	}
	
	m_rOwner.commitProxyOperation();
}

void
ForwardingConfigWindow::ProxyEditPanel::onCancel(wxCommandEvent&)
{
	m_rOwner.stopProxyOperation();
}


/*================= ForwardingConfigWindow::BypasssPanel =================*/

ForwardingConfigWindow::BypassPanel::BypassPanel(
	wxWindow* parent, Impl& owner)
:	wxPanel(parent, -1),
	m_rOwner(owner),
	m_pBypassHosts(0),
	m_pBypassLocal(0),
	m_pOKBtn(0),
	m_pCancelBtn(0),
	m_optionIdx(-1),
	m_ignoreModifications(0),
	m_bypassLocalInitial(false),
	m_wasModified(false)
{
	wxBoxSizer *topsizer = new wxBoxSizer(wxVERTICAL);
	SetSizer(topsizer);
	
	{
		wxBoxSizer* sizer = new wxBoxSizer(wxHORIZONTAL);
		topsizer->Add(sizer, 0, wxEXPAND|wxBOTTOM, 3);

		sizer->Add(
			new wxStaticText(
				this, -1, _T("Direct connection to certain hosts:")
			)
		);
		
		wxHyperlinkCtrl* link = new wxHyperlinkCtrl(
			this, ID_BYPASS_HOSTS_HELP, _T(" (?)"), wxEmptyString,
			wxDefaultPosition, wxDefaultSize, wxNO_BORDER|wxHL_ALIGN_LEFT
		);
		sizer->Add(link);
		link->SetVisitedColour(link->GetNormalColour());
	
		wxFont font(link->GetFont());
		font.SetUnderlined(false);
		link->SetFont(font);
	}

	m_pBypassHosts = new wxTextCtrl(
		this, ID_BYPASS_HOSTS, wxEmptyString,
		wxDefaultPosition, wxDefaultSize, wxTE_MULTILINE|wxTE_RICH
	);
	topsizer->Add(m_pBypassHosts, 1, wxEXPAND);

	topsizer->Add(3, 3);
	
	{
		wxBoxSizer* sizer = new wxBoxSizer(wxHORIZONTAL);
		topsizer->Add(sizer, 0, wxEXPAND);

		m_pBypassLocal = new wxCheckBox(
			this, ID_BYPASS_LOCAL, _T("Direct connection to local addresses")
		);
		sizer->Add(m_pBypassLocal);
		
		wxHyperlinkCtrl* link = new wxHyperlinkCtrl(
			this, ID_BYPASS_LOCAL_HELP, _T(" (?)"), wxEmptyString,
			wxDefaultPosition, wxDefaultSize, wxNO_BORDER|wxHL_ALIGN_LEFT
		);
		sizer->Add(link);
		link->SetVisitedColour(link->GetNormalColour());
	
		wxFont font(link->GetFont());
		font.SetUnderlined(false);
		link->SetFont(font);
	}
	
	topsizer->Add(5, 5);

	wxBoxSizer *buttonsizer = new wxBoxSizer(wxHORIZONTAL);
	topsizer->Add(buttonsizer, 0, wxALIGN_CENTRE);
	m_pOKBtn = new wxButton(this, wxID_OK, _T("OK"));
	buttonsizer->Add(m_pOKBtn, 0, wxLEFT|wxRIGHT, 5);
	m_pCancelBtn = new wxButton(this, wxID_CANCEL, _T("Cancel"));
	buttonsizer->Add(m_pCancelBtn, 0, wxLEFT|wxRIGHT, 5);
	m_pOKBtn->Disable();
	m_pCancelBtn->Disable();

	topsizer->SetSizeHints(this);

	m_helpProvider.AddHelp(
		m_pBypassHosts,
		_T("Examples:\n")
		_T("host.com, *.host.com, 192.168.*.*")
	);

	m_helpProvider.AddHelp(
		m_pBypassLocal,
		_T("Examples of local addresses:\n")
		_T("http://files/upload/\n")
		_T("http://webmail/login.php")
	);
}

ForwardingConfigWindow::BypassPanel::~BypassPanel()
{
}

void
ForwardingConfigWindow::BypassPanel::loadFrom(
	Forwarding::Option const& opt, int const option_idx)
{	
	ScopedIncDec<unsigned char> guard(m_ignoreModifications);
	m_optionIdx = option_idx;
	m_pBypassHosts->Clear();
	m_pBypassLocal->SetValue(false);
	opt.bypassList().accept(*this);
	m_pBypassHosts->DiscardEdits();
	m_bypassLocalInitial = m_pBypassLocal->GetValue();
	m_wasModified = false;
	m_pOKBtn->Disable();
	m_pCancelBtn->Disable();
}

void
ForwardingConfigWindow::BypassPanel::visit(
	Forwarding::HostMaskMatchPattern const& p)
{
	if (m_pBypassHosts->GetLastPosition() != 0) {
		m_pBypassHosts->AppendText(_T(", "));
	}
	m_pBypassHosts->AppendText(utf8ToWxString(p.getPattern()));
}

void
ForwardingConfigWindow::BypassPanel::visit(
	Forwarding::SimpleHostnamesMatchPattern const& p)
{
	m_pBypassLocal->SetValue(true);
}

void
ForwardingConfigWindow::BypassPanel::onBypassHostsHelp(wxHyperlinkEvent&)
{
	m_helpProvider.ShowHelp(m_pBypassHosts);
}

void
ForwardingConfigWindow::BypassPanel::onBypassLocalHelp(wxHyperlinkEvent&)
{
	m_helpProvider.ShowHelp(m_pBypassLocal);
}

void
ForwardingConfigWindow::BypassPanel::onBypassHostsModified(wxCommandEvent&)
{
	if (m_ignoreModifications) {
		return;
	}

	if (m_pBypassHosts->IsModified()) {
		if (!m_wasModified) {
			onModified(true);
		}
	} else if (m_wasModified) {
		if (m_pBypassLocal->GetValue() == m_bypassLocalInitial) {
			onModified(false);
		}
	}
}

void
ForwardingConfigWindow::BypassPanel::onBypassLocalModified(wxCommandEvent& evt)
{
	if (m_ignoreModifications) {
		return;
	}
	
	if (evt.IsChecked() != m_bypassLocalInitial) {
		if (!m_wasModified) {
			onModified(true);
			
		}
	} else if (m_wasModified) {
		if (!m_pBypassHosts->IsModified()) {
			onModified(false);
		}
	}
}

void
ForwardingConfigWindow::BypassPanel::onModified(bool const modified)
{
	if (modified == m_wasModified) {
		return;
	}

	if (modified) {
		m_rOwner.enterBypassEditingMode();
	} else {
		m_rOwner.leaveBypassEditingMode();
	}

	m_pOKBtn->Enable(modified);
	m_pCancelBtn->Enable(modified);

	m_wasModified = modified;
}

void
ForwardingConfigWindow::BypassPanel::onOK(wxCommandEvent&)
{
	Forwarding::Config new_config(m_rOwner.forwardingConfig());
	Forwarding::OptionList::iterator opt(new_config.options().begin());
	assert(m_optionIdx >= 0);
	assert(m_optionIdx < (int)new_config.options().size());
	std::advance(opt, m_optionIdx);
	opt->bypassList().clear();

	wxString hosts(m_pBypassHosts->GetValue());
	for (;;) {
		wxString const pattern(nextPattern(hosts));
		if (pattern.IsEmpty()) {
			break;
		}
		opt->bypassList().push_back(
			Forwarding::BypassPatternConstPtr(
				new Forwarding::HostMaskMatchPattern(wxStringToUtf8(pattern))
			)
		);
	}

	if (m_pBypassLocal->GetValue()) {
		opt->bypassList().push_back(
			Forwarding::BypassPatternConstPtr(
				new Forwarding::SimpleHostnamesMatchPattern()
			)
		);
	}

	Impl::ensureProperSelection(new_config);
	if (!FCF().applyAndSave(new_config)) {
		return;
	} else {
		m_rOwner.forwardingConfig().swap(new_config);
	}
	
	onModified(false);
}

void
ForwardingConfigWindow::BypassPanel::onCancel(wxCommandEvent&)
{
	onModified(false);
	m_rOwner.reloadBypassData();
}

wxString
ForwardingConfigWindow::BypassPanel::nextPattern(wxString& patterns)
{
	static wxChar const sep[] = _T(",; \t\r\n");

	size_t const begin = patterns.find_first_not_of(sep);
	if (begin == patterns.npos) {
		patterns.Clear();
		return wxEmptyString;
	}

	size_t end = patterns.find_first_of(sep, begin + 1);
	if (end == patterns.npos) {
		end = patterns.size();
	}

	wxString const pattern(patterns.substr(begin, end - begin));
	patterns.substr(end).swap(patterns);
	return pattern;
}

} // namespace wxGUI
