/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "RequestLogHandler.h"
#include "Application.h"
#include "RequestLogIcons.h"
#include "RequestLog.h"
#include "RequestTag.h"
#include "MvcListModel.h"
#include "MvcListView.h"
#include "StrongPtr.h"
#include "WeakPtr.h"
#include "DownloadProgress.h"
#include "ArraySize.h"
#include "types.h"
#include <wx/listctrl.h>
#include <wx/string.h>
#include <wx/colour.h>
#include <wx/dataobj.h>
#include <wx/clipbrd.h>
#include <memory>
#include <string>
#include <deque>
#include <map>
#include <stddef.h>

using namespace std;

namespace wxGUI
{

struct RequestLogHandler::ListRow
{
	ListRow(RequestId const& id, unsigned seq_id);
	
	~ListRow();
	
	RequestId id;
	unsigned seq_id;
	wxString size;
	wxString url;
	RequestType type;
	int status_code;
	wxColour background;
	int image;
};


class RequestLogHandler::ListModel : public MvcListModel
{
public:
	enum { MAX_ROWS = 400 };

	typedef WeakPtr<ACE_NULL_SYNCH, MvcListModel, MvcListModel*> WeakHandle;
	
	ListModel();
	
	virtual ~ListModel();
	
	WeakHandle getHandle() const { return m_handle; }

	void processRequest(
		RequestId const& req_id, std::string const& uri, RequestType type);
	
	void processResponseBegin(
		RequestId const& req_id, int status_code,
		DownloadProgress const& progress);
	
	void processResponseEnd(
		RequestId const& req_id, int req_flags,
		DownloadProgress const& progress, bool error);
	
	void processRequestCancel(
		RequestId const& req_id, DownloadProgress const& progress);
	
	void clear();

	void copyToClipboard();

	void copyUrlAtRow(long row);
private:
	typedef std::deque<ListRow> Rows;
	typedef std::map<RequestId, ListRow*> RowsById;
	
	virtual long getRowCount() const;

	virtual wxString getCellText(long row, long col) const;

	virtual int getRowImage(long row) const;

	virtual wxListItemAttr* getRowAttr(long row) const;

	static wxString formatSize(uintmax_t size);
	
	typedef StrongPtr<ACE_NULL_SYNCH, MvcListModel, MvcListModel*> StrongHandle;
	StrongHandle m_handle;
	Rows m_rows;
	RowsById m_rowsById;
	unsigned m_nextSeqId;
	mutable wxListItemAttr m_normalRequestAttr;
	mutable wxListItemAttr m_substRequestAttr;
	mutable wxListItemAttr m_analyzeRequestAttr;
	mutable wxListItemAttr m_internalRequestAttr;
};


RequestLogHandler::RequestLogHandler(InterthreadCommandQueue& command_queue)
:	m_rCommandQueue(command_queue),
	m_ptrModel(new ListModel)
{
	RequestLog::setHandler(this);
}

RequestLogHandler::~RequestLogHandler()
{
}

void
RequestLogHandler::attachView(MvcListView& view)
{
	view.ClearAll();
	
	// Too bad wxLIST_AUTOSIZE doesn't work in virtual mode.
	view.InsertColumn(0, wxEmptyString, wxLIST_FORMAT_LEFT, 20);
	view.InsertColumn(1, wxEmptyString, wxLIST_FORMAT_RIGHT, 45);
	view.InsertColumn(2, wxEmptyString, wxLIST_FORMAT_LEFT, 1000);

	view.setModel(m_ptrModel->getHandle());

	// scroll to bottom
	int delta = view.GetScrollRange(wxVERTICAL) -
		view.GetScrollPos(wxVERTICAL) - view.GetCountPerPage();
	if (delta > 0) {
		view.ScrollLines(delta);
	}
}

void
RequestLogHandler::clearLog()
{
	m_ptrModel->clear();
}

void
RequestLogHandler::copyToClipboard()
{
	m_ptrModel->copyToClipboard();
}

void
RequestLogHandler::copyUrlAtRow(long row)
{
	m_ptrModel->copyUrlAtRow(row);
}

wxColour
RequestLogHandler::getColorFor(RequestType type)
{
	switch (type) {
		case RequestLog::SUBST_REQUEST: {
			return wxColour(0xfd, 0xf7, 0x9e);
		}
		case RequestLog::ANALYZE_REQUEST: {
			return wxColour(0xff, 0xe1, 0xbe);
		}
		case RequestLog::INTERNAL_REQUEST: {
			return wxColour(0xd6, 0xd1, 0xee);
		}
		default: {
			return wxColour(0xff, 0xff, 0xff);
		}
	}
}

InterthreadCommandQueue&
RequestLogHandler::getCommandQueue()
{
	return m_rCommandQueue;
}

void
RequestLogHandler::processRequest(
	RequestId const& req_id, std::string const& uri, RequestType type)
{
	m_ptrModel->processRequest(req_id, uri, type);
}

void
RequestLogHandler::processResponseBegin(
	RequestId const& req_id, int status_code,
	DownloadProgress const& progress)
{
	m_ptrModel->processResponseBegin(req_id, status_code, progress);
}

void
RequestLogHandler::processResponseEnd(
	RequestId const& req_id, int req_flags,
	DownloadProgress const& progress, bool error)
{
	m_ptrModel->processResponseEnd(req_id, req_flags, progress, error);
}

void
RequestLogHandler::processRequestCancel(
	RequestId const& req_id, DownloadProgress const& progress)
{
	m_ptrModel->processRequestCancel(req_id, progress);
}


/*==================== RequestLogHandler::ListRow =======================*/

RequestLogHandler::ListRow::ListRow(RequestId const& id, unsigned seq_id)
:	id(id),
	seq_id(seq_id),
	image(RequestLogIcons::ICON_NONE)
{
}

RequestLogHandler::ListRow::~ListRow()
{
}


/*=================== RequestLogHandler::ListModel ======================*/

RequestLogHandler::ListModel::ListModel()
:	m_handle(this),
	m_nextSeqId(0)
{
	typedef RequestLog RL;
	m_normalRequestAttr.SetBackgroundColour(getColorFor(RL::NORMAL_REQUEST));
	m_substRequestAttr.SetBackgroundColour(getColorFor(RL::SUBST_REQUEST));
	m_analyzeRequestAttr.SetBackgroundColour(getColorFor(RL::ANALYZE_REQUEST));
	m_internalRequestAttr.SetBackgroundColour(getColorFor(RL::INTERNAL_REQUEST));
}

RequestLogHandler::ListModel::~ListModel()
{
}

void
RequestLogHandler::ListModel::processRequest(
	RequestId const& req_id, std::string const& uri, RequestType type)
{
	bool const excess_row = (m_rows.size() == MAX_ROWS);
	if (excess_row) {
		m_rowsById.erase(m_rows.front().id);
		m_rows.pop_front();
	}
	
	m_rows.push_back(ListRow(req_id, m_nextSeqId++));
	ListRow& row = m_rows.back();
	m_rowsById.insert(RowsById::value_type(req_id, &row));
	row.type = type;
	row.url = uri.c_str();
	row.background = getColorFor(type);
	
	if (excess_row) {
		// the total number of rows didn't change
		informRowsModified(0, m_rows.size());
	} else {
		informRowInserted(m_rows.size() - 1);
	}
}

void
RequestLogHandler::ListModel::processResponseBegin(
	RequestId const& req_id, int status_code,
	DownloadProgress const& progress)
{
	RowsById::iterator it = m_rowsById.find(req_id);
	if (it == m_rowsById.end()) {
		return;
	}
	
	ListRow& row = *it->second;
	row.status_code = status_code;
}

void
RequestLogHandler::ListModel::processResponseEnd(
	RequestId const& req_id, int req_flags,
	DownloadProgress const& progress, bool error)
{
	RowsById::iterator it = m_rowsById.find(req_id);
	if (it == m_rowsById.end()) {
		return;
	}
	
	ListRow& row = *it->second;
	row.size = formatSize(progress.received());
	
	int icon = RequestLogIcons::ICON_OK;
	if (error) {
		icon = RequestLogIcons::ICON_ERROR;
	} else if (row.type == RequestLog::SUBST_REQUEST ||
		   (req_flags & RequestTag::RESPONSE_CRAFTED)) {
		icon = RequestLogIcons::ICON_AD;
	} else {
		int const scode = row.status_code;
		if (scode >= 400) {
			icon = RequestLogIcons::ICON_ERROR;
		} else if (scode >= 300) {
			if (scode == 304) {
				icon = RequestLogIcons::ICON_CACHE;
			} else {
				icon = RequestLogIcons::ICON_REDIRECT;
			}
		}
	}
	row.image = icon;

	informRowModified(row.seq_id - m_rows.front().seq_id);
}

void
RequestLogHandler::ListModel::processRequestCancel(
	RequestId const& req_id, DownloadProgress const& progress)
{
	RowsById::iterator it = m_rowsById.find(req_id);
	if (it == m_rowsById.end()) {
		return;
	}
	
	ListRow& row = *it->second;
	row.size = formatSize(progress.received());
	row.image = RequestLogIcons::ICON_CANCEL;

	informRowModified(row.seq_id - m_rows.front().seq_id);
}

void
RequestLogHandler::ListModel::clear()
{
	long num_rows = m_rows.size();
	m_rows.clear();
	m_rowsById.clear();
	informRowsDeleted(0, num_rows);
}

void
RequestLogHandler::ListModel::copyToClipboard()
{
	static char const padding[5] = {' ', ' ', ' ', ' ', ' ' };
	wxString str;
	
	Rows::iterator it = m_rows.begin();
	Rows::iterator const end = m_rows.end();
	for (; it != end; ++it) {
		wxString size = it->size;
		wxString url = it->url;
		int const pad_width = ARRAY_SIZE(padding) - size.size();
		if (pad_width > 0) {
			str.append(padding, pad_width);
		}
		str << size << "  " << url << "\r\n";
	}
	
	auto_ptr<wxTextDataObject> data(new wxTextDataObject(str));
	wxTheClipboard->Open();
	wxTheClipboard->SetData(data.release());
	wxTheClipboard->Close();
}

void
RequestLogHandler::ListModel::copyUrlAtRow(long row)
{
	if (row < 0 || size_t(row) >= m_rows.size()) {
		return;
	}

	ListRow const& rec = m_rows[row];
	auto_ptr<wxTextDataObject> data(new wxTextDataObject(rec.url));
	wxTheClipboard->Open();
	wxTheClipboard->SetData(data.release());
	wxTheClipboard->Close();
}

long
RequestLogHandler::ListModel::getRowCount() const
{
	return m_rows.size();
}

wxString
RequestLogHandler::ListModel::getCellText(long row, long col) const
{
	if (row < 0 || size_t(row) >= m_rows.size()) {
		return wxEmptyString;
	}

	ListRow const& rec = m_rows[row];
	if (col == 1) {
		return rec.size;
	} else if (col == 2) {
		return rec.url;
	} else {
		return wxEmptyString;
	}
}

int
RequestLogHandler::ListModel::getRowImage(long row) const
{
	if (row < 0 || size_t(row) >= m_rows.size()) {
		return -1;
	}

	return m_rows[row].image;
}

wxListItemAttr*
RequestLogHandler::ListModel::getRowAttr(long row) const
{
	if (row < 0 || size_t(row) >= m_rows.size()) {
		return 0;
	}

	switch (m_rows[row].type) {
		case RequestLog::NORMAL_REQUEST:
			return &m_normalRequestAttr;
		case RequestLog::SUBST_REQUEST:
			return &m_substRequestAttr;
		case RequestLog::ANALYZE_REQUEST:
			return &m_analyzeRequestAttr;
		case RequestLog::INTERNAL_REQUEST:
			return &m_internalRequestAttr;
	}

	return 0;
}

wxString
RequestLogHandler::ListModel::formatSize(uintmax_t size)
{
	static char const suffix[] = { 'B', 'K', 'M', 'G', 'T' };
	
	wxString str;
	uintmax_t divisor = 1;
	
	for (int i = 0;; ++i, divisor <<= 10) {
		if (size < divisor * 1000u || i + 1 == ARRAY_SIZE(suffix)) {
			str << (unsigned long)((size + divisor - 1u) / divisor) << ' ' << suffix[i];
			break;
		}
	}
	
	return str;
}

} // namespace wxGUI
