/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef POLLREACTOR_H_
#define POLLREACTOR_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// Here we actually mean: #if <we don't have to support Panther>
#if defined(__APPLE__) && defined(__i386__)
#  ifndef HAVE_NATIVE_POLL
#    define HAVE_NATIVE_POLL
#  endif
#  ifndef HAVE_POLL_H
#    define HAVE_POLL_H
#  endif
#endif

#ifdef HAVE_NATIVE_POLL

#define HAVE_POLL_REACTOR

#include "Reactor.h"
#include "WakeupPipe.h"
#include "NonCopyable.h"
#include "types.h"
#include <ace/config-lite.h>
#include <sigc++/sigc++.h>
#include <memory>
#include <stddef.h>
#ifdef HAVE_POLL_H
#  include <poll.h>
#endif
#ifdef HAVE_SYS_POLL_H
#  include <sys/poll.h>
#endif

class AbstractSynchFactory;
class ACE_Lock;

namespace ReactorHelpers {
	struct Handler;
	class HandlerRepository;
	class TimerQueue;
	
	
	class PollWaitSet
	{
		DECLARE_NON_COPYABLE(PollWaitSet)
	public:
		PollWaitSet(ACE_HANDLE wakeup_handle);
		
		~PollWaitSet();
		
		size_t size() const { return m_size; }
		
		size_t capacity() const { return m_capacity; }
		
		void ensureCapacity(size_t capacity);
		
		void reduceCapacity(size_t capacity);
		
		void trim(size_t to);
		
		void append(pollfd const& fd);
		
		int poll(int timeout, bool& woken_up);
		
		pollfd& operator[](ptrdiff_t idx) { return m_fdArray[idx + 1]; }
		
		pollfd const& operator[](ptrdiff_t idx) const { return m_fdArray[idx + 1]; }
	private:
		void resizeFdArray(size_t user_requested_size);
		
		size_t adviceFdArraySize(size_t requested_size);
		
		size_t m_size; // not counting the wakeup handle
		size_t m_capacity; // not counting the wakeup handle
		pollfd* m_fdArray; // the first one is the wakeup fd
	};

} // namespace ReactorHelpers


class PollReactor : public Reactor
{
	DECLARE_NON_COPYABLE(PollReactor)
public:
	PollReactor(
		AbstractSynchFactory const& synch_factory,
		bool wakeup_on_signal = false);
	
	virtual ~PollReactor();
	
	virtual ReactorHandlerId findHandler(ACE_HANDLE handle) const;
	
	virtual ReactorHandlerId registerHandler(
	    ACE_HANDLE handle, EventHandlerPtr const& handler, IOEvents events);
	
	virtual void unregisterHandler(ReactorHandlerId const& id);
	
	virtual void unregisterAllHandlers();
	
	virtual void enableEvents(ReactorHandlerId const& id, IOEvents events);
	
	virtual void disableEvents(ReactorHandlerId const& id, IOEvents events);
	
	virtual void setEvents(ReactorHandlerId const& id, IOEvents events);
	
	virtual IOEvents getEvents(ReactorHandlerId const& id) const;
	
	virtual ReactorTimerId registerTimer(
		EventHandlerPtr const& handler, TimeDelta const* timeout = 0);
	
	virtual void rescheduleTimer(
		ReactorTimerId const& id, TimeDelta const* timeout = 0);
	
	virtual void unregisterTimer(ReactorTimerId const& id);
	
	virtual void unregisterAllTimers();
	
	virtual TimeDelta getRemainingTime(ReactorTimerId const& timer_id) const;
	
	virtual Status handleEvents();
	
	virtual Status runEventLoop();
	
	virtual void wakeup();
	
	virtual void stop();
	
	virtual void restart();
	
	virtual sigc::signal<void>& beforeSleepSignal();
private:
	typedef void (EventHandlerBase::*HandlerFuncPtr)(ACE_HANDLE);
	
	struct IODispatchIter
	{
		ptrdiff_t pos;
		unsigned todo;
		short revents;
		unsigned char phase;
		
		static ptrdiff_t const INITIAL_POS = -1;
		
		static unsigned char const WRITE_PHASE = 0;
		static unsigned char const EXCEPT_PHASE = 1;
		static unsigned char const READ_PHASE = 2;
		static unsigned char const INITIAL_PHASE = 2;
		static unsigned char const MIN_PHASE = 0;
		static unsigned char const MAX_PHASE = 2;
		
		IODispatchIter(unsigned todo);
		
		void reset(unsigned todo);

		bool next(ReactorHelpers::PollWaitSet const& ws);
	};
	
	void updateWaitSet();
	
	int getPollTimeout() const;
	
	void continueIODispatching();
	
	void dispatchIO(ReactorHelpers::Handler const& handler);
	
	void dispatchIOEvent(
		ACE_HANDLE handle, EventHandlerBase* eh, HandlerFuncPtr func);
	
	std::auto_ptr<ReactorHelpers::HandlerRepository> m_ptrHandlers;
	std::auto_ptr<ReactorHelpers::TimerQueue> m_ptrTimers;
	std::auto_ptr<ACE_Lock> m_ptrMutex; // available in poll() and in event handlers
	std::auto_ptr<ACE_Lock> m_ptrDemuxMutex; // available outside of poll()
	ReactorHelpers::WakeupPipe m_wakeupPipe;
	ReactorHelpers::PollWaitSet m_waitSet1;
	ReactorHelpers::PollWaitSet m_waitSet2;
	ReactorHelpers::PollWaitSet* m_pActiveWaitSet;
	ReactorHelpers::PollWaitSet* m_pInactiveWaitSet;
	unsigned m_dispatchWave;
	IODispatchIter m_ioIter;
	bool m_wakeupOnSignal;
	bool m_isInsideDemux;
	bool m_isStopped;
	int32_t m_isStoppedAtomic;
	sigc::signal<void> m_beforeSleepSignal;
};

#endif // HAVE_NATIVE_POLL

#endif
