/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef SELECTREACTOR_H_
#define SELECTREACTOR_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Reactor.h"
#include "WakeupPipe.h"
#include "NonCopyable.h"
#include "types.h"
#include <ace/config-lite.h>
#include <ace/Handle_Set.h>
#include <sigc++/sigc++.h>
#include <memory>
#include <stddef.h>

class AbstractSynchFactory;
class ACE_Lock;

namespace ReactorHelpers {
	struct Handler;
	class HandlerRepository;
	class TimerQueue;
}


class SelectReactor : public Reactor
{
	DECLARE_NON_COPYABLE(SelectReactor)
public:
	SelectReactor(
		AbstractSynchFactory const& synch_factory,
		bool wakeup_on_signal = false);
	
	virtual ~SelectReactor();
	
	virtual ReactorHandlerId findHandler(ACE_HANDLE handle) const;
	
	virtual ReactorHandlerId registerHandler(
	    ACE_HANDLE handle, EventHandlerPtr const& handler, IOEvents events);
	
	virtual void unregisterHandler(ReactorHandlerId const& id);
	
	virtual void unregisterAllHandlers();
	
	virtual void enableEvents(ReactorHandlerId const& id, IOEvents events);
	
	virtual void disableEvents(ReactorHandlerId const& id, IOEvents events);
	
	virtual void setEvents(ReactorHandlerId const& id, IOEvents events);
	
	virtual IOEvents getEvents(ReactorHandlerId const& id) const;
	
	virtual ReactorTimerId registerTimer(
		EventHandlerPtr const& handler, TimeDelta const* timeout = 0);
	
	virtual void rescheduleTimer(
		ReactorTimerId const& id, TimeDelta const* timeout = 0);
	
	virtual void unregisterTimer(ReactorTimerId const& id);
	
	virtual void unregisterAllTimers();
	
	virtual TimeDelta getRemainingTime(ReactorTimerId const& timer_id) const;
	
	virtual Status handleEvents();
	
	virtual Status runEventLoop();
	
	virtual void wakeup();
	
	virtual void stop();
	
	virtual void restart();
	
	virtual sigc::signal<void>& beforeSleepSignal();
private:
	typedef void (EventHandlerBase::*HandlerFuncPtr)(ACE_HANDLE);
	
	struct HandleSet
	{
		ACE_Handle_Set read;
		ACE_Handle_Set write;
		ACE_Handle_Set except;
		
		void reset() { read.reset(); write.reset(); except.reset(); }
	};
	
	struct IODispatchIter
	{
		ACE_Handle_Set_Iterator write_iter;
		ACE_Handle_Set_Iterator except_iter;
		ACE_Handle_Set_Iterator read_iter;
		unsigned todo;
		ACE_HANDLE handle;
		unsigned char phase;
		
		static unsigned char const WRITE_PHASE = 0;
		static unsigned char const EXCEPT_PHASE = 1;
		static unsigned char const READ_PHASE = 2;
		static unsigned char const INITIAL_PHASE = 2;
		
		void nextPhase() {
			phase = static_cast<unsigned char>(0x00000201 >> (phase << 3));
		}
		
		IODispatchIter(HandleSet const& hs);
		
		void reset(unsigned todo);

		bool next();
	};
	
	int select(bool& woken_up, bool& interrupted);
	
	TimeDelta getSelectTimeout() const;
	
	void nextDispatchWave();
	
	void continueIODispatching();
	
	void dispatchIO(ReactorHelpers::Handler const& handler);
	
	void dispatchIOEvent(
		ACE_HANDLE handle, EventHandlerBase* eh, HandlerFuncPtr func);
	
	static void setBits(HandleSet& hs, ACE_HANDLE handle, IOEvents events);
	
	std::auto_ptr<ReactorHelpers::HandlerRepository> m_ptrHandlers;
	std::auto_ptr<ReactorHelpers::TimerQueue> m_ptrTimers;
	std::auto_ptr<ACE_Lock> m_ptrMutex; // available in select() and in event handlers
	std::auto_ptr<ACE_Lock> m_ptrDemuxMutex; // available outside of select()
	ReactorHelpers::WakeupPipe m_wakeupPipe;
	HandleSet m_waitSet;
	HandleSet m_dispatchSet;
	unsigned m_dispatchWave; // must have the same type as Handler::flags
	IODispatchIter m_ioIter;
	bool m_wakeupOnSignal;
	bool m_isInsideDemux;
	bool m_isStopped;
	int32_t m_isStoppedAtomic;
	sigc::signal<void> m_beforeSleepSignal;
};

#endif
