/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "SHA1.h"
#include <algorithm>
#include <string.h> // for memcpy() and memset()
#include <assert.h>

SHA1::SHA1()
{
	reset();
}

void
SHA1::reset()
{
	m_inputSize = 0;
	m_bufPos = 0;
	m_state[0] = 0x67452301;
	m_state[1] = 0xefcdab89;
	m_state[2] = 0x98badcfe;
	m_state[3] = 0x10325476;
	m_state[4] = 0xc3d2e1f0;
}

void SHA1::feed(uint8_t const *data, size_t len)
{
	m_inputSize += len;
	
	if (m_bufPos != 0) {
		size_t const todo = std::min<size_t>(len, BLOCK_SIZE - m_bufPos);
		memcpy(m_buf + m_bufPos, data, todo);
		data += todo;
		len -= todo;
		m_bufPos += todo;
		if (m_bufPos == BLOCK_SIZE) {
			transform(m_buf);
			m_bufPos = 0;
		}
	}
	
	assert(m_bufPos == 0 || len == 0);
	
	while (len >= BLOCK_SIZE) {
		transform(data);
		data += BLOCK_SIZE;
		len -= BLOCK_SIZE;
	}
	
	if (len != 0) {
		assert(m_bufPos == 0);
		memcpy(m_buf, data, len);
		m_bufPos = len;
	}
}

void
SHA1::finalize(HashValue<HASH_SIZE>& output)
{
	if (m_bufPos < BLOCK_SIZE - 8) {
		m_buf[m_bufPos] = 0x80;
		++m_bufPos;
		for (; m_bufPos < BLOCK_SIZE - 8; ++m_bufPos) {
			m_buf[m_bufPos] = 0;
		}
	} else {
		m_buf[m_bufPos] = 0x80;
		++m_bufPos;
		for (; m_bufPos < BLOCK_SIZE; ++m_bufPos) {
			m_buf[m_bufPos] = 0;
		}
		transform(m_buf);
		memset(m_buf, 0, BLOCK_SIZE - 8);
	}
	
	uint64_t const input_bits = m_inputSize << 3;
	m_buf[BLOCK_SIZE - 8] = input_bits >> 56;
	m_buf[BLOCK_SIZE - 7] = input_bits >> 48;
	m_buf[BLOCK_SIZE - 6] = input_bits >> 40;
	m_buf[BLOCK_SIZE - 5] = input_bits >> 32;
	m_buf[BLOCK_SIZE - 4] = input_bits >> 24;
	m_buf[BLOCK_SIZE - 3] = input_bits >> 16;
	m_buf[BLOCK_SIZE - 2] = input_bits >> 8;
	m_buf[BLOCK_SIZE - 1] = input_bits;
	transform(m_buf);
	
#if defined(__BIG_ENDIAN__)
	output.set((uint8_t const*)m_state);
#else
	{
		// This code works with any byte order.
		uint8_t buf[HASH_SIZE];
		uint8_t *p = buf;
		for (int i = 0; i < 5; ++i) {
			uint32_t const state = m_state[i];
			*p++ = state >> 24;
			*p++ = state >> 16;
			*p++ = state >> 8;
			*p++ = state;
		}
		output.set(buf);
	}
#endif
}

namespace
{

inline uint32_t rol(uint32_t val, int bits) {
	return (val << bits) | (val >> (32 - bits));
}

} // anonymouse namespace

void SHA1::transform(uint8_t const* data)
{
	uint32_t a, b, c, d, e, tm;
	uint32_t x[16];
	
	a = m_state[0];
	b = m_state[1];
	c = m_state[2];
	d = m_state[3];
	e = m_state[4];

#if defined(__BIG_ENDIAN__)
	memcpy(x, data, BLOCK_SIZE);
#else
	{
		// This code works with any byte order.
		uint32_t* dst = x;
		uint8_t const* src = data;
		uint8_t const* const src_end = src + BLOCK_SIZE;
		for (; src != src_end; src += 4, ++dst) {
			uint32_t val = src[0];
			val <<= 8;
			val |= src[1];
			val <<= 8;
			val |= src[2];
			val <<= 8;
			val |= src[3];
			*dst = val;
		}
	}
#endif

#define SHA1_K1  0x5A827999L
#define SHA1_K2  0x6ED9EBA1L
#define SHA1_K3  0x8F1BBCDCL
#define SHA1_K4  0xCA62C1D6L
#define SHA1_F1(x,y,z)   ( z ^ ( x & ( y ^ z ) ) )
#define SHA1_F2(x,y,z)   ( x ^ y ^ z )
#define SHA1_F3(x,y,z)   ( ( x & y ) | ( z & ( x | y ) ) )
#define SHA1_F4(x,y,z)   ( x ^ y ^ z )

#define M(i) ( tm =   x[i&0x0f] ^ x[(i-14)&0x0f] \
		    ^ x[(i-8)&0x0f] ^ x[(i-3)&0x0f] \
	       , (x[i&0x0f] = rol(tm, 1)) )

#define SHA1_R(a,b,c,d,e,f,k,m)  do { e += rol( a, 5 )     \
				      + f( b, c, d )  \
				      + k	      \
				      + m;	      \
				 b = rol( b, 30 );    \
			       } while(0)
	SHA1_R(a, b, c, d, e, SHA1_F1, SHA1_K1, x[0]);
	SHA1_R(e, a, b, c, d, SHA1_F1, SHA1_K1, x[1]);
	SHA1_R(d, e, a, b, c, SHA1_F1, SHA1_K1, x[2]);
	SHA1_R(c, d, e, a, b, SHA1_F1, SHA1_K1, x[3]);
	SHA1_R(b, c, d, e, a, SHA1_F1, SHA1_K1, x[4]);
	SHA1_R(a, b, c, d, e, SHA1_F1, SHA1_K1, x[5]);
	SHA1_R(e, a, b, c, d, SHA1_F1, SHA1_K1, x[6]);
	SHA1_R(d, e, a, b, c, SHA1_F1, SHA1_K1, x[7]);
	SHA1_R(c, d, e, a, b, SHA1_F1, SHA1_K1, x[8]);
	SHA1_R(b, c, d, e, a, SHA1_F1, SHA1_K1, x[9]);
	SHA1_R(a, b, c, d, e, SHA1_F1, SHA1_K1, x[10]);
	SHA1_R(e, a, b, c, d, SHA1_F1, SHA1_K1, x[11]);
	SHA1_R(d, e, a, b, c, SHA1_F1, SHA1_K1, x[12]);
	SHA1_R(c, d, e, a, b, SHA1_F1, SHA1_K1, x[13]);
	SHA1_R(b, c, d, e, a, SHA1_F1, SHA1_K1, x[14]);
	SHA1_R(a, b, c, d, e, SHA1_F1, SHA1_K1, x[15]);
	SHA1_R(e, a, b, c, d, SHA1_F1, SHA1_K1, M(16));
	SHA1_R(d, e, a, b, c, SHA1_F1, SHA1_K1, M(17));
	SHA1_R(c, d, e, a, b, SHA1_F1, SHA1_K1, M(18));
	SHA1_R(b, c, d, e, a, SHA1_F1, SHA1_K1, M(19));
	SHA1_R(a, b, c, d, e, SHA1_F2, SHA1_K2, M(20));
	SHA1_R(e, a, b, c, d, SHA1_F2, SHA1_K2, M(21));
	SHA1_R(d, e, a, b, c, SHA1_F2, SHA1_K2, M(22));
	SHA1_R(c, d, e, a, b, SHA1_F2, SHA1_K2, M(23));
	SHA1_R(b, c, d, e, a, SHA1_F2, SHA1_K2, M(24));
	SHA1_R(a, b, c, d, e, SHA1_F2, SHA1_K2, M(25));
	SHA1_R(e, a, b, c, d, SHA1_F2, SHA1_K2, M(26));
	SHA1_R(d, e, a, b, c, SHA1_F2, SHA1_K2, M(27));
	SHA1_R(c, d, e, a, b, SHA1_F2, SHA1_K2, M(28));
	SHA1_R(b, c, d, e, a, SHA1_F2, SHA1_K2, M(29));
	SHA1_R(a, b, c, d, e, SHA1_F2, SHA1_K2, M(30));
	SHA1_R(e, a, b, c, d, SHA1_F2, SHA1_K2, M(31));
	SHA1_R(d, e, a, b, c, SHA1_F2, SHA1_K2, M(32));
	SHA1_R(c, d, e, a, b, SHA1_F2, SHA1_K2, M(33));
	SHA1_R(b, c, d, e, a, SHA1_F2, SHA1_K2, M(34));
	SHA1_R(a, b, c, d, e, SHA1_F2, SHA1_K2, M(35));
	SHA1_R(e, a, b, c, d, SHA1_F2, SHA1_K2, M(36));
	SHA1_R(d, e, a, b, c, SHA1_F2, SHA1_K2, M(37));
	SHA1_R(c, d, e, a, b, SHA1_F2, SHA1_K2, M(38));
	SHA1_R(b, c, d, e, a, SHA1_F2, SHA1_K2, M(39));
	SHA1_R(a, b, c, d, e, SHA1_F3, SHA1_K3, M(40));
	SHA1_R(e, a, b, c, d, SHA1_F3, SHA1_K3, M(41));
	SHA1_R(d, e, a, b, c, SHA1_F3, SHA1_K3, M(42));
	SHA1_R(c, d, e, a, b, SHA1_F3, SHA1_K3, M(43));
	SHA1_R(b, c, d, e, a, SHA1_F3, SHA1_K3, M(44));
	SHA1_R(a, b, c, d, e, SHA1_F3, SHA1_K3, M(45));
	SHA1_R(e, a, b, c, d, SHA1_F3, SHA1_K3, M(46));
	SHA1_R(d, e, a, b, c, SHA1_F3, SHA1_K3, M(47));
	SHA1_R(c, d, e, a, b, SHA1_F3, SHA1_K3, M(48));
	SHA1_R(b, c, d, e, a, SHA1_F3, SHA1_K3, M(49));
	SHA1_R(a, b, c, d, e, SHA1_F3, SHA1_K3, M(50));
	SHA1_R(e, a, b, c, d, SHA1_F3, SHA1_K3, M(51));
	SHA1_R(d, e, a, b, c, SHA1_F3, SHA1_K3, M(52));
	SHA1_R(c, d, e, a, b, SHA1_F3, SHA1_K3, M(53));
	SHA1_R(b, c, d, e, a, SHA1_F3, SHA1_K3, M(54));
	SHA1_R(a, b, c, d, e, SHA1_F3, SHA1_K3, M(55));
	SHA1_R(e, a, b, c, d, SHA1_F3, SHA1_K3, M(56));
	SHA1_R(d, e, a, b, c, SHA1_F3, SHA1_K3, M(57));
	SHA1_R(c, d, e, a, b, SHA1_F3, SHA1_K3, M(58));
	SHA1_R(b, c, d, e, a, SHA1_F3, SHA1_K3, M(59));
	SHA1_R(a, b, c, d, e, SHA1_F4, SHA1_K4, M(60));
	SHA1_R(e, a, b, c, d, SHA1_F4, SHA1_K4, M(61));
	SHA1_R(d, e, a, b, c, SHA1_F4, SHA1_K4, M(62));
	SHA1_R(c, d, e, a, b, SHA1_F4, SHA1_K4, M(63));
	SHA1_R(b, c, d, e, a, SHA1_F4, SHA1_K4, M(64));
	SHA1_R(a, b, c, d, e, SHA1_F4, SHA1_K4, M(65));
	SHA1_R(e, a, b, c, d, SHA1_F4, SHA1_K4, M(66));
	SHA1_R(d, e, a, b, c, SHA1_F4, SHA1_K4, M(67));
	SHA1_R(c, d, e, a, b, SHA1_F4, SHA1_K4, M(68));
	SHA1_R(b, c, d, e, a, SHA1_F4, SHA1_K4, M(69));
	SHA1_R(a, b, c, d, e, SHA1_F4, SHA1_K4, M(70));
	SHA1_R(e, a, b, c, d, SHA1_F4, SHA1_K4, M(71));
	SHA1_R(d, e, a, b, c, SHA1_F4, SHA1_K4, M(72));
	SHA1_R(c, d, e, a, b, SHA1_F4, SHA1_K4, M(73));
	SHA1_R(b, c, d, e, a, SHA1_F4, SHA1_K4, M(74));
	SHA1_R(a, b, c, d, e, SHA1_F4, SHA1_K4, M(75));
	SHA1_R(e, a, b, c, d, SHA1_F4, SHA1_K4, M(76));
	SHA1_R(d, e, a, b, c, SHA1_F4, SHA1_K4, M(77));
	SHA1_R(c, d, e, a, b, SHA1_F4, SHA1_K4, M(78));
	SHA1_R(b, c, d, e, a, SHA1_F4, SHA1_K4, M(79));

	m_state[0] += a;
	m_state[1] += b;
	m_state[2] += c;
	m_state[3] += d;
	m_state[4] += e;
}
