/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef ERRORFACTORY_H_
#define ERRORFACTORY_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ErrorCodes.h"
#include <memory>
#include <string>
#include <map>

class ErrorDescriptor;
class ImmediateResponse;
class CraftedResponse;
class SplittableBuffer;
class BString;
class URI;

class ErrorFactory
{
public:
	static std::auto_ptr<ErrorDescriptor> errConnectionFailed(
		ErrorCodes::Code code, std::string const& debug_msg,
		URI const& url, std::string host, int port,
		char const* syserr);
	
	static std::auto_ptr<ErrorDescriptor> errConnectionTimeout(
		ErrorCodes::Code code, std::string const& debug_msg,
		URI const& url, std::string const& host, int port);
	
	static std::auto_ptr<ErrorDescriptor> errHostNameResolution(
		ErrorCodes::Code code, std::string const& debug_msg,
		URI const& url, std::string const& host);
	
	static std::auto_ptr<ErrorDescriptor> errServerTimeout(
		ErrorCodes::Code code, std::string const& debug_msg,
		URI const& url);
	
	static std::auto_ptr<ErrorDescriptor> errParsingRequest(
		ErrorCodes::Code code, std::string const& debug_msg,
		URI const& url, std::string const& problem);
	
	static std::auto_ptr<ErrorDescriptor> errParsingResponse(
		ErrorCodes::Code code, std::string const& debug_msg,
		URI const& url, std::string const& problem);
	
	static std::auto_ptr<ErrorDescriptor> errUnsupportedProtocol(
		ErrorCodes::Code code, std::string const& debug_msg,
		URI const& url, std::string const& protocol);
	
	static std::auto_ptr<ErrorDescriptor> errUnsupportedMethod(
		ErrorCodes::Code code, std::string const& debug_msg,
		URI const& url, std::string const& method);
	
	static std::auto_ptr<ErrorDescriptor> errExpectationFailed(
		ErrorCodes::Code code, std::string const& debug_msg,
		URI const& url, std::string const& expectation);
	
	static std::auto_ptr<ErrorDescriptor> errGenericError(
		ErrorCodes::Code code, std::string const& debug_msg,
		URI const& url, std::string const& error,
		int status_code);
	
	static std::auto_ptr<CraftedResponse> getGenericError(
		URI const& url, std::string const& error, int status_code);
	
	static std::auto_ptr<ErrorDescriptor> errUrlForbidden(
		ErrorCodes::Code code, std::string const& debug_msg,
		URI const& url);
	
	static std::auto_ptr<ErrorDescriptor> errRemoteProxyConfig(
		ErrorCodes::Code code, std::string const& debug_msg,
		URI const& url, std::string const& problem);
	
	static std::auto_ptr<ErrorDescriptor> errForwardingThroughProxy(
		ErrorCodes::Code code, std::string const& debug_msg,
		URI const& url, std::string const& problem,
		std::string const& proxy_type,
		std::string const& proxy_host, int proxy_port);
	
	/**
	 * \brief Creates an error that is not supposed to reach the client.
	 */
	static std::auto_ptr<ErrorDescriptor> errDummyError(
		ErrorCodes::Code code = ErrorCodes::UNKNOWN_ERROR,
		std::string const& debug_msg = std::string());
private:
	typedef std::map<BString, std::string> MacroMap;
	
	ErrorFactory(); // forbid construction
	
	static void prepareCommonMacros(MacroMap& m);
	
	static void prepareContent(CraftedResponse& response,
		char const* const templ[], MacroMap const& macros);
	
	static std::auto_ptr<ImmediateResponse> prepareResponse(
		int status_code, char const* const templ[],
		MacroMap const& macros);
	
	static std::auto_ptr<CraftedResponse> prepareCraftedResponse(
		int status_code, char const* const templ[],
		MacroMap const& macros);
	
	static std::string getFullHostName();
};

#endif
