/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ResponseFilterChain.h"
#include "AbstractResponseFilter.h"
#include "ResponseFilterBase.h"
#include "FilterTryList.h"
#include "ErrorDescriptor.h"
#include "HttpResponseMetadata.h"
#include "HttpRequestMetadata.h"
#include "JsFilterContext.h"
#include "RefCountable.h"
#include "RefCounter.h"
#include <cassert>

using namespace std;

class ResponseFilterChain::Terminator : public ResponseFilterBase
{
public:
	Terminator(ResponseFilterChain& chain);
	
	virtual ~Terminator();

	virtual void processMetadata(
		RequestStatus& status,
		std::auto_ptr<HttpResponseMetadata> metadata);
	
	virtual void processBodyData(
		RequestStatus& status,
		SplittableBuffer& data, bool eof);
};


class ResponseFilterChain::RcJsFilterContext :
	public JsFilterContext,
	public RefCountable<RefCounter<ACE_NULL_SYNCH> >
{
};


ResponseFilterChain::ResponseFilterChain(
	ServiceContext& context,
	ConstRequestPtr const& request,
	RequestTag const& request_tag,
	auto_ptr<FilterTryList> filter_try_list,
	IntrusivePtr<AbstractResponseHandler> const& final_recipient)
:	m_rContext(context),
	m_ptrRequest(request),
	m_requestTag(request_tag),
	m_ptrFilterTryList(filter_try_list),
	m_ptrFinalRecipient(final_recipient),
	m_ptrTerminator(new Terminator(*this))
{
	m_ptrFilterTryList->append(FilterTryList::FilterConstructorPtr(
		new FilterTryList::FilterConstructor(
			sigc::ptr_fun(&FilterTryList::ensureTransferEncodingUnderstood)
		)
	));
	m_ptrFilterTryList->append(FilterTryList::FilterConstructorPtr(
		new FilterTryList::FilterConstructor(
			sigc::ptr_fun(&FilterTryList::ensureContentEncodingUnderstood)
		)
	));
}

ResponseFilterChain::~ResponseFilterChain()
{
}

void
ResponseFilterChain::processProvisionalResponse(
	RequestStatus& status, auto_ptr<HttpResponseMetadata> metadata)
{
	m_ptrFinalRecipient->processProvisionalResponse(status, metadata);
}

void
ResponseFilterChain::processResponseMetadata(
	RequestStatus& status, auto_ptr<HttpResponseMetadata> metadata)
{
	if (!m_ptrFirstFilter) {
		m_ptrFirstFilter = nextFilter(*metadata);
	}
	m_ptrFirstFilter->processMetadata(status, metadata);
}

void
ResponseFilterChain::processBodyData(
	RequestStatus& status, SplittableBuffer& data, bool eof)
{
	assert(m_ptrFirstFilter.get());
	m_ptrFirstFilter->processBodyData(status, data, eof);
}

void
ResponseFilterChain::processError(
	RequestStatus& status, std::auto_ptr<ErrorDescriptor> edesc)
{
	m_ptrFinalRecipient->processError(status, edesc);
}

void
ResponseFilterChain::appendFilter(
	IntrusivePtr<AbstractResponseFilter> const& filter)
{
	m_filterQueue.push_back(filter);
}

IntrusivePtr<AbstractResponseFilter>
ResponseFilterChain::nextFilter(HttpResponseMetadata const& metadata)
{
	while (m_filterQueue.empty() && !m_ptrFilterTryList->empty()) {
		FilterTryList::FilterConstructorPtr filter_constructor(
			m_ptrFilterTryList->front()
		);
		m_ptrFilterTryList->shift();
		// The filter constructor must be removed from the FilterTryList
		// before being called, because it may alter this list.
		(*filter_constructor)(*this, metadata);
	}
	if (!m_filterQueue.empty()) {
		IntrusivePtr<AbstractResponseFilter> filter = m_filterQueue.front();
		m_filterQueue.pop_front();
		return filter;
	}
	return m_ptrTerminator;
}

bool
ResponseFilterChain::isFlagSet(
	FilterGroupTag const& group, std::string const& flag) const
{
	if (m_globalFlags.isSet(flag)) {
		return true;
	} else if (m_groupLocalFlags.isSet(group, flag)) {
		return true;
	} else {
		return false;
	}
}


JsFilterContext&
ResponseFilterChain::getJsContextFor(FilterGroupTag const& group)
{
	IntrusivePtr<RcJsFilterContext>& context = m_jsContexts[group];
	if (!context) {
		context.reset(new RcJsFilterContext);
	}
	return *context;
}


/*======================= ResponseFilterChain::Terminator ======================*/

ResponseFilterChain::Terminator::Terminator(ResponseFilterChain& chain)
:	ResponseFilterBase(chain)
{
}

ResponseFilterChain::Terminator::~Terminator()
{
}

void
ResponseFilterChain::Terminator::processMetadata(
	RequestStatus& status, auto_ptr<HttpResponseMetadata> metadata)
{
	getFinalRecipient().processResponseMetadata(status, metadata);
}

void
ResponseFilterChain::Terminator::processBodyData(
	RequestStatus& status, SplittableBuffer& data, bool eof)
{
	getFinalRecipient().processBodyData(status, data, eof);
}
