/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef HTTPCACHE_ABSTRACTITEM_H_
#define HTTPCACHE_ABSTRACTITEM_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "RequestPtr.h"
#include "types.h"

namespace HttpCache
{

class RequestResolution;

/**
 * \brief Represents the exclusive owner of a cached resource.
 */
class AbstractItem
{
public:
	enum GCFlags {
		GC_DEFAULT,
		GC_ONLY_UNCACHED /**< Only garbage collect items that
		don't have a file on disk. */
	};
	
	virtual ~AbstractItem() {}
	
	/**
	 * \brief Add a reference.
	 */
	virtual void ref() const = 0;
	
	/**
	 * \brief Remove a reference.
	 * \note If the number of references reaches zero, the object is destroyed.
	 */
	virtual void unref() const = 0;
	
	/**
	 * \brief Handle request metadata.
	 *
	 * \param request Non-null smart pointer to request metadata.
	 *        Request metadata is considered immutable and must not
	 *        be modified through non-const pointers or references.
	 * \note This method is thread-safe.
	 */
	virtual RequestResolution handleRequest(ConstRequestPtr const& request) = 0;
	
	/**
	 * \brief If possible, prepare the item for garbage collection.
	 *
	 * This method assumes that the item's owner holds exactly one
	 * reference to it, and that the owner provides mutual exclusion
	 * for handleRequest() and prepareForGC().
	 *
	 * \return true on success, false on failure. The failure may be caused
	 *         by the item being in use or because of limitations imposed
	 *         by /p flags.
	 */
	virtual bool prepareForGC(GCFlags flags = GC_DEFAULT) = 0;
};

} // namespace HttpCache

#endif
