/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "FileStorage.h"
#include "AbstractFileIO.h"
#include "FileIO.h"
#include "HexDecoder.h"
#include "StringUtils.h"
#include "FileHandle.h"
#include <ace/config-lite.h>
#include <ace/Dirent.h>
#include <ace/OS_NS_sys_stat.h> // for ACE_OS::lstat()
#include <ace/OS_NS_unistd.h> // for ACE_OS::unlink()
#include <ace/OS_NS_stdio.h> // for ACE_OS::snprintf()
#include <ace/OS_NS_fcntl.h> // for ACE_OS::open()
#include <ace/os_include/os_limits.h> // for PATH_MAX
#include <string>
#include <stdexcept>
#include <string.h>
#include <stddef.h>
#include <assert.h>

namespace HttpCache
{

namespace
{

struct FilePath
{
	char path[PATH_MAX];
	
	FilePath(char const* dir, char const *fname);
	
	bool isOk() const { return path[0] != '\0'; }
	
	operator char const*() const { return path; }
};


FilePath::FilePath(char const* dir, char const *fname)
{
	int const len = ACE_OS::snprintf(path, sizeof(path), "%s/%s", dir, fname);
	if (len < 0 || len >= int(sizeof(path))) { // len does not include the trailing '\0'.
		path[0] = '\0';
	}
}

} // anonymous namespace


FileStorage::FileStorage(std::string const& dir, uint64_t const space_limit,
	IntrusivePtr<AbstractCommand> const& gc_invoker)
:	m_ptrSpaceManager(SpaceManager::create(space_limit, gc_invoker)),
	m_dirPath(dir)
{
	ACE_Dirent dh;
	if (dh.open(dir.c_str()) == -1) {
		throw std::runtime_error("could not open directory");
	}
	
	if (ACE_OS::access(dir.c_str(), R_OK|W_OK|X_OK) == -1) {
		throw std::runtime_error("directory has wrong permissions");
	}
		
	for (dirent* ent; (ent = dh.read()); ) {
		char const* const fname = ent->d_name;
		FilePath path(dir.c_str(), fname);
		if (!path.isOk()) {
			continue;
		}
		
		char const* const pdot = strchr(fname, '.');
		if (!pdot) {
			ACE_OS::unlink(path);
			continue;
		} else if (pdot == fname) {
			if (fname[1] == '\0' || (fname[1] == '.' && fname[2] == '\0')) {
				// "." or ".."
				continue;
			}
		}
		
		HexDecoder decoder;
		if (!decoder.feed(fname, pdot - fname) || !decoder.finalize()) {
			ACE_OS::unlink(path);
			continue;
		}
		
		if (decoder.buffer().size() != FileId::Key::HASH_SIZE) {
			ACE_OS::unlink(path);
			continue;
		}
		
		FileId::Key key;
		key.set(&decoder.buffer()[0]);
		
		char const* const begin = pdot + 1;
		char const* const end = fname + strlen(fname);
		char const* e = end;
		uint32_t const version = StringUtils::parseUnsigned<uint32_t>(begin, e);
		if (e != end) {
			ACE_OS::unlink(path);
			continue;
		}
		
		ACE_stat st;
		if (ACE_OS::lstat(path, &st) == -1) {
			ACE_OS::unlink(path);
			continue;
		}
		
		FileId const file_id(key, version);
		if (!m_ptrSpaceManager->newObject(file_id, st.st_size)) {
			ACE_OS::unlink(path);
			continue;
		}
	}
}

FileStorage::~FileStorage()
{
}

IntrusivePtr<AbstractFileStorage>
FileStorage::create(
	std::string const& dir, uint64_t const space_limit,
	IntrusivePtr<AbstractCommand> const& gc_invoker)
{
	return IntrusivePtr<AbstractFileStorage>(
		new FileStorage(dir, space_limit, gc_invoker)
	);
}

void
FileStorage::ref() const
{
	++m_numRefs;
}

void
FileStorage::unref() const
{
	if (--m_numRefs == 0) {
		delete this;
	}
}

IntrusivePtr<FileStorage::SpaceManager> const&
FileStorage::spaceManager()
{
	return m_ptrSpaceManager;
}

std::auto_ptr<AbstractFileIO>
FileStorage::openExisting(FileId const& id, Mode const mode)
{
	FilePath path(m_dirPath.c_str(), id.toFileName().c_str());
	if (!path.isOk()) {
		return std::auto_ptr<AbstractFileIO>();
	}
	
	IntrusivePtr<LimitedSpaceObject> object(m_ptrSpaceManager->openObject(id));
	if (!object.get()) {
		return std::auto_ptr<AbstractFileIO>();
	}
	
	FileHandle handle(ACE_OS::open(path, translateMode(mode), ACE_DEFAULT_FILE_PERMS));
	if (handle.get() == ACE_INVALID_HANDLE) {
		object.reset(0); // Better done before eraseObject().
		m_ptrSpaceManager->eraseObject(id);
		return std::auto_ptr<AbstractFileIO>();
	}
	
	return std::auto_ptr<AbstractFileIO>(new FileIO(handle, object));
}

std::auto_ptr<AbstractFileIO>
FileStorage::createNew(FileId const& id, Mode const mode)
{
	FilePath path(m_dirPath.c_str(), id.toFileName().c_str());
	if (!path.isOk()) {
		return std::auto_ptr<AbstractFileIO>();
	}
	
	int const m = translateMode(mode)|O_CREAT|O_TRUNC;
	FileHandle handle(ACE_OS::open(path, m, ACE_DEFAULT_FILE_PERMS));
	if (handle.get() == ACE_INVALID_HANDLE) {
		return std::auto_ptr<AbstractFileIO>();
	}
	
	IntrusivePtr<LimitedSpaceObject> object(
		m_ptrSpaceManager->newObject(id, 0, SpaceManager::ERASE_EXISTING)
	);
	if (!object.get()) {
		return std::auto_ptr<AbstractFileIO>();
	}
	
	return std::auto_ptr<AbstractFileIO>(new FileIO(handle, object));
}

bool
FileStorage::unlink(FileId const& id)
{
	FilePath path(m_dirPath.c_str(), id.toFileName().c_str());
	if (!path.isOk()) {
		errno = ENAMETOOLONG;
		return false;
	}
	
	if (ACE_OS::unlink(path) == -1 && errno != ENOENT && errno != ENOTDIR) {
		return false;
	}
	
	m_ptrSpaceManager->eraseObject(id);
	
	return true;
}

int
FileStorage::translateMode(Mode const mode)
{
	switch (mode) {
		case RDONLY:
		return O_RDONLY;
		case WRONLY:
		return O_WRONLY;
		case RW:
		return O_RDWR;
	}
	
	assert(0);
	return O_RDWR;
}

} // namespace HttpCache
