/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ContentFilters.h"
#include "ContentFilterGroup.h"
#include "FilterGroupTag.h"
#include "OperationLog.h"
#include "Log.h"
#include "EncOps.h"
#include "AvailableContentFilters.h"
#include "AvailableFiltersOrdered.h"
#include "GlobalState.h"
#include <ace/config-lite.h>
#include <ace/Dirent.h>
#include <boost/multi_index_container.hpp>
#include <boost/multi_index/ordered_index.hpp>
#include <boost/multi_index/sequenced_index.hpp>
#include <boost/multi_index/mem_fun.hpp>
#include <vector>
#include <utility>
#include <algorithm>

namespace GtkGUI
{

using namespace boost::multi_index;

struct ContentFilters::QualifiedFileName
{
	std::string name;
	ContentFilterGroup::FileLocation location;
	
	QualifiedFileName(std::string const& name,
		ContentFilterGroup::FileLocation location)
	: name(name), location(location) {}
	
	bool operator<(QualifiedFileName const& rhs) const {
		return name < rhs.name;
	}
};

class ContentFilters::Impl
{
	DECLARE_NON_COPYABLE(Impl)
public:
	Impl(std::string const& user_dir, std::string const& global_dir);
	
	~Impl();
	
	void load();
	
	void apply() const;
	
	ContentFilterGroup const* findGroup(FilterGroupTag const& tag) const;
	
	bool addGroup(ContentFilterGroup const& group);
	
	bool eraseGroup(FilterGroupTag const& tag);
	
	bool updateGroup(ContentFilterGroup const& group);
	
	void foreachImpl(OutputFunction<ContentFilterGroup>& func) const;
	
	std::vector<QualifiedFileName> readFilterFileNames(
		std::string const& dir, ContentFilterGroup::FileLocation loc);
private:
	class SequenceTag;
	class EquivTag;
	class OrderTag;
	
	
	typedef multi_index_container<
		ContentFilterGroup,
		indexed_by<
			sequenced<
				tag<SequenceTag>
			>,
			ordered_unique<
				tag<EquivTag>,
				const_mem_fun<
					ContentFilterGroup, FilterGroupTag const&,
					&ContentFilterGroup::getTag
				>,
				FilterGroupTag::EquivComp
			>,
			ordered_non_unique<
				tag<OrderTag>,
				const_mem_fun<
					ContentFilterGroup, FilterGroupTag const&,
					&ContentFilterGroup::getTag
				>,
				FilterGroupTag::OrderComp
			>
		>
	> Container;
	typedef Container::index<SequenceTag>::type SequenceIdx;
	typedef Container::index<EquivTag>::type EquivIdx;
	typedef Container::index<OrderTag>::type OrderIdx;
	
	std::string m_userDir;
	std::string m_globalDir;
	Container m_groups;
};


ContentFilters::ContentFilters(
	std::string const& user_dir, std::string const& global_dir)
:	m_ptrImpl(new Impl(user_dir, global_dir))
{
}

ContentFilters::Impl::Impl(
	std::string const& user_dir, std::string const& global_dir)
:	m_userDir(user_dir),
	m_globalDir(global_dir)
{
}

ContentFilters::~ContentFilters()
{
}

ContentFilters::Impl::~Impl()
{
}

void
ContentFilters::swap(ContentFilters& other)
{
	Impl* impl1 = m_ptrImpl.release();
	Impl* impl2 = other.m_ptrImpl.release();
	m_ptrImpl.reset(impl2);
	other.m_ptrImpl.reset(impl1);
}

void
ContentFilters::load()
{
	m_ptrImpl->load();
}

void
ContentFilters::Impl::load()
{
	Log* log = OperationLog::instance();
	log->appendRecord("Loading content filters ... ");
	size_t const num_records = log->getNumRecords();
	
	std::vector<QualifiedFileName> global_fnames(
		readFilterFileNames(m_globalDir, ContentFilterGroup::GLOBAL_DIR)
	);
	std::vector<QualifiedFileName> user_fnames(
		readFilterFileNames(m_userDir, ContentFilterGroup::USER_DIR)
	);
	std::vector<QualifiedFileName> merged_fnames;
	
	std::sort(global_fnames.begin(), global_fnames.end());
	std::sort(user_fnames.begin(), user_fnames.end());
	std::set_union(
		user_fnames.begin(), user_fnames.end(),
		global_fnames.begin(), global_fnames.end(),
		std::back_inserter(merged_fnames)
	);
	
	Container new_groups;
	
	std::vector<QualifiedFileName>::const_iterator it(merged_fnames.begin());
	std::vector<QualifiedFileName>::const_iterator const end(merged_fnames.end());
	for (; it != end; ++it) {
		ContentFilterGroup group(it->location, it->name);
		if (group.load()) {
			new_groups.push_back(group);
		}
	}
	
	m_groups.swap(new_groups);
	
	if (num_records == log->getNumRecords()) {
		log->appendToLastRecord("done", log->getSuccessStyle());
	}
}

void
ContentFilters::apply() const
{
	m_ptrImpl->apply();
}

void
ContentFilters::Impl::apply() const
{
	AvailableContentFilters filters;
	
	SequenceIdx const& seq_idx = m_groups.get<SequenceTag>();
	SequenceIdx::const_iterator it(seq_idx.begin());
	SequenceIdx::const_iterator const end(seq_idx.end());
	for (; it != end; ++it) {
		filters.add(it->getFilters());
	}
	
	AvailableFiltersOrdered ordered_filters(filters);
	GlobalState::WriteAccessor()->swapContentFilters(ordered_filters);
}

ContentFilterGroup const*
ContentFilters::findGroup(FilterGroupTag const& tag) const
{
	return m_ptrImpl->findGroup(tag);
}

ContentFilterGroup const*
ContentFilters::Impl::findGroup(FilterGroupTag const& tag) const
{
	EquivIdx const& equiv_idx = m_groups.get<EquivTag>();
	EquivIdx::const_iterator it(equiv_idx.find(tag));
	if (it == equiv_idx.end()) {
		return 0;
	} else {
		return &*it;
	}
}

bool
ContentFilters::addGroup(ContentFilterGroup const& group)
{
	return m_ptrImpl->addGroup(group);
}

bool
ContentFilters::Impl::addGroup(ContentFilterGroup const& group)
{
	EquivIdx& equiv_idx = m_groups.get<EquivTag>();
	std::pair<EquivIdx::iterator, bool> const res(
		equiv_idx.insert(group)
	);
	return res.second;
}

bool
ContentFilters::eraseGroup(FilterGroupTag const& tag)
{
	return m_ptrImpl->eraseGroup(tag);
}

bool
ContentFilters::Impl::eraseGroup(FilterGroupTag const& tag)
{
	EquivIdx& equiv_idx = m_groups.get<EquivTag>();
	return equiv_idx.erase(tag) != 0;
}

bool
ContentFilters::updateGroup(ContentFilterGroup const& group)
{
	return m_ptrImpl->updateGroup(group);
}

bool
ContentFilters::Impl::updateGroup(ContentFilterGroup const& group)
{
	EquivIdx& equiv_idx = m_groups.get<EquivTag>();
	EquivIdx::iterator it(equiv_idx.find(group.getTag()));
	if (it == equiv_idx.end()) {
		return false;
	} else {
		return equiv_idx.replace(it, group);
	}
}

void
ContentFilters::foreachImpl(OutputFunction<ContentFilterGroup>& func) const
{
	m_ptrImpl->foreachImpl(func);
}

void
ContentFilters::Impl::foreachImpl(
	OutputFunction<ContentFilterGroup>& func) const
{
	OrderIdx const& order_idx = m_groups.get<OrderTag>();
	OrderIdx::const_iterator it(order_idx.begin());
	OrderIdx::const_iterator const end(order_idx.end());
	for (; it != end; ++it) {
		func(*it);
	}
}

std::vector<ContentFilters::QualifiedFileName>
ContentFilters::Impl::readFilterFileNames(
	std::string const& dir, ContentFilterGroup::FileLocation loc)
{
	std::vector<QualifiedFileName> fnames;
	
	ACE_Dirent dir_h;
	if (dir_h.open(dir.c_str()) == -1) {
		Log* log = OperationLog::instance();
		log->appendRecord(
			"Could not open directory "+EncOps::filenameToUtf8(dir),
			log->getErrorStyle()
		);
		return fnames;
	}
	
	for (dirent* ent; (ent = dir_h.read()); ) {
		std::string const fname(ent->d_name);
		if (fname.find('.') != std::string::npos) {
			// skip files with extensions
			continue;
		}
		fnames.push_back(QualifiedFileName(fname, loc));
	}
	
	return fnames;
}

} // namespace GtkGUI
