/*
** Copyright (C) 2003-2006 Teus Benschop.
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#include "libraries.h"
#include "utilities.h"
#include "indexchapter.h"
#include <config.h>
#include <glib.h>
#include "gwrappers.h"
#include <sqlite3.h>
#include "constants.h"
#include "shell.h"


#define TABLE_LINES "lines"


IndexChapter::IndexChapter (ustring& filename, unsigned int book, unsigned int chapter, vector<ustring>& lines)
{
  // Database variables.
  sqlite3 *db;
  int rc;
  char *error = NULL;
  try
  {
    // Filenames for input, and raw and categorized output.
    // We use Scripture Checks to produce this data from the lines we have.
    ustring inputfile = temporary_file ("indexer_inputfile");
    unlink (inputfile.c_str());
    ustring raw_outputfile = temporary_file ("indexer_raw_outputfile");
    unlink (raw_outputfile.c_str());
    ustring categorized_outputfile = temporary_file ("indexer_categorized_outputfile");
    unlink (categorized_outputfile.c_str());
    // Save the lines and produce the output: raw and categorized data.
    write_lines (inputfile, lines);
    ustring scripturechecks = "sc-input-usfm ";
    ustring command (scripturechecks);
    command.append (shell_quote_space (inputfile) + ">");
    command.append (shell_quote_space (raw_outputfile));
    system (command.c_str());
    command = scripturechecks;
    command.append (shell_quote_space (inputfile) + "--categorize >");
    command.append (shell_quote_space (categorized_outputfile));
    system (command.c_str());
    // Raw parsing first.
    parseraw = true;
    parse (raw_outputfile);
    // Categorized parsing next;
    parseraw = false;
    parse (categorized_outputfile);
    // Connect to database, set busy timeout, and start transaction
    rc = sqlite3_open(filename.c_str (), &db);
    if (rc) {
      throw runtime_error (sqlite3_errmsg(db));
    }
    sqlite3_busy_timeout (db, 1000);
    rc = sqlite3_exec (db, "begin;", NULL, NULL, &error);
    if (rc != SQLITE_OK) {
      throw runtime_error (error);
    }
    // Delete possible previous data for this chapter;
    char * sql;
    sql = g_strdup_printf  ("delete from '%s' where book = %d and chapter = %d;", TABLE_LINES, book, chapter);
    rc = sqlite3_exec (db, sql, NULL, NULL, &error);
    g_free (sql);
    if (rc != SQLITE_OK) {
      throw runtime_error (error);
    }
    // Go through each verse.
    for (unsigned int i = 0; i < verses.size(); i++) {
      // Normalize the text for better result when comparing text.
      rawtext[i] = rawtext[i].normalize ();
      id_texts[i] = id_texts[i].normalize ();
      intro_texts[i] = intro_texts[i].normalize ();
      head_texts[i] = head_texts[i].normalize ();
      chap_texts[i] = chap_texts[i].normalize ();
      study_texts[i] = study_texts[i].normalize ();
      note_texts[i] = note_texts[i].normalize ();
      ref_texts[i] = ref_texts[i].normalize ();
      verse_texts[i] = verse_texts[i].normalize ();
      // Any apostrophe (') should be prefixed by another one.
      rawtext[i] = double_apostrophy (rawtext[i]);
      id_texts[i] = double_apostrophy (id_texts[i]);
      intro_texts[i] = double_apostrophy (intro_texts[i]);
      head_texts[i] = double_apostrophy (head_texts[i]);
      chap_texts[i] = double_apostrophy (chap_texts[i]);
      study_texts[i] = double_apostrophy (study_texts[i]);
      note_texts[i] = double_apostrophy (note_texts[i]);
      ref_texts[i] = double_apostrophy (ref_texts[i]);
      verse_texts[i] = double_apostrophy (verse_texts[i]);
      // Prepare for case insensitive search.
      ustring casefolded (rawtext[i]);
      casefolded = casefolded.casefold ();
      ustring idcf (id_texts[i]);
      idcf = idcf.casefold ();
      ustring introcf (intro_texts[i]);
      introcf = introcf.casefold ();
      ustring headcf (head_texts[i]);
      headcf = headcf.casefold ();
      ustring chapcf (chap_texts[i]);
      chapcf = chapcf.casefold ();
      ustring studycf (study_texts[i]);
      studycf = studycf.casefold ();
      ustring notecf (note_texts[i]);
      notecf = notecf.casefold ();
      ustring refcf (ref_texts[i]);
      refcf = refcf.casefold ();
      ustring versecf (verse_texts[i]);
      versecf = versecf.casefold ();
      // Store this verse in the database.
      sql = g_strdup_printf ("insert into '%s' values (%d, %d, '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s');", TABLE_LINES, book, chapter, verses[i].c_str(), rawtext[i].c_str(), casefolded.c_str(), id_texts[i].c_str(), idcf.c_str(), intro_texts[i].c_str(), introcf.c_str(), head_texts[i].c_str(), headcf.c_str(), chap_texts[i].c_str(), chapcf.c_str(), study_texts[i].c_str(), studycf.c_str(), note_texts[i].c_str(), notecf.c_str(), ref_texts[i].c_str(), refcf.c_str(), verse_texts[i].c_str(), versecf.c_str());
      rc = sqlite3_exec (db, sql, NULL, NULL, &error);
      g_free (sql);
      if (rc != SQLITE_OK) {
        throw runtime_error (error);
      }
    }
    // Commit transaction.
    rc = sqlite3_exec (db, "commit;", NULL, NULL, &error);
    if (rc != SQLITE_OK) {
      throw runtime_error (error);
    }
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
  // Close connection.  
  sqlite3_close (db);
}


IndexChapter::~IndexChapter ()
{
}


void IndexChapter::parse (const ustring& filename)
{
  // Some variables for parsing.
  bool was_error = false;
  gchar *contents;
  gsize  length;
  GError *gerror;
  gerror = NULL;
  // Read the xml file.
  if (!g_file_get_contents (filename.c_str(), &contents, &length, &gerror)) {
    gw_critical (gerror->message);
    g_error_free (gerror);
    was_error = true;
  }
  // If length is (about) zero, don't parse it, because it does not have data,
  // and parsing an empty file gives a segmentation error.
  if (length < 10)
    was_error = true;
  // Set up parser.
  GMarkupParseContext *context;
  GMarkupParser parser = {
    start_element_handler,
    end_element_handler,
    text_handler,
    passthrough_handler,
    error_handler
  };
  // Parse xml file.
  if (!was_error) {
    context = g_markup_parse_context_new (&parser, GMarkupParseFlags (0), gpointer (this), NULL);
    if (!g_markup_parse_context_parse (context, contents, length, NULL)) {
      g_markup_parse_context_free (context);
      was_error = true;
      gw_critical (gerror->message);
    }
  }
  if (!was_error) {
    if (!g_markup_parse_context_end_parse (context, NULL)) {
      g_markup_parse_context_free (context);
      was_error = true;
      gw_critical (gerror->message);
    }
  }
  if (!was_error)
    g_markup_parse_context_free (context);
}


void IndexChapter::start_element_handler (GMarkupParseContext *context,
                                   const gchar         *element_name,
                                   const gchar        **attribute_names,
                                   const gchar        **attribute_values,
                                   gpointer             user_data,
                                   GError             **error)
{
  ((IndexChapter *) user_data)->start_element_handler (element_name, attribute_values);
}


void IndexChapter::end_element_handler (GMarkupParseContext *context,
                                 const gchar         *element_name,
                                 gpointer             user_data,
                                 GError             **error)
{
  ((IndexChapter *) user_data)->end_element_handler (element_name);
}


void IndexChapter::text_handler (GMarkupParseContext *context,
                          const gchar         *text,
                          gsize                text_len,
                          gpointer             user_data,
                          GError             **error)
{
  ((IndexChapter *) user_data)->text_handler (text);
}


void IndexChapter::passthrough_handler (GMarkupParseContext *context,
                                    const gchar         *passthrough_text,
                                    gsize                text_len,
                                    gpointer             user_data,
                                    GError             **error)
{
}


void IndexChapter::error_handler (GMarkupParseContext *context,
                                  GError              *error,
                                  gpointer             user_data)
{
  gw_critical (error->message);
}


void IndexChapter::start_element_handler (const gchar  *element_name,
                                          const gchar **attribute_values)
{
  currentelement = element_name;
  if (currentelement == "verse") {
    verse = attribute_values[0];
  }
}


void IndexChapter::end_element_handler (const gchar *element_name)
{
  ustring element (element_name);
  if (element == "verse") {
    if (parseraw) {
      // Store data.
      verses.push_back (verse);
      rawtext.push_back (xmltext);      
      // Clear text.   
      xmltext.clear();
    } else {
      // Store data.
      id_texts.push_back (id_text);
      intro_texts.push_back (intro_text);
      head_texts.push_back (head_text);
      chap_texts.push_back (chap_text);
      study_texts.push_back (study_text);
      note_texts.push_back (note_text);
      ref_texts.push_back (ref_text);
      verse_texts.push_back (verse_text);
      // Clear text.    
      id_text.clear();
      intro_text.clear();
      head_text.clear();
      chap_text.clear();
      study_text.clear();
      note_text.clear();
      ref_text.clear();
      verse_text.clear();
    }
  }
}


void IndexChapter::text_handler (const gchar *text)
{
  if (parseraw) {
    xmltext.append (text);
  } else {
     if       (currentelement == IDENTIFIER_TEXT_TAG) {
       id_text.append (text);
    } else if (currentelement == INTRODUCTION_TEXT_TAG) {
       intro_text.append (text);
    } else if (currentelement == HEADING_TEXT_TAG) {
       head_text.append (text);
    } else if (currentelement == CHAPTER_TEXT_TAG) {
       chap_text.append (text);
    } else if (currentelement == STUDY_NOTE_TEXT_TAG) {
       study_text.append (text);
    } else if (currentelement == NOTE_TEXT_TAG) {
       note_text.append (text);
    } else if (currentelement == CROSSREFERENCE_TEXT_TAG) {
       ref_text.append (text);
    } else if (currentelement == VERSE_TEXT_TAG) {
       verse_text.append (text);
    }
  }
}
