/*
** Copyright (C) 2003-2006 Teus Benschop.
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#include <libgen.h>
#include <glib.h>
#include <config.h>
#include "uname.h"
#include "utilities.h"


UNameType uname_get ()
{
  string sysname;
  struct utsname info;
  if(uname(&info) == -1)
    perror("uname");
  else
    sysname = info.sysname;
  UNameType result;
  sysname = lowerCase (sysname);
  if (sysname.find ("linux") != string::npos)
    result = untLinux;
  else if (sysname.find ("darwin") != string::npos)
    result = untDarwin;
  else if (sysname.find ("cygwin") != string::npos)
    result = untCygwin;
  else
    result = untUnknown;
  return result;
}


ustring uname_apostrophy (const ustring& filename, bool windows)
// Normally, that is on Linux, a filename can be put in apostrophies, like
// '/usr/bin/bibledit'. On Windows this should be "\usr\bin\bibledit".
// This function puts the right apostrophies around the filename and returns
// the resulting filename.
{
  ustring localfilename (filename);
  if (windows) {
    localfilename.insert (0, "\"");
    localfilename.append ("\"");
  } else {
    localfilename.insert (0, "'");
    localfilename.append ("'");
  }
  return localfilename;
}


ustring filename_modify_if_cygwin (const ustring& filename)
// A file in the Cygwin system is called /usr/share/bibledit/index.html.
// To open it from a Windows browser, it must now be called:
// C:\Cygwin\usr\share\bibledit\index.html.
// Similarly, a file in Cygwin is called /cygdrive/c/license.xml.
// To open that from a Windows program, it must now be callled:
// C:\license.xml.
{
  // If we run in Linux or Darwin, return the file unmodified.
  if (uname_get() != untCygwin)
    return filename;
  
  // We run in Windows / Cygwin.
  ustring windowsfilename (filename);

  // Is it on the C: drive? Or D: if the user mounts it, and so on.
  #define WINDOWS_DRIVE "/cygdrive/"
  size_t position = windowsfilename.find (WINDOWS_DRIVE);
  if (position == 0) {
    windowsfilename.erase (0, strlen (WINDOWS_DRIVE));
    if (windowsfilename.length() > 0)
      windowsfilename.insert (1, ":");
  } else {
    // In Cygwin system: insert Cygwin prefix.
    windowsfilename.insert (0, "C:/Cygwin");
  }
  
  // Change forwardslashes to backslashes.
  replace_text (windowsfilename, "/", "\\");
  
  // Return result.
  return windowsfilename;  
}
