#!/usr/bin/env python
#file test_lyxserver.py
"""Test functions and classes in the LyX.lyxserver module

Revision History:

2005-02-23 G. Milde first version with the unittest framework
"""

from unittest import TestCase, main 
import os, sys, types

from LyX.lyxserver import *
logger.setLevel(logging.DEBUG) 


class lyxserverTests(TestCase):
    """Test public functions in lyxserver.py"""
    # Common testcases
    # a list of file name arguments
    filelist = ["file1", 'file2']
    #
    def test_start_lyx_cmd_with_options(self):
        """start_lyx("lyx -version") should display version and exit with 0"""
        pid = start_lyx("lyx -version")
        (pid, rv) = os.waitpid(pid, 0)
        self.assertEqual(0, rv)
    #
    def test_start_lyx_with_args(self):
        """start_lyx(args=["-help"]) should display help and exit with 0"""
        pid = start_lyx(args=["-help"])
        (pid, rv) = os.waitpid(pid, 0)
        self.assertEqual(0, rv)
    #
    def test_lyxpipe_autoopen(self):
        """lyxpipe with lyxcmd should open lyx and return file object"""
        pipe = lyxpipe(lyxcmd=LYXCMD)
        self.assertEqual(types.FileType, type(pipe))
    #
    def test_lyxpipe_write(self):
        """writing to the lyxpipe(s) should work"""
        inpipe = lyxpipe(lyxcmd=LYXCMD)
        inpipe.write('LYXCMD:test:message:test lyxpipe write\n')
    #
        
    
class LyXServerTests(TestCase):
    """Test the lyxserver.LyXServer class"""
    
    cmd_string = "LYXCMD:test:message:%s\n"
    
    def setUp(self):
        self.lyxserver = LyXServer()  # lyxserver object with defaults
    def tearDown(self):
        del(self.lyxserver)
    
    def test_file_likeness(self):
        """LyXServer should provide all methods of a FileType-like object
        
        see Python Library Reference section 2.3.8 File Objects
        """
        file_methods = ['close', '__iter__', 'read', 'readline',
                        'readlines', 'write', 'writelines']
        file_attributes = ['closed', 'encoding', 'mode', 'name', 'newlines',
                           'softspace']
        for method in file_methods + file_attributes:
            self.assert_(method in dir(self.lyxserver), method + " missing")
        
    def test__init__(self):
        """LyxServer.__init__ should set a unique name"""
        lyx2 = LyXServer()
        self.assertNotEqual(self.lyxserver.name, lyx2.name)
        
    def test__init__attributes(self):
        """LyXServer.__init__ should set path and lyxcmd"""
        lyx2 = LyXServer(lyxcmd="foo", path="/tmp/foo")
        self.assertEqual(lyx2.lyxcmd, 'foo')
        self.assertEqual(lyx2.path, '/tmp/foo')
        
    def test_open(self):
        """LyXServer.open should open in- and outpipe"""
        self.lyxserver.open()
        self.assertEqual(self.lyxserver.closed, False)
        self.assertEqual(types.FileType, type(self.lyxserver.inpipe))
        self.assertEqual(types.FileType, type(self.lyxserver.outpipe))
        
    def test_close(self):
        """LyXServer.close should prevent further reading and writing
        
        (If the instance is not explicitely closed, reading and writing 
         will auto-open the pipes (starting LyX if necessary))
        """
        self.lyxserver.close()
        self.assertEqual(self.lyxserver.closed, True)
        self.assertRaises(IOError, self.lyxserver.write, "foo")
        self.assertRaises(IOError, self.lyxserver.read)
        
    def test_poll(self):
        """LyXServer.poll waits timeout seconds and signals waiting data"""
        # stuff something into LyX to make sure there is a response
        self.lyxserver.write("LYXCMD:test:message:hi\n")
        self.assertEqual(True, bool(self.lyxserver.poll()))
        # read the data, so none is left
        self.lyxserver.read()
        self.assertEqual(False, bool(self.lyxserver.poll(timeout=0)))
    
    def test_read_write(self):
        """LyXServer.read and LyXServer.write should read/write data"""
        # read to empty the pipe and check if a string is returned
        self.assertEqual(type(''), type(self.lyxserver.read(timeout=0)))
        # write a command to the inpipe
        self.lyxserver.write(self.cmd_string%"test read write")
        # read the reply
        self.assert_("message" in self.lyxserver.read())
        # now nothing should be left
        self.assertEqual('', self.lyxserver.read(timeout=0))
    
    def test_readline(self):
        """LyXServer.readline should return 1 line of data or empty string"""
        for i in xrange(3):
            self.lyxserver.write(self.cmd_string%"test readline")
        self.assertEqual(1, self.lyxserver.readline().count('\n'))
        self.lyxserver.read(timeout=0) # empty the outpipe
    
    def test_readlines(self):
        """LyXServer.readlines should return data as list of lines"""
        self.lyxserver.read(timeout=0) # empty the outpipe
        for i in xrange(3):
            self.lyxserver.write(self.cmd_string%"test readlines")
        time.sleep(0.5) # give LyX some time for responding
        lines = self.lyxserver.readlines()
        self.assertEqual(types.ListType, type(lines))
        # the pipe should now be empty
        lines = self.lyxserver.readlines()
        self.assertEqual([], lines)
        
    def test_writelines(self):
        """LyXServer.writelines should write a sequence as lines"""
        self.lyxserver.read(timeout=0) # empty the outpipe
        self.lyxserver.writelines([self.cmd_string%"test writelines"]*3)
        time.sleep(0.5) # give LyX some time for responding
        self.assertEqual(3, len(self.lyxserver.readlines()))
        
    def test__iter__(self):
        """LyXServer.__iter__ should return iterator"""
        self.assertEqual(types.GeneratorType, type(self.lyxserver.__iter__()))
        self.lyxserver.read(timeout=0) # empty the outpipe
        self.lyxserver.writelines([self.cmd_string%"test __iter__"]*3)
        for line in self.lyxserver:
            self.assert_("message" in line)
    

if __name__ == '__main__':    #run tests if called from command-line
    main()

