/*
 * Bickley - a meta data management framework.
 * Copyright © 2008, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <dbus/dbus-protocol.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-bindings.h>

#include "bkl-db.h"
#include "bkl-internal-marshal.h"
#include "bkl-source-manager-client.h"

#include "bkl-source-manager-bindings.h"

enum {
    PROP_0,
};

enum {
    SOURCE_ADDED,
    SOURCE_REMOVED,
    READY,
    LAST_SIGNAL
};

struct _BklSourceManagerClientPrivate {
    DBusGConnection *connection;
    DBusGProxy *proxy;
};

#define ORBITER_NAME "org.moblin.Bickley.Orbiter"

#define GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), BKL_TYPE_SOURCE_MANAGER_CLIENT, BklSourceManagerClientPrivate))
G_DEFINE_TYPE (BklSourceManagerClient, bkl_source_manager_client, G_TYPE_OBJECT);

static guint32 signals[LAST_SIGNAL] = {0, };

static void source_added (DBusGProxy             *proxy,
                          const char             *path,
                          BklSourceManagerClient *client);
static void source_removed (DBusGProxy             *proxy,
                            const char             *path,
                            BklSourceManagerClient *client);

static void
bkl_source_manager_client_finalize (GObject *object)
{
    G_OBJECT_CLASS (bkl_source_manager_client_parent_class)->finalize (object);
}

static void
bkl_source_manager_client_dispose (GObject *object)
{
    BklSourceManagerClient *self = (BklSourceManagerClient *) object;
    BklSourceManagerClientPrivate *priv = self->priv;

    if (priv->proxy) {
        dbus_g_proxy_disconnect_signal (priv->proxy, "SourceAdded",
                                        G_CALLBACK (source_added), self);
        dbus_g_proxy_disconnect_signal (priv->proxy, "SourceRemoved",
                                        G_CALLBACK (source_removed), self);
        g_object_unref (priv->proxy);
        priv->proxy = NULL;
    }

    G_OBJECT_CLASS (bkl_source_manager_client_parent_class)->dispose (object);
}

static void
bkl_source_manager_client_class_init (BklSourceManagerClientClass *klass)
{
    GObjectClass *o_class = (GObjectClass *)klass;

    o_class->dispose = bkl_source_manager_client_dispose;
    o_class->finalize = bkl_source_manager_client_finalize;

    g_type_class_add_private (klass, sizeof (BklSourceManagerClientPrivate));

    signals[SOURCE_ADDED] = g_signal_new ("source-added",
                                          G_TYPE_FROM_CLASS (klass),
                                          G_SIGNAL_RUN_FIRST |
                                          G_SIGNAL_NO_RECURSE,
                                          0, NULL, NULL,
                                          g_cclosure_marshal_VOID__STRING,
                                          G_TYPE_NONE, 1, G_TYPE_STRING);
    signals[SOURCE_REMOVED] = g_signal_new ("source-removed",
                                            G_TYPE_FROM_CLASS (klass),
                                            G_SIGNAL_RUN_FIRST |
                                            G_SIGNAL_NO_RECURSE,
                                            0, NULL, NULL,
                                            g_cclosure_marshal_VOID__STRING,
                                            G_TYPE_NONE, 1, G_TYPE_STRING);
    signals[READY] = g_signal_new ("ready",
                                   G_TYPE_FROM_CLASS (klass),
                                   G_SIGNAL_RUN_FIRST |
                                   G_SIGNAL_NO_RECURSE,
                                   0, NULL, NULL,
                                   g_cclosure_marshal_VOID__VOID,
                                   G_TYPE_NONE, 0);
}

static void
source_added (DBusGProxy             *proxy,
              const char             *path,
              BklSourceManagerClient *client)
{
    g_signal_emit (client, signals[SOURCE_ADDED], 0, path);
}

static void
source_removed (DBusGProxy             *proxy,
                const char             *path,
                BklSourceManagerClient *client)
{
    g_signal_emit (client, signals[SOURCE_REMOVED], 0, path);
}

static void
orbiter_started_reply (DBusGProxy *proxy,
                       guint       OUT_arg2,
                       GError     *error,
                       gpointer    userdata)
{
    if (error != NULL) {
        g_warning ("Error starting orbiter: %s", error->message);
    }

    g_signal_emit (userdata, signals[READY], 0);
}

static void
bkl_source_manager_client_init (BklSourceManagerClient *self)
{
    BklSourceManagerClientPrivate *priv;
    DBusGProxy *proxy;
    GError *error = NULL;

    priv = self->priv = GET_PRIVATE (self);

    priv->connection = dbus_g_bus_get (DBUS_BUS_SESSION, &error);
    if (priv->connection == NULL) {
        g_warning ("Unable to get connection to D-Bus: %s",
                   error->message);
        g_error_free (error);
        return;
    }

    proxy = dbus_g_proxy_new_for_name (priv->connection,
                                       DBUS_SERVICE_DBUS,
                                       DBUS_PATH_DBUS,
                                       DBUS_INTERFACE_DBUS);
    org_freedesktop_DBus_start_service_by_name_async (proxy,
                                                      ORBITER_NAME, 0,
                                                      orbiter_started_reply,
                                                      self);

    priv->proxy = dbus_g_proxy_new_for_name (priv->connection,
                                             BKL_SOURCE_MANAGER_DBUS_SERVICE,
                                             BKL_SOURCE_MANAGER_DBUS_PATH,
                                             BKL_SOURCE_MANAGER_DBUS_INTERFACE);
    dbus_g_proxy_add_signal (priv->proxy, "SourceAdded",
                             G_TYPE_STRING, G_TYPE_INVALID);
    dbus_g_proxy_add_signal (priv->proxy, "SourceRemoved",
                             G_TYPE_STRING, G_TYPE_INVALID);
    dbus_g_proxy_connect_signal (priv->proxy, "SourceAdded",
                                 G_CALLBACK (source_added), self, NULL);
    dbus_g_proxy_connect_signal (priv->proxy, "SourceRemoved",
                                 G_CALLBACK (source_removed), self, NULL);
}

struct _ListSourcesData {
    BklSourceManagerClient *client;
    BklSourceManagerClientGetSourcesCallback cb;
    gpointer userdata;
};

static void
list_sources_reply (DBusGProxy *proxy,
                    char      **OUT_sources,
                    GError     *error,
                    gpointer    userdata)
{
    struct _ListSourcesData *data = (struct _ListSourcesData *) userdata;
    GList *slist = NULL;
    int i;

    if (error != NULL) {
        data->cb (data->client, NULL, error, data->userdata);

        g_free (data);
        return;
    }

    for (i = 0; OUT_sources[i]; i++) {
        slist = g_list_prepend (slist, OUT_sources[i]);
    }

    slist = g_list_reverse (slist);
    data->cb (data->client, slist, error, data->userdata);

    g_strfreev (OUT_sources);

    g_free (data);
}

void
bkl_source_manager_client_get_sources (BklSourceManagerClient *client,
                                       BklSourceManagerClientGetSourcesCallback cb,
                                       gpointer                userdata)
{
    struct _ListSourcesData *data;
    BklSourceManagerClientPrivate *priv = client->priv;

    g_return_if_fail (priv->proxy != NULL);

    data = g_new (struct _ListSourcesData, 1);
    data->client = client;
    data->cb = cb;
    data->userdata = userdata;

    org_moblin_Bickley_SourceManager_list_sources_async (priv->proxy,
                                                         list_sources_reply,
                                                         data);
}
