;*=====================================================================*/
;*    serrano/prgm/project/bigloo/api/pthread/recette/recette.scm      */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Mon Feb  4 14:28:58 2002                          */
;*    Last change :  Tue Jun  7 09:32:15 2005 (serrano)                */
;*    Copyright   :  2002-05 Manuel Serrano                            */
;*    -------------------------------------------------------------    */
;*    A test module that deploys the examples of SRFI18.               */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    The module                                                       */
;*---------------------------------------------------------------------*/
(module recette
   (library pthread)
   (main    main))

;*---------------------------------------------------------------------*/
;*    err ...                                                          */
;*---------------------------------------------------------------------*/
(define (err . msg)
   (with-output-to-port (current-error-port)
      (lambda ()
	 (for-each write msg)
	 (newline))))

;*---------------------------------------------------------------------*/
;*    do-something-else ...                                            */
;*---------------------------------------------------------------------*/
(define (do-something-else)
   #unspecified)

;*---------------------------------------------------------------------*/
;*    *tests* ...                                                      */
;*---------------------------------------------------------------------*/
(define *tests* '())

;*---------------------------------------------------------------------*/
;*    *failure* and *success* ...                                      */
;*---------------------------------------------------------------------*/
(define *failure* '())
(define *success* 0)

;*---------------------------------------------------------------------*/
;*    test ...                                                         */
;*---------------------------------------------------------------------*/
(define (test name prgm::procedure res)
   (display* name "...")
   (flush-output-port (current-output-port))
   (let ((provided (with-exception-handler
		      (lambda (e)
			 (error-notify e)
			 (vector res))
		      prgm)))
      (if (or (eq? res #unspecified)
	      (and (procedure? res) (res provided))
	      (equal? res provided))
	  (begin
	     (set! *success* (+fx 1 *success*))
	     (print "ok."))
	  (begin
	     (set! *failure* (cons name *failure*))
	     (print "error.")
	     (print "   ==> provided: [" provided
		    "]\n       expected: ["
		    (if (procedure? res) (res 'result) res)
		    "]")))))

;*---------------------------------------------------------------------*/
;*    define-test ...                                                  */
;*---------------------------------------------------------------------*/
(define-macro (define-test id prgm . rest)
   (let ((t (match-case rest
	       ((:result ?result)
		`(list ',id (lambda () ,prgm) ,result))
	       (()
		`(list ',id (lambda () ,prgm) #unspecified))
	       (else
		(error "define-test" "Illegal rest argument" rest)))))
      `(set! *tests* (cons ,t *tests*))))

;*---------------------------------------------------------------------*/
;*    cond-expand ...                                                  */
;*---------------------------------------------------------------------*/
(define-test cond-expand
   (cond-expand
      (pthread #t)
      (else #f))
   :result #t)

;*---------------------------------------------------------------------*/
;*    make-thread ...                                                  */
;*---------------------------------------------------------------------*/
(define-test make-thread
   (make-thread (lambda () (write 'hello)))
   :result (lambda (v)
	      (if (eq? v 'result)
		  "a thread"
		  (thread? v))))

;*---------------------------------------------------------------------*/
;*    thread-name ...                                                  */
;*---------------------------------------------------------------------*/
(define-test thread-name
   (thread-name (make-thread (lambda () #f) 'foo))
   :result 'foo)

;*---------------------------------------------------------------------*/
;*    thread-specific ...                                              */
;*---------------------------------------------------------------------*/
(define-test thread-specific
   (let ((t (make-thread (lambda () #f))))
      (thread-specific-set! t "hello")
      (thread-specific t))
   :result "hello")
   
;*---------------------------------------------------------------------*/
;*    thread-start! ...                                                */
;*---------------------------------------------------------------------*/
(define-test thread-start
   (with-output-to-string
      (lambda ()
	 (letrec ((t0 (make-thread (lambda ()
				      (write 'b)
				      (thread-join! t1))))
		  (t1 (make-thread (lambda ()
				      (write 'a)))))
	    (thread-start-joinable! t0)
	    (thread-start-joinable! t1)
	    (thread-join! t0))))
   :result (lambda (v)
	      (if (eq? v 'result)
		  "ab or ba"
		  (or (equal? v "ab") (equal? v "ba")))))

;*---------------------------------------------------------------------*/
;*    bind-exit ...                                                    */
;*---------------------------------------------------------------------*/
(define-test bind-exit
   (let ((res #f))
      (let ((val (thread-join!
		  (thread-start-joinable!
		   (make-thread (lambda ()
				   (set! res
					 (bind-exit (stop)
					    (+ 1 (stop #t))))))))))
	 res))
   :result #t)

;*---------------------------------------------------------------------*/
;*    mutex ...                                                        */
;*---------------------------------------------------------------------*/
(define-test mutex
   (let ((m (make-mutex)))
      (mutex-lock! m)
      (mutex-unlock! m)
      #t)
   :result #t)

;*---------------------------------------------------------------------*/
;*    thread-yield! ...                                                */
;*---------------------------------------------------------------------*/
(define-test thread-yield
   (with-output-to-string
      (lambda ()
	 (let ((m (make-mutex)))
	    (thread-join!
	     (thread-start-joinable!
	      (make-thread (lambda ()
			      (let loop ()
				 (if (mutex-lock! m)
				     (begin
					(display "locked")
					(mutex-unlock! m))
				     (begin
					(thread-yield!)
					(loop)))))))))))
   :result "locked")

;*---------------------------------------------------------------------*/
;*    thread-join1 ...                                                 */
;*---------------------------------------------------------------------*/
(define-test thread-join1
   (thread-join! (thread-start-joinable! (make-thread (lambda () 23))))
   :result 23)

;*---------------------------------------------------------------------*/
;*    thread-join2 ...                                                 */
;*---------------------------------------------------------------------*/
(define-test thread-join2
   (let* ((res 0)
	  (t (thread-start-joinable! (make-thread
				      (lambda () (expt 2 10))
				      'thread-join1.1)))
	  (t2 (thread-start-joinable! (make-thread
				       (lambda ()
					  (do-something-else)
					  (set! res (thread-join! t)))
				       'thread-join1.2))))
      (thread-join! t2)
      res)
   :result 1024)

;*---------------------------------------------------------------------*/
;*    thread-join3 ...                                                 */
;*---------------------------------------------------------------------*/
(define-test thread-join3
   (let ((t (thread-start-joinable!
	     (make-thread (lambda ()
			     (with-error-to-port (open-output-string)
				(lambda ()
				   (raise 123))))
			  'thread-join2.1)))
	 (res 0))
      (do-something-else)
      (thread-join!
       (thread-start-joinable!
	(make-thread
	 (lambda ()
	    (with-exception-handler
	       (lambda (exc)
		  (if (uncaught-exception? exc)
		      (* 10 (uncaught-exception-reason exc))
		      99999))
	       (lambda ()
		  (set! res (+ 1 (thread-join! t))))))
	 'thread-join2.2)))
      res)
   :result 1231)

;*---------------------------------------------------------------------*/
;*    thread-join4 ...                                                 */
;*---------------------------------------------------------------------*/
(define-test thread-join4
   (let ((res '()))
      (define (wait-for-termination! thread)
	 (let ((eh (current-exception-handler)))
	    (with-exception-handler
	       (lambda (exc)
		  (if (not (or (terminated-thread-exception? exc)
			       (uncaught-exception? exc)))
		      (eh exc)))
	       (lambda ()
		  (thread-join! thread)
		  #f))))
      (let* ((t1 (thread-start-joinable!
		  (make-thread (lambda ()
				  (sleep 5))
			       'thread-join4.1)))
	     (t2 (thread-start-joinable!
		  (make-thread (lambda ()
				  (sleep 10))
			       'thread-join4.2)))
	     (t3 (thread-start-joinable!
		  (make-thread (lambda ()
				  (sleep 3)
				  (thread-terminate! t2))
			       'thread-join4.3)))
	     (t4 (thread-start-joinable!
		  (make-thread (lambda ()
				  (sleep 3)
				  (with-error-to-port (open-output-string)
				     (lambda ()
					(raise #t))))
			       'thread-join4.4)))
	     (t5 (thread-start-joinable!
		  (make-thread
		   (lambda ()
		      (set! res (cons (wait-for-termination! t1)
				      res))
		      (set! res (cons (wait-for-termination! t3)
				      res))
		      (set! res (cons (wait-for-termination! t4)
				      res)))
		   'thread-join4.5))))
	 (thread-join! t5)
	 res))
   :result '(#f #f #f))

;*---------------------------------------------------------------------*/
;*    mutex ...                                                        */
;*---------------------------------------------------------------------*/
(define-test mutex
   (list (mutex? (make-mutex))
	 (mutex? 'foo)
	 (mutex-name (make-mutex 'foo)))
   :result '(#t #f foo))

;*---------------------------------------------------------------------*/
;*    mutex-specific ...                                               */
;*---------------------------------------------------------------------*/
(define-test mutex-specific
   (let ((m (make-mutex)))
      (mutex-specific-set! m "hello")
      (mutex-specific m))
   :result "hello")

;*---------------------------------------------------------------------*/
;*    mutex-specific2 ...                                              */
;*---------------------------------------------------------------------*/
(define-test mutex-specific2
   (cond-expand
      (bigloo-jvm
       '(0 2))
      (else
       (let ((res '()))
	  (define (mutex-lock-recursively! mutex)
	     (if (eq? (mutex-state mutex) (current-thread))
		 (let ((n (mutex-specific mutex)))
		    (mutex-specific-set! mutex (+ n 1)))
		 (begin
		    (mutex-lock! mutex)
		    (mutex-specific-set! mutex 0))))
	  (define (mutex-unlock-recursively! mutex)
	     (let ((n (mutex-specific mutex)))
		(if (= n 0)
		    (mutex-unlock! mutex)
		    (mutex-specific-set! mutex (- n 1)))))
	  (let ((t (thread-start-joinable!
		    (make-thread
		     (lambda ()
			(let ((m (make-mutex)))
			   (mutex-lock-recursively! m)
			   (mutex-lock-recursively! m)
			   (mutex-lock-recursively! m)
			   (set! res (cons (mutex-specific m) res))
			   (mutex-unlock-recursively! m)
			   (mutex-unlock-recursively! m)
			   (mutex-unlock-recursively! m)
			   (set! res (cons (mutex-specific m) res))))))))
	     (thread-join! t)
	     res))))
   :result '(0 2))

;*---------------------------------------------------------------------*/
;*    mutex-state ...                                                  */
;*---------------------------------------------------------------------*/
(define-test mutex-state
   (mutex-state (make-mutex))
   :result 'not-abandoned)

;*---------------------------------------------------------------------*/
;*    mutex-lock ...                                                   */
;*---------------------------------------------------------------------*/
(define-test mutex-lock
   (begin
      (define m (make-mutex))
      (define (mutex-toggle m bool next)
	 (if bool
	     (begin
		(mutex-lock! m)
		#f)
	     (begin
		(mutex-unlock! m)
		#t)))
      (let ((th1 (thread-start-joinable!
		  (make-thread
		   (lambda ()
		      (mutex-toggle m #t mutex-toggle))))))
	 (thread-join! th1)
	 (mutex-state m)))
   :result 'abandoned)

;*---------------------------------------------------------------------*/
;*    mutex-lock2 ...                                                  */
;*---------------------------------------------------------------------*/
(define-test mutex-lock2
   (begin
      (define m (make-mutex))
      (let* ((res #unspecified)
	     (th1 (thread-start-joinable!
		   (make-thread
		    (lambda ()
		       (set! res (mutex-lock! m 100)))))))
	 (thread-join! th1)
	 res))
   :result #t)

;*---------------------------------------------------------------------*/
;*    mutex-lock3 ...                                                  */
;*---------------------------------------------------------------------*/
(define-test mutex-lock3
   (begin
      (define m (make-mutex))
      (mutex-lock! m)
      (let* ((res #unspecified)
	     (th1 (thread-start-joinable!
		   (make-thread
		    (lambda ()
		       (set! res (mutex-lock! m 100)))))))
	 (thread-join! th1)
	 res))
   :result #f)

;*---------------------------------------------------------------------*/
;*    mutex-lock4                                                      */
;*---------------------------------------------------------------------*/
(define-test mutex-lock4
   (with-output-to-string
      (lambda ()
	 (let ((m (make-mutex)))
	    (thread-join!
	     (thread-start-joinable!
	      (make-thread (lambda ()
			      (let loop ()
				 (if (mutex-lock! m 0)
				     (begin
					(display "locked")
					(mutex-unlock! m))
				     (begin
					(thread-yield!)
					(loop)))))))))))
   :result "locked")

;*---------------------------------------------------------------------*/
;*    condition-variable ...                                           */
;*---------------------------------------------------------------------*/
(define-test condition-variable
   (list (condition-variable? (make-condition-variable))
	 (condition-variable? 'foo)
	 (condition-variable-name (make-condition-variable 'foo))
	 (let ((cv (make-condition-variable)))
	    (condition-variable-specific-set! cv 'bar)
	    (condition-variable-specific cv)))
   :result `(#t #f foo bar))

;*---------------------------------------------------------------------*/
;*    condition-variable2                                              */
;*---------------------------------------------------------------------*/
(define-test condition-variable2
   (let ((res #f)
	 (lock (make-mutex))
	 (cv (make-condition-variable)))
      (let* ((th1 (thread-start-joinable!
		   (make-thread
		    (lambda ()
		       (mutex-lock! lock)
		       (condition-variable-signal! cv)
		       (mutex-unlock! lock)
		       (set! res 23))))))
	 (thread-join! th1))
      res)
   :result 23)

;*---------------------------------------------------------------------*/
;*    condition-variable3                                              */
;*---------------------------------------------------------------------*/
(define-test condition-variable3
   (let ((res #f)
	 (lock0 (make-mutex))
	 (lock (make-mutex))
	 (cv (make-condition-variable)))
      (mutex-lock! lock0)
      (let* ((th1 (thread-start-joinable!
		   (make-thread
		    (lambda ()
		       (mutex-lock! lock)
		       (mutex-unlock! lock0)
		       (condition-variable-wait! cv lock)
		       (mutex-unlock! lock)
		       (set! res 23)))))
	     (th2 (thread-start!
		   (make-thread
		    (lambda ()
		       (mutex-lock! lock0)
		       (mutex-lock! lock)
		       (condition-variable-signal! cv)
		       (mutex-unlock! lock)
		       (mutex-unlock! lock0))))))
	 (thread-join! th1))
      res)
   :result 23)

;*---------------------------------------------------------------------*/
;*    condition-variable4 ...                                          */
;*---------------------------------------------------------------------*/
(define-test condition-variable4
   (let ((res #f)
	 (lock (make-mutex))
	 (cv (make-condition-variable)))
      (let* ((th1 (thread-start-joinable!
		   (make-thread
		    (lambda ()
		       (mutex-lock! lock)
		       (condition-variable-signal! cv)
		       (mutex-unlock! lock)
		       (set! res 23))))))
	 (thread-join! th1))
      res)
   :result 23)

;*---------------------------------------------------------------------*/
;*    condition-variable5 ...                                          */
;*---------------------------------------------------------------------*/
(define-test condition-variable5
   (let ((res #f)
	 (lock (make-mutex))
	 (cv (make-condition-variable)))
      (let* ((th1 (thread-start-joinable!
		   (make-thread
		    (lambda ()
		       (mutex-lock! lock)
		       (set! res (condition-variable-wait! cv lock 100))
		       (mutex-unlock! lock))))))
	 (thread-join! th1))
      res)
   :result #f)

;*---------------------------------------------------------------------*/
;*    condition-variable6 ...                                          */
;*---------------------------------------------------------------------*/
(define-test condition-variable6
   (let ((res #f)
	 (lock (make-mutex))
	 (lock2 (make-mutex))
	 (cv (make-condition-variable)))
      (mutex-lock! lock2)
      (let* ((th1 (thread-start-joinable!
		   (make-thread
		    (lambda ()
		       (mutex-lock! lock)
		       (mutex-unlock! lock2)
		       (set! res (condition-variable-wait! cv lock 10000))
		       (mutex-unlock! lock))))))
	 (mutex-lock! lock2)
	 (mutex-lock! lock)
	 (condition-variable-signal! cv)
	 (mutex-unlock! lock)
	 (mutex-unlock! lock2)
	 (thread-join! th1))
      res)
   :result #t)

;*---------------------------------------------------------------------*/
;*    current-exception-handler ...                                    */
;*---------------------------------------------------------------------*/
(define-test current-exception-handler
   (current-exception-handler)
   :result procedure?)

;*---------------------------------------------------------------------*/
;*    with-exception-handler ...                                       */
;*---------------------------------------------------------------------*/
(define-test with-exception-handler
   (with-exception-handler list current-exception-handler)
   :result list)

;*---------------------------------------------------------------------*/
;*    raise ...                                                        */
;*---------------------------------------------------------------------*/
(define-test raise
   (let* ((res '())
	  (v2 (begin
		 (define (f n)
		    (if (< n 0) (raise "negative arg") (sqrt n)))
		 (define (g)
		    (call-with-current-continuation
		     (lambda (return)
			(with-exception-handler
			 (lambda (exc)
			    (return
			     (if (string? exc)
				 (string-append "error: " exc)
				 "unknown error")))
			 (lambda ()
			    (set! res (cons (f 4.) res))
			    (set! res (cons (f -1.) res))
			    (set! res (cons (write (f 9.)) res)))))))
		 (g))))
      (cons v2 res))
   :result '("error: negative arg" 2.))
	   
;*---------------------------------------------------------------------*/
;*    main ...                                                         */
;*---------------------------------------------------------------------*/
(define (main argv)
   (let ((tests '()))
      (args-parse (cdr argv)
	 ((("-h" "--help") (help "This help message"))
	  (args-parse-usage #f)
	  (exit 0))
	 (else
	  (set! tests (cons (string->symbol else) tests))))
      ;; run all the tests
      (for-each (lambda (pvn)
		   (apply test pvn))
		(if (null? tests)
		    (reverse *tests*)
		    (reverse (filter (lambda (t) (memq (car t) tests))
				     *tests*))))
      ;; if we reach that point, we are done
      (print "\n"
	     (if (null? tests) "All" (reverse tests))
	     " tests executed...\n"
	     (if (null? *failure*)
		 "all succeeded"
		 (format " ~a succeeded\n ~a failed ~a"
			 *success*
			 (length *failure*)
			 (reverse *failure*))))))


   

