/*=====================================================================*/
/*    .../prgm/project/bigloo/api/pthread/src/Posix/bglpmutex.c        */
/*    -------------------------------------------------------------    */
/*    Author      :  Manuel Serrano                                    */
/*    Creation    :  Wed Nov  3 07:58:16 2004                          */
/*    Last change :  Tue Jun 21 11:57:14 2005 (serrano)                */
/*    Copyright   :  2004-05 Manuel Serrano                            */
/*    -------------------------------------------------------------    */
/*    The Posix mutex implementation                                   */
/*=====================================================================*/
#ifdef  _MINGW_VER
#  include <sys/timeb.h>
#endif
#include <pthread.h>
#include <sched.h>
#include <stdlib.h>
#include <string.h>

#define GC_PRIVATE_H
#include <gc.h>
#include <bglpthread.h>

#ifdef __CYGWIN__
#  include <sys/time.h>
#endif
#ifdef _MSC_VER
#  include <sys/timeb.h>
#endif

/*---------------------------------------------------------------------*/
/*    Imports                                                          */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DECL void bgl_mutex_init_register( obj_t (*)(obj_t) );
BGL_RUNTIME_DECL void bgl_mutex_lock_register( bool_t (*)(obj_t) );
BGL_RUNTIME_DECL void bgl_mutex_timed_lock_register( bool_t (*)(obj_t, long) );
BGL_RUNTIME_DECL void bgl_mutex_unlock_register( bool_t (*)(obj_t) );

BGL_RUNTIME_DECL void bgl_sleep( long );

/*---------------------------------------------------------------------*/
/*    Mutex symbols                                                    */
/*---------------------------------------------------------------------*/
static obj_t sym_not_owned = 0L;
static obj_t sym_abandoned = 0L;
static obj_t sym_not_abandoned = 0L;

static obj_t mutexes = BNIL;

/*---------------------------------------------------------------------*/
/*    static void                                                      */
/*    symbols_init ...                                                 */
/*---------------------------------------------------------------------*/
static void
symbols_init() {
   if( !sym_not_owned ) {
      sym_not_owned = string_to_symbol( "not-owned" );
      sym_abandoned = string_to_symbol( "abandoned" );
      sym_not_abandoned = string_to_symbol( "not-abandoned" );
   }
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    bglpth_mutex_state ...                                           */
/*---------------------------------------------------------------------*/
obj_t
bglpth_mutex_state( obj_t m ) {
   bglpmutex_t mut = BGLPTH_MUTEX_BGLPMUTEX( m );

   if( mut->locked ) {
      if( mut->thread ) {
	 return mut->thread->bglthread;
      }
      
      symbols_init();
      
      return sym_not_owned;
   } else {
      symbols_init();
      
      if( mut->thread ) 
	 return sym_abandoned;
      else 
	 return sym_not_abandoned;
   }
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    bglpth_mutex_init ...                                            */
/*---------------------------------------------------------------------*/
obj_t
bglpth_mutex_init( obj_t m ) {
   bglpmutex_t mut = (bglpmutex_t)GC_MALLOC( sizeof( struct bglpmutex ) );

   mut->thread = 0L;
   mut->locked = 0;
   mut->specific = BUNSPEC;
   
   m->mutex_t.mutex = mut;
   pthread_mutex_init( &(mut->pmutex), 0L );

   mutexes = MAKE_PAIR( m, mutexes );
   
   return m;
}

/*---------------------------------------------------------------------*/
/*    void                                                             */
/*    bglpth_mutexes_unlock ...                                        */
/*---------------------------------------------------------------------*/
void
bglpth_mutexes_unlock( bglpthread_t thread ) {
   obj_t w = mutexes;

   while( PAIRP( w ) ) {
      obj_t m = CAR( w );
      bglpmutex_t mut = m->mutex_t.mutex;

      if( mut->thread == thread ) {
	 bglpth_mutex_unlock( m );
	 mut->thread = thread;
      }

      w = CDR( w );
   }
}
   
/*---------------------------------------------------------------------*/
/*    bool_t                                                           */
/*    bglpth_mutex_lock ...                                            */
/*---------------------------------------------------------------------*/
bool_t
bglpth_mutex_lock( obj_t m ) {
   bglpmutex_t mut = BGLPTH_MUTEX_BGLPMUTEX( m ); 
   bool_t res = !pthread_mutex_lock( &(mut->pmutex) );

   mut->locked = res;
   
   if( res ) {
      bglpthread_t cth = bglpth_current_pthread();

      if( cth ) mut->thread = cth;
   }
   
   return res;
}

/*---------------------------------------------------------------------*/
/*    bool_t                                                           */
/*    bglpth_mutex_timed_lock ...                                      */
/*---------------------------------------------------------------------*/
bool_t
bglpth_mutex_timed_lock( obj_t m, long ms ) {
#if BGL_HAVE_MUTEX_TIMEDLOCK
   bglpmutex_t mut = BGLPTH_MUTEX_BGLPMUTEX( m );
   struct timespec timeout;
   bool_t res;
#if defined( _MINGW_VER ) || defined( _MSC_VER )
   struct timeb tb;
   ftime( &tb );
   timeout.tv_sec = tb.time + (ms / 1000);
   timeout.tv_nsec = (tb.millitm * 1000000) + ((ms % 1000) * 1000000); 
#else
   struct timeval now;
   gettimeofday( &now, 0 );
   timeout.tv_sec = now.tv_sec + (ms / 1000);
   timeout.tv_nsec = (now.tv_usec * 1000) + ((ms % 1000) * 1000000);
#endif

   res = !pthread_mutex_timedlock( &(mut->pmutex), &timeout );
   mut->locked = res;
   
   if( res  ) {
      bglpthread_t cth = bglpth_current_pthread();

      if( cth ) mut->thread = cth;
   }
   
   return res;
#else
   int res;
   bglpmutex_t mut = BGLPTH_MUTEX_BGLPMUTEX( m );
   while( (ms > 0) && (pthread_mutex_trylock( &(mut->pmutex) ) == EBUSY ) ) {
      ms = ms - 100;
      bgl_sleep( 100 * 1000000);
   }

   return (ms > 0);
#endif
}

/*---------------------------------------------------------------------*/
/*    bool_t                                                           */
/*    bglpth_mutex_unlock ...                                          */
/*---------------------------------------------------------------------*/
bool_t
bglpth_mutex_unlock( obj_t m ) {
   bglpmutex_t mut = BGLPTH_MUTEX_BGLPMUTEX( m );
   bglpthread_t cth = bglpth_current_pthread();
   bool_t res; 

   res = pthread_mutex_unlock( &(mut->pmutex) );

   if( !res ) {
      mut->thread = 0L;
      mut->locked = 0;
   } 
}

/*---------------------------------------------------------------------*/
/*    void                                                             */
/*    bglpth_setup_mutex ...                                           */
/*---------------------------------------------------------------------*/
void
bglpth_setup_mutex() {
   bgl_mutex_init_register( &bglpth_mutex_init );
   bgl_mutex_lock_register( &bglpth_mutex_lock );
   bgl_mutex_timed_lock_register( &bglpth_mutex_timed_lock );
   bgl_mutex_unlock_register( &bglpth_mutex_unlock );
}
