/*=====================================================================*/
/*    serrano/prgm/project/bigloo/runtime/Clib/csymbol.c               */
/*    -------------------------------------------------------------    */
/*    Author      :  Manuel Serrano                                    */
/*    Creation    :  Wed Feb 12 14:51:41 1992                          */
/*    Last change :  Thu May 12 09:14:16 2005 (serrano)                */
/*    -------------------------------------------------------------    */
/*    Symbol handling (creation and hash tabling).                     */
/*=====================================================================*/
#include <string.h>
#include <bigloo.h>

/*---------------------------------------------------------------------*/
/*    Imports                                                          */
/*---------------------------------------------------------------------*/
extern obj_t make_vector();
extern long get_hash_power_number( char *, unsigned long );

/*---------------------------------------------------------------------*/
/*    Global C variables                                               */
/*---------------------------------------------------------------------*/
obj_t c_symtab = BUNSPEC;

/*---------------------------------------------------------------------*/
/*    Symbol mutex                                                     */
/*---------------------------------------------------------------------*/
static obj_t symbol_mutex = BUNSPEC;
DEFINE_STRING( symbol_mutex_name, _1, "symbol-mutex", 12 );

/*---------------------------------------------------------------------*/
/*    bgl_init_symbol_table ...                                        */
/*---------------------------------------------------------------------*/
void
bgl_init_symbol_table() {
   if( !VECTORP( c_symtab ) ) {
      c_symtab = make_vector( SYMBOL_HASH_TABLE_SIZE, BNIL );
      symbol_mutex = bgl_make_mutex( symbol_mutex_name );
   }
}
          
/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    bgl_get_symtab ...                                               */
/*---------------------------------------------------------------------*/
obj_t
bgl_get_symtab() {
   if( !VECTORP( c_symtab ) ) {
      bgl_init_symbol_table();
   }

   return c_symtab;
}

/*---------------------------------------------------------------------*/
/*    make_symbol ...                                                  */
/*    -------------------------------------------------------------    */
/*    Symbols are not allocated in the first generation because        */
/*    they can't be collected.                                         */
/*---------------------------------------------------------------------*/
static obj_t
make_symbol( obj_t name ) {
   obj_t symbol;

   symbol = GC_MALLOC( SYMBOL_SIZE );
   
   symbol->symbol_t.header = MAKE_HEADER( SYMBOL_TYPE, SYMBOL_SIZE );
   symbol->symbol_t.string = name;
   symbol->symbol_t.cval   = BNIL;

   return BREF( symbol );
}
   
/*---------------------------------------------------------------------*/
/*    bstring_to_symbol ...                                            */
/*    obj_t --> obj_t                                                  */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF
obj_t
bstring_to_symbol( obj_t name ) {
   long hash_number;
   obj_t bucket;
   char *cname = BSTRING_TO_STRING( name );

   hash_number = get_hash_power_number( cname, SYMBOL_HASH_TABLE_SIZE_SHIFT );
   
   bgl_mutex_lock( symbol_mutex );
   bucket = VECTOR_REF( c_symtab, hash_number );
   
   if( NULLP( bucket ) ) {
      obj_t symbol = make_symbol( name );
      obj_t pair = MAKE_PAIR( symbol, BNIL );
      
      VECTOR_SET( c_symtab, hash_number, pair );
      
      bgl_mutex_unlock( symbol_mutex );
      return symbol;
   } else {
      obj_t run = bucket, back = bucket;
      
      while( !NULLP( run ) &&
	     strcmp( (char *)BSTRING_TO_STRING( SYMBOL( CAR( run ) ).string ),
		     cname ) )
         back = run, run = CDR( run );
      
      if( !NULLP( run ) ) {
	 bgl_mutex_unlock( symbol_mutex );
         return CAR( run );
      }
      else {
         obj_t symbol = make_symbol( name );
	 obj_t pair = MAKE_PAIR( symbol, BNIL );
	 
         SET_CDR( back, pair );

	 bgl_mutex_unlock( symbol_mutex );
         return symbol;
      }
   }
}

/*---------------------------------------------------------------------*/
/*    string_to_symbol ...                                             */
/*    char * --> obj_t                                                 */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF
obj_t
string_to_symbol( char *cname ) {
   long hash_number;
   obj_t bucket;

   hash_number = get_hash_power_number( cname, SYMBOL_HASH_TABLE_SIZE_SHIFT );
   
   bgl_mutex_lock( symbol_mutex );
   bucket = VECTOR_REF( c_symtab, hash_number );
   
   if( NULLP( bucket ) ) {
      obj_t symbol = make_symbol( string_to_bstring( cname ) );
      obj_t pair   = MAKE_PAIR( symbol, BNIL );
      
      VECTOR_SET( c_symtab, hash_number, pair );
      
      bgl_mutex_unlock( symbol_mutex );
      return symbol;
   } else {
      obj_t run = bucket, back = bucket;
      
      while( !NULLP( run ) &&
	     strcmp( (char *)BSTRING_TO_STRING( SYMBOL( CAR( run ) ).string ),
		     cname ) )
         back = run, run = CDR( run );
       
      if( !NULLP( run ) ) {
	 bgl_mutex_unlock( symbol_mutex );
         return CAR( run );
      }
      else {
         obj_t symbol = make_symbol( string_to_bstring( cname ) );
	 obj_t pair   = MAKE_PAIR( symbol, BNIL );
	 
         SET_CDR( back, pair );

	 bgl_mutex_unlock( symbol_mutex );
         return symbol;
      }
   }
}

/*---------------------------------------------------------------------*/
/*    int                                                              */
/*    symbol_exists_p ...                                              */
/*    -------------------------------------------------------------    */
/*    Cette fonction retourne vrai s'il existe un symbol de nom        */
/*    `string' et faux sinon                                           */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF
int
symbol_exists_p( char *name ) {
   long hash_number;
   obj_t bucket;

   hash_number = get_hash_power_number( name, SYMBOL_HASH_TABLE_SIZE_SHIFT );
   bucket = VECTOR_REF( c_symtab, hash_number );
   
   if( NULLP( bucket ) ) {
      return 0;
   } else {
      bgl_mutex_lock( symbol_mutex );
      while( strcmp( (char *)BSTRING_TO_STRING(SYMBOL( CAR( bucket ) ).string),
		     name ) ) {
	 bucket = CDR( bucket );
	 
	 if( NULLP( bucket) ) {
	    bgl_mutex_unlock( symbol_mutex );
	    return 0;
	 }
      }
      
      bgl_mutex_unlock( symbol_mutex );
      return 1;
   }
}
