/*
 * Copyright 2009 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
/* Lots and lots of debugging information */

const EXPORTED_SYMBOLS = ["Bindwood"];

const Cc = Components.classes;
const Ci = Components.interfaces;
const Cu = Components.utils;

Cu.import("resource://bindwood/desktopcouch.jsm");
Cu.import("resource://bindwood/logging.jsm");
Cu.import("resource://bindwood/sync.jsm");

var envService = Cc["@mozilla.org/process/environment;1"]
    .getService(Ci.nsIEnvironment);
var directoryService = Cc["@mozilla.org/file/directory_service;1"]
    .getService(Ci.nsIProperties);
// Technically, a branch, rather than the actual service, but
// consistency wins, I think
var prefsService = Cc["@mozilla.org/preferences-service;1"]
    .getService(Ci.nsIPrefService).getBranch("bindwood.");

var Bindwood = {

    sync_timer: Cc["@mozilla.org/timer;1"]
        .createInstance(Ci.nsITimer),

    couch: null,
    synchroniser: null,
    running: false,

    extractProfileName: function(path) {
        // We want the last part of the profile path
        // ('default' for '/home/zbir/.mozilla/firefox/ixlw0nvl.default')
        // For profiles that have not been created via the Profile Manager,
        // we just take the last path segment as the profile name.
        //
        // Actually, there's a degenerate case, where the last segment
        // doesn't have the Firefox-provided random string:
        //   '/home/zbir/canonical.com' => 'com'
        // But as I said, this is maybe degenerate.

        var segments = path.split('/');
        var possible_profile = segments[segments.length - 1];
        var first_period = possible_profile.indexOf('.');
        if (first_period == -1) {
            // no periods in the last segment, return as is
            return possible_profile;
        } else {
            return possible_profile.substr(first_period + 1);
        }
    },

    // Setup the environment and go
    init: function() {
        // Start the process and de-register ourself
        // http://forums.mozillazine.org/viewtopic.php?f=19&t=657911&start=0
        // It ensures that we're only running that code on the first window.

        if (!this.running) {
            Log.debug("Getting started in init.");
            this.currentProfile = Bindwood.extractProfileName(
                directoryService.get('ProfD', Ci.nsIFile).path);

            Log.debug("Got our profile: " + this.currentProfile);

            this.running = true;
            this.connectToCouchDB();
        }
    },

    uninit: function() {
        if (this.running) {
            this.running = false;
            this.sync_timer.cancel();
            if (this.synchroniser) {
                this.synchroniser.uninit();
                this.synchroniser = null;
            }
        }
    },

    connectToCouchDB: function() {
        var db_name = 'bookmarks';
        if (envService.exists('BINDWOOD_DB')) {
            db_name = envService.get('BINDWOOD_DB');
        }

        connect_desktopcouch(
            db_name,
            function (database) {
                if (database) {
                    Log.debug("Got our db name: " + database.name);
                    Bindwood.couch = database;

                    try {
                        Bindwood.startProcess();
                    } catch(e) {
                        Log.exception(
                            "Something wrong with the process, exiting.", e);
                        return;
                    }
                }
            }, function (message) {
                Log.debug(message);
            });
    },

    getLastSequence: function() {
        var seq;
        try {
            seq = prefsService.getIntPref('last_seq');
        } catch(e) {
            seq = 0;
        }
        return seq;
    },

    setLastSequence: function(seq) {
        prefsService.setIntPref('last_seq', seq);
        return seq;
    },

    startProcess: function() {
        // If we were shut down while initialising CouchDB, just return.
        if (!this.running)
            return;
        Log.debug("Setting up synchroniser.");
        this.synchroniser = new Synchroniser(this.couch, this.currentProfile);
        this.synchroniser.last_seq = this.getLastSequence();
        Log.debug("Got our last known sequence number: " +
                  this.synchroniser.last_seq);
        this.synchroniser.init();

        var repeater = {
            notify: function(timer) {
                Bindwood.synchroniser.sync();
                Bindwood.setLastSequence(Bindwood.synchroniser.last_seq);
                Log.debug(
                    "Successful run, rescheduling ourself");
            }
        };
        try {
            repeater.notify(); // Prime the pump, then schedule us out.
        } catch (e) {
            Log.exception("Problem synchronising bookmarks", e);
            return;
        }

        // reschedule ourself
        try {
            this.sync_timer.initWithCallback(
                repeater, 30000, Ci.nsITimer.TYPE_REPEATING_SLACK);
        } catch(e) {
            Log.exception("Problem setting up repeater.", e);
        }
    },
};
