/*
 * Copyright 2009-2011 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
/* Utility functions to establish a connection to a DesktopCouch
 * CouchDB instance.
 */

const EXPORTED_SYMBOLS = [
    "connect_desktopcouch", "_get_desktopcouch_environment"];

var Cc = Components.classes;
var Ci = Components.interfaces;
var Cu = Components.utils;

Cu.import("resource://bindwood/couch.jsm");
Cu.import("resource://bindwood/oauth.jsm");

try {
    Cu.import("resource://gre/modules/AddonManager.jsm");
    var have_addonmanager = true;
} catch (e) {
    var have_addonmanager = false;
}

const EXTENSION_ID = "bindwood@ubuntu.com";

/**
 * Retrieve the DesktopCouch environment.
 *
 * @param callback
 *     A callback that will be passed the environment on success, or
 *     null on failure.
 * @param log
 *     A callback that will be called to log debug messages.
 */
function _get_desktopcouch_environment(callback, log) {
    // find the D-Bus bash script, which is in our extension folder
    if (have_addonmanager) {
        AddonManager.getAddonByID(EXTENSION_ID, function (addon) {
                var couchdb_env_script = addon.getResourceURI("couchdb_env.sh")
                    .QueryInterface(Ci.nsIFileURL).file;
                _continue_get_desktopcouch_environment(
                    couchdb_env_script, callback, log);
            });
    } else {
        // AddonManager is not available (i.e. we're on Firefox 3.x),
        // so use the old nsIExtensionManager service.
        var em = Cc["@mozilla.org/extensions/manager;1"]
            .getService(Ci.nsIExtensionManager);
        var couchdb_env_script = em.getInstallLocation(EXTENSION_ID)
            .getItemFile(EXTENSION_ID, "couchdb_env.sh");
        _continue_get_desktopcouch_environment(
            couchdb_env_script, callback, log);
    }
}

function _continue_get_desktopcouch_environment(couchdb_env_script,
                                                callback, log) {
    // find OS temp dir to put the tempfile in
    // https://developer.mozilla.org/index.php?title=File_I%2F%2FO#Getting_special_files
    var tmpdir = Cc["@mozilla.org/file/directory_service;1"]
        .getService(Ci.nsIProperties).get("TmpD", Ci.nsIFile);
    // create a randomly named tempfile in the tempdir
    var tmpfile = Cc["@mozilla.org/file/local;1"]
        .createInstance(Ci.nsILocalFile);
    tmpfile.initWithPath(tmpdir.path + "/desktopcouch." + Math.random());
    tmpfile.createUnique(tmpfile.NORMAL_FILE_TYPE, 0600);

    // create an nsILocalFile for the shell
    var shell = Cc["@mozilla.org/file/local;1"]
        .createInstance(Ci.nsILocalFile);
    shell.initWithPath("/bin/sh");

    // create an nsIProcess2 to execute this bash script
    var process = Cc["@mozilla.org/process/util;1"]
        .createInstance(Ci.nsIProcess2 || Ci.nsIProcess);
    process.init(shell);

    // Run the process, passing the tmpfile path
    var args = [couchdb_env_script.path, tmpfile.path];
    process.runAsync(args, args.length, {
            observe: function(process, finishState, unused_data) {
                // If the script exists cleanly, we should have a file
                // containing the port couch is running on as well as
                // the various OAuth tokens necessary to talk to it.
                var shouldProceed = true;
                if (finishState == "process-finished") {
                    // read temp file to find couch environment
                    // https://developer.mozilla.org/en/Code_snippets/File_I%2f%2fO#Reading_from_a_file
                    var environment;
                    var fstream = Cc["@mozilla.org/network/file-input-stream;1"]
                        .createInstance(Ci.nsIFileInputStream);
                    var cstream = Cc["@mozilla.org/intl/converter-input-stream;1"]
                        .createInstance(Ci.nsIConverterInputStream);
                    fstream.init(tmpfile, -1, 0, 0);
                    cstream.init(fstream, "UTF-8", 0, 0);
                    let (str = {}) {
                        // read the whole file and put it in str.value
                        cstream.readString(-1, str);
                        environment = str.value;
                    };
                    cstream.close(); // this closes fstream
                    environment = environment.replace(/^\s\s*/, '')
                        .replace(/\s\s*$/, '');
                } else {
                    // If we fail, we should just return
                    log("D-Bus port find failed");
                    shouldProceed = false;
                }
                tmpfile.remove(false);

                if (environment == 'ENOCOUCH') {
                    // No Couch environment found. Just spit out a
                    // message and return, stopping Bindwood from
                    // doing anything further.
                    log(
                        "No suitable Couch environment found." +
                            " Not proceeding.", e);
                    shouldProceed = false;
                }

                if (shouldProceed && environment) {
                    log("Got our environment, proceeding.");
                    callback(environment);
                } else {
                    // Unregister our observer for bookmark events; we're done
                    log("No environment. Unregistering observer.");
                    callback(null);
                }
            }
        });
}

/**
 * Connect to the given database in DesktopCouch using the given environment.
 *
 * @param name
 *     The name of the database in DesktopCouch
 * @param environment
 *     The DesktopCouch environment, as produced by
 *     _get_desktopcouch_environment.
 * @returns A new CouchDB instance configured to talk to DesktopCouch.
 */
function DesktopCouchDB(name, environment) {
    var env_array = environment.split(":");
    var host = "http://localhost:" + env_array[0];
    var consumer_key = env_array[1];
    var consumer_secret = env_array[2];
    var token = env_array[3];
    var token_secret = env_array[4];

    CouchDB.call(this, name);
    var accessor = {
        consumerKey: consumer_key,
        consumerSecret: consumer_secret,
        token: token,
        tokenSecret: token_secret,
    };
    this.request = function(method, uri, requestOptions) {
        // prepend full host name to URI.
        uri = host + uri;

        // Construct OAuth signature, and add to request headers.
        var message = {
            method: method,
            action: uri,
            parameters: {
                oauth_signature_method: "PLAINTEXT",
                oauth_version: "1.0",
            },
        };
        OAuth.setTimestampAndNonce(message);
        OAuth.completeRequest(message, accessor);
        var auth_header = OAuth.getAuthorizationHeader(
            "desktopcouch", message.parameters);

        requestOptions = requestOptions || {};
        requestOptions.headers = CouchDB.combine(
            requestOptions.headers, {Authorization: auth_header});

        // Issue request.
        return CouchDB.request(method, uri, requestOptions);
    }
}

DesktopCouchDB.prototype = CouchDB.prototype;

var _desktopcouch_environment = null;

/**
 * Connect to the given database in DesktopCouch.
 *
 * @param name
 *     The name of the database in DesktopCouch
 * @param callback
 *     A callback called with the CouchDB instance on succes, or null
 *     on failure.
 * @param log
 *     A callback that will be called to log debug messages.
 */
function connect_desktopcouch(name, callback, log) {
    if (_desktopcouch_environment == null) {
        _get_desktopcouch_environment(function(environment) {
                _desktopcouch_environment = environment;
                if (environment) {
                    var db = new DesktopCouchDB(name, environment);
                } else {
                    db = null;
                }
                callback(db);
            }, log);
        return;
    }
    log("Connecting to DesktopCouch using saved environment.");
    callback(new DesktopCouchDB(name, _desktopcouch_environment));
}
