/*
 * Copyright 2009-2011 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
/* Logging helpers for use in other Bindwood modules. */

const EXPORTED_SYMBOLS = ["Log"];

const Cc = Components.classes;
const Ci = Components.interfaces;

var envService = Cc["@mozilla.org/process/environment;1"]
    .getService(Ci.nsIEnvironment);
var consoleService = Cc["@mozilla.org/consoleservice;1"]
    .getService(Ci.nsIConsoleService);
var prefsBranch = Cc["@mozilla.org/preferences-service;1"]
    .getService(Ci.nsIPrefService).getBranch("bindwood.");


const PR_WRONLY = 0x02;
const PR_CREATE_FILE = 0x08;
const PR_APPEND = 0x10;


const LEVEL_DEBUG = 0;
const LEVEL_INFO = 1;
const LEVEL_WARNING = 2;
const LEVEL_ERROR = 3;

const LEVEL_NAMES = ["DEBUG", "INFO", "WARNING", "ERROR"];


var Log = {
    _log_opened: false,
    _log_stream: null,

    _get_log_stream: function() {
        if (!this._log_opened) {
            if (envService.exists('BINDWOOD_LOGFILE')) {
                try {
                    var logfile = Cc["@mozilla.org/file/local;1"].
                        createInstance(Ci.nsILocalFile);
                    logfile.initWithPath(envService.get('BINDWOOD_LOGFILE'));

                    // Create an output stream appending to the log
                    // file with default permissions.
                    var ostream = Cc["@mozilla.org/network/file-output-stream;1"].
                        createInstance(Ci.nsIFileOutputStream);
                    ostream.init(
                        logfile, PR_WRONLY|PR_CREATE_FILE | PR_APPEND, -1, 0);

                    // Wrap the output stream with a UTF-8 encoder.
                    this._log_stream = Cc["@mozilla.org/intl/converter-output-stream;1"].
                        createInstance(Ci.nsIConverterOutputStream);
                    this._log_stream.init(
                        ostream, "UTF-8", 4096,
                        Ci.nsIConverterInputStream.DEFAULT_REPLACEMENT_CHARACTER);
                } catch (e) {
                    dump("Error setting up log file: " + e + "\n");
                    this._log = null;
                }
            }
            this._log_opened = true;
        }
        return this._log_stream;
    },

    _write: function(level, message) {
        var log = this._get_log_stream();
        if (log) {
            log.writeString((new Date()).toUTCString() + " " +
                            LEVEL_NAMES[level] + ": " + message + "\n");
        }

        if (level <= LEVEL_DEBUG) {
            try {
                var debug = prefsBranch.getBoolPref('debug');
            } catch(e) {
                var debug = false;
            }
            if (!debug) {
                // The user has not requested logging of debug
                // messages to the error console.
                return;
            }
        }
        consoleService.logStringMessage("Bindwood: " + message);
    },

    debug: function(message) {
        this._write(LEVEL_DEBUG, message);
    },

    error: function(message) {
        this._write(LEVEL_ERROR, message);
    },

    exception: function(message, error) {
        this.error(message +
                   " message: '" + error.message +
                   "', reason: '" + error.reason +
                   "', description: '" + error.description +
                   "', error: '" + error.error +
                   "', raw error: '" + JSON.stringify(error) + "'");
    },
};
