/*
*
*  A2DPD - Bluetooth A2DP daemon for Linux
*
*  Copyright (C) 2006-2007  Frédéric DALLEAU <frederic.dalleau@palmsource.com>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#ifndef __A2DPD_PROTOCOL_H__
#define __A2DPD_PROTOCOL_H__

#include <stdio.h>
#include <stdint.h>
#include <time.h>
#include <unistd.h>
#include <sys/time.h>
#include <errno.h>
#include <string.h>

extern FILE* g_fdout;
extern int g_bdebug;
extern char* g_prefix;

#define DEFAULTFDOUT NULL

#ifndef NDEBUG
#	define DEFAULTDEBUG 0
#else
#	define DEFAULTDEBUG 1
#endif

#ifdef S_SPLINT_S
// Splint don't like the arg...  syntax
#	define DBG printf 
#else
#if 1
//#ifndef NDEBUG
#	define DBGCORE(filter, fmt, arg...)\
do {\
	if(g_bdebug & filter) {\
		int errno0 = errno;\
		struct timeval dbg_time;\
		char szTime [16]="[";\
		time_t nowt = time(NULL);\
		(void)strftime(szTime+1, 9, "%H:%M:%S", localtime(&nowt));\
		(void)gettimeofday(&dbg_time, 0);\
		(void)snprintf(szTime+9, 6, ".%03d]", (int)(dbg_time.tv_usec/1000));\
		errno = errno0;\
		if(g_fdout<0) g_fdout = DEFAULTFDOUT;\
		if(errno!=0 && errno!=EAGAIN)\
			if(g_fdout)\
				(void)fprintf(g_fdout, "%s%s: %s: (errno=%d:%s)" fmt "\r\n" , g_prefix, szTime, __FUNCTION__ , errno, strerror(errno), ## arg);\
			else\
				(void)printf("%s%s: %s: (errno=%d:%s)" fmt "\r\n" , g_prefix, szTime, __FUNCTION__ , errno, strerror(errno), ## arg);\
		else\
			if(g_fdout)\
				(void)fprintf(g_fdout, "%s%s: %s: " fmt "\n", g_prefix, szTime, __FUNCTION__ , ## arg);\
			else\
				(void)printf("%s%s: %s: " fmt "\n", g_prefix, szTime, __FUNCTION__ , ## arg);\
		if(g_fdout)\
				fflush(g_fdout);\
		else\
				fflush(stdout);\
	}\
	errno=0;\
} while(0)
#else
#	define DBGCORE(var, fmt, arg...) do { (void)0; } while(0)
#endif
#endif

#define DBG(arg...)  DBGCORE(1, ##arg)
#define DBG2(arg...) DBGCORE(2, ##arg)
#define DBG3(arg...) DBGCORE(4, ##arg)
#define DBG4(arg...) DBGCORE(8, ##arg)
#define DBG5(arg...) DBGCORE(16, ##arg)
#define DBG6(arg...) DBGCORE(32, ##arg)
#define DBG7(arg...) DBGCORE(64, ##arg)
#define DBG8(arg...) DBGCORE(128, ##arg)

#define RETURNERROR(fmt, arg...) do { DBG(fmt, ## arg); return -1; } while (0)

#define min(X, Y)  ((X) < (Y) ? (X) : (Y))
#define max(X, Y)  ((X) > (Y) ? (X) : (Y))

// parameters used to describe device state
typedef struct {
	int16_t volume_speaker_right;
	int16_t volume_speaker_left;
	int16_t volume_micro_right;
	int16_t volume_micro_left;
} __attribute__ ((packed)) AUDIOMIXERDATA;

#define INVALIDAUDIOMIXERDATA   { -1, -1, -1, -1 }

#define A2DPMAXIMUMTRANSFERUNITSIZE 610


// PCM formats defined in alsa, we will restrict our selves to 8 and 16 bits
#define A2DPD_PCM_FORMAT_UNKNOWN 0x00000000
#define A2DPD_PCM_FORMAT_S8      0x00000001
#define A2DPD_PCM_FORMAT_U8      0x00000002
#define A2DPD_PCM_FORMAT_S16_LE  0x00000003
//#define A2DPD_FORMAT_S16_BE        0x00000004
//#define A2DPD_FORMAT_U16_LE        0x00000005
//#define A2DPD_FORMAT_U16_BE        0x00000006

// parameters used to describe device state
typedef struct {
	uint32_t format;
	uint16_t rate;
	uint8_t channels;
	uint16_t bitspersample;
	uint32_t streamid;
	uint32_t buffer_size;
	uint32_t start_threshold;
} __attribute__ ((packed)) AUDIOSTREAMINFOS;

typedef struct {
	struct timeval packet_date;
	uint32_t pcm_buffer_size;
} __attribute__ ((packed)) AUDIOPACKETHEADER;

typedef struct {
	uint32_t action;
	uint32_t param;
} __attribute__ ((packed)) GETDELAYREQ;

typedef struct {
	uint32_t delay;
} __attribute__ ((packed)) GETDELAYRESP;

typedef struct {
	uint32_t pcm_state;
	uint64_t app_ptr;
	uint64_t hw_ptr;
} __attribute__ ((packed)) GETPOINTERRESP;

// Theses state are the pending of alsa states
// They should be kept in sync between alsa and the daemon
#define A2DPD_PCM_STATE_OPEN         0
#define A2DPD_PCM_STATE_SETUP        1
#define A2DPD_PCM_STATE_PREPARED     2
#define A2DPD_PCM_STATE_RUNNING      3
#define A2DPD_PCM_STATE_XRUN         4
#define A2DPD_PCM_STATE_DRAINING     5
#define A2DPD_PCM_STATE_PAUSED       6
#define A2DPD_PCM_STATE_SUSPENDED    7
#define A2DPD_PCM_STATE_DISCONNECTED 8

#define INVALIDAUDIOSTREAMINFOS   { 0, 0, 0 }

// Different types of client plugin for the daemon
#define INVALID_CLIENT_TYPE       0xFFFFFFFF
#define A2DPD_PLUGIN_CTL_WRITE    0x00000001
#define A2DPD_PLUGIN_CTL_READ     0x00000002
#define A2DPD_PLUGIN_PCM_WRITE    0x00000003
#define A2DPD_PLUGIN_PCM_READ     0x00000004
#define A2DPD_PLUGIN_PCM_WRITE_3  0x00000005
#define A2DPD_PLUGIN_GET_DELAY    0x00000006
#define A2DPD_PLUGIN_GET_POINTER  0x00000007
#define A2DPD_PLUGIN_PREPARE      0x00000008
#define A2DPD_PLUGIN_START        0x00000009
#define A2DPD_PLUGIN_STOP         0x0000000A
#define A2DPD_PLUGIN_DRAIN        0x0000000B
#define A2DPD_PLUGIN_PAUSE_0      0x0000000C
#define A2DPD_PLUGIN_PAUSE_1      0x0000000D
#define A2DPD_PLUGIN_RESUME       0x0000000E

#define A2DPD_VOLUME_MIN          0
#define A2DPD_VOLUME_MAX          16

#define A2DPD_FRAME_BYTES         4	// 16bits * 2 channels
#define A2DPD_FRAME_RATE          44100	// Can be 32000, tested with HP, iPhono and needed Sonorix, but quality decreases, 48000 nearly never works
// a2dp->sbc.channels*44100*2/(size*a2dp->frame_bytes);
// 344.53125=channels*freq*16 bits/sizeof(buf)
#define A2DPD_BLOCK_SIZE          (512*1)
// This is the number of blocks enqueued by the client
#define MAXCLIENTSRINGSIZE        (128)

#define PIDFILE                   "/tmp/a2dp.pid"

#endif
