//  BMPx - The Dumb Music Player
//  Copyright (C) 2005 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License Version 2
//  as published by the Free Software Foundation.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <iostream>
#include <cstring>
#include <glibmm.h>
#include <glib/gi18n.h>

#include <gst/gst.h>
#include <gst/gstelement.h>
#include <gst/interfaces/mixer.h>
#include <gst/interfaces/mixertrack.h>
#include <gst/interfaces/mixeroptions.h>

#include <boost/shared_ptr.hpp>
#include <boost/algorithm/string.hpp>
#include <boost/format.hpp>
#include <boost/optional.hpp>

#include "main.hh"
#include "uri++.hh"
#include "util.hh"
#include "play.hh"
#include "debug.hh"

#include "dialog-gsterror.hh"
#include "x_service_core.hh"

#define BMP_GST_BUFFER_TIME   ((gint64) 50000)
#define BMP_GST_PLAY_TIMEOUT  ((gint64) 3000000000)

namespace
{
  const char *NAME_DECODER  = _("Stream Decoder");
  const char *NAME_CONVERT  = _("Raw Audio Format Converter");
  const char *NAME_QUEUE    = _("Data Buffer (Queue)");
  const char *NAME_VOLUME   = _("Volume Adjustment");
  const char *NAME_RESAMPLE = _("Resampler");

  gboolean
  drop_data (GstPad *pad, GstMiniObject *mini_obj, gpointer data)
  {
    return FALSE;
  }
}

namespace Bmp
{
    Play::Play () 
        : Glib::ObjectBase (typeid(Play)),
          property_stream_        (*this, "stream", ""),
          property_lastfm_mode_   (*this, "lastfm-mode", false),
          property_volume_        (*this, "volume", 50),
          property_position_      (*this, "position", 0),
          property_status_        (*this, "playstatus", PLAYSTATUS_STOPPED),
          property_sane_          (*this, "sane", false),
          property_length_        (*this, "length", 0), 
          m_seeking               (false),
          http_elmt               (0),
          play_elmt               (0)

    {

      property_volume() = mcs->key_get<int>("bmp", "volume"); 

      property_volume().signal_changed().connect 
                (sigc::mem_fun(this, &Bmp::Play::on_volume_changed));

      property_stream().signal_changed().connect
                (sigc::mem_fun(this, &Bmp::Play::on_stream_changed));

      property_lastfm_mode().signal_changed().connect
                (sigc::mem_fun(this, &Bmp::Play::on_lastfm_mode_changed));

      bin_file      = 0;
      bin_http      = 0;
      bin_http_mad  = 0;
      bin_mmsx      = 0;
      bin_cdda      = 0;
      bin_output    = 0;

#ifdef HAVE_VISUALIZATIONS
      bin_vis       = 0;
      vis_buffer_probe_handler_id = 0;
#endif //HAVE_VISUALIZATIONS

      pipeline      = 0;

      reset ();
    }

    //dtor
    Play::~Play ()
    {
      request_status (PLAYSTATUS_STOPPED);
      bins_destroy ();
    }

    bool
    Play::timeout_handler ()
    {
      GstQuery       *query;
      GstFormat       format = GST_FORMAT_TIME;
      gint64          time_in_nanoseconds;
      gint            time_in_seconds;

      if (!(GST_STATE (pipeline) == GST_STATE_PLAYING))
        {
          return false;
        }

      query = gst_query_new_position (format);
      gst_element_query (pipeline, query);
      gst_query_parse_position (query, &format, &time_in_nanoseconds);
      time_in_seconds = time_in_nanoseconds / GST_SECOND;
      gst_query_unref (query);

      if ((time_in_seconds >= 0) && (GST_STATE(pipeline) >= GST_STATE_PLAYING))
        {
          signal_position_.emit( time_in_seconds );
          return true;
        }
      else
        {
          signal_position_.emit( 0 );
          return false;
        }

      return false;
    }

    bool
    Play::is_audio_file (std::string const& str)
    {
      char* const extensions[] =
      { "mp3",
        "aac",
        "mp4",
        "m4a",
        "m4b",
        "m4p",
        "m4v",
        "mp4v",
        "flac",
        "wma",
        "asf",
        "sid",
        "psid",
        "mod",
        "oct",
        "xm",
        "669",
        "sht", 
        "mpc",
        "ogg",
        "wav",
        0 };

      return Bmp::Util::str_has_suffixes_nocase (str.c_str(), extensions); 
    }

    void 
    Play::stop_stream (bool set_state)
    {
      gst_element_set_state (pipeline, GST_STATE_NULL);
      if (set_state)
        {
          property_status_ = PLAYSTATUS_STOPPED;
        }
    }
    
    void 
    Play::play_stream ()
    {
      g_object_set (G_OBJECT (gst_bin_get_by_name (GST_BIN(bin_output), NAME_VOLUME)), 
                    "volume", property_volume().get_value()/100. , NULL);

      GstStateChangeReturn statechange;
      statechange = gst_element_set_state (pipeline, GST_STATE_PLAYING);

      if (statechange != GST_STATE_CHANGE_FAILURE)
        {
          property_status_ = PLAYSTATUS_PLAYING;
          return;
        }

      stop_stream ();
      return;
    }
    
    void 
    Play::pause_stream ()
    {
      GstStateChangeReturn statechange;
      if (GST_STATE (pipeline) == GST_STATE_PAUSED)
        {
          statechange = gst_element_set_state (pipeline, GST_STATE_PLAYING);
          if (statechange != GST_STATE_CHANGE_FAILURE)
            {
              property_status_ = PLAYSTATUS_PLAYING;
              return;
            }
        }
      else
        {
          statechange = gst_element_set_state (pipeline, GST_STATE_PAUSED);
          if (statechange != GST_STATE_CHANGE_FAILURE)
            {
              property_status_ = PLAYSTATUS_PAUSED;
              return;
            }
        }

      stop_stream ();
      return;
    }

    void
    Play::on_volume_changed () 
    {
      gdouble _volume;
      _volume = property_volume().get_value()/100.; 
      g_object_set (G_OBJECT (gst_bin_get_by_name (GST_BIN (bin_output), NAME_VOLUME)), "volume", _volume, NULL);
    }

    void
    Play::on_lastfm_mode_changed () 
    {
      g_object_set (G_OBJECT (gst_bin_get_by_name (GST_BIN (bin_http), "src")), "lastfm-mode",
                        property_lastfm_mode_.get_value() ? TRUE : FALSE, NULL);
    }

    void
    Play::pipeline_preconfigure (std::string const& name)
    {
     if (play_elmt)
        {
          gst_element_unlink (play_elmt, bin_output);
          gst_element_set_state (play_elmt, GST_STATE_NULL);
          gst_element_set_state (bin_output, GST_STATE_NULL);
          gst_bin_remove_many (GST_BIN (pipeline), play_elmt, bin_output, NULL);
        }

      gst_element_set_state (pipeline, GST_STATE_NULL);
      gst_element_set_name (pipeline, name.c_str());
    }

    void
    Play::on_stream_changed () 
    {
      if (property_sane().get_value() == false) return;

      Bmp::URI uri (property_stream().get_value());

      switch (uri.get_protocol())
      {
          case Bmp::URI::PROTOCOL_FILE:
          { 
            try
              {
                g_object_set (gst_bin_get_by_name (GST_BIN (bin_file), "src"), 
                              "location", Glib::filename_from_uri (property_stream().get_value()).c_str(), NULL);
                             
                if (std::string (gst_element_get_name (pipeline)) != "file")
                {
                  pipeline_preconfigure ("file");

#ifdef HAVE_VISUALIZATIONS

                  gst_bin_add_many (GST_BIN (pipeline), bin_file, bin_output, bin_vis, NULL);
                  gst_element_link_many (bin_file, bin_output, NULL);
                  gst_element_link (gst_bin_get_by_name (GST_BIN (bin_file), "tee"), bin_vis);

#else
                  gst_bin_add_many (GST_BIN (pipeline), bin_file, bin_output, NULL);
                  gst_element_link_many (bin_file, bin_output, NULL);

#endif //HAVE_VISUALIZATIONS

                  play_elmt = bin_file;

                }
              }
            catch (Glib::ConvertError& cxe)
              {
                static boost::format format ("Unable to play stream '%s': Conversion error (%s)");
                Gtk::MessageDialog dialog ((format % property_stream().get_value().c_str() % 
                                            cxe.what()).str(),
                                            false, Gtk::MESSAGE_ERROR, Gtk::BUTTONS_OK, true);
                dialog.run();
                dialog.hide();
              }
            break;
          }
  
          case Bmp::URI::PROTOCOL_MMS:
          case Bmp::URI::PROTOCOL_MMSU:
          case Bmp::URI::PROTOCOL_MMST:
          {
            g_object_set (gst_bin_get_by_name (GST_BIN (bin_mmsx), "src"), 
                          "location", property_stream().get_value().c_str(), NULL);

            if (std::string (gst_element_get_name (pipeline)) != "mmsx")
            {
              pipeline_preconfigure ("mmsx");

#ifdef HAVE_VISUALIZATIONS

              gst_bin_add_many (GST_BIN (pipeline), bin_mmsx, bin_output, bin_vis, NULL);
              gst_element_link_many (bin_mmsx, bin_output, NULL);
              gst_element_link (gst_bin_get_by_name (GST_BIN (bin_mmsx), "tee"), bin_vis);


#else
              gst_bin_add_many (GST_BIN (pipeline), bin_mmsx, bin_output, NULL);
              gst_element_link_many (bin_mmsx, bin_output, NULL);

#endif // HAVE_VISUALIZATIONS

              play_elmt = bin_mmsx;

            }
            break;
          }
 
          case Bmp::URI::PROTOCOL_HTTP:
          {
            http_elmt = bin_http;
            if (m_stream_type.size())
              {
                if (m_stream_type == "audio/mpeg")
                  http_elmt = bin_http_mad;
              }

            g_object_set (gst_bin_get_by_name (GST_BIN (http_elmt), "src"), 
                          "lastfm-mode", gboolean(property_lastfm_mode().get_value()), NULL);
            g_object_set (gst_bin_get_by_name (GST_BIN (http_elmt), "src"), 
                          "location", property_stream().get_value().c_str(), NULL);
            g_object_set (gst_bin_get_by_name (GST_BIN (http_elmt), "src"), 
                          "prebuffer", TRUE, NULL);

            pipeline_preconfigure ("http");

#ifdef HAVE_VISUALIZATIONS

            gst_bin_add_many (GST_BIN (pipeline), http_elmt, bin_output, bin_vis, NULL);
            gst_element_link_many (http_elmt, bin_output, NULL);
            gst_element_link (gst_bin_get_by_name (GST_BIN (http_elmt), "tee"), bin_vis);

#else

            gst_bin_add_many (GST_BIN (pipeline), http_elmt, bin_output, NULL);
            gst_element_link_many (http_elmt, bin_output, NULL);

#endif // HAVE_VISUALIZATIONS

            play_elmt = http_elmt;
            break;
          }
           
          case Bmp::URI::PROTOCOL_CDDA:
          {
            unsigned int track;
            track = (unsigned int)atoi((uri.path.c_str())+1);
            g_object_set (gst_bin_get_by_name (GST_BIN (bin_cdda), "src"), 
                          "track", (unsigned int)(track), NULL);

            if (std::string (gst_element_get_name (pipeline)) != "cdda")
            {
              pipeline_preconfigure ("cdda");

#ifdef HAVE_VISUALIZATIONS

              gst_bin_add_many (GST_BIN (pipeline), bin_cdda, bin_output, bin_vis, NULL);
              gst_element_link_many (bin_cdda, bin_output, NULL);
              gst_element_link (gst_bin_get_by_name (GST_BIN (bin_cdda), "tee"), bin_vis);

#else
              gst_bin_add_many (GST_BIN (pipeline), bin_cdda, bin_output, NULL);
              gst_element_link_many (bin_cdda, bin_output, NULL);

#endif // HAVE_VISUALIZATIONS

              play_elmt = bin_cdda;
              break;
            }

            case URI::PROTOCOL_UNKNOWN:
            case URI::PROTOCOL_FTP:
            case URI::PROTOCOL_QUERY:
            case URI::PROTOCOL_TRACK:
            case URI::PROTOCOL_LASTFM:
                // Yer all are not getting played here dudes
                break;
          }
      }
    }

    void
    Play::switch_stream (Glib::ustring const& stream, Glib::ustring const& type)
    {
      request_status (PLAYSTATUS_WAITING);
      m_stream_type = type;
      property_stream_ = stream; 
      request_status (PLAYSTATUS_PLAYING);
    }

    void
    Play::request_status (Playstatus status) 
    {
      Glib::Mutex::Lock L (state_lock); //XXX: This might deadlock, it's intentionally placed to find race conditions

      switch (status)
      {
          case PLAYSTATUS_PAUSED:
            pause_stream (); 
            return;

          case PLAYSTATUS_STOPPED:
            m_stream_type = Glib::ustring(); 
            stop_stream ();
            return;

          case PLAYSTATUS_WAITING:
            m_stream_type = Glib::ustring(); 
            stop_stream (false);
            return;

          case PLAYSTATUS_PLAYING:
            play_stream ();
            return;

          default: 
            g_log (G_LOG_DOMAIN, G_LOG_LEVEL_WARNING, "%s: Unhandled Playback status request: %d", G_STRFUNC, int(status));
            return;
      }
    }

    void
    Play::seek (int position) 
    {
      m_seeking = true;
      conn_stream_position.disconnect ();

      if (!gst_element_seek_simple ( GST_ELEMENT (pipeline), GST_FORMAT_TIME,
            GstSeekFlags (GST_SEEK_FLAG_FLUSH), guint64 (position * GST_SECOND)))
        {
          m_seeking = false;
          return;
        }

      GstStateChangeReturn rstate = GstStateChangeReturn (-1); 

      int ctr = 0;
      do{
        rstate = gst_element_get_state (pipeline, NULL, NULL, guint64 (50000 * GST_MSECOND));
        ctr++;
       }
      while ((rstate == GST_STATE_CHANGE_ASYNC) && ctr <= 5);
  
      if (ctr == 5)
        m_seeking = false;
    }

    void
    Play::bins_destroy ()    
    {
    
#ifdef HAVE_VISUALIZATIONS

      if (bin_vis)
        {
          gst_element_set_state (bin_vis, GST_STATE_NULL);
          gst_object_unref (bin_vis);
          bin_vis = 0;
        }

#endif //HAVE_VISUALIZATIONS

      if (bin_file)
        {
          gst_element_set_state (bin_file, GST_STATE_NULL);
          gst_object_unref (bin_file);
          bin_file = 0;
        }

      if (bin_http)
        {
          gst_element_set_state (bin_http, GST_STATE_NULL);
          gst_object_unref (bin_http);
          bin_http = 0;
        }

      if (bin_http_mad)
        {
          gst_element_set_state (bin_http_mad, GST_STATE_NULL);
          gst_object_unref (bin_http_mad);
          bin_http_mad = 0;
        }

      if (bin_mmsx)
        {
          gst_element_set_state (bin_mmsx, GST_STATE_NULL);
          gst_object_unref (bin_mmsx);
          bin_mmsx = 0;
        }

      if (bin_cdda)
        {
          gst_element_set_state (bin_cdda, GST_STATE_NULL);
          gst_object_unref (bin_cdda);
          bin_cdda = 0;
        }

      if (bin_output)
        {
          gst_element_set_state (bin_output, GST_STATE_NULL);
          gst_object_unref (bin_output);
          bin_output = 0;
        }
    }

    void
    Play::link_pad (GstElement *element,
                    GstPad     *pad,
                    gboolean    last,
                    gpointer    data) 
    {
      gst_pad_link (pad, GST_PAD (data));
    }

#ifdef HAVE_VISUALIZATIONS
    void
    Play::bus_sync_watch (GstBus     *bus,
                          GstMessage *msg,
                          gpointer    data)
    {
      Bmp::Play *play = reinterpret_cast<Bmp::Play *>(data);

      if (gst_structure_has_name (msg->structure, "prepare-xwindow-id"))
      {
        play->signal_xoverlay_setup_.emit (GST_ELEMENT (GST_MESSAGE_SRC(msg)));
      }
    }
#endif //HAVE_VISUALIZATIONS

    gboolean
    Play::bus_watch (GstBus     *bus,
                     GstMessage *message,
                     gpointer    data)
    {
        static GstState old_state=GstState (0), new_state=GstState(0), pending_state=GstState(0);
        Bmp::Play *play = reinterpret_cast<Bmp::Play*>(data);
        
        switch (GST_MESSAGE_TYPE (message))
          {
            case GST_MESSAGE_APPLICATION:
              {
                const GstStructure *s = gst_message_get_structure (message);
                debug ("play", "Got GST_MESSAGE_APPLICATION: %s", gst_structure_get_name (s));
                  
                if (std::string(gst_structure_get_name (s)) == "lastfm-sync")
                  {
                    play->signal_lastfm_sync_.emit ();
                  }
                else
                if (std::string(gst_structure_get_name (s)) == "lastfm-status")
                  {
                    int status;
                    gst_structure_get_int (s, "status", &status);
                    play->signal_http_status_.emit (status);
                  }
                else
                if (std::string(gst_structure_get_name (s)) == "buffering")
                  {
                    double size;
                    gst_structure_get_double (s, "size", &size);
                    play->signal_buffering_.emit (size);
                  }
                else
                if (std::string(gst_structure_get_name (s)) == "buffering-done")
                  {
                  }
                break;
              }

            case GST_MESSAGE_TAG:
              {
                debug ("play", "Got GST_MESSAGE_TAG");

                if (std::string (gst_element_get_name (play->pipeline)) != std::string ("http") &&
                    std::string (gst_element_get_name (play->pipeline)) != std::string ("mmsx"))
                  break;

                if (play->property_lastfm_mode_.get_value())
                  break;

                GstTagList *tag_list;
                gst_message_parse_tag (message, &tag_list);

                if (tag_list)
                  {
                    char *title = 0;
                    if (gst_tag_list_get_string (tag_list, GST_TAG_TITLE, &title));
                    {
                        if (title) play->signal_title_.emit ((const char*)title);
                        g_free (title);
                    }
                  }
                break;
          }

          case GST_MESSAGE_STATE_CHANGED:
            {
              debug ("play", "Got GST_MESSAGE_STATE_CHANGED");

              gst_message_parse_state_changed  (message, &old_state, &new_state, &pending_state);
              if (new_state == GST_STATE_PLAYING)
                {
                  if (play->m_seeking)
                    {
                      play->signal_seek_.emit (play->property_position().get_value());
                      play->m_seeking = false;
                    }

                  if (!play->conn_stream_position.connected())
                    {
                       play->conn_stream_position = Glib::signal_timeout().connect
                                                      (sigc::mem_fun(play, &Bmp::Play::timeout_handler), 500);
                    }
                }
              play->signal_pipeline_state_.emit (new_state);
              break;
            }

            case GST_MESSAGE_ERROR:
            {
                GstElement    *element;
                GError        *error = 0;
                Glib::ustring  main, name, location, helpstring;
                char          *debug_string;
  
                if (!play->property_sane_.get_value()) break;
          
                play->property_sane_ = false;
                play->request_status (PLAYSTATUS_STOPPED);

                element = GST_ELEMENT(GST_MESSAGE_SRC (message));
                name = gst_object_get_name (GST_OBJECT(element));
                location = play->property_stream().get_value(); 

                gst_message_parse_error (message,
                                         &error,
                                         &debug_string);
        
                if (error->domain == GST_CORE_ERROR)
                  {
                            switch (error->code)
                                {
                                  case GST_CORE_ERROR_MISSING_PLUGIN:
                                  {
                                    main += "1.01  [No plugin available to play stream]";
                                    break;
                                  }
        
                                  case GST_CORE_ERROR_SEEK:
                                  {
                                    main += "1.02  [Error during Seek]";
                                    break;
                                  }
        
                                  case GST_CORE_ERROR_STATE_CHANGE:
                                  {
                                        struct StringPairs
                                        {
                                            Glib::ustring   state_pre;
                                            Glib::ustring   state_post;
                                        };
        
                                        static StringPairs string_pairs[] =
                                        {
                                            {"0",         "READY"},
                                            {"READY",     "PAUSED"},
                                            {"PAUSED",    "PLAYING"},
                                            {"PLAYING",   "PAUSED"},
                                            {"PAUSED",    "READY"},
                                            {"READY",     "0"},
                                        };
       
                                    main += "1.03  [Error occured during state change from "+string_pairs[GST_STATE_PENDING(message->src)].state_pre+" to "
                                                        +string_pairs[GST_STATE_PENDING(message->src)].state_post+"]"; 
                                    break;
                                  }
        
                                  case GST_CORE_ERROR_PAD:
                                  {
                                    main += "1.04  [Error while performing pad linking]";
                                    break;
                                  }
        
                                  case GST_CORE_ERROR_NEGOTIATION:
                                  {
                                    main =+ "1.05  [Error occured during element(s) negotiation]";
                                    break;
                                  }
        
                                    default: break;
                                }
                  }
                else if (error->domain == GST_RESOURCE_ERROR)
                  {
                            switch (error->code)
                            {
        
                                  case GST_RESOURCE_ERROR_SEEK:
                                  {
                                    main += "2.01  [Error during Seek (resource; end of stream?)]";
                                    break;
                                  }
        
                                  case GST_RESOURCE_ERROR_NOT_FOUND:
                                  case GST_RESOURCE_ERROR_OPEN_READ:
                                  case GST_RESOURCE_ERROR_OPEN_READ_WRITE:
                                  case GST_RESOURCE_ERROR_READ:
                                  {
                                    main += "2.02  [Unable to Open Resource (stream)]";
                                    helpstring = _("The requested resource was not found (The file/stream does not exist/is invalid?)");
                                    break;
                                  }

                                  case GST_RESOURCE_ERROR_SYNC:
                                  {
                                    main += "2.03  [Synchronization Error]";
                                    break;
                                  }

                                  case GST_RESOURCE_ERROR_BUSY:
                                  {
                                    main += "2.04  [Device is Busy]";

                                    if (name == "sink")
                                    {
                                      helpstring = _("The audio device seems to be busy. (Maybe another is player running?)");
                                    }

                                    break;
                                  }
      
                                  default: break;
                                }
                    }
                else if (error->domain == GST_STREAM_ERROR)
                    {
                            switch (error->code)
                                {
                                  case GST_STREAM_ERROR_FAILED:
                                  {
                                    main += "3.01  [Unable Play Stream (Perhaps missing plugin?)]";
                                    break;
                                  }

                                  case GST_STREAM_ERROR_TYPE_NOT_FOUND:
                                  {
                                    main += "3.02  [Unable to Determine Stream Type (invalid stream?)\n";
                                    helpstring = _("GStreamer was unable to determine the type of the stream and hence couldn't play it back");
                                    break;
                                  }

                                  case GST_STREAM_ERROR_WRONG_TYPE:
                                  {
                                    main += "3.03  [Element Unable to Handle this Stream Type]";
                                    break;
                                  }

                                  case GST_STREAM_ERROR_CODEC_NOT_FOUND:
                                  {
                                    main += "3.04  [No Codec installed to Handle this Stream]";
                                    helpstring = _("GStreamer is not able to play this kind of stream (Install missing plugin perhaps?)");
                                    break;
                                  }
      
                                  case GST_STREAM_ERROR_DECODE:
                                  {
                                    main += "3.05  [Error Decoding the Stream]";
                                    break;
                                  }
      
                                  case GST_STREAM_ERROR_DEMUX:
                                  {
                                    main += "3.06  [Error Demultiplexing the Stream]";
                                    break;
                                  }
      
                                  case GST_STREAM_ERROR_FORMAT:
                                  {
                                    main += "3.07  [Error Parsing the Stream Format]";
                                    break;
                                  }
    
                                default: break;
                                }
                    }
        
                g_error_free (error);

                Glib::signal_idle().connect (sigc::mem_fun (play, &Bmp::Play::reset_idle));

                Bmp::DialogGSTError *dialog = Bmp::DialogGSTError::create ();
                dialog->run (main, name, location, debug_string, helpstring);
                g_free (debug_string);
                delete dialog;

                break;
            }
        
            case GST_MESSAGE_WARNING:
              {
                GError  *error = 0;
                gchar   *debug;
        
                gst_message_parse_warning (message, &error, &debug);
                g_print ("%s WARNING: %s (%s)\n", G_STRLOC, error->message, debug);
                g_error_free (error);
                break;
              }
        
            case GST_MESSAGE_EOS:
              {
                play->signal_eos_.emit();
                break;
              }
        
            default: break;
          }
        
          return TRUE;
    }

    void
    Play::queue_underrun (GstElement *element,
                          gpointer data)
    {
      unsigned int level;
      g_object_get (G_OBJECT (element), "current-level-bytes", &level, NULL);
      //std::cerr << "Queue has underrun!!\n";
      //std::cerr << "Current fill state is: " << level << "\n"; 
    }

    void
    Play::bins_create ()
    {
      boost::optional<std::string> device, propname;
      std::string sink;
                                  
      guint64 buffer_time_value;
      bool    buffer_time_set;

      sink = mcs->key_get <std::string> ("audio", "sink");

      buffer_time_set = false;
      buffer_time_value = 200000;

      if (sink == "osssink")
        {
          buffer_time_set = true;
          buffer_time_value = mcs->key_get<int>("audio", "oss-buffer-time");
          propname = "device";
          device = mcs->key_get<std::string>( "audio", "device-oss");
        }
      else
      if (sink == "esdsink")
        {
          buffer_time_set = true;
          buffer_time_value = mcs->key_get<int>("audio", "esd-buffer-time");
          propname = "host";
          device = mcs->key_get<std::string>( "audio", "device-esd");
        }

#ifdef HAVE_ALSA
      else
      if (sink == "alsasink")
        {
          buffer_time_set = true;
          buffer_time_value = mcs->key_get<int>("audio", "alsa-buffer-time");
          propname = "device";
          device = mcs->key_get<std::string>( "audio", "device-alsa");
        }
#endif

#ifdef HAVE_SUN
      else
      if (sink == "sunaudiosink")
        {
          buffer_time_set = true;
          buffer_time_value = mcs->key_get<int>("audio", "sun-buffer-time");
          propname = "device";
          device = mcs->key_get<std::string>( "audio", "device-sun");
        }
#endif

#ifdef HAVE_HAL
      else
      if (sink == "halaudiosink")
        {
          buffer_time_set = false;
        }
#endif

      // Output bin 
      { 
        bin_output = gst_bin_new ("output");

        GstElement *resample = gst_element_factory_make ("audioresample", NAME_RESAMPLE); 
        GstElement *volume = gst_element_factory_make ("volume", NAME_VOLUME); 
        GstElement *sink_elmt = gst_element_factory_make (sink.c_str(), "sink"); 

        if (propname && device)          
          {
            g_object_set (G_OBJECT(sink_elmt), propname.get().c_str(), device.get().c_str(), NULL);
          }
        
        if (buffer_time_set)
          {
            g_object_set (G_OBJECT(sink_elmt), "buffer-time", buffer_time_value, NULL);
          }

        gst_bin_add_many (GST_BIN (bin_output), resample, volume, sink_elmt, NULL);
        gst_element_link_many (resample, volume, sink_elmt, NULL);

        GstPad *pad = gst_element_get_pad (resample, "sink");
        gst_element_add_pad (bin_output, gst_ghost_pad_new ("sink", pad));
        gst_object_unref (pad);

        gst_object_ref (bin_output);
      }

      // File bin
      {
        bin_file = gst_bin_new ("play-file");

#ifdef HAVE_VISUALIZATIONS

        GstElement *src = gst_element_factory_make ("filesrc", "src"); 
        GstElement *decoder = gst_element_factory_make ("decodebin", NAME_DECODER);  
        GstElement *tee = gst_element_factory_make ("tee", "tee");
        GstElement *queue = gst_element_factory_make ("queue", NAME_QUEUE);
        GstElement *convert = gst_element_factory_make ("audioconvert", NAME_CONVERT); 

        gst_bin_add_many (GST_BIN (bin_file), src, decoder, tee, queue, convert, NULL);

        gst_element_link_many (src, decoder, NULL);
        gst_element_link_many (tee, queue, convert, NULL);

        g_signal_connect (G_OBJECT(decoder), "new-decoded-pad",
                          G_CALLBACK (Bmp::Play::link_pad),
                          gpointer(gst_element_get_pad (tee, "sink")));

#else

        GstElement *src = gst_element_factory_make ("filesrc", "src"); 
        GstElement *decoder = gst_element_factory_make ("decodebin", NAME_DECODER);  
        GstElement *convert = gst_element_factory_make ("audioconvert", NAME_CONVERT); 

        gst_bin_add_many (GST_BIN (bin_file), src, decoder, convert, NULL);
        gst_element_link_many (src, decoder, NULL);

        g_signal_connect (G_OBJECT(decoder), "new-decoded-pad",
                          G_CALLBACK (Bmp::Play::link_pad),
                          gpointer(gst_element_get_pad (convert, "sink")));

#endif //HAVE_VISUALIZATIONS

        GstPad *pad = gst_element_get_pad (convert, "src");
        gst_element_add_pad (bin_file, gst_ghost_pad_new ("src", pad));
        gst_object_unref (pad);

        gst_object_ref (bin_file);
      }

      // HTTP bin
      {
        bin_http = gst_bin_new ("play-http");

#ifdef HAVE_VISUALIZATIONS 

        GstElement *src = gst_element_factory_make ("bmpx-neonhttpsrc", "src"); 
        GstElement *decoder = gst_element_factory_make ("decodebin", NAME_DECODER);  
        GstElement *tee = gst_element_factory_make ("tee", "tee");
        GstElement *queue1 = gst_element_factory_make ("queue", "queue1"); 
        GstElement *queue = gst_element_factory_make ("queue", NAME_QUEUE); 
        GstElement *convert = gst_element_factory_make ("audioconvert", NAME_CONVERT); 

        gst_bin_add_many (GST_BIN (bin_http), src, decoder, tee, queue1, queue, convert, NULL);
        gst_element_link_many (src, decoder, NULL);
        gst_element_link_many (tee, queue1, queue, convert, NULL);

        g_signal_connect (G_OBJECT(decoder), "new-decoded-pad",
                          G_CALLBACK (Bmp::Play::link_pad),
                          gpointer(gst_element_get_pad (tee, "sink")));

#else //HAVE_VISUALIZATIONS

        GstElement *src = gst_element_factory_make ("bmpx-neonhttpsrc", "src"); 
        GstElement *decoder = gst_element_factory_make ("decodebin", NAME_DECODER);  
        GstElement *queue1 = gst_element_factory_make ("queue", "queue1"); 
        GstElement *queue = gst_element_factory_make ("queue", NAME_QUEUE); 
        GstElement *convert = gst_element_factory_make ("audioconvert", NAME_CONVERT); 

        gst_bin_add_many (GST_BIN (bin_http), src, decoder, queue1, queue, convert, NULL);
        gst_element_link_many (src, decoder, NULL);
        gst_element_link_many (queue1, queue, convert, NULL);

        g_signal_connect (G_OBJECT(decoder), "new-decoded-pad",
                          G_CALLBACK (Bmp::Play::link_pad),
                          gpointer(gst_element_get_pad (queue1, "sink")));

#endif // HAVE_VISUALIZATIONS

        g_object_set (G_OBJECT (queue1),
                      "min-threshold-bytes",
                       (unsigned int)(8 * 1024), 
                      "max-size-bytes",
                       (unsigned int)(32 * 1024), 
                       NULL);

        g_signal_connect (G_OBJECT (queue1), "underrun", G_CALLBACK (Bmp::Play::queue_underrun), this);

        g_object_set (G_OBJECT (src),
                      "iradio-mode",
                       property_lastfm_mode_.get_value() ? FALSE : TRUE,
                      "lastfm-mode",
                       property_lastfm_mode_.get_value() ? TRUE : FALSE,
                       NULL);

        GstPad *pad = gst_element_get_pad (convert, "src");
        gst_element_add_pad (bin_http, gst_ghost_pad_new ("src", pad));
        gst_object_unref (pad);

        gst_object_ref (bin_http);
      }

      // HTTP MP3 bin
      {
        bin_http_mad = gst_bin_new ("play-http-mad");

#ifdef HAVE_VISUALIZATIONS 

        GstElement *src = gst_element_factory_make ("bmpx-neonhttpsrc", "src"); 
        GstElement *decoder = gst_element_factory_make ("mad", NAME_DECODER);  
        GstElement *tee = gst_element_factory_make ("tee", "tee");
        GstElement *queue1 = gst_element_factory_make ("queue", "queue1"); 
        GstElement *queue = gst_element_factory_make ("queue", NAME_QUEUE); 
        GstElement *convert = gst_element_factory_make ("audioconvert", NAME_CONVERT); 

        gst_bin_add_many (GST_BIN (bin_http_mad), src, decoder, tee, queue1, queue, convert, NULL);
        gst_element_link_many (src, decoder,tee, queue1, queue, convert, NULL);

#else

        GstElement *src = gst_element_factory_make ("bmpx-neonhttpsrc", "src"); 
        GstElement *decoder = gst_element_factory_make ("mad", NAME_DECODER);  
        GstElement *queue1 = gst_element_factory_make ("queue", "queue1"); 
        GstElement *queue = gst_element_factory_make ("queue", NAME_QUEUE); 
        GstElement *convert = gst_element_factory_make ("audioconvert", NAME_CONVERT); 

        gst_bin_add_many (GST_BIN (bin_http_mad), src, decoder, queue1, queue, convert, NULL);
        gst_element_link_many (src, decoder, queue1, queue, convert, NULL);

#endif // HAVE_VISUALIZATIONS

        g_object_set (G_OBJECT (queue1),
                      "min-threshold-bytes",
                       (unsigned int)(8 * 1024), 
                      "max-size-bytes",
                       (unsigned int)(32 * 1024), 
                       NULL);

        g_signal_connect (G_OBJECT (queue1), "underrun", G_CALLBACK (Bmp::Play::queue_underrun), this);

        g_object_set (G_OBJECT (src),
                      "iradio-mode",
                       property_lastfm_mode_.get_value() ? FALSE : TRUE,
                      "lastfm-mode",
                       property_lastfm_mode_.get_value() ? TRUE : FALSE,
                       NULL);

        GstPad *pad = gst_element_get_pad (convert, "src");
        gst_element_add_pad (bin_http_mad, gst_ghost_pad_new ("src", pad));
        gst_object_unref (pad);
        gst_object_ref (bin_http_mad);
      }

      // MMS* bin
      {
        bin_mmsx = gst_bin_new ("play-mmsx");
        GstElement *src = gst_element_factory_make ("mmssrc", "src"); 
        if (src)
          {

#ifdef HAVE_VISUALIZATIONS 

            GstElement *decoder = gst_element_factory_make ("decodebin", NAME_DECODER);  
            GstElement *tee = gst_element_factory_make ("tee", "tee");  
            GstElement *queue = gst_element_factory_make ("queue", "queue1");  
            GstElement *convert = gst_element_factory_make ("audioconvert", NAME_CONVERT); 

            gst_bin_add_many (GST_BIN (bin_mmsx), src, decoder, tee, queue, convert, NULL);

            gst_element_link_many (src, decoder, NULL);
            gst_element_link_many (tee, queue, convert, NULL);

            g_signal_connect (G_OBJECT(decoder), "new-decoded-pad",
                              G_CALLBACK (Bmp::Play::link_pad),
                              gpointer(gst_element_get_pad (tee, "sink")));
#else

            GstElement *decoder  = gst_element_factory_make ("decodebin", NAME_DECODER);  
            GstElement *convert  = gst_element_factory_make ("audioconvert", NAME_CONVERT); 

            gst_bin_add_many (GST_BIN (bin_mmsx), src, decoder, convert, NULL);
            gst_element_link_many (src, decoder, NULL);
            g_signal_connect (G_OBJECT(decoder), "new-decoded-pad",
                              G_CALLBACK (Bmp::Play::link_pad),
                              gpointer(gst_element_get_pad (convert, "sink")));

#endif // HAVE_VISUALIZATIONS

            GstPad *pad = gst_element_get_pad (convert, "src");
            gst_element_add_pad (bin_mmsx, gst_ghost_pad_new ("src", pad));
            gst_object_unref (pad);

            gst_object_ref (bin_mmsx);
          }
        else
          { 
            gst_object_unref (bin_mmsx);
            bin_mmsx = 0;
          }
      }

      // CDDA bin
      {
        bin_cdda = gst_bin_new ("play-cdda");

        GstElement *src = gst_element_factory_make ("cdparanoiasrc", "src"); 
        if (src)
          {

#ifdef HAVE_VISUALIZATIONS

            GstElement *tee = gst_element_factory_make ("tee", "tee"); 
            GstElement *queue = gst_element_factory_make ("queue", "queue1"); 

            gst_bin_add_many (GST_BIN (bin_cdda), src, tee, queue, NULL);
            gst_element_link_many (src, tee, queue, NULL);

            GstPad *pad = gst_element_get_pad (queue, "src");
            gst_element_add_pad (bin_cdda, gst_ghost_pad_new ("src", pad));
            gst_object_unref (pad);

#else
            gst_bin_add_many (GST_BIN (bin_cdda), src, NULL);

            GstPad *pad = gst_element_get_pad (src, "src");
            gst_element_add_pad (bin_cdda, gst_ghost_pad_new ("src", pad));
            gst_object_unref (pad);

#endif // HAVE_VISUALIZATIONS

            gst_object_ref (bin_cdda);
          }
        else
          {
            gst_object_unref (bin_cdda);
            bin_cdda = 0;
          }
      }

#ifdef HAVE_VISUALIZATIONS

      // Vis bin
      {
        bin_vis = gst_bin_new ("vis");
        GstElement *queue = gst_element_factory_make ("queue", "queue-vis"); 
        GstElement *convert = gst_element_factory_make ("audioconvert", "convert-vis"); 
        GstElement *vis = gst_element_factory_make ("libvisual_infinite", "infinite"); 
        GstElement *scale = gst_element_factory_make ("ffmpegcolorspace", "ffmpegcolorspace1"); 
        GstElement *sink = gst_element_factory_make ("xvimagesink", "vissink"); 

        g_object_set (G_OBJECT (scale), "method", int(0), NULL);

        gst_bin_add_many (GST_BIN (bin_vis), queue, convert, vis, scale, sink, NULL);
        gst_element_link_many (queue, convert, vis, scale, sink, NULL);

        GstPad *pad = gst_element_get_pad (queue, "sink");
        gst_element_add_pad (bin_vis, gst_ghost_pad_new ("sink", pad));
        gst_object_unref (pad);

        gst_object_ref (bin_vis);
      }
#endif //HAVE_VISUALIZATIONS

      pipeline = gst_pipeline_new ("");
      gst_bus_add_watch (gst_pipeline_get_bus (GST_PIPELINE (pipeline)),
                         GstBusFunc(Bmp::Play::bus_watch), gpointer(this));

#ifdef HAVE_VISUALIZATIONS

      gst_bus_enable_sync_message_emission (GST_ELEMENT_BUS(pipeline));
      g_signal_connect (GST_ELEMENT_BUS(pipeline),
                "sync-message::element",
                G_CALLBACK(Bmp::Play::bus_sync_watch),
                gpointer(this));

#endif //HAVE_VISUALIZATIONS

    }

#ifdef HAVE_VISUALIZATIONS

    void
    Play::unplug_vis ()
    {
      if (!pipeline)
      {
        g_log (G_LOG_DOMAIN, G_LOG_LEVEL_INFO, "%s: No pipeline", G_STRFUNC);
        return;
      }

      GstElement *vis  = gst_bin_get_by_name (GST_BIN (pipeline), "vis");
      if (!vis) return;
     
      GstElement *vis_queue = gst_bin_get_by_name (GST_BIN (vis), "queue-vis"); 
      GstPad *pad = gst_element_get_pad (GST_ELEMENT (vis_queue), "sink"); 

      vis_buffer_probe_handler_id =
          gst_pad_add_buffer_probe (pad, G_CALLBACK(drop_data), NULL);
    }

    void
    Play::replug_vis ()
    {
      if (!pipeline)
      {
        g_log (G_LOG_DOMAIN, G_LOG_LEVEL_INFO, "%s: No pipeline", G_STRFUNC);
        return;
      }

      if (!vis_buffer_probe_handler_id) return;

      GstElement *vis  = gst_bin_get_by_name (GST_BIN (pipeline), "vis");
      if (!vis) return;

      GstElement *vis_queue = gst_bin_get_by_name (GST_BIN (vis), "queue-vis"); 
      GstPad *pad = gst_element_get_pad (GST_ELEMENT (vis_queue), "sink"); 

      gst_pad_remove_buffer_probe (pad, vis_buffer_probe_handler_id);
      vis_buffer_probe_handler_id = 0;
    }

    bool
    Play::vis_playing ()
    {
      return (GST_STATE (bin_vis) == GST_STATE_PLAYING);
    }

#endif // HAVE_VISUALIZATIONS

    bool
    Play::reset_idle ()
    {
      reset ();
      return false;
    }

    void
    Play::reset ()
    {
      GstElement *element;
      Glib::ustring sink;

      property_status_ = PLAYSTATUS_STOPPED;

      if (pipeline)
        {
          gst_element_set_state (pipeline, GST_STATE_NULL);
          gst_object_unref (pipeline);
          pipeline = 0; 
        }
    
      bins_destroy ();

      // Check if the sink can be created
      sink = mcs->key_get<std::string>( "audio", "sink");
      element = gst_element_factory_make (sink.c_str(), "sinktest0");

      if (!element)
        {
          Gtk::MessageDialog dialog (_("BMP was not able to initialize the playback system. Please check the audio "
                                       "settings in the Preferences panel."),
                                       false, Gtk::MESSAGE_ERROR, Gtk::BUTTONS_OK, true);
          dialog.run();
          dialog.hide();
          property_sane_ = false;
          return;
        }

      //FIXME: Possibly even more sanity checks
      property_sane_ = true; 
      bins_create ();
    }

    ///////////////////////////////////////////////
    /// Glib::Object Properties
    ///////////////////////////////////////////////

    Glib::PropertyProxy<bool>
    Play::property_lastfm_mode()
      {
        return property_lastfm_mode_.get_proxy();
      }

    Glib::PropertyProxy<Glib::ustring>
    Play::property_stream()
      {
        return property_stream_.get_proxy();
      }

    Glib::PropertyProxy<int>
    Play::property_volume()
      {
        return property_volume_.get_proxy();
      }

    /// RO Proxies

    Glib::PropertyProxy_ReadOnly<int>
    Play::property_status() const
      {
        return Glib::PropertyProxy_ReadOnly<int>(const_cast<const Bmp::Play *>(this), "playstatus");
      }

    Glib::PropertyProxy_ReadOnly<int>
    Play::property_position() const
      {
        GstQuery       *query;
        GstFormat       format = GST_FORMAT_TIME;
        gint64          length_in_nanoseconds;

        query = gst_query_new_position (format);
        gst_element_query (pipeline, query);
        gst_query_parse_position (query, &format, &length_in_nanoseconds);
        const int value = length_in_nanoseconds / GST_SECOND;
        gst_query_unref (query);

        g_object_set (G_OBJECT (gobj()), "position", int(value), NULL);
        return Glib::PropertyProxy_ReadOnly<int>(const_cast<const Bmp::Play *>(this), "position");
      }

    Glib::PropertyProxy_ReadOnly<bool>
    Play::property_sane() const
      {
        return Glib::PropertyProxy_ReadOnly<bool>(const_cast<const Bmp::Play *>(this), "sane");
      }

    Glib::PropertyProxy_ReadOnly<int>
    Play::property_length() const
      {
        GstQuery       *query;
        GstFormat       format = GST_FORMAT_TIME;
        gint64          length_in_nanoseconds;

        query = gst_query_new_duration (format);
        gst_element_query (pipeline, query);
        gst_query_parse_duration (query, &format, &length_in_nanoseconds);
        const int value = length_in_nanoseconds / GST_SECOND;
        gst_query_unref (query);

        g_object_set (G_OBJECT (gobj()), "length", int(value), NULL);
        return Glib::PropertyProxy_ReadOnly<int>(const_cast<const Bmp::Play *>(this), "length");
      }

    // Signals
#ifdef HAVE_VISUALIZATIONS
    Play::SignalXOverlaySetup&
    Play::signal_xoverlay_setup ()
    {
      return signal_xoverlay_setup_;
    }
#endif //HAVE_VISUALIZATIONS

    Play::SignalPipelineState&
    Play::signal_pipeline_state ()
    {
      return signal_pipeline_state_;
    }

    Play::SignalTitle&
    Play::signal_title ()
    {
      return signal_title_;
    }

    Play::SignalEos&
    Play::signal_eos ()
    { 
      return signal_eos_;
    }

    Play::SignalSeek&
    Play::signal_seek ()
    {
      return signal_seek_;
    }

    Play::SignalPosition&
    Play::signal_position ()
    {
      return signal_position_;
    }

    Play::SignalHttpStatus&
    Play::signal_http_status ()
    {
      return signal_http_status_;
    }

    Play::SignalLastFMSync&
    Play::signal_lastfm_sync ()
    {
      return signal_lastfm_sync_;
    }

    Play::SignalBuffering&
    Play::signal_buffering ()
    {
      return signal_buffering_;
    }
};
