//  BMPx - The Dumb Music Player
//  Copyright (C) 2005 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License Version 2
//  as published by the Free Software Foundation.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifndef BMP_PLAY_HH
#define BMP_PLAY_HH

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <cstring>
#include <glibmm.h>

#ifndef BMP_PLUGIN_BUILD
#  include <gst/gst.h>
#  include <gst/gstelement.h>
#  include <gst/interfaces/mixer.h>
#  include <gst/interfaces/mixertrack.h>
#  include <gst/interfaces/mixeroptions.h>
#  include "uri++.hh"
#  include "util.hh"
#endif

namespace Bmp
{
  enum Playstatus
  {
    PLAYSTATUS_NONE    = 0,
    PLAYSTATUS_STOPPED = 1 << 0,
    PLAYSTATUS_PLAYING = 1 << 1,
    PLAYSTATUS_PAUSED  = 1 << 2,
    PLAYSTATUS_SEEKING = 1 << 3,
    PLAYSTATUS_WAITING = 1 << 4
  };

};

namespace Bmp
{
    /** Playback Engine
     *
     * Bmp::Play is the playback engine of BMP. It is based on GStreamer 0.10
     * using a rather simple design. (http://www.gstreamer.net)
     *
     */
    class Play  : public Glib::Object
    {

      public:

        Play ();
        ~Play ();

        void request_status (Playstatus status);
        void switch_stream (Glib::ustring const& stream, Glib::ustring const& type = Glib::ustring());

        /** Determine whether a stream/file specified by uri is a playable file or not 
         *
         * @param uri The URI to determine whether a stream is an audio file or not 
         *
         */
        bool is_audio_file (std::string const& str);

#ifndef BMP_PLUGIN_BUILD

        void reset ();
        void seek (int position);

        Glib::PropertyProxy<bool> property_lastfm_mode();
        Glib::PropertyProxy<Glib::ustring> property_stream();
        Glib::PropertyProxy<int> property_volume();

        Glib::PropertyProxy_ReadOnly<int>   property_status()   const;
        Glib::PropertyProxy_ReadOnly<bool>  property_sane()     const;
        Glib::PropertyProxy_ReadOnly<int>   property_position() const;
        Glib::PropertyProxy_ReadOnly<int>   property_length()   const;

        typedef sigc::signal<void, const char*>	SignalTitle;
        typedef sigc::signal<void> SignalEos;
        typedef sigc::signal<void, int>	SignalSeek;
        typedef sigc::signal<void, int>	SignalPosition;
        typedef sigc::signal<void, int> SignalHttpStatus;
        typedef sigc::signal<void> SignalLastFMSync;
        typedef sigc::signal<void, double> SignalBuffering;
        typedef sigc::signal<void, GstState> SignalPipelineState;

#ifdef HAVE_VISUALIZATIONS
        typedef sigc::signal<void, GstElement *> SignalXOverlaySetup;

        void unplug_vis ();
        void replug_vis ();
        bool vis_playing ();

        gulong vis_buffer_probe_handler_id;

#endif //HAVE_VISUALIZATIONS

        /** Signal emitted when a stream title is incoming 
         *
         */
        SignalPipelineState& signal_pipeline_state();

        /** Signal emitted when a stream title is incoming 
         *
         */
        SignalTitle& signal_title();

        /** Signal emitted on end of stream
         *
         */
        SignalEos& signal_eos();

        /** Signal emitted on a successful seek event
         *
         */
        SignalSeek& signal_seek();

        /** Signal emitted on stream position change
         *
         */
        SignalPosition& signal_position();

        /** Signal emitted on stream position change
         *
         */
        SignalHttpStatus& signal_http_status();

        /** Signal emitted on stream position change
         *
         */
        SignalLastFMSync& signal_lastfm_sync();

        /** Signal emitted on stream position change
         *
         */
        SignalBuffering& signal_buffering();

#ifdef HAVE_VISUALIZATIONS
        /** Signal emitted upon request to set an xwindow id for the vis xoverlay
         *
         */
        SignalXOverlaySetup& signal_xoverlay_setup();
#endif

      private:

        Glib::Mutex state_lock;

        SignalPipelineState         signal_pipeline_state_;
        SignalTitle                 signal_title_;
        SignalEos                   signal_eos_;
        SignalSeek                  signal_seek_;
        SignalPosition              signal_position_;
        SignalHttpStatus            signal_http_status_;
        SignalLastFMSync            signal_lastfm_sync_;
        SignalBuffering             signal_buffering_;

#ifdef HAVE_VISUALIZATIONS
        SignalXOverlaySetup         signal_xoverlay_setup_;
#endif //HAVE_VISUALIZATIONS

        bool reset_idle ();

        static void
        queue_underrun (GstElement *element,
                        gpointer data);

        static void
        http_status  (GstElement *element,
                      int status,
                      gpointer data);

        static void
        lastfm_sync  (GstElement *element,
                      gpointer data);

        static void
        link_pad  (GstElement *element,
                   GstPad     *pad,
                   gboolean    last,
                   gpointer    data);

        static gboolean
        bus_watch (GstBus     *bus,
                   GstMessage *message,
                   gpointer    data);

#ifdef HAVE_VISUALIZATIONS
        static void
        bus_sync_watch (GstBus     *bus,
                        GstMessage *msg,
                        gpointer    data);

#endif //HAVE_VISUALIZATIONS

        static gboolean
        foreach_structure (GQuark	       field_id,
                           const GValue	*value,
                           gpointer	     data);

        // Properties
        Glib::Property<Glib::ustring> property_stream_;
        Glib::Property<bool> property_lastfm_mode_;
        Glib::Property<int>  property_volume_;
        Glib::Property<int>  property_position_;
        Glib::Property<int>  property_status_;
        Glib::Property<bool> property_sane_;
        Glib::Property<int>  property_length_;

        GstElement           *pipeline;

        GstElement	         *bin_file;
        GstElement	         *bin_mmsx;
        GstElement	         *bin_cdda;
        GstElement	         *bin_output;

        GstElement	         *bin_http;
        GstElement	         *bin_http_mad;

#ifdef HAVE_VISUALIZATIONS
        GstElement	         *bin_vis;
#endif //HAVE_VISUALIZATIONS
  
        sigc::connection      conn_stream_position;
        bool                  m_seeking;

        std::map<std::string, bool> extensions;

        bool sane;

        void bins_destroy ();
        void bins_create ();
        void stop_stream (bool set_state = true); //Whether to change the playstatus property, for PLAYSTATUS_WAITING we don't want that
        void play_stream ();
        void pause_stream ();

        void on_stream_changed ();
        void on_lastfm_mode_changed ();
        void on_volume_changed ();
        bool timeout_handler ();

        Glib::ustring m_stream_type;
        GstElement * http_elmt;
        GstElement * play_elmt;

        void pipeline_preconfigure (std::string const& name);
#endif
    };
}

#endif // BMP_PLAY_HH
