//  BMPx - The Dumb Music Player
//  Copyright (C) 2005 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <glibmm/i18n.h>
#include <glib/gstdio.h>
#include <gtkmm.h>
#include <cstdlib>
#include <cstring>
#include <iostream>
#include <sstream>
#include <fstream>

#include <boost/format.hpp>
#include <boost/optional.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/tuple/tuple.hpp>
#include <boost/tuple/tuple_comparison.hpp>

#include <mcs/mcs.h>

#include "main.hh"
#include "paths.hh"
#include "debug.hh"

#include "xds.hh"
#include "uri++.hh"
#include "mbxml.hh"
#include "stock.hh"
#include "util.hh"
#include "util_file.hh"

#include "x_library.hh"
#include "x_play.hh"
#include "x_vfs.hh"
#include "x_mcsbind.hh"

#ifdef HAVE_HAL
#  include "x_hal.hh"
#endif //HAVE_HAL

#include "library-ui-modify.hh"
#include "library-ui-import.hh"
#include "library-ui-relocate-album.hh"
#include "ui_toolbox.hh"
#include "ui-part-albums.hh"

#include "dialog-filelist.hh"
#include "dialog-progress.hh"

#include "taskdialog.hh"

#define KEY_PRESS_MAX_TIME 0.2
#define BOOST_ENABLE_ASSERT_HANDLER 1

namespace boost
{
  void assertion_failed(char const * expr, char const * function, char const * file, long line)
  {
    g_message ("expr: %s", expr);
    g_message ("function: %s", function);
    g_message ("file: %s", file);
    g_message ("line: %ld", line);
  }
}

using namespace Gtk;
using namespace Gdk;
using namespace Glib;

using namespace boost;
using namespace std;

using namespace Bmp::DB;
using namespace Bmp::Library;
using namespace Bmp::MusicBrainzXML;
using namespace Bmp::Util;
using namespace Bmp::VFS;

namespace
{

#define ACTION_IMPORT_MUSIC       "library-action-import-music"
#define ACTION_IMPORT_RETAG       "library-action-retag-import"

#define ACTION_EDIT_ALBUMS        "library-action-edit-album"
#define ACTION_REMOVE_ALBUMS      "library-action-remove-albums"
#define ACTION_MOVE_ALBUM         "library-action-relocate-album"

#define ACTION_NEW_APPROVE_ALL    "library-action-approve-items"
#define ACTION_APPROVE_SELECTED   "library-action-approve-selected"
#define ACTION_NEW_DROP_ALL       "library-action-drop-all"

#define ACTION_VIEW_SIZE_NORMAL   "library-action-view-size-normal"
#define ACTION_VIEW_SIZE_SMALL    "library-action-view-size-small"

#define ACTION_SHOW_ALBUM_LIST    "library-action-show-album-list"
//#define ACTION_SHOW_ARTIST_LIST   "library-action-show-artist-list"

#define ACTION_CONTINUOUS_PLAY    "library-action-continuous-play"
#define ACTION_FETCH_ALL_COVERS   "library-action-fetch-all-covers"

#define ACTION_UNSELECT_ALL       "library-action-unselect-all"

#define ACTION_VIEW               "library-action-view"

// Popup Menu for Track List

#define ACTION_REMOVE_TRACKS      "library-action-remove-tracks"
#define ACTION_ENQUEUE_TRACKS     "library-action-enqueue-tracks"
#define ACTION_MENU_UI_PART_ALBUMS "menu-ui-part-albums"

    const char *albums_albumlist_menu =
    "<ui>"
    ""
    "<menubar name='popup-albums-albumlist'>"
    "   <menu action='dummy' name='menu-albums-albumlist'>"
    ""
    "     <menuitem action='" ACTION_EDIT_ALBUMS "'/>"
    "     <menuitem action='" ACTION_REMOVE_ALBUMS "'/>"
    "     <menuitem action='" ACTION_MOVE_ALBUM "'/>"
    ""
    "     <separator name='sep1'/>"
    ""
    "     <menuitem action='" ACTION_APPROVE_SELECTED "'/>"
    ""
    "     <separator name='sep2'/>"
    ""
    "     <menuitem action='" ACTION_UNSELECT_ALL "'/>"
    ""
    "   </menu>"
    "</menubar>"
    ""
    "</ui>";

    const char *albums_tracklist_menu =
    "<ui>"
    ""
    "<menubar name='popup-albums-tracklist'>"
    ""
    "   <menu action='dummy' name='menu-albums-tracklist'>"
    ""
    "     <menuitem action='" ACTION_ENQUEUE_TRACKS "'/>"
    "     <separator name='sep5'/>"
    "     <menuitem action='" ACTION_REMOVE_TRACKS "'/>"
    ""
    "   </menu>"
    ""
    "</menubar>"
    ""
    "</ui>";

    const char *albums_menu =
    "<ui>"
    ""
    "<menubar name='MenuBarMain'>"
    ""
    "   <menu action='" ACTION_MENU_UI_PART_ALBUMS "'>"
    ""
    "     <menu action='" ACTION_VIEW "'>"
    "         <menuitem action='" ACTION_SHOW_ALBUM_LIST "'/>"
/*
    "         <menuitem action='" ACTION_SHOW_ARTIST_LIST "'/>"
*/
    "             <separator name='sep413'/>"
    "         <menuitem action='" ACTION_VIEW_SIZE_NORMAL "'/>"
    "         <menuitem action='" ACTION_VIEW_SIZE_SMALL "'/>"
    "     </menu>"
    ""
    "     <menuitem action='" ACTION_CONTINUOUS_PLAY "'/>"
    ""
    "       <separator name='sep402'/>"
    ""
    "     <menuitem action='" ACTION_IMPORT_MUSIC "'/>"
    "     <menuitem action='" ACTION_IMPORT_RETAG "'/>"
    ""
    "       <separator name='sep400'/>"
    ""
    "     <menuitem action='" ACTION_FETCH_ALL_COVERS "'/>"
    ""
    "       <separator name='sep401'/>"
    ""
    "     <menuitem action='" ACTION_NEW_APPROVE_ALL "'/>"
    "     <menuitem action='" ACTION_NEW_DROP_ALL "'/>"
    ""
    "   </menu>"
    "</menubar>"
    ""
    "</ui>";

    enum Page
    {
      PAGE_VIEW,
      PAGE_PROCESSING,
    };

    static boost::format format_int           ("%d");
    static boost::format format_uint          ("%u");
    static boost::format format_progress_text ("%d / %d");

    void clear_entry (Gtk::Entry *entry)    
    {
      entry->set_text ("");
    }

    Gtk::Widget * get_popup (Glib::RefPtr<Gtk::UIManager> ui_manager,
                             Glib::ustring const&         path) 
    {
      Gtk::Widget *menu_item;
      menu_item = ui_manager->get_widget (path);
      if ( menu_item )
          return dynamic_cast<Gtk::MenuItem*>(menu_item)->get_submenu ();
      else
          return 0;
    }

    void
    unselect_all (Glib::RefPtr<Gtk::TreeView::Selection> selection)
    {
      selection->unselect_all ();
    }

    void print_album (Album const& a)
    {
      static boost::format fint ("%d");

      std::cerr << "Artist: " << (a.artist ? a.artist.get() : "(unset)");
      std::cerr << "\nAlbum: " << (a.album ? a.album.get() : "(unset)");
      std::cerr << "\nASIN: " << (a.asin ? a.asin.get() : "(unset)");
      std::cerr << "\nRelease Id: " << (a.release_id ? a.release_id.get() : "(unset)");
      std::cerr << "\nArtist Id: " << (a.artist_id ? a.artist_id.get() : "(unset)");
      std::cerr << "\nNew Item: " << (a.new_item ? (a.new_item.get() ? "true" : "false") : "(unset)");
      std::cerr << "\nTracks: " << (a.tracks ? (fint % a.tracks.get()).str() : "(unset)"); 
      std::cerr << "\nSortname: " << (a.sortname ? a.sortname.get() : "(unset)");
    }
}

namespace Bmp
{
      AlbumView::AlbumView (BaseObjectType                  *cobject,
                            RefPtr<Gnome::Glade::Xml> const& xml)

            : TreeView                  (cobject),
              m_viewsize                (ViewSize (mcs->key_get<int>("bmp","library-view-size"))),
              m_selection_list_modified (false),
              m_selection_change_block  (false),
              m_show_only_new           (false),
              m_show_only_sel           (false),
              m_filter_artist_only      (false),
              m_guid (0)
      {

#if GTK_CHECK_VERSION(2, 10, 0)
          gtk_tree_view_set_rubber_banding ( GTK_TREE_VIEW (gobj()), TRUE);
#endif

          m_emblem_new  = Gdk::Pixbuf::create_from_file
                            (Glib::build_filename (BMP_IMAGE_DIR_LIBRARY, "new.png"));

          m_disc_normal = Gdk::Pixbuf::create_from_file
                            (Glib::build_filename (BMP_IMAGE_DIR,
                              BMP_COVER_IMAGE_DEFAULT))->scale_simple (70, 70, Gdk::INTERP_BILINEAR);

          m_disc_small  = Gdk::Pixbuf::create_from_file
                            (Glib::build_filename (BMP_IMAGE_DIR,
                              BMP_COVER_IMAGE_DEFAULT))->scale_simple (32, 32, Gdk::INTERP_BILINEAR);

          enum Renderer
          {
            R_TEXT,
            R_PIXBUF,
          };

         struct { 
              char    *title;
              double   xalign;
              int      width;
              Renderer renderer;
          } headers_albums[] = {
              { _("New"),        0.5, -1,  R_PIXBUF },
              { _("Cover"),      0.5, -1,  R_PIXBUF },
              { _("Album"),      0.0, 250, R_TEXT   },
          };

          for (unsigned int n = 0; n < G_N_ELEMENTS(headers_albums); ++n)
          {
            switch (n)
            {
              case 0:
              {
                Gtk::CellRendererPixbuf *cell = manage (new CellRendererPixbuf());
                cell->property_xpad() = 2; 
                cell->property_ypad() = 2; 
                cell->property_yalign() = 0.0;
                cell->property_xalign() = 0.0; 
                TreeViewColumn *column = manage (new Gtk::TreeViewColumn (headers_albums[n].title, *cell));
                column->set_cell_data_func (*cell, sigc::mem_fun
                          (this, &Bmp::AlbumView::album_new_cell_data_func));
                column->set_resizable (false);
                column->set_expand (false);
                column->set_sizing (TREE_VIEW_COLUMN_FIXED);
                column->set_fixed_width (22);
                append_column (*column);
                break;
              }

              case 1:
              {
                Gtk::CellRendererPixbuf *cell = manage (new CellRendererPixbuf());
                cell->property_xalign() = headers_albums[n].xalign;
                cell->property_xpad() = 0; 
                cell->property_ypad() = 2; 
                TreeViewColumn *column = manage (new TreeViewColumn (headers_albums[n].title, *cell));
                column->set_cell_data_func (*cell, sigc::mem_fun
                          (this, &Bmp::AlbumView::album_cover_cell_data_func));
                column->set_resizable (false);
                column->set_expand (false);
                column->set_sizing (TREE_VIEW_COLUMN_FIXED);
                column->set_fixed_width (m_viewsize?50:76);
                append_column (*column);
                break;
              }

              case 2:
              {
                m_cell_album_text = manage (new CellRendererText());
                m_cell_album_text->property_xalign() = 0.0; 
                m_cell_album_text->property_yalign() = 0.5; 
                m_cell_album_text->property_xpad() = 12; 
                m_cell_album_text->property_size() = (m_viewsize?8:14) * PANGO_SCALE;
                TreeViewColumn *column  = manage (new TreeViewColumn (headers_albums[n].title, *m_cell_album_text));
                column->set_cell_data_func (*m_cell_album_text, sigc::mem_fun
                          (this, &Bmp::AlbumView::album_title_cell_data_func));
                column->set_resizable (false);
                column->set_expand (false);
                column->set_sizing (TREE_VIEW_COLUMN_FIXED);
                column->set_fixed_width(headers_albums[n].width);
                append_column (*column);
                break;
              }
            }
          }

          m_store_base = ListStore::create (m_album);
          m_store_base->set_default_sort_func
                (sigc::mem_fun (*this, &Bmp::AlbumView::default_sort_func));
          m_store_base->set_sort_column (-1, SORT_ASCENDING);

          m_store_filter = TreeModelFilter::create (m_store_base);
          m_store_filter->set_visible_func
                (sigc::mem_fun (*this, &Bmp::AlbumView::visible_func));

          get_selection()->set_mode (SELECTION_MULTIPLE);
          get_selection()->set_select_function
                (sigc::mem_fun (*this, &Bmp::AlbumView::record_selection_change));

          set_model (m_store_filter);
      }

      bool
      AlbumView::has_new_items ()
      {
        return (m_new_items_map.size() != 0);
      }

      void
      AlbumView::save_context ()
      {
        if (m_selection_list_modified)
        {
          m_selection_list_albums_saved = m_selection_list_albums;
          m_selection_list_modified = false;
        }
      }

      void 
      AlbumView::restore_context ()
      {
        m_sort_string.clear();
        m_store_filter->refilter ();
        get_selection()->unselect_all ();

        m_selection_list_albums = m_selection_list_albums_saved;
        restore_selection ();
      }

      void
      AlbumView::set_viewsize (ViewSize viewsize)
      {
        m_viewsize = viewsize;
        m_cell_album_text->property_size() = (m_viewsize?8:14) * PANGO_SCALE;
        TreeViewColumn *column = get_column (1);
        std::vector<Gtk::CellRenderer*> renderers = column->get_cell_renderers();
        renderers[0]->set_fixed_size (m_viewsize?50:76, m_viewsize?50:76);
        column->set_fixed_width (m_viewsize?50:76);
        column->queue_resize ();
        queue_draw ();
      }

      void
      AlbumView::new_approve_one (guint64 uid)
      {
        using namespace Gtk;
        using namespace Bmp::Library;

        NewItemsMap::iterator i (m_new_items_map.find (uid));
        if (i != m_new_items_map.end())
          {
            TreeModel::iterator const& m_iter (m_store_base->get_iter (i->second.get_path()));
            Album a ((*m_iter)[m_album.album]);
            a.new_item = false;
            (*m_iter)[m_album.album] = a;
            m_new_items_map.erase (i);
          }
      }

      void
      AlbumView::new_remove_one (guint64 uid)
      {
        using namespace Gtk;
        NewItemsMap::iterator i (m_new_items_map.find (uid));
        if (i != m_new_items_map.end())
          {
            TreeModel::iterator const& m_iter = m_store_base->get_iter (i->second.get_path());
            TreeModel::Path const& m_path = m_store_filter->convert_child_path_to_path
              (m_store_base->get_path (m_iter)); 
            if (m_path.gobj() && get_selection()->is_selected (m_path)) get_selection()->unselect (m_path);
            m_new_items_map.erase (i);
          }
      }

      void
      AlbumView::new_approve_all ()
      {
        using namespace Gtk;
        for (NewItemsMap::const_iterator i = m_new_items_map.begin(); i != m_new_items_map.end(); ++i)
          {
            TreeModel::iterator const& m_iter = m_store_base->get_iter (i->second.get_path());
            Library::Album a ((*m_iter)[m_album.album]);
            a.new_item = false;
            (*m_iter)[m_album.album] = a;

          }
        m_new_items_map.clear();
      }

      void
      AlbumView::new_drop_all ()
      {
        for (NewItemsMap::const_iterator i = m_new_items_map.begin(); i != m_new_items_map.end(); ++i)
        {
          TreeModel::iterator const& m_iter = m_store_base->get_iter (i->second.get_path());
          TreeModel::Path const& m_path = m_store_filter->convert_child_path_to_path (m_store_base->get_path (m_iter)); 
          if (m_path.gobj() && get_selection()->is_selected (m_path)) get_selection()->unselect (m_path);
        
          Bmp::Library::Album const& a ((*m_iter)[m_album.album]);
          m_album_map.erase (a.tpl);

          m_store_base->erase (m_iter);
        }
        m_new_items_map.clear();
      }

      void
      AlbumView::restore_selection ()
      {
        m_selection_change_block = true;
        for (SelectionList::iterator p  = m_selection_list_albums.begin();
                                     p != m_selection_list_albums.end(); ++p)
          {
            TreeModel::Path const& c_path = p->second.get_path();
            TreeModel::Path const& m_path = m_store_filter->convert_child_path_to_path (c_path);
            if (m_path.gobj() && !get_selection()->is_selected (m_path)) get_selection()->select (m_path);
          }
        m_selection_change_block = false;
      }

      void
      AlbumView::set_filtering (bool show_only_new, bool show_only_sel)
      {
        m_show_only_new = show_only_new;
        m_show_only_sel = show_only_sel;
        m_store_filter->refilter ();      
        restore_selection ();
      }

      void
      AlbumView::set_sorting (ustring const& sort_text)
      {
        m_sort_string = sort_text;
        m_store_filter->refilter ();      
        restore_selection ();
      }

/*
      void
      AlbumView::set_sorting (ustring const& sort_text, bool artist_only)
      {
        m_sort_string         = sort_text;
        m_filter_artist_only  = artist_only;
        m_store_filter->refilter ();      
        restore_selection ();
      }
*/

      bool
      AlbumView::visible_func (TreeModel::const_iterator const& c_iter)
      {

        Album const& a = (*c_iter)[m_album.album];
        unsigned int guid = (*c_iter)[m_album.guid];

        if (m_show_only_sel)
        {
            bool predicate = false;
            for (SelectionList::iterator  i  = m_selection_list_albums.begin() ; 
                                          i != m_selection_list_albums.end() ; ++i)
            {
              if (i->first == guid)
              {
                  predicate = true;
                  break;
              }
            }

            if (!predicate)
              return false;
        }
        else if (m_show_only_new)
        {
          if (a.new_item) 
            {
              if (!a.new_item.get())
                return false;
            }
        }

/*
        if (m_filter_artist_only)
          {
            if (a.artist)
              {
                return (m_sort_string.empty() || (a.artist.get() == m_sort_string));
              }
          }
        else
          {
*/

            if (a.artist && a.album)
              return (m_sort_string.empty() || match_keys (a.artist.get() + " " + a.album.get(), m_sort_string));

/*
          }
*/

        return false;
      }

      int
      AlbumView::default_sort_func (TreeModel::iterator const& iter_a, TreeModel::iterator const& iter_b)
      {
        Library::Album const& a ((*iter_a)[m_album.album]);
        Library::Album const& b ((*iter_b)[m_album.album]);

        return a.key.compare(b.key);
      }

      void
      AlbumView::album_new_cell_data_func (Gtk::CellRenderer* cell_, Gtk::TreeModel::iterator const& m_iter)
      {
        using namespace Gtk;
        using namespace Gdk;
        using namespace Glib;

        CellRendererPixbuf *cell = dynamic_cast<CellRendererPixbuf *>(cell_);
        Album const& a = ((*m_iter)[m_album.album]);
        if (a.new_item && a.new_item.get())
            cell->property_pixbuf() = m_emblem_new;
        else
            cell->property_pixbuf() = Glib::RefPtr<Gdk::Pixbuf>(0); 
      }

      void
      AlbumView::album_cover_cell_data_func (Gtk::CellRenderer* cell_, Gtk::TreeModel::iterator const& m_iter)
      {
        Gtk::CellRendererPixbuf * cell = dynamic_cast<Gtk::CellRendererPixbuf *>(cell_);
        Glib::RefPtr<Gdk::Pixbuf> const& p ((*m_iter)[m_album.cover]);
        cell->property_pixbuf() = m_viewsize? p->scale_simple (44, 44, Gdk::INTERP_BILINEAR) : p;
      }

      void
      AlbumView::album_title_cell_data_func (Gtk::CellRenderer* cell_, Gtk::TreeModel::iterator const& m_iter)
      {
        CellRendererText *cell = dynamic_cast<CellRendererText *>(cell_);
        Album const& a = ((*m_iter)[m_album.album]);

        ustring title;

        title . append ("<b>")
              . append (Markup::escape_text (a.sortname? a.sortname.get() : a.artist.get()))
              . append ("</b>\n")
              . append ("<i>") 
              . append (Markup::escape_text (a.album.get()))
              . append ("</i>")
              . append ("\n<small>(")
              . append ((format_int % a.tracks.get()).str())
              . append ((a.tracks.get() > 1)? _(" Tracks") : _(" Track"))
              . append (")</small>");

        cell->property_markup() = title;
      }

      bool
      AlbumView::record_selection_change  (RefPtr<TreeModel> const& model,
                                           TreeModel::Path const&   path,
                                           bool                     selected)
      {
        if (m_selection_change_block)
          return true;

        RefPtr<TreeModelFilter> const& model_filter = RefPtr<TreeModelFilter>::cast_static (model);
        RefPtr<TreeModel> const& model_child = model_filter->get_model(); 
        TreeModel::Path const& c_path = model_filter->convert_path_to_child_path (path); 
        TreeModel::iterator const& m_iter = model_child->get_iter (c_path);

        unsigned int guid = (*m_iter)[m_album.guid];

        if (!selected)
          {
            m_selection_list_albums.push_back (make_pair (guid,
                                                          TreeModel::RowReference (model_child, c_path)));
          }
        else
          {
            for (SelectionList::iterator i  = m_selection_list_albums.begin();
                                         i != m_selection_list_albums.end(); ++i)
              {
                if (i->first == guid)
                  {
                    m_selection_list_albums.erase (i);
                    break;
                  }
              }
          }
        m_selection_list_modified = true;
        return true;
      }

      bool
      AlbumView::matching_album (Album const& a)
      {
        AlbumMap::iterator i = m_album_map.find (a.tpl);
        return (i != m_album_map.end()); 
      }

      bool
      AlbumView::matching_album (Album const& a, TreeModel::iterator & m_iter)
      {
        AlbumMap::iterator i = m_album_map.find (a.tpl);
        bool exists (i != m_album_map.end()); 
        if (exists)
          m_iter = i->second;
        return exists;
      }

      void
      AlbumView::get_cover (Gtk::TreeModel::iterator const& m_iter, oustring const& asin, bool only_cached)
      {
        if (!asin)
          {
            (*m_iter)[m_album.cover] = m_disc_normal;
            return;
          }

        try{
            RefPtr<Pixbuf> cover;
            Amazon::fetch( asin.get(), cover, only_cached );
            (*m_iter)[m_album.cover] = cover->scale_simple (70, 70, INTERP_BILINEAR);
          }
        catch (...)
          {
            (*m_iter)[m_album.cover] = m_disc_normal;
          }
      }

      void
      AlbumView::update_new ()
      {
        VRows vector;
        library->new_albums (vector);
        for (VRows::iterator i = vector.begin(); i != vector.end(); ++i)
        {
          Album a (*i);
          if (!matching_album (a))
            insert (a, a.new_item.get());
          while (gtk_events_pending ())
            gtk_main_iteration ();
        }
      }

      void
      AlbumView::update_all (bool get_covers)
      {
        unset_model ();

        m_selection_list_albums.clear();
        m_store_base->clear ();

        VRows vector;
        library->all_albums (vector);

        for (VRows::iterator i = vector.begin(); i != vector.end(); ++i)
        {
          Album a (*i);
          insert (a, a.new_item.get());              
          while (gtk_events_pending ())
            gtk_main_iteration ();
        }
        set_model (m_store_filter);
      }

      bool
      AlbumView::relocate ()
      {
        PathList const& paths = get_selection()->get_selected_rows ();

        if (paths.size () == 1)
          {
            TreeModel::iterator const& m_iter = m_store_filter->get_iter (paths[0]);
            shared_ptr<LibraryUiRelocateAlbum> dialog (LibraryUiRelocateAlbum::create ());

            Album const& a ((*m_iter)[m_album.album]);

            int response = dialog->run (a);
            dialog->hide ();

            if (response == RESPONSE_OK)
              {
                string path_new;

                ustring uri = dialog->m_widget_fcb_move->get_current_folder_uri ();

                if (dialog->m_widget_cb_append_albumname->get_active())
                  {
                    string path = filename_from_uri (uri);
                    string append = filename_from_utf8 (a.album.get());
                    path_new = build_filename (path, append);
                    g_mkdir (path_new.c_str(), mode_t(S_IRWXU)); 
                  }
                else
                  {
                    path_new = filename_from_uri (uri);
                  }

                VRows v;
                library->query( album_to_attrs (a, false), v );

                UTrackV update_tracks;
                for (VRows::const_iterator i = v.begin() ; i != v.end() ; ++i)
                  {
                    try{
                        Track track (*i);

                        string filename = filename_from_uri (track.location.get());
                        string basename = path_get_basename (filename);
                        string pathname = path_get_dirname  (filename);

                        if (pathname != path_new) 
                          {
                            UTrack update_track;
                            update_track.location_cur = track.location.get(); 
                            update_track.location_new = filename_to_uri (build_filename (path_new, basename));
#ifdef HAVE_HAL
                            update_track.volume_relative_path = track.volume_relative_path;
                            update_track.volume_udi = track.volume_udi;
#endif //HAVE_HAL
                            update_tracks.push_back (update_track); 
                          }
                      }
                    catch (ConvertError& cxe)
                      {
                        /* FIXME: We should convert all names beforehand so this 
                         *  will result in a mostly atomic procedure
                         */

                        MessageDialog message_dialog (_("Unable to proceed: A conversion error "
                                                        "occured during processing"),
                                                        false, MESSAGE_ERROR, BUTTONS_OK, true);
                        message_dialog.run ();
                        return false;
                      }
                  }
                library->mod (update_tracks, false);
                return true;
              }
          }
          return false;
      }

      void
      AlbumView::current_selection (AlbumV & list)
      {
        if (get_selection()->count_selected_rows() == 0) return;

        for (SelectionList::const_iterator p  = m_selection_list_albums.begin(); 
                                           p != m_selection_list_albums.end(); ++p)
          {
            TreeModel::Path const& c_path = p->second.get_path ();
            TreeModel::Path m_path = m_store_filter->convert_child_path_to_path (c_path);
            if (m_path.gobj())
              {
                TreeModel::iterator const& m_iter = m_store_filter->get_iter (m_path);
                Album const& a ((*m_iter)[m_album.album]);
                list.push_back (a);
              }
          }
      }

#ifdef HAVE_HAL
      void AlbumView::modify (VUriVrpPair const& list)
#else
      void AlbumView::modify (VUri const& list)
#endif //HAVE_HAL

      {
        MBReleaseV releases; 
        PathList const& paths = get_selection()->get_selected_rows ();

        for (PathList::const_iterator path = paths.begin(), end = paths.end(); path != end; ++path)
          {
            TreeModel::iterator const& m_iter (m_store_filter->get_iter (*path)); 
            Album const& a ((*m_iter)[m_album.album]);
            mb_query_releases ((a.sortname? a.sortname.get() : a.artist.get()), a.album.get(), releases);
          }

        LibraryUiModify * modify = LibraryUiModify::create();
        LibraryUiModify::ModifyTracks modify_tracks;
        int response = modify->run (releases, list, modify_tracks);
        delete modify;

        if (response == GTK_RESPONSE_OK)
          {
            ReferenceList rl;

            for (PathList::const_iterator i = paths.begin (); i != paths.end (); ++i)
              { 
								get_selection()->unselect (*i);
                rl.push_back (TreeRowReference (m_store_base,
                              m_store_filter->convert_path_to_child_path (*i)));
              }

            for (ReferenceList::const_iterator i = rl.begin(); i != rl.end(); ++i) 
							{
                TreeModel::iterator m_iter (m_store_base->get_iter (i->get_path()));

                Library::Album const& a ((*m_iter)[m_album.album]);
                m_album_map.erase (a.tpl);

							  new_remove_one ((*m_iter)[m_album.guid]);
								m_store_base->erase (m_iter);
							}

            library->mod (modify_tracks.tracks, true, true, false);
            update_new ();
        
            TreeModel::iterator m_iter;
            if (matching_album (modify_tracks.album, m_iter))
            {
              get_selection()->select (m_store_filter->convert_child_iter_to_iter (m_iter));
              scroll_to_row (m_store_filter->get_path (m_store_filter->convert_child_iter_to_iter (m_iter)), 0.5);
            }
          }
      }

      void
      AlbumView::insert (Bmp::Library::Album const& album,
                         bool                       new_album)
      {
        using namespace Gtk;
        using namespace Glib;

        TreeModel::iterator m_iter (m_store_base->append());
    
        m_guid++;
      
        if (new_album)
          m_new_items_map[m_guid] = TreeModel::RowReference (m_store_base, m_store_base->get_path (m_iter));

        (*m_iter)[m_album.album]   = album; 
        (*m_iter)[m_album.guid]  = m_guid;

        get_cover(m_iter, album.asin); 
        m_album_map.insert(std::make_pair (album.tpl, m_iter));
      }

      void
      AlbumView::insert (Bmp::Library::Album const& album,
                         bool                       new_album,
                         Gtk::TreeModel::iterator & m_iter)
      {
        using namespace Gtk;
        using namespace Glib;

        m_iter = m_store_base->append();
    
        m_guid++;
      
        if (new_album)
          m_new_items_map[m_guid] = TreeModel::RowReference (m_store_base, m_store_base->get_path (m_iter));

        (*m_iter)[m_album.album]   = album; 
        (*m_iter)[m_album.guid]  = m_guid;
        get_cover (m_iter, album.asin); 
        m_album_map.insert (std::make_pair (album.tpl, m_iter));
      }

      void
      AlbumView::remove (bool unattended)
      {
        ReferenceList references;
        PathList const& paths = get_selection()->get_selected_rows ();

        for (PathList::const_iterator path = paths.begin (), end = paths.end (); path != end; ++path)
            references.push_back (TreeRowReference (m_store_base,
              m_store_filter->convert_path_to_child_path (*path)));

        if (!unattended)
          {
            TaskDialog dialog (_("Music Library - BMP"),
                               _("Remove Selected Albums?"),
                               Gtk::MESSAGE_QUESTION);

            dialog.add_button (_("Remove"), _("Remove selected Albums"), Gtk::Stock::GO_FORWARD, GTK_RESPONSE_OK);
            dialog.add_button (_("Keep"), _("Cancels the Request"), Gtk::Stock::CANCEL, GTK_RESPONSE_CANCEL);
            dialog.set_default_response (GTK_RESPONSE_CANCEL);
            if (dialog.run () != GTK_RESPONSE_OK) return;
          }
  
        for (ReferenceList::const_iterator r = references.begin (), end = references.end (); r != end; ++r)
          {
            TreeModel::const_iterator const& m_iter (m_store_base->get_iter (r->get_path()));
            Album const& a ((*m_iter)[m_album.album]); 

            if (unattended)
              {
                if (library->album_exists (a))
                {
                  (*m_iter)[m_album.album] = library->album_refetch (a);
                }
                else
                {
                  get_selection()->unselect (m_store_filter->convert_child_iter_to_iter (m_iter));

                  Bmp::Library::Album const& a ((*m_iter)[m_album.album]);
                  m_album_map.erase (a.tpl);

                  new_remove_one ((*m_iter)[m_album.guid]);
                  m_store_base->erase (m_iter);
                }
              }
            else
              {
                get_selection()->unselect (m_store_filter->convert_child_iter_to_iter (m_iter));

                Bmp::Library::Album const& a ((*m_iter)[m_album.album]);
                m_album_map.erase (a.tpl);

                new_remove_one ((*m_iter)[m_album.guid]);
                m_store_base->erase (m_iter);
                library->del( album_to_attrs (a, false) );
              }
            while (gtk_events_pending()) gtk_main_iteration();
          }
      }

      void
      AlbumView::approve_sel ()
      {
        ReferenceList references;
        PathList const& paths = get_selection()->get_selected_rows ();

        for (PathList::const_iterator p = paths.begin(); p != paths.end(); 
          references.push_back (TreeModel::RowReference (m_store_base,
                                m_store_filter->convert_path_to_child_path (*p))), ++p);

        for (ReferenceList::const_iterator r  = references.begin (); r != references.end (); ++r)
          {
            TreeModel::const_iterator const& m_iter (m_store_base->get_iter (r->get_path()));
            Album const& a ((*m_iter)[m_album.album]);

            library->new_approve (a);
            get_selection()->unselect (m_store_filter->convert_child_iter_to_iter (m_iter));
            new_approve_one ((*m_iter)[m_album.guid]);
            m_store_filter->row_changed (r->get_path(), m_iter);
          }
      }
}

namespace Bmp
{
    TrackView::TrackView (BaseObjectType                 *  obj,
                          RefPtr<Gnome::Glade::Xml> const&  xml)

          : TreeView          (obj),
            m_pixbuf_playing  (Gdk::Pixbuf::create_from_file (Glib::build_filename (BMP_IMAGE_DIR_MAIN, "playing.png"))),
            m_guid            (0)
    {

#if GTK_CHECK_VERSION(2, 10, 0)
      gtk_tree_view_set_rubber_banding (GTK_TREE_VIEW (gobj()), TRUE);
      gtk_tree_view_set_grid_lines (GTK_TREE_VIEW (gobj()), GTK_TREE_VIEW_GRID_LINES_VERTICAL); 
#endif

      const unsigned int N_RATINGS = 6;
      for (unsigned int n = 0; n < N_RATINGS; ++n)
        {
          static boost::format star ("stars_%d.png");
          m_rating[n] = Gdk::Pixbuf::create_from_file (build_filename (BMP_IMAGE_DIR_RATING, (star % n).str()));
        }

      enum Renderer
      {
        R_TEXT,
        R_PIXBUF,
      };

      struct { 
          char    *title;
          double   xalign;
          int      width;
          Renderer renderer;
          int      column;
      } headers_tracks[] = {
          { "",              0.5, 30,   R_PIXBUF,  -1},
          { "",              0.5, 30,   R_PIXBUF,  -1},
          { _("Track"),      1.0, 30,   R_TEXT,     0},
          { _("Title"),      0.0, 100,  R_TEXT,     1},
          { _("Rating"),     0.0, 80,   R_PIXBUF,   2},
          { _("Artist"),     0.0, 80,   R_TEXT,     3},
          { _("Album"),      0.0, 80,   R_TEXT,     4},
          { _("Genre"),      0.0, 80,   R_TEXT,     5},
          { _("Play Count"), 0.0, 30,   R_TEXT,     6},
          { _("Bitrate"),    0.0, 30,   R_TEXT,     7},
          { _("Samplerate"), 0.0, 30,   R_TEXT,     8},
      };

      for (unsigned int n = 0; n < G_N_ELEMENTS(headers_tracks); ++n)
      {
          TreeViewColumn *column = 0;
          CellRenderer *cell_ = 0;

          if (headers_tracks[n].renderer == R_PIXBUF)
          {
              CellRendererPixbuf *cell = manage (new CellRendererPixbuf());
              cell->property_xalign() = headers_tracks[n].xalign;
              append_column (headers_tracks[n].title, *cell);
              column = get_column (n);
              column->set_resizable (false);
              column->set_expand (false);
              column->property_min_width() = headers_tracks[n].width; 
              column->property_max_width() = headers_tracks[n].width; 
              if (headers_tracks[n].column != -1) column->set_sort_column (headers_tracks[n].column);
              cell_ = cell;

              Gtk::Image * image = 0; 
              switch (n)
                {
                    case 0:
                          cell_new = cell;
                          break;

                    case 1:
                          cell_playing = cell;
                          image = Gtk::manage ( new Gtk::Image() );
                          image->set (Glib::build_filename (BMP_IMAGE_DIR, "blue-speaker.png")); 
                          column->set_widget (*image); 
                          image->show();
                          break;

                    case 4:
                          cell_rating = cell;
                          break;
                }
          }
          else 
          {
              CellRendererText *cell = manage ( new CellRendererText() );
              cell->property_xalign() = headers_tracks[n].xalign;
              append_column (headers_tracks[n].title, *cell);
              column = get_column (n);
              column->add_attribute (*cell, "text", headers_tracks[n].column);
              column->set_resizable (true);
              column->set_sort_column (headers_tracks[n].column);
              column->set_expand (true);
              column->property_min_width() = headers_tracks[n].width;
              column->signal_clicked().connect (sigc::mem_fun (this, &Bmp::TrackView::column_toggled));
              cell_ = cell;
          }
        column->set_cell_data_func (*cell_, sigc::mem_fun
              (this, &Bmp::TrackView::cell_data_func_tracks));
      }

      m_store = ListStore::create (m_track);
      set_model (m_store);

      get_selection()->set_mode (SELECTION_MULTIPLE);

      get_selection()->set_select_function
          (sigc::mem_fun (this, &Bmp::TrackView::select_func_tracks));

      library->signal_row_updated().connect
          (sigc::mem_fun (this, &Bmp::TrackView::check_update));
    }

    void
    TrackView::notify_remove (ustring const& uri)
    {
      if (playing_uri && playing_uri.get() == uri)
        {
          playing_uri.reset ();
          playing_row.reset ();
        }
    }

    void
    TrackView::check_update (ustring const& location, UTrack track)
    {
      RowMapTracks::iterator rmt_iter = m_row_map_tracks.find (location); 
      if (rmt_iter != m_row_map_tracks.end())
        {
          set_sensitive (false);

          TreeModel::iterator const& m_iter (m_store->get_iter (rmt_iter->second.get_path()));
          TreeModel::Row row (*m_iter);

          if (track.title)          row[m_track.title]     = track.title.get (); 
          if (track.artist)         row[m_track.artist]    = track.artist.get (); 
          if (track.album)          row[m_track.album]     = track.album.get (); 
          if (track.tracknumber)    row[m_track.track]     = track.tracknumber.get (); 
          if (track.genre)          row[m_track.genre]     = track.genre.get (); 
          if (track.location_new)   row[m_track.location]  = track.location_new.get (); 
          if (track.count)          row[m_track.count]     = track.count.get (); 

          row[m_track.rating] = track.rating? track.rating.get () : 0;

#ifdef HAVE_HAL
          if (track.volume_relative_path)   row[m_track.volume_relative_path] = track.volume_relative_path.get (); 
          if (track.volume_udi)             row[m_track.volume_udi] = track.volume_udi.get (); 
#endif //HAVE_HAL

          if (track.location_new) 
          {
              m_row_map_tracks.erase  (location);
              m_row_map_tracks.insert (std::make_pair (track.location_new.get(),
                                                       TreeRowReference (m_store, TreePath (m_iter))));

              (*m_iter)[m_track.present] = 
                      Glib::file_test(Glib::filename_from_uri(track.location_new.get()), Glib::FILE_TEST_EXISTS); 
               
              if (playing_uri && (playing_uri.get() == location)) 
                {
                  playing_uri = track.location_new.get();
                }
            }
          set_sensitive (true);
        }
    }

    bool
    TrackView::select_func_tracks (RefPtr<TreeModel> const& model, TreeModel::Path const& path, bool selected)
    {
      TreeModel::iterator const& m_iter = model->get_iter (path);
      return ((*m_iter)[m_track.present]);
    }

    void
    TrackView::cell_data_func_tracks (CellRenderer * cell_, TreeModel::iterator const& m_iter)
    {
      cell_->property_sensitive() = ((*m_iter)[m_track.present]); 

      if (cell_ == cell_rating)
        {
          CellRendererPixbuf *cell = dynamic_cast<CellRendererPixbuf *>(cell_);
          cell->property_pixbuf() = m_rating[((*m_iter)[m_track.rating])];
          return;
        }
      else if (cell_ == cell_playing)
        {
          CellRendererPixbuf * cell = dynamic_cast<CellRendererPixbuf *>(cell_);
          if (playing_uri && ((*m_iter)[m_track.location] == playing_uri.get()))
          {
            cell->property_pixbuf() = m_pixbuf_playing; 
            return;
          }
          cell->property_pixbuf() = RefPtr<Gdk::Pixbuf>(0);
        }
      else if (cell_ == cell_new)
        {
          CellRendererPixbuf * cell = dynamic_cast<CellRendererPixbuf *>(cell_);
          if ((*m_iter)[m_track.new_item])
          {
            cell->property_pixbuf() = render_icon (Gtk::StockID (BMP_STOCK_NEW), Gtk::ICON_SIZE_SMALL_TOOLBAR); 
            return;
          }
          cell->property_pixbuf() = RefPtr<Gdk::Pixbuf>(0);
        }
    }

    bool
    TrackView::on_event (GdkEvent * ev)
    {
      GdkEventButton *event = 0;

      if (ev->type == GDK_BUTTON_PRESS)
        {
          event = reinterpret_cast<GdkEventButton *>(ev);
          TreeViewColumn *column, *column_rating = get_column (4);
          TreeModel::Path path;
          int cell_x, cell_y;

          if (get_path_at_pos (int(event->x),
                               int(event->y),
                               path,
                               column,
                               cell_x,
                               cell_y)) 
            {
              if (column == column_rating)
                {
                  TreeModel::iterator const& m_iter = m_store->get_iter (path);

                  int n = (int(cell_x+7)/14);

                  UTrackV update_tracks;
                  UTrack  update_track;

#ifdef HAVE_HAL
                  update_track.volume_udi = ((*m_iter)[m_track.volume_udi]); 
                  update_track.volume_relative_path = ((*m_iter)[m_track.volume_relative_path]); 
#endif //HAVE_HAL

                  update_track.location_cur = ((*m_iter)[m_track.location]); 

                  update_track.rating = n;
                  update_tracks.push_back (update_track);

                  try {
                      library->mod (update_tracks, false, true, false);
                      (*m_iter)[m_track.rating] = n; 
                    }
                  catch (...) {}

                  return true;
                }
            } 
        }
      return false;
    }

    bool
    TrackView::remove ()
    {
      bool all_missing = true;
      ReferenceList references;
      PathList const& paths = get_selection()->get_selected_rows ();

      for (PathList::const_iterator p = paths.begin (); p != paths.end (); ++p)
        {
          TreeModel::iterator const& m_iter = m_store->get_iter (*p);
          references.push_back (TreeModel::RowReference (m_store, (*p)));
          if (bool((*m_iter)[m_track.present])) all_missing = false;
        }

      bool physically_delete_files = mcs->key_get<bool>("bmp", "physically-delete-files");

      if (physically_delete_files && !all_missing)
        {
          TaskDialog dialog (_("Music Library - BMP"),
                             _("Remove selected Files?"),
                             Gtk::MESSAGE_QUESTION,
                             _("<b>You are about to delete the selected Files also <i>physically</i> from "
                               "the filesystem. "
                               "If you do <i>NOT</i> wish that, please use 'Keep Files' now.</b>\n\nNOTE: You "
                               "can change this "
                               "behaviour in Preferences, section 'Library'"));
                                                    
          dialog.add_button ("Remove",  "Removes all selected Files <b>irreversibely</b>",
                                          Gtk::Stock::GO_FORWARD, GTK_RESPONSE_OK);

          dialog.add_button ("Keep",    "Cancels the Request",
                                          Gtk::Stock::CANCEL, GTK_RESPONSE_CANCEL);

          dialog.set_default_response (GTK_RESPONSE_CANCEL);

          if (dialog.run() != GTK_RESPONSE_OK)
          {
            dialog.hide ();
            return false;
          }
        }

      for (ReferenceList::const_iterator r = references.begin (); r != references.end(); ++r)
        {
          TreeModel::const_iterator const& m_iter = m_store->get_iter (r->get_path());

          ustring const& uri = (*m_iter)[m_track.location];
          notify_remove (uri);

          VAttributes attrs;

#ifdef HAVE_HAL

					Attribute a1 (Bmp::DB::EXACT,
                        Bmp::MetadatumId (DATUM_HAL_VOLUME_UDI),
                        ustring((*m_iter)[m_track.volume_udi]));

          Attribute a2 (Bmp::DB::EXACT,
                        Bmp::MetadatumId (DATUM_VOLUME_RELATIVE_PATH),
                        ustring((*m_iter)[m_track.volume_relative_path]));

          attrs.push_back (a1);
          attrs.push_back (a2);

#else
          Attribute a1 (Bmp::DB::EXACT, Bmp::MetadatumId (DATUM_LOCATION), uri);
          attrs.push_back (a1);

#endif //HAVE_HAL

          library->del (attrs);

          if (physically_delete_files && bool((*m_iter)[m_track.present]))
            {
              string filename = filename_from_uri (uri);
              if (file_test (filename, FILE_TEST_EXISTS))
                {
                  g_unlink (filename.c_str());    
                }
            }
          m_store->erase (m_iter);
        }
      return true;
    }

    void
    TrackView::add_tracks (Bmp::DB::VRows const& vector)
    {
      m_row_map_tracks.clear ();
  
      for (VRows::const_iterator v = vector.begin (), end = vector.end (); v != end; ++v) 
      { 
        try {

            Track track (*v);

            TreeModel::iterator m_iter (m_store->append ());
            TreeModel::Row row (*m_iter); 

            if (track.title)        row[m_track.title]       = track.title.get(); 
            if (track.artist)       row[m_track.artist]      = track.artist.get(); 
            if (track.album)        row[m_track.album]       = track.album.get(); 
            if (track.tracknumber)  row[m_track.track]       = track.tracknumber.get(); 
            if (track.genre)        row[m_track.genre]       = track.genre.get(); 
            if (track.count)        row[m_track.count]       = track.count.get(); 
            if (track.bitrate)      row[m_track.bitrate]     = track.bitrate.get(); 
            if (track.samplerate)   row[m_track.samplerate]  = track.samplerate.get(); 
            if (track.new_item)     row[m_track.new_item]    = track.new_item.get(); 

            if (track.location)
              {
                row[m_track.location] = track.location.get (); 
                if (playing_uri && (playing_uri.get() == track.location.get()))
                  {
                    playing_row = m_iter; 
                  }
              }

#ifdef HAVE_HAL

            if (track.volume_relative_path)
              {
                row[m_track.volume_relative_path] = track.volume_relative_path.get(); 
              }

            if (track.volume_udi)
              {
                row[m_track.volume_udi] = track.volume_udi.get(); 
              }

            if (hal->is_initialized())
              {
                row[m_track.present] = (((track.location && // location is not initialized if the volume is not present 
                    Glib::file_test (Glib::filename_from_uri (track.location.get()), Glib::FILE_TEST_EXISTS)))); 
              }
#else
                row[m_track.present] = 
                    Glib::file_test (Glib::filename_from_uri (track.location.get()), Glib::FILE_TEST_EXISTS); 
                                       
#endif //HAVE_HAL

            if (track.location)
              {
                m_row_map_tracks.insert (std::make_pair (track.location.get(),
                    TreeModel::RowReference (m_store, TreePath (m_iter))));
              }
            row[m_track.rating] = track.rating? track.rating.get () : 0;
            row[m_track.guid] = m_guid++;
          }
        catch (...) { continue; }
      }
    }

    ////////////////////////////////// Public API    

    bool
    TrackView::all_tracks_present ()
    {

#ifdef HAVE_HAL
      if (!hal->is_initialized()) return false;
#endif //HAVE_HAL

      TreeModel::Children const& nodes = m_store->children ();
      if (nodes.size() == 0) return false;

      for (TreeModel::Children::const_iterator m_iter = nodes.begin(); m_iter != nodes.end(); ++m_iter)
        {
          if (!((*m_iter)[m_track.present])) return false;
        }

      return true;
    }

    void
    TrackView::get_files (Util::FileList & list)
    {
      TreeModel::Children const& nodes = m_store->children ();

      for (TreeModel::Children::const_iterator m_iter = nodes.begin (); m_iter != nodes.end (); ++m_iter) 
        {
          list.push_back (filename_from_uri (ustring((*m_iter)[m_track.location])));
        }
    }

    void
    TrackView::clear ()
    {
      m_store->clear ();
      m_row_map_tracks.clear ();
      playing_row.reset ();
    }

    void
    TrackView::unsort ()
    {
      m_store->set_sort_column (TreeSortable::DEFAULT_UNSORTED_COLUMN_ID, SORT_ASCENDING);
    }

#ifdef HAVE_HAL

    void
    TrackView::get_tracks (VUriVrpPair& pairs)
    {
      TreeModel::Children const& nodes = m_store->children ();
      for (TreeModel::Children::const_iterator m_iter = nodes.begin(), end = nodes.end(); m_iter != end; ++m_iter)
        {
          pairs.push_back (std::make_pair (ustring((*m_iter)[m_track.volume_udi]),
                                           ustring((*m_iter)[m_track.volume_relative_path])));
        } 
    }

#else //HAVE_HAL

    void
    TrackView::get_tracks (Bmp::VUri & locations)
    {
      TreeModel::Children const& nodes = m_store->children ();

      for (TreeModel::Children::const_iterator m_iter = nodes.begin(); m_iter != nodes.end(); ++m_iter)
        {
          locations.push_back (ustring((*m_iter)[m_track.location]));
        }
    }

#endif //HAVE_HAL

    ustring
    TrackView::playback_uri ()
    {
      if (!playing_uri) throw NoCurrentUriError();

      return playing_uri.get();
    }

    void
    TrackView::notify_stop ()
    {
      if (playing_uri) playing_uri.reset ();
      if (playing_row) playing_row.reset ();
      queue_draw ();
    }

    bool
    TrackView::notify_next (unsigned int & position, unsigned int & size)
    {
      TreeModel::Children const& children = m_store->children();
      size = children.size();

      if (!playing_row && size) 
        {
          TreeModel::Path path; 
          path.append_index (0);

          TreeModel::iterator m_iter = m_store->get_iter (path);

          playing_row = m_iter; 
          playing_uri = ((*m_iter)[m_track.location]);

          m_store->row_changed (path, m_iter);
          position = path.get_indices().data()[0]; 
          return true; 
        }

      TreeModel::Path path (m_store->get_path (playing_row.get()));
      position = path.get_indices().data()[0] + 1;

      if (position != size) 
        {
          TreeModel::Path path_old (1, position-1);
          TreeModel::Path path_new (1, position);

          TreeModel::iterator const& m_iter = m_store->get_iter (path_new);
          playing_row = m_iter; 
          playing_uri = (*m_iter)[m_track.location]; 

          m_store->row_changed (path_old, m_iter);
          m_store->row_changed (path_new, m_iter);

          return true;
        }
      else
        {
          return false;
        }
    }

    bool
    TrackView::notify_prev (unsigned int & position, unsigned int & size)
    {
      if (!playing_row)
        return false;

      TreeModel::Children const& children = m_store->children();
      TreeModel::Path path (m_store->get_path (playing_row.get()));

      size = children.size();
      position = path.get_indices().data()[0] - 1;

      if (position >= 0) 
        {
          TreeModel::Path path_old (1, position+1);
          TreeModel::Path path_new (1, position);

          TreeModel::iterator const& m_iter = m_store->get_iter (path_new);
          playing_row = m_iter; 
          playing_uri = (*m_iter)[m_track.location]; 

          m_store->row_changed (path_old, m_iter);
          m_store->row_changed (path_new, m_iter);

          return true;
        }
      else
        {
          return false;
        }
    }

    void
    TrackView::has_next_prev (bool & next, bool & prev)
    {
      next = false;
      prev = false;

      if (playing_row && playing_row.get())
        {
          TreeModel::Children const& children = m_store->children();
          unsigned int position = m_store->get_path (playing_row.get()).get_indices().data()[0];

          prev = (position > 0);
          next = (position+1 != children.size());
        }
      else if (!playing_row || (playing_row && !playing_row.get()))
        {
          playing_row.reset ();
        }
    }

    void
    TrackView::column_toggled ()
    {
      signal_column_toggled_.emit ();
    }

    bool
    TrackView::notify_play (unsigned int & position, unsigned int & size)
    {
      if (get_selection()->count_selected_rows() == 0)
        {
          TreeModel::Path path; 
          path.append_index (0);
          get_selection()->select (path);
        }

      PathList paths = get_selection()->get_selected_rows ();

      TreeModel::Path     const& path     = paths[0]; 
      TreeModel::iterator const& m_iter   = m_store->get_iter (path);
      TreeModel::Children const& children = m_store->children();

      size = children.size();

      if (!playing_row) 
        {
          playing_row = m_iter; 
          playing_uri = ((*m_iter)[m_track.location]);
          m_store->row_changed (path, m_iter);
          position = path.get_indices().data()[0]; 
          return true; 
        }
      else
        {
          TreeModel::iterator l_iter = m_store->get_iter (path);
          if ((*l_iter)[m_track.location] != playing_uri.get())
            {
              playing_row = l_iter; 
              playing_uri = ((*l_iter)[m_track.location]);
              m_store->row_changed (path, l_iter);
              position = path.get_indices().data()[0]; 
              return true; 
            }
        }
      return false;
    }
}

namespace Bmp
{
  namespace UiPart
  {
      Albums::~Albums () {}
 
      guint 
      Albums::add_ui ()
      {
        return m_ui_manager->add_ui_from_string (albums_menu);
      };

      Albums::Albums (RefPtr<Gnome::Glade::Xml> const& xml, RefPtr<UIManager> ui_manager)

          : PlaybackSource (PlaybackSource::Caps (PlaybackSource::CAN_SEEK | PlaybackSource::CAN_PROVIDE_METADATA)), Base (xml, ui_manager),
            m_busy (false),
            m_init (true)
      {
          m_audio_empty =
            Gdk::Pixbuf::create_from_file (Glib::build_filename
                (BMP_IMAGE_DIR, BMP_COVER_IMAGE_EMPTY))->scale_simple (96, 96, Gdk::INTERP_HYPER); 

          m_audio_multiple =
            Gdk::Pixbuf::create_from_file (Glib::build_filename
                (BMP_IMAGE_DIR, BMP_COVER_IMAGE_MULTIPLE))->scale_simple (96, 96, Gdk::INTERP_HYPER); 

          m_audio_default =
            Gdk::Pixbuf::create_from_file (Glib::build_filename
                (BMP_IMAGE_DIR, BMP_COVER_IMAGE_DEFAULT))->scale_simple (96, 96, Gdk::INTERP_HYPER); 

          struct ButtonActionProxyI
          {
            char  *widget;
            char  *action;
          };

          struct ImageStockIdI
          {
            char  *widget;
            char  *stockid;
          };

          m_actions = ActionGroup::create ("LibraryActions");

          m_actions->add ( Action::create (ACTION_MENU_UI_PART_ALBUMS, _("_Albums")));

          m_actions->add (  Action::create (ACTION_IMPORT_MUSIC,
                            Gtk::StockID (BMP_STOCK_ADD_MUSIC), 
                            _("_Add Music")),
                            sigc::mem_fun (this, &Bmp::UiPart::Albums::on_tracks_import));

          m_actions->add (  Action::create (ACTION_IMPORT_RETAG,
                            Gtk::StockID (BMP_STOCK_ADD_MUSIC), 
                            _("Re_tag Files and Add")),
                            sigc::mem_fun (this, &Bmp::UiPart::Albums::on_tracks_import_mbtag));

          m_actions->add (  Action::create (ACTION_EDIT_ALBUMS,
                            Gtk::StockID (BMP_STOCK_EDIT_ALBUM), 
                            _("_Edit Metadata")),
                            sigc::mem_fun (this, &Bmp::UiPart::Albums::on_modify));

          m_actions->add (  Action::create (ACTION_REMOVE_ALBUMS,
                            Gtk::StockID (BMP_STOCK_REMOVE_MUSIC), 
                            _("_Remove Selected")),
                            sigc::mem_fun (this, &Bmp::UiPart::Albums::on_remove));

          m_actions->add (  Action::create (ACTION_MOVE_ALBUM,
                            Gtk::StockID (BMP_STOCK_MOVE_ALBUM), 
                            _("Re_locate Selected")),
                            sigc::mem_fun (this, &Bmp::UiPart::Albums::on_relocate));


          m_actions->add ( Action::create ("dummy", "dummy", "dummy"));
          m_actions->add ( Action::create (ACTION_VIEW, Gtk::StockID (BMP_STOCK_VIEW), _("View Options")));

          m_actions->add (  Action::create (ACTION_REMOVE_TRACKS,
                            Gtk::StockID (BMP_STOCK_DELETE),
                            _("Remove Selected Tracks")),
                            sigc::mem_fun (this, &Bmp::UiPart::Albums::on_tracks_remove));

          m_actions->add (  Action::create (ACTION_ENQUEUE_TRACKS,
                            Gtk::StockID (BMP_STOCK_ADD),
                            _("Enqueue Selected Tracks To Playlist")),
                            sigc::mem_fun (this, &Bmp::UiPart::Albums::on_tracks_enqueue));


          m_actions->add (  Action::create (ACTION_NEW_APPROVE_ALL,
                            Gtk::StockID (BMP_STOCK_APPROVE),
                            _("Approve All Pending")),
                            sigc::mem_fun (this, &Bmp::UiPart::Albums::on_new_approve_all));

          m_actions->add (  Action::create (ACTION_APPROVE_SELECTED,
                            Gtk::StockID (BMP_STOCK_APPROVE),
                            _("Approve Selected")),
                            sigc::mem_fun (this, &Bmp::UiPart::Albums::on_new_approve_sel));

          m_actions->add (  Action::create (ACTION_NEW_DROP_ALL,
                            Gtk::StockID (BMP_STOCK_DELETE),
                            _("Remove All Pending")),
                            sigc::mem_fun (this, &Bmp::UiPart::Albums::on_new_drop_all));

          m_actions->add (  Action::create (ACTION_FETCH_ALL_COVERS,
                            Gtk::StockID (GTK_STOCK_REFRESH),
                            _("Fetch Covers for all Albums")),
                            sigc::mem_fun (this, &Bmp::UiPart::Albums::on_get_covers));


          Gtk::RadioButtonGroup group1;
          m_actions->add (  RadioAction::create  (group1, ACTION_VIEW_SIZE_NORMAL, 
                            _("Large Icons")),
                            sigc::mem_fun (this, &Bmp::UiPart::Albums::on_view_size_large_activated));

          m_actions->add (  RadioAction::create  (group1, ACTION_VIEW_SIZE_SMALL, 
                            _("Small Icons")),
                            sigc::mem_fun (this, &Bmp::UiPart::Albums::on_view_size_small_activated));


          m_actions->add (  ToggleAction::create  (ACTION_SHOW_ALBUM_LIST, 
                            _("Show Album List")),
                            sigc::mem_fun (this, &Bmp::UiPart::Albums::on_show_album_list_toggled));

/*
          m_actions->add (  ToggleAction::create  (ACTION_SHOW_ARTIST_LIST, 
                            _("Show Artist Selection")),
                            sigc::mem_fun (this, &Bmp::UiPart::Albums::on_show_artist_list_toggled));
*/

          m_actions->add (  ToggleAction::create  (ACTION_CONTINUOUS_PLAY, 
                            _("Continuous Play")),
                            sigc::mem_fun (this, &Bmp::UiPart::Albums::on_continuous_play_toggled));

          Glib::RefPtr<Gtk::ToggleAction>::cast_static(m_actions->get_action (ACTION_SHOW_ALBUM_LIST))->set_active();
//          Glib::RefPtr<Gtk::ToggleAction>::cast_static(m_actions->get_action (ACTION_SHOW_ARTIST_LIST))->set_active(false);

          // Throbbers
          {
            dynamic_cast<Gtk::Image *>(m_ref_xml->get_widget ("i_throbber-1"))->set
            (Glib::build_filename (BMP_IMAGE_DIR, BMP_THROBBER));
          }

          m_ref_xml->get_widget ("albums-metadata-label-current-album",   metadata_label_current_album);
          m_ref_xml->get_widget ("albums-metadata-label-current-artist",  metadata_label_current_artist);
          m_ref_xml->get_widget ("albums-metadata-label-current-date",    metadata_label_current_date);

          m_ref_xml->get_widget ("albums-metadata-cover-current-album",   metadata_cover_current_album);

          // Progress for modifications
          m_ref_xml->get_widget ("progress", m_progress);

          library->signal_modify_start().connect
              (sigc::mem_fun (this, &Bmp::UiPart::Albums::progress_start));

          library->signal_modify_step().connect
              (sigc::mem_fun (this, &Bmp::UiPart::Albums::progress_step));

          library->signal_modify_end().connect
              (sigc::mem_fun (this, &Bmp::UiPart::Albums::progress_stop));

          m_ref_xml->get_widget ("albums-notebook", notebook_albums);

          m_ref_xml->get_widget_derived ("albums-track-view", m_view_tracks);
          m_view_tracks->signal_row_activated().connect
                (sigc::mem_fun (this, &Bmp::UiPart::Albums::activate_default));
          m_view_tracks->get_selection()->signal_changed().connect
            (sigc::mem_fun (this, &Bmp::UiPart::Albums::on_tracks_selection_changed));
          m_view_tracks->signal_event().connect
                (sigc::mem_fun (this, &Bmp::UiPart::Albums::popup_menu_tracks));
          m_view_tracks->signal_column_toggled().connect
                (sigc::mem_fun (this, &Bmp::UiPart::Albums::on_continuous_play_toggled)); //FIXME: Slight abuse

          m_ref_xml->get_widget_derived ("albums-album-view", m_view_albums); 
          m_view_albums->signal_row_activated().connect
                (sigc::mem_fun (this, &Bmp::UiPart::Albums::activate_default));
          m_view_albums->get_selection()->signal_changed().connect
                (sigc::mem_fun (this, &Bmp::UiPart::Albums::on_albums_selection_changed));

          m_view_albums->signal_event().connect
                (sigc::mem_fun (this, &Bmp::UiPart::Albums::popup_menu_albums));

          m_actions->add ( Action::create (ACTION_UNSELECT_ALL,
                           Gtk::StockID (GTK_STOCK_CLEAR),
                           _("Clear Selection")),
                (sigc::bind (sigc::ptr_fun (&unselect_all), m_view_albums->get_selection())));

          m_ui_manager->insert_action_group (m_actions);
          m_ui_manager->add_ui_from_string (albums_tracklist_menu);
          m_ui_manager->add_ui_from_string (albums_albumlist_menu);
  
          // Search/Filter Entry
          m_ref_xml->get_widget ("albums-search", m_albums_search);
          m_albums_search->signal_changed().connect
                (sigc::mem_fun (this, &Bmp::UiPart::Albums::on_sort_entry_changed));

          // Artists View
          m_albums_search_completion_model = Gtk::ListStore::create (completion_model);
          m_albums_search_completion_model->set_default_sort_func
                (sigc::mem_fun (*this, &Bmp::UiPart::Albums::completion_sort_func));
          m_albums_search_completion_model->set_sort_column (-1, SORT_ASCENDING);

/*
          m_ref_xml->get_widget ("albums-view-artists", m_view_artists);
          m_ref_xml->get_widget_derived ("albums-sw-artists-alignment", m_alignment_artists);

          Gtk::CellRendererText * cell = Gtk::manage (new CellRendererText());
          TreeViewColumn *column  = manage (new TreeViewColumn ("", *cell));
          column->add_attribute (*cell, "text", 0); 
          m_view_artists->append_column (*column);
          m_view_artists->get_selection ()->signal_changed().connect
            (sigc::mem_fun (this, &Bmp::UiPart::Albums::artist_selected));
          m_view_artists->get_selection ()->set_mode (Gtk::SELECTION_BROWSE);
          m_view_artists->set_model (m_albums_search_completion_model);
*/
        
          // + Completion
          m_albums_search_completion = Gtk::EntryCompletion::create ();
          m_albums_search_completion->set_minimum_key_length (1);
          m_albums_search_completion->set_popup_completion (false);
          m_albums_search_completion->set_inline_completion (true);
          m_albums_search_completion->set_popup_set_width (false);
          m_albums_search_completion->set_text_column (0);

          m_albums_search_completion->set_model (m_albums_search_completion_model);
          m_albums_search->set_completion (m_albums_search_completion);    

          m_ref_xml->get_widget ("albums-cbox-viewmode", albums_cbox_viewmode);
          albums_cbox_viewmode->signal_changed().connect
                (sigc::mem_fun (this, &Bmp::UiPart::Albums::on_viewmode_changed));
          albums_cbox_viewmode->set_active (M_EVERYTHING);

#ifdef HAVE_HAL

          // Connect Bmp::HAL

          if (hal->is_initialized())
            {
              hal->signal_volume_added().connect
                (sigc::mem_fun (this, &Bmp::UiPart::Albums::hal_volume_added));
              hal->signal_volume_removed().connect
                (sigc::mem_fun (this, &Bmp::UiPart::Albums::hal_volume_removed));
            }

          bool enabled = hal->is_initialized(); 

#else
          bool enabled = true;

#endif //HAVE_HAL

          m_actions->get_action (ACTION_IMPORT_MUSIC)->set_sensitive (enabled);
          m_actions->get_action (ACTION_IMPORT_RETAG)->set_sensitive (enabled);
          m_actions->get_action (ACTION_MOVE_ALBUM)->set_sensitive (enabled);

          dynamic_cast<Gtk::Image *>(m_ref_xml->get_widget ("albums-image-filter-clear"))->set
                  (Gdk::Pixbuf::create_from_file (Glib::build_filename (BMP_IMAGE_DIR_STOCK, "entry-clear.png")));

          dynamic_cast<Gtk::Button *>(m_ref_xml->get_widget ("albums-button-filter-clear"))->signal_clicked().connect
                  (sigc::bind (sigc::ptr_fun (&clear_entry), m_albums_search));

          if (mcs->key_get<int>("bmp", "library-view-size") == 1)
              RefPtr<ToggleAction>::cast_static(m_actions->get_action (ACTION_VIEW_SIZE_SMALL))->set_active ();
          else
              RefPtr<ToggleAction>::cast_static(m_actions->get_action (ACTION_VIEW_SIZE_NORMAL))->set_active ();

          mcs_bind->bind_toggle_action
                    (Glib::RefPtr<Gtk::ToggleAction>::cast_static(m_actions->get_action (ACTION_CONTINUOUS_PLAY)),
                     "albums", "continuous-play");

          on_albums_selection_changed ();

          m_init = false;

          update (false);
      }

      int
      Albums::completion_sort_func (TreeModel::iterator const& iter_a, TreeModel::iterator const& iter_b)
      {
        std::string const& a ((*iter_a)[completion_model.key]);
        std::string const& b ((*iter_b)[completion_model.key]);

        return a.compare(b);
      }

/*
      void
      Albums::artist_selected ()
      {
        if (!m_view_artists->get_selection()->count_selected_rows ())
          {
            m_view_albums->set_sorting ("", true);
            return;
          }
        TreeModel::iterator m_iter = m_view_artists->get_selection()->get_selected ();
        m_view_albums->set_sorting (((*m_iter)[completion_model.name]), true);
      }
*/

      void
      Albums::update_completion ()
      {
        m_albums_search_completion_model->clear ();

        ::Bmp::DB::VRows rows;
        ::Bmp::DB::VAttributes attrs;

        library->project (MetadatumId (DATUM_ARTIST), rows,  attrs);

        for (::Bmp::DB::VRows::const_iterator i = rows.begin() ; i != rows.end() ; ++i)
          {
              ::Bmp::DB::Row::const_iterator r = i->find (MetadatumId (DATUM_ARTIST));
              if (r != i->end())
                {
                  Gtk::TreeModel::iterator m_iter = m_albums_search_completion_model->append ();
                  (*m_iter)[completion_model.name] = boost::get<std::string>(r->second); 
                  (*m_iter)[completion_model.key]  = Glib::ustring (boost::get<std::string>(r->second)).casefold_collate_key(); 
                }
          }
      }

#ifdef HAVE_HAL
      void
      Albums::hal_volume_removed (HAL::Volume volume)
      {
        debug ("library-ui", "HAL Volume Removed: %s", volume.volume_udi.c_str());
        on_albums_selection_changed (); 
      }

      void
      Albums::hal_volume_added (HAL::Volume volume)
      {
        debug ("library-ui", "HAL Volume Added: %s", volume.volume_udi.c_str());
        on_albums_selection_changed (); 
      }
#endif //HAVE_HAL

      void
      Albums::activate_default (TreeModel::Path const& path, TreeViewColumn* column)
      {
        dynamic_cast<Gtk::Window *>(m_ref_xml->get_widget ("main-ui"))->activate_default ();
      }

      void
      Albums::on_continuous_play_toggled ()
      {
        bool active (Glib::RefPtr<Gtk::ToggleAction>::cast_static
            (m_actions->get_action (ACTION_CONTINUOUS_PLAY))->get_active());

        unsigned int count = m_view_albums->get_selection()->count_selected_rows();

        bool next, prev;
        m_view_tracks->has_next_prev (next, prev);

        if (active)
          {
            if ((count) && (m_caps & PlaybackSource::CAN_PAUSE))
              m_caps = Caps (m_caps |  PlaybackSource::CAN_GO_NEXT);
            else
              m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_NEXT);
          }
        else
          {
            if (next)
              m_caps = Caps (m_caps |  PlaybackSource::CAN_GO_NEXT);
            else
              m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_NEXT);
          }

        if (prev)
          m_caps = Caps (m_caps |  PlaybackSource::CAN_GO_PREV);
        else
          m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_PREV);

        s_caps_.emit (m_caps);
      }

/*
      void
      Albums::on_show_artist_list_toggled ()
      {
        if (m_init) return;

        bool active (Glib::RefPtr<Gtk::ToggleAction>::cast_static
          (m_actions->get_action (ACTION_SHOW_ARTIST_LIST))->get_active());

        if (active)
          {
            m_ref_xml->get_widget ("hbox-filter-entry")->set_sensitive (false);
            m_ref_xml->get_widget ("albums-sw-artists-alignment")->show_all ();
          }
        else
          {
            m_ref_xml->get_widget ("hbox-filter-entry")->set_sensitive (true);
            m_ref_xml->get_widget ("albums-sw-artists-alignment")->hide ();
            m_view_albums->set_sorting (m_albums_search->get_text());
          }
      }
*/

      void
      Albums::on_show_album_list_toggled ()
      {
        if (m_init) return;

        bool active (Glib::RefPtr<Gtk::ToggleAction>::cast_static
          (m_actions->get_action (ACTION_SHOW_ALBUM_LIST))->get_active());

    
        GtkWidget * w = GTK_WIDGET (m_ref_xml->get_widget ("albums-tracklist-vbox")->gobj());
        g_object_ref (G_OBJECT (w));

        if (active)
          {
            m_ref_xml->get_widget ("hbox-album-info")->hide ();

            dynamic_cast  <Gtk::Alignment *>
                (m_ref_xml->get_widget ("albums-alignment-tracklist-single"))->
                remove();

            dynamic_cast  <Gtk::Notebook *>
                (m_ref_xml->get_widget ("albums-notebook-inner"))->
                set_current_page (0);

            dynamic_cast  <Gtk::Alignment *>
                (m_ref_xml->get_widget ("albums-alignment-tracklist"))->
                add (*(m_ref_xml->get_widget ("albums-tracklist-vbox")));
          }
        else
          {
            dynamic_cast  <Gtk::Alignment *>
                (m_ref_xml->get_widget ("albums-alignment-tracklist"))->
                remove ();

            dynamic_cast  <Gtk::Notebook *>
                (m_ref_xml->get_widget ("albums-notebook-inner"))->
                set_current_page (1);

            dynamic_cast  <Gtk::Alignment *>
                (m_ref_xml->get_widget ("albums-alignment-tracklist-single"))->
                add (*(m_ref_xml->get_widget ("albums-tracklist-vbox")));

            m_ref_xml->get_widget ("hbox-album-info")->show ();
          }
      }

      void
      Albums::on_view_size_large_activated ()
      {
        m_view_albums->set_viewsize (AlbumView::NORMAL);
        mcs->key_set<int>("bmp","library-view-size", int(0));
      }

      void
      Albums::on_view_size_small_activated ()
      {
        m_view_albums->set_viewsize (AlbumView::SMALL);
        mcs->key_set<int>("bmp","library-view-size", int(1));
      }
     
      void
      Albums::on_albums_selection_changed ()
      {
        m_view_tracks->clear ();
        unsigned int count = m_view_albums->get_selection()->count_selected_rows();

#ifdef HAVE_HAL
        m_actions->get_action (ACTION_REMOVE_ALBUMS)->set_sensitive (count && hal->is_initialized());
#else
        m_actions->get_action (ACTION_REMOVE_ALBUMS)->set_sensitive (count);
#endif //HAVE_HAL

        AlbumV list;
        m_view_albums->current_selection (list);

        if (list.size() == 1)
          {
            Library::Album first (list[0]);

            if (first.asin)
              {
                try {
                    Glib::RefPtr<Gdk::Pixbuf> cover;
                    Amazon::fetch( first.asin.get(), cover, true );
                    metadata_cover_current_album->set( cover->scale_simple( 128, 128, Gdk::INTERP_BILINEAR) );
                  }
                catch (...) {}
              }
            else
              {
                metadata_cover_current_album->set( m_audio_default );
              }

            metadata_label_current_artist->set_markup
                  ("<big><b>" + Glib::Markup::escape_text
                                (first.sortname? first.sortname.get() : first.artist.get())
                              + "</b></big>");

            metadata_label_current_album->set_markup
                  ("<big><i>" + Glib::Markup::escape_text (first.album.get()) + "</i></big>");

            if (first.date && first.date.get() != 0)
                metadata_label_current_date->set_markup ("<b>" + (format_int % first.date.get()).str() + "</b>");
            else
                metadata_label_current_date->set_text ("");

          }
        else if (list.size() == 0)
          {
            metadata_cover_current_album->set (m_audio_empty);
            metadata_label_current_album->set_text ("");
            metadata_label_current_artist->set_text ("");
            metadata_label_current_date->set_text ("");
          }
        else if (list.size() > 1)
          {
            m_view_tracks->unsort ();
            metadata_cover_current_album->set (m_audio_multiple);
            metadata_label_current_album->set_markup (_("<big><b>(Multiple Albums)</b></big>")); 
            metadata_label_current_artist->set_text ("");
            metadata_label_current_date->set_text ("");
          }

        for (AlbumV::const_iterator n = list.begin(); n != list.end(); ++n)
          {
            VRows v;
            library->query(album_to_attrs (*n, false), v);
            m_view_tracks->add_tracks( v );
          }

          // Check State Of Tracks
          bool all = m_view_tracks->all_tracks_present();

          m_actions->get_action (ACTION_EDIT_ALBUMS)->set_sensitive (all);
          m_actions->get_action (ACTION_MOVE_ALBUM)->set_sensitive (all && list.size() == 1);

          if (count)
            {
              bool approve = false;

              if (library->new_exist())
                {
                  PathList paths = m_view_albums->get_selection()->get_selected_rows ();
                  PathList::size_type new_items = 0;

                  for (PathList::const_iterator p = paths.begin(); p != paths.end(); ++p)
                    {
                      TreeModel::iterator const& m_iter (m_view_albums->get_model()->get_iter(*p));
                      Album const& a = (*m_iter)[m_view_albums->m_album.album]; 
                      if (a.new_item.get() == true) ++new_items;
                    }

                  approve = (new_items == paths.size()) ? true : false;
                }

              m_actions->get_action (ACTION_APPROVE_SELECTED)->set_sensitive (approve);
              m_actions->get_action (ACTION_REMOVE_ALBUMS)->set_sensitive (true);

              m_caps = Caps (m_caps | PlaybackSource::CAN_PLAY);
            }
          else
            {
              m_actions->get_action (ACTION_APPROVE_SELECTED)->set_sensitive (false);
              m_actions->get_action (ACTION_REMOVE_ALBUMS)->set_sensitive (false);

              m_caps = Caps (m_caps & ~PlaybackSource::CAN_PLAY);
            }

          m_actions->get_action (ACTION_NEW_APPROVE_ALL)->set_sensitive
                (library->new_exist());
          m_actions->get_action (ACTION_NEW_DROP_ALL)->set_sensitive
                (library->new_exist());

          on_continuous_play_toggled ();
      }

			void
			Albums::on_get_covers ()
			{
          notebook_albums->set_current_page (PAGE_PROCESSING);
          m_actions->set_sensitive (false);

          m_view_tracks->clear ();

          m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_NEXT);
          m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_PREV);
          m_caps = Caps (m_caps & ~PlaybackSource::CAN_PLAY);

          s_caps_.emit  (m_caps);

          m_view_albums->update_all (false);

          m_actions->get_action (ACTION_NEW_APPROVE_ALL)->set_sensitive
                (library->new_exist());
          m_actions->get_action (ACTION_NEW_DROP_ALL)->set_sensitive
                (library->new_exist());

          notebook_albums->set_current_page (PAGE_VIEW);
          m_actions->set_sensitive (true);
			}

      void
      Albums::on_new_drop_all ()
      {
        TaskDialog dialog (_("Music Library - BMP"),
                           _("Remove remaining Pending albums?"),
                           Gtk::MESSAGE_QUESTION,
                           _("TIP: If you just want to remove one or several "
                             "albums use the regular 'Remove Album' option."));
                                                  
        dialog.add_button ("Remove",  "Remove remaining Pending albums",
                                        Gtk::Stock::GO_FORWARD,
                                        GTK_RESPONSE_OK);

        dialog.add_button ("Keep",    "Cancels the Request",
                                        Gtk::Stock::CANCEL,
                                        GTK_RESPONSE_CANCEL);

        dialog.set_default_response (GTK_RESPONSE_CANCEL);

        if (dialog.run() == GTK_RESPONSE_OK)
          {
            dialog.hide ();

            library->new_drop_all ();
            m_view_albums->new_drop_all ();
  
            m_actions->get_action (ACTION_NEW_APPROVE_ALL)->set_sensitive
                (library->new_exist());
            m_actions->get_action (ACTION_NEW_DROP_ALL)->set_sensitive
                (library->new_exist());

            albums_cbox_viewmode->set_active (M_EVERYTHING);
#ifdef HAVE_HAL
            s_rescan_devices_.emit ();
#endif //HAVE_HAL
          }
      }

      void
      Albums::on_new_approve_sel ()
      {
        m_ref_xml->get_widget("albums-hpaned")->set_sensitive (false);
        m_view_albums->approve_sel ();
        m_ref_xml->get_widget("albums-hpaned")->set_sensitive (true);
      }

      void
      Albums::on_new_approve_all ()
      {
        m_ref_xml->get_widget("albums-hpaned")->set_sensitive (false);

        library->new_approve ();
        m_view_albums->new_approve_all ();

        m_actions->get_action (ACTION_NEW_APPROVE_ALL)->set_sensitive (false);
        m_actions->get_action (ACTION_NEW_DROP_ALL)->set_sensitive (false);

        albums_cbox_viewmode->set_active (M_EVERYTHING);

        m_ref_xml->get_widget("albums-hpaned")->set_sensitive (true);
      }

      void
      Albums::remove (bool unattended)
      {
        m_modify_lock.lock();
        if (m_busy)
          {
            while (m_busy)
             m_modify_cond.wait(m_modify_lock); 
          }
        m_busy = true;

        m_view_albums->remove (unattended);
#ifdef HAVE_HAL
        s_rescan_devices_.emit ();
#endif //HAVE_HAL

        m_busy = false;
        m_modify_cond.signal();
        m_modify_lock.unlock();
      }

      void
      Albums::on_relocate ()
      {
        m_modify_lock.lock();
        if (m_busy)
          {
            while (m_busy)
             m_modify_cond.wait(m_modify_lock); 
          }

        m_busy = 1;

        m_view_albums->set_sensitive (false); 
        m_view_tracks->set_sensitive (false); 

        if (m_view_albums->relocate()) on_albums_selection_changed ();

        m_view_albums->set_sensitive (true); 
        m_view_tracks->set_sensitive (true); 

        m_busy = 0;

        m_modify_cond.signal();
        m_modify_lock.unlock();
      }

      void
      Albums::on_modify ()
      {

#ifdef HAVE_HAL
        Bmp::VUriVrpPair l;
#else
        Bmp::VUri        l;
#endif //HAVE_HAL

        m_modify_lock.lock();

        if (m_busy)
          {
            while (m_busy)
             m_modify_cond.wait(m_modify_lock); 
          }

        m_busy = 1;

        Util::FileList list;
        m_view_tracks->get_files (list);
  
        Util::FileList non_writable;
        Util::files_writable (list, non_writable);
        if (!non_writable.empty ())
          {
            shared_ptr<DialogFilelist> dialog (DialogFilelist::create ());
            dialog->run (non_writable,
                         _("BMP is unable to perform the requested modifications "
                          "due to the following files being not writable:"));
            goto out;
          }

        if (!Util::files_taggable (list))
          {
            MessageDialog dialog (_("Unable to modify: Some of the selected/specified files are "
                                    "currently not taggable with BMP"),
                                    false, MESSAGE_ERROR, BUTTONS_OK, true);
            dialog.run();
            goto out;
          }


        m_view_albums->set_sensitive (false); 
        m_view_tracks->set_sensitive (false); 

        m_view_tracks->get_tracks (l);
        m_view_albums->modify (l);

        out:

        m_busy = 0;

        m_modify_cond.signal();
        m_modify_lock.unlock();

        m_view_albums->set_sensitive (true); 
        m_view_tracks->set_sensitive (true); 

        on_albums_selection_changed ();
        send_title ();
      }

      bool
      Albums::popup_menu_tracks (GdkEvent *ev)
      {
        using namespace Gtk;

        GdkEventButton *event = 0;
        if (ev->type == GDK_BUTTON_PRESS)
          {
            event = reinterpret_cast<GdkEventButton *>(ev);
            if ((event->button == 3) && (event->type == GDK_BUTTON_PRESS))
              {
                Menu *menu = dynamic_cast<Menu *>
                      (get_popup(m_ui_manager, "/popup-albums-tracklist/menu-albums-tracklist"));
                m_actions->get_action (ACTION_REMOVE_TRACKS)->set_sensitive
                      (m_view_tracks->get_selection()->count_selected_rows() > 0);
                m_actions->get_action (ACTION_ENQUEUE_TRACKS)->set_sensitive
                      (m_view_tracks->get_selection()->count_selected_rows() > 0);
                menu->popup (event->button, event->time);
                return true;
              }
          } 
        return false;
      }

      bool
      Albums::popup_menu_albums (GdkEvent *ev)
      {
        using namespace Gtk;

        GdkEventButton *event = 0;
        if (ev->type == GDK_BUTTON_PRESS)
          {
            event = reinterpret_cast<GdkEventButton *>(ev);
            if ((event->button == 3) && (event->type == GDK_BUTTON_PRESS))
              {
                bool i = m_view_albums->get_selection()->count_selected_rows();

                m_actions->get_action (ACTION_UNSELECT_ALL)->set_sensitive (i);

                dynamic_cast<Menu *>(get_popup(m_ui_manager, "/popup-albums-albumlist/menu-albums-albumlist"))->popup
                  (event->button, event->time);

                return true;
              }
          } 
        return false;
      }

      void
      Albums::on_tracks_remove ()
      {
        if (m_view_tracks->remove ())
        {
          remove (true);
#ifdef HAVE_HAL
          s_rescan_devices_.emit ();
#endif //HAVE_HAL
        }
      }

      void
      Albums::on_tracks_enqueue ()
      {
        using namespace Gtk;
        using namespace Glib;

        PathList const& paths (m_view_tracks->get_selection()->get_selected_rows());
        VUri list;
        for (PathList::const_iterator p = paths.begin(), e = paths.end(); p != e; ++p)
          {
            TreeModel::iterator const& m_iter = m_view_tracks->get_model()->get_iter (*p);
            list.push_back ((*m_iter)[m_view_tracks->m_track.location]);
          }
        s_uri_enqueue_request_.emit (list);
      }

      void
      Albums::import_external (VUri & uri_list)
      {
        if (uri_list.empty()) return;

        m_modify_lock.lock();
        if (m_busy)
          {
            while (m_busy)
             m_modify_cond.wait(m_modify_lock); 
          }
        m_busy = true;

        guint64   n_items   = uri_list.size();
        guint64   new_items = 0;
        guint64   upd_items = 0;
        guint64   nth_item  = 0; 

        ProgressDialog * progress_dialog = ProgressDialog::create ();

        for (VUri::const_iterator i = uri_list.begin (); i != uri_list.end(); ++i)
          {
            LibraryAddOp op = LIBRARY_ADD_NOOP;
            try {
                op = library->add ((*i));
                progress_dialog->add_file (op, *i,
                                            _("OK"));
              }
            catch (Library::NoMetadataError& cxe)
              {
                progress_dialog->add_file (LIBRARY_ADD_ERROR, *i,
                                            (boost::format (_("No Metadata for this File: %s")) % cxe.what()).str()); 
              }
#ifdef HAVE_HAL
            catch (Library::NoHALInformationError& cxe)
              {
                progress_dialog->add_file (LIBRARY_ADD_ERROR, *i,
                                            _("No HAL Volume/Device Information for this File")); 
              }
#endif //HAVE_HAL
            catch (Library::DatabaseError& cxe)
              {
                progress_dialog->add_file (LIBRARY_ADD_ERROR, *i,
                                            (boost::format (_("Database Error Occured: %s")) % cxe.what()).str()); 
              }
            catch (ConvertError& cxe) {} // FIXME:

            switch (op)
            {
              case LIBRARY_ADD_IMPORT: new_items++; break;
              case LIBRARY_ADD_UPDATE: upd_items++; break;
              default: break;
            }

            progress_dialog->set_progress(n_items, ++nth_item);
            while (gtk_events_pending ()) gtk_main_iteration ();
          }
      
        progress_dialog->done(new_items, upd_items, n_items);
        delete progress_dialog;

#ifdef HAVE_HAL
        s_rescan_devices_.emit ();
#endif //HAVE_HAL

        if (!new_items)
          {
            on_albums_selection_changed ();
          }
        else
          {
            albums_cbox_viewmode->set_active (M_NEW);
            update ();
          }

        m_busy = false;
        m_modify_cond.signal();
        m_modify_lock.unlock();
      }

      void
      Albums::on_tracks_import_mbtag ()
      {
        FileChooserDialog dialog (_("Select Tracks to Add as Album - BMP"), FILE_CHOOSER_ACTION_OPEN);

        dialog.set_select_multiple (true);
        dialog.add_button (StockID (Stock::CANCEL), RESPONSE_CANCEL);
        dialog.add_button (StockID (Stock::OPEN), RESPONSE_OK);
        dialog.set_default_response (RESPONSE_CANCEL);
        dialog.set_current_folder (mcs->key_get<string>("bmp", "file-chooser-path"));

        if (dialog.run () == RESPONSE_OK)
          {
            dialog.hide ();
            mcs->key_set<string>("bmp", "file-chooser-path", dialog.get_current_folder());

            SListHandle<ustring> handle = dialog.get_uris (); 
            VUri uri_list (handle.begin(), handle.end());

            if (uri_list.empty()) 
              return;

            // Check for writability
            Util::FileList list, non_writable;
            for (VUri::const_iterator iter = uri_list.begin (); iter != uri_list.end(); ++iter)
              list.push_back (filename_from_uri(*iter));

            Util::files_writable (list, non_writable);
            if (!non_writable.empty ())
              {
                shared_ptr<DialogFilelist> dialog (DialogFilelist::create ());
                dialog->run (non_writable,
                             _("BMP is unable to perform the requested modifications "
                                "due to the following files being not writable:"));
                return;
              }

            // Check for taggability
            if (!Util::files_taggable (list))
              {
                MessageDialog dialog (_("Unable to modify: Some of the selected/specified files are currently "
                                          "not taggable with BMP"),
                                          true, MESSAGE_ERROR, BUTTONS_OK, true);
                dialog.run ();
                return;
              }

            shared_ptr<LibraryUiImport> dialog (LibraryUiImport::create());
            Library::UTrackV tracks;

            if (dialog->run (uri_list, tracks) == RESPONSE_OK)
              {
                dialog->hide();
                m_modify_lock.lock();
                if (m_busy)
                  {
                    while (m_busy)
                     m_modify_cond.wait(m_modify_lock); 
                  }
                m_busy = true;

                guint64   n_items   = uri_list.size();
                guint64   new_items = 0;
                guint64   upd_items = 0;
                guint64   nth_item  = 0; 

                ProgressDialog * progress_dialog = ProgressDialog::create ();

                for (Library::UTrackV::iterator i = tracks.begin (); i != tracks.end(); ++i)
                  {
                    Library::UTrack & track (*i);
                    LibraryAddOp op = LIBRARY_ADD_NOOP;

                    try{
                        library->metadata_set_taglib (track.location_cur.get(), track); 
                        op = library->add (track.location_cur.get());
                        progress_dialog->add_file (op, Glib::filename_from_uri (track.location_cur.get()),
                                                    _("OK"));
                      }
                   catch (Library::MetadataWriteError& cxe)
                      {
                        progress_dialog->add_file (LIBRARY_ADD_ERROR, Glib::filename_from_uri (track.location_cur.get()),
                                                    cxe.what()); 
                      }
                   catch (Library::NoMetadataError& cxe)
                      {
                        progress_dialog->add_file (LIBRARY_ADD_ERROR, Glib::filename_from_uri (track.location_cur.get()),
                                                    _("No Metadata for this File")); 
                      }
#ifdef HAVE_HAL
                    catch (Library::NoHALInformationError& cxe)
                      {
                        progress_dialog->add_file (LIBRARY_ADD_ERROR, Glib::filename_from_uri (track.location_cur.get()),
                                                    _("No HAL Volume/Device Information for this File")); 
                      }
#endif //HAVE_HAL
                    catch (Library::DatabaseError& cxe)
                      {
                        progress_dialog->add_file (LIBRARY_ADD_ERROR, Glib::filename_from_uri (track.location_cur.get()),
                                                    _("Database Error Occured")); 
                      }
                    catch (ConvertError& cxe) {} // FIXME:

                    switch (op)
                    {
                      case LIBRARY_ADD_IMPORT: new_items++; break;
                      case LIBRARY_ADD_UPDATE: upd_items++; break;
                      default: break;
                    }

                    progress_dialog->set_progress(n_items, ++nth_item);

                  }

                progress_dialog->done(new_items, upd_items, n_items); 
                delete progress_dialog;

#ifdef HAVE_HAL
                s_rescan_devices_.emit ();
#endif //HAVE_HAL

                if (!new_items)
                  {
                    on_albums_selection_changed ();
                  }
                else
                  {
                    albums_cbox_viewmode->set_active (M_NEW);
                    update ();
                  }

                m_busy = false;
                m_modify_cond.signal();
                m_modify_lock.unlock();
              }
          }
      }

      void
      Albums::on_tracks_import ()
      {
        FileChooserDialog dialog (_("Select Path To Add - BMP"),
                                  FILE_CHOOSER_ACTION_SELECT_FOLDER);

        dialog.add_button (Stock::CANCEL, RESPONSE_CANCEL);
        dialog.add_button (Stock::ADD, RESPONSE_OK);
        dialog.set_current_folder (mcs->key_get<string>("bmp", "file-chooser-path"));
        dialog.set_default_response (RESPONSE_CANCEL);
        dialog.set_default_size (750, 570);
        dialog.set_position (WIN_POS_CENTER);

        if (dialog.run () == RESPONSE_OK)
          {
            mcs->key_set<string>("bmp", "file-chooser-path", dialog.get_current_folder());
            ustring folder = dialog.get_filename ();
            dialog.hide ();

            m_modify_lock.lock();
            if (m_busy)
              {
                while (m_busy)
                 m_modify_cond.wait(m_modify_lock); 
              }
            m_busy = true;

            Util::FileList list;
            Util::collect_path (folder, list);

            guint64   n_items   = list.size();
            guint64   new_items = 0;
            guint64   upd_items = 0;
            guint64   nth_item  = 0; 

            ProgressDialog * progress_dialog = ProgressDialog::create ();

            LibraryAddOp op = LIBRARY_ADD_NOOP;

            for(Util::FileList::const_iterator i = list.begin (); i != list.end(); ++i)
              {
                try{
                    debug ("library-ui", "Adding URI '%s' to Library", (*i).c_str()); 
                    op = library->add (filename_to_uri (*i));
                    progress_dialog->add_file (op, *i, _("OK"));
                  }
                catch (Library::NoMetadataError& cxe)
                  {
                    progress_dialog->add_file (LIBRARY_ADD_ERROR, *i,
                                                (boost::format (_("No Metadata for this File: %s")) % cxe.what()).str()); 
                  }
#ifdef HAVE_HAL
                catch (Library::NoHALInformationError& cxe)
                  {
                    progress_dialog->add_file (LIBRARY_ADD_ERROR, *i,
                                                _("No HAL Volume/Device Information for this File")); 
                  }
#endif //HAVE_HAL
                catch (Library::DatabaseError& cxe)
                  {
                    progress_dialog->add_file (LIBRARY_ADD_ERROR, *i,
                                                (boost::format (_("Database Error Occured")) % cxe.what()).str()); 
                  }
                catch (ConvertError& cxe) {} // FIXME:

                switch (op)
                {
                  case LIBRARY_ADD_IMPORT: new_items++; break;
                  case LIBRARY_ADD_UPDATE: upd_items++; break;
                  default: break;
                }

                progress_dialog->set_progress (n_items, ++nth_item);
              }
          
            progress_dialog->done (new_items, upd_items, n_items);
            delete progress_dialog;

#ifdef HAVE_HAL
            s_rescan_devices_.emit ();
#endif //HAVE_HAL

            if (!new_items)
              {
                on_albums_selection_changed ();
              }
            else
              {
                albums_cbox_viewmode->set_active (M_NEW);
                update ();
              }

            m_busy = false;
            m_modify_cond.signal();
            m_modify_lock.unlock();
          }
      }

      void
      Albums::update (bool update_new)
      {
          notebook_albums->set_current_page (PAGE_PROCESSING);
          m_actions->set_sensitive (false);

          if (!update_new)
            {
              m_view_tracks->clear ();
              m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_NEXT);
              m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_PREV);
              m_caps = Caps (m_caps & ~PlaybackSource::CAN_PLAY);
              s_caps_.emit  (m_caps);
            }

          if (update_new)
            m_view_albums->update_new();
          else
            m_view_albums->update_all();

          update_completion ();

          m_actions->get_action (ACTION_NEW_APPROVE_ALL)->set_sensitive
                (library->new_exist());
          m_actions->get_action (ACTION_NEW_DROP_ALL)->set_sensitive
                (library->new_exist());

          notebook_albums->set_current_page (PAGE_VIEW);
          m_actions->set_sensitive (true);
      }

      void
      Albums::on_remove ()
      {
        remove (false);
      }

      bool
      Albums::on_sort_entry_changed_timeout_func ()
      {
        if (entry_changed_timer.elapsed() >= KEY_PRESS_MAX_TIME) 
          {
            entry_changed_timer.stop();
            m_view_albums->set_sorting (m_albums_search->get_text());
            return false;
          }
        return true;
      }

      void
      Albums::on_sort_entry_changed ()
      {
        sort_entry_changed_timeout_conn.disconnect();
        entry_changed_timer.reset();
        entry_changed_timer.start();
        sort_entry_changed_timeout_conn = Glib::signal_timeout().connect
              (sigc::mem_fun (this, &Bmp::UiPart::Albums::on_sort_entry_changed_timeout_func),
                                    (unsigned int)(1000. * KEY_PRESS_MAX_TIME));
      }

      void
      Albums::on_viewmode_changed ()
      {
        switch (albums_cbox_viewmode->get_active_row_number())
          {
                case 0:
                    m_view_albums->set_filtering (false, false);
                    break;
                case 1:
                    m_view_albums->set_filtering (false, true);
                    break;
                case 2:
                    m_view_albums->set_filtering (true, false);
                    break;
          }

        m_actions->get_action (ACTION_NEW_APPROVE_ALL)->set_sensitive
              (library->new_exist());
        m_actions->get_action (ACTION_NEW_DROP_ALL)->set_sensitive
              (library->new_exist());
      }

      // Bmp::PlaybackSource
      ustring
      Albums::get_uri ()
      {
        return m_view_tracks->playback_uri ();
      }

      GHashTable*
      Albums::get_metadata ()
      {
        return library->get_hashtable (::play->property_stream().get_value());
      }
      
      void
      Albums::send_title ()
      {
        ustring uri;

        try { uri = m_view_tracks->playback_uri(); }
        catch (Bmp::TrackView::NoCurrentUriError& cxe) { return; }

        Track track;
        library->get (uri, track);

        oustring eartist, ealbum, etitle;
        ustring artist, album, title;

        SimpleTrackInfo sti;

        if (track.artist)
          {
            eartist = Markup::escape_text (track.artist.get()); 
            sti.artist = track.artist;
          }

        if (track.album)
          {
            ealbum  = Markup::escape_text (track.album.get()); 
            sti.album = track.album;
          }

        if (track.title)
          {
            etitle = Markup::escape_text (track.title.get()); 
            sti.title = track.title;
          }

        if (track.asin)
          {
            sti.asin = track.asin;
          }

        if (track.duration)
          {
            sti.duration = track.duration;
          }


        if (eartist)
          {
            artist.append (eartist.get());
          }

        if (ealbum)
          {
            album.append ("<i>")
                 .append (ealbum.get())
                 .append ("</i>"); 
          }

        if (etitle)
          {
            title.append ("<b>")
                 .append (etitle.get())
                 .append ("</b>");
          }

        if (track.date && (track.date.get() > 0))
          {
            album.append (", ")
                 .append ((format_int % track.date.get()).str()); 
          }

        char *text = g_strdup_printf ("%s (%s%s%s)", (etitle                    ? title.c_str()       : ""  ),
                                                     (eartist                   ? artist.c_str()      : ""  ),
                                                     ((!eartist || (!ealbum))   ? ""                  : ": "),
                                                     (ealbum                    ? album.c_str()       : ""  ));

        s_track_info_.emit (ustring(text), sti); 
        g_free (text);
      }

      bool
      Albums::go_next ()
      {
        unsigned int position, size;

        if (m_view_tracks->notify_next (position, size))
          {

            m_view_albums->save_context ();
            m_view_tracks->save_context ();
            m_saved_context_filter = m_albums_search->get_text ();

            if (position+1 != size)
              m_caps = Caps (m_caps |  PlaybackSource::CAN_GO_NEXT);
            else
              m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_NEXT);

            if (position > 0) 
              m_caps = Caps (m_caps |  PlaybackSource::CAN_GO_PREV);
            else
              m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_PREV);

            s_caps_.emit (m_caps);
            send_title ();    
            return true;
          }
        return false;
      }

      bool
      Albums::go_prev ()
      {
        unsigned int position, size;
        if (m_view_tracks->notify_prev (position, size))
          {

            m_view_albums->save_context ();
            m_view_tracks->save_context ();
            m_saved_context_filter = m_albums_search->get_text ();

            if (position+1 != size)
              m_caps = Caps (m_caps |  PlaybackSource::CAN_GO_NEXT);
            else
              m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_NEXT);

            if (position > 0) 
              m_caps = Caps (m_caps |  PlaybackSource::CAN_GO_PREV);
            else
              m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_PREV);

            s_caps_.emit (m_caps);
            send_title ();    
            return true;
          }
        return false;
      }

      void
      Albums::stop ()
      {
        m_view_tracks->notify_stop ();

        m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_NEXT);
        m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_PREV);
        m_caps = Caps (m_caps & ~PlaybackSource::CAN_PAUSE);
        m_caps = Caps (m_caps & ~PlaybackSource::CAN_PROVIDE_METADATA);

        s_caps_.emit (m_caps);
      }

      void
      Albums::play ()
      {
        m_view_albums->save_context ();
        m_view_tracks->save_context ();
        m_saved_context_filter = m_albums_search->get_text ();
        
        unsigned int position, size;
        if (m_view_tracks->notify_play (position, size))
          {
            if (position+1 != size)
              m_caps = Caps (m_caps |  PlaybackSource::CAN_GO_NEXT);
            else
              m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_NEXT);

            if (position > 0) 
              m_caps = Caps (m_caps |  PlaybackSource::CAN_GO_PREV);
            else
              m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_PREV);
        }

        m_caps = Caps (m_caps | PlaybackSource::CAN_PAUSE);
        m_caps = Caps (m_caps | PlaybackSource::CAN_PROVIDE_METADATA);
        s_caps_.emit (m_caps);
        send_title ();
      }

      void
      Albums::restore_context ()
      {
        if (m_saved_context_filter)
          {
            m_albums_search->set_text (m_saved_context_filter.get());
            m_view_albums->set_sorting (m_saved_context_filter.get());
          }

        m_view_tracks->restore_context ();
        m_view_albums->restore_context ();
      } 

      void
      Albums::on_tracks_selection_changed ()
      {
        bool albums_has_selection = (m_view_albums->get_selection()->count_selected_rows() > 0);
        bool tracks_has_selection = (m_view_tracks->get_selection()->count_selected_rows() == 1);

        m_actions->get_action (ACTION_REMOVE_TRACKS)->set_sensitive
                      ((m_view_tracks->get_selection ()->count_selected_rows() > 0));

        if (albums_has_selection && tracks_has_selection)
          m_caps = Caps (m_caps | PlaybackSource::CAN_PLAY);
        else
          m_caps = Caps (m_caps & ~PlaybackSource::CAN_PLAY);

        s_caps_.emit (m_caps);
      }

      void
      Albums::progress_start (int n_mod_tracks)
      {
        m_ref_xml->get_widget("albums-hpaned")->set_sensitive (false);
        m_actions->set_sensitive (false);

        m_n_mod_tracks = n_mod_tracks; 
        m_n_mod_current = 0;
        m_progress->set_fraction (0.0);
        m_progress->set_text ((format_progress_text % 0 % n_mod_tracks).str());
        m_progress->show ();

        while (gtk_events_pending()) gtk_main_iteration ();
      }

      void
      Albums::progress_step ()
      {
        m_n_mod_current++;
        m_progress->set_fraction ((m_n_mod_current*1.)/(m_n_mod_tracks*1.));
        m_progress->set_text ((format_progress_text % m_n_mod_current % m_n_mod_tracks).str());
        while (gtk_events_pending()) gtk_main_iteration ();
      }

      void
      Albums::progress_stop ()
      {
        m_progress->hide ();
        m_ref_xml->get_widget("albums-hpaned")->set_sensitive (true);
        m_actions->set_sensitive (true);

        m_n_mod_tracks = 0;
        m_n_mod_current = 0;
        m_progress->set_fraction (0.0);
        m_progress->set_text ("");

        while (gtk_events_pending()) gtk_main_iteration ();
      }
  }
}
