//  BMPx - The Dumb Music Player
//  Copyright (C) 2005 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <cstdlib>
#include <cstring>
#include <cctype>

#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <vector>
#include <iostream>
#include <iterator>
#include <string>

#include <gtkmm.h>
#include <glib.h>
#include <glib/gstdio.h>
#include <glib/gi18n.h>

#include <boost/algorithm/string.hpp>
#include <boost/regex.hpp>

#include "util.hh"
#include "md5.h"
#include "uri++.hh"

#include "neon++/request.hh"

#define BOOST_REGEX_MATCH_EXTRA 1

namespace
{
  const struct {
      const char *exp;
      const char *fmt;
  } regexices[] = {

     { "(<\\s*div\\s*[^>]*?\\s*>)([^>]*?)(<\\s*/\\s*div\\s*>)",
       "(?1)(?2)(?3)"},

     { "(<img\\s+src\\s*=\")([^\">]*?)(\"\\s*?/?>)",
       "Image '$2'"},

     { "(<a\\s+href\\s*=\")([^\">]*?)(\".*?>)([^>]*?)(<\\s*/\\s*a>)",
       "$2\\: \"$4\""},

     { "(<\\s*p\\s*>)|(<\\s*/\\s*p\\s*>)",
       "(?1\\n)(?2\\n)" },

     { "(<br\\s*/\\s*>)",
       "(?1\\n)" },
  };

  // This table of RFC822 timezones is from gmime-utils.c of the gmime API */
  const struct {
    char *name;
    int   offset;
  } tz_offsets [] = {
    { "UT", 0 },
    { "GMT", 0 },
    { "EST", -500 },        // These are all US timezones.  Bloody Yanks!!
    { "EDT", -400 },
    { "CST", -600 },
    { "CDT", -500 },
    { "MST", -700 },
    { "MDT", -600 },
    { "PST", -800 },
    { "PDT", -700 },
    { "Z", 0 },
    { "A", -100 },
    { "M", -1200 },
    { "N", 100 },
    { "Y", 1200 }
  };

  // Returns timezone offset in seconds
  // Code (C) Liferea Developers
  time_t common_parse_rfc822_tz (char *token)
  {
    int offset = 0;
    const char *inptr = token;

    if (*inptr == '+' || *inptr == '-')
    {
      offset = atoi (inptr);
    }
    else
    {
      int t;

      if (*inptr == '(') inptr++;

      for (t = 0; t < 15; t++)
      {
        if (!strncmp (inptr, tz_offsets[t].name, strlen (tz_offsets[t].name))) offset = tz_offsets[t].offset;
      }
    }
    return 60 * ((offset / 100) * 60 + (offset % 100));
  }
}

namespace Bmp
{
  namespace Util
  {

    bool
    str_has_prefix_nocase (const gchar *str, const gchar *prefix)
    {
        if ((!str) || (!prefix)) return false;
        return (strncasecmp (str, prefix, std::strlen (prefix)) == 0);
    }

    bool
    str_has_suffix_nocase (const gchar *str, const gchar *_suffix)
    {
        gchar *suffix_fq;
        gboolean rv;

        if ((!str) || (!_suffix)) return false;

        suffix_fq = g_strconcat (".", _suffix, NULL);
        rv = (strcasecmp (str + std::strlen (str) - std::strlen (suffix_fq), suffix_fq) == 0);
        g_free (suffix_fq);
        return rv;
    }

    bool
    str_has_suffixes_nocase (const char * str,
                             char * const * suffixes)
    {
        char *const *suffix;

        g_return_val_if_fail (str != 0, false);
        g_return_val_if_fail (suffixes != 0, false);

        for (suffix = suffixes; *suffix; suffix++)
          {
            if (str_has_suffix_nocase (str, *suffix)) return true;
          }

        return false;
    }

    bool
    str_has_suffixes_nocase_on_crack (std::string const& str,
                                      StrV const& strv)
    {
        if (str.empty())
          return false;

        for (StrV::const_iterator i = strv.begin() ; i != strv.end() ; ++i)
          {
            if (str_has_suffix_nocase (str.c_str(), i->c_str()))
              return true;
          }
        return false;
    }

    bool
    match_keys (Glib::ustring const& _h,
                Glib::ustring const& _n)
    {
        using boost::algorithm::split;
        using boost::algorithm::is_any_of;
        using boost::algorithm::find_first;

        StrV m;    

        std::string n (_n.lowercase());
        std::string h (_h.lowercase());

        split( m, n, is_any_of(" ") );

        for (StrV::const_iterator i = m.begin (); i != m.end (); ++i)
          {
            if (!find_first (h, (*i)))
              return false;
          }

        return true;
    }

    char*
    hexify (char *pass, int len)
    {
      char *hash;
      char *bp;
      char hexchars[] = "0123456789abcdef";
      int i;

      hash = static_cast<char *> (g_malloc0 (33));
      bp = hash;

      for(i = 0; i < len; i++)
        {
          *(bp++) = hexchars[(pass[i] >> 4) & 0x0f];
          *(bp++) = hexchars[pass[i] & 0x0f];
        }
      *bp = 0;
      return hash;
    }

    std::string
    md5_hex (char* const data, size_t data_size)
    {
        md5_state_t  md5state;
        char         md5pword[16];

        md5_init (&md5state);
        md5_append (&md5state, (unsigned const char *)data, data_size); 
        md5_finish (&md5state, (unsigned char *)md5pword);
        char *md5_hex = Util::hexify (md5pword, sizeof(md5pword));
        std::string md5_hex_std = md5_hex;
        free (md5_hex);
        return md5_hex_std;
    }

    std::string 
    get_timestr (time_t t, int gmt)
    {
      struct tm *tm;
      static char buf[30];

      tm = gmt ? gmtime(&t) : localtime(&t);
      snprintf(buf, sizeof(buf), "%d-%.2d-%.2d %.2d:%.2d:%.2d",
          tm->tm_year + 1900,
          tm->tm_mon + 1,
          tm->tm_mday,
          tm->tm_hour,
          tm->tm_min,
          tm->tm_sec);
      return std::string(buf);
    }

    // HTML Sanitizer based on regex example code Copyright (c) 1998-2002 John Maddock
    Glib::ustring 
    sanitize_html (Glib::ustring const& in)
    {

      std::string out = in;

      try {

        boost::regex e1;

        for (unsigned int n = 0; n < G_N_ELEMENTS(regexices); ++n)
          {
            e1.assign(regexices[n].exp);
            out = boost::regex_replace(out, e1, regexices[n].fmt,
                             boost::match_default | boost::format_all | boost::match_extra);
          }

       }
      catch (...) {}

      return Glib::ustring(out);
    }

    // Converts a RFC822 time string to a time_t value
    // Code (C) Liferea Developers
    time_t parseRFC822Date (const char * date)
    {
      struct tm	tm;
      time_t t, t2;
      char 	 *oldlocale;
      char	 *pos;
      gboolean	success = FALSE;

      memset (&tm, 0, sizeof(struct tm));

      // We expect at least something like "03 Dec 12 01:38:34" 
      // and don't require a day of week or the timezone.
      //
      // The most specific format we expect:  "Fri, 03 Dec 12 01:38:34 CET"

      // Skip day of week
      if (NULL != (pos = g_utf8_strchr(date, -1, ','))) date = ++pos;

      // We expect English month names, so we set the locale
      oldlocale = g_strdup (setlocale (LC_TIME, NULL));
      setlocale (LC_TIME, "C");
      
      // Standard format with seconds and 4 digit year
      if (0 != (pos = strptime((const char *)date, " %d %b %Y %T", &tm)))
        success = TRUE;
      // Non-standard format without seconds and 4 digit year
      else if (0 != (pos = strptime((const char *)date, " %d %b %Y %H:%M", &tm)))
        success = TRUE;
      // Non-standard format with seconds and 2 digit year
      else if (0 != (pos = strptime((const char *)date, " %d %b %y %T", &tm)))
        success = TRUE;
      // Non-standard format without seconds 2 digit year
      else if (0 != (pos = strptime((const char *)date, " %d %b %y %H:%M", &tm)))
        success = TRUE;
      
      while (pos != 0 && *pos != '\0' && isspace((int)*pos)) pos++; // skip whitespaces before timezone
      
      if (0 != oldlocale)
      {
        setlocale (LC_TIME, oldlocale);	// Reset Locale 
        g_free (oldlocale);
      }
      
      if (TRUE == success)
      {
        if((time_t)(-1) != (t = mktime(&tm)))
        {

          //
          // GMT time, with no daylight savings time
          // correction. (Usually, there is no daylight savings
          // time since the input is GMT.) 
          //

          t = t - common_parse_rfc822_tz (pos);
          t2 = mktime (gmtime(&t));
          t = t - (t2 - t);
          return t;
        }
        else
        {
          g_warning ("Internal error! time conversion error! mktime failed!\n");
        }
      }
      
      return 0;
    }




    std::string 
    get_time ()
    {
      struct tm *tm;
      static char buf[30];

      time_t t = time (NULL);
      tm = gmtime (&t);
      snprintf(buf, sizeof(buf), "%d-%.2d-%.2dT%.2d:%.2d:%.2d",
          tm->tm_year + 1900,
          tm->tm_mon + 1,
          tm->tm_mday,
          tm->tm_hour,
          tm->tm_min,
          tm->tm_sec);
      return std::string(buf);
    }

    Glib::RefPtr<Gdk::Pixbuf> 
    get_image_from_uri (Glib::ustring const& uri)
    {
      Bmp::URI u (uri);
      Neon::Request request (u.hostname, u.path);

      try {
        request.dispatch ();
        }
      catch (Neon::Request::Exception &cxe)
        {
          g_log (G_LOG_DOMAIN, G_LOG_LEVEL_WARNING, "Error loading image from uri %s: %s", uri.c_str(), cxe.what());
          return Glib::RefPtr<Gdk::Pixbuf>(0);
        }

      std::string format;

      if ((str_has_suffix_nocase (uri.c_str(), "jpeg")) || (str_has_suffix_nocase (uri.c_str(), "jpg")))
          format = "jpeg";
      else if (str_has_suffix_nocase (uri.c_str(), "gif"))
          format = "gif";
      else if (str_has_suffix_nocase (uri.c_str(), "png"))
          format = "png";

      Glib::RefPtr<Gdk::Pixbuf> image;
    
      try
        {
          Glib::RefPtr<Gdk::PixbufLoader> loader = Gdk::PixbufLoader::create (format);
          loader->write (reinterpret_cast<const guint8*>(request.get_data()), request.get_data_size());
          request.clear ();
          loader->close ();
          image = loader->get_pixbuf();
        }
      catch (...) {}
        return image;
    }
  }
}

