//-*- Mode: C++; indent-tabs-mode: nil; -*-

//  BMPx - The Dumb Music Player
//  Copyright (C) 2005-2007 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License Version 2
//  as published by the Free Software Foundation.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <gtk/gtk.h>
#include <gtkmm.h>
#include <glibmm.h>
#include <glibmm/i18n.h>

#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <cctype>
#include <iostream>
#include <fstream>
#include <vector>

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <strings.h>

#include <boost/format.hpp>
#include <boost/regex.hpp>
#include <boost/algorithm/string.hpp>
#include <boost/algorithm/string/regex.hpp>

#include "x_core.hh"

#include "main.hh"
#include "paths.hh"
#include "stock.hh"
#include "util.hh"

#include "lastfm.hh"

#include "lastfm-tag-dialog.hh"

using namespace Gtk;
using namespace Glib;
using namespace Markup;
using namespace std;
using namespace Bmp;
using namespace LastFM;
using namespace LastFM::XMLRPC;
using namespace LastFM::WS;
using boost::algorithm::split;
using boost::algorithm::split_regex;
using boost::algorithm::is_any_of;
using boost::algorithm::find_nth;
using boost::iterator_range;

namespace Bmp
{
  namespace LastFM
  {
      TagDialog*
      TagDialog::create ()
      {
        const std::string path (build_filename (BMP_GLADE_DIR, "dialog-lastfm-tag.glade"));
        RefPtr<Gnome::Glade::Xml> glade_xml = Gnome::Glade::Xml::create (path);
        TagDialog *i = 0;
        glade_xml->get_widget_derived ("tag", i);
        return i;
      }

      TagDialog::~TagDialog ()
      {}

      TagDialog::TagDialog (BaseObjectType*                  obj,
                            RefPtr<Gnome::Glade::Xml> const& xml)
      : Gtk::Dialog (obj)
      , m_ref_xml   (xml)
      , m_selection_clear_block (0)
      {
        dynamic_cast <Gtk::Image *>(m_ref_xml->get_widget ("image"))->set
            (build_filename (BMP_IMAGE_DIR, BMP_THROBBER));

        m_ref_xml->get_widget ("e_artist", e_artist);
        m_ref_xml->get_widget ("e_album",  e_album);
        m_ref_xml->get_widget ("e_title",  e_title);

        m_ref_xml->get_widget ("l_artist", l_artist);
        m_ref_xml->get_widget ("l_album",  l_album);
        m_ref_xml->get_widget ("l_title",  l_title);

        m_ref_xml->get_widget ("notebook", notebook);
        notebook->set_current_page (0);

        m_ref_xml->get_widget ("treeview", tag_view);
        {
          CellRendererText *cell_text = Gtk::manage (new Gtk::CellRendererText ());
          tag_view->append_column ("", *cell_text);
          tag_view->get_column (0)->add_attribute (*cell_text, "text", 0);
          tag_view->get_column (0)->set_resizable (false);
        }
        tag_view->get_selection()->signal_changed().connect (sigc::mem_fun (*this, &LastFM::TagDialog::on_tag_selected));

        m_ref_xml->get_widget ("tag_name", m_tag_name);
        m_tag_name->signal_changed().connect (sigc::mem_fun (*this, &LastFM::TagDialog::on_tag_name_modified));

        m_ref_xml->get_widget ("tag_cbox", m_tag_cbox);
        m_tag_cbox->signal_changed().connect (sigc::mem_fun (*this, &LastFM::TagDialog::on_tag_cbox_changed));
        m_ref_xml->get_widget ("cover_image", m_cover);
      }

      void
      TagDialog::run (XSPF::Item const& item)
      {
        m_item = item;
        m_cover->clear ();

        if (!item.image.empty())
        {
          try{
              m_cover->set (Util::get_image_from_uri (item.image)->scale_simple (64, 64, Gdk::INTERP_BILINEAR));
            }
          catch (...) {}
        }
        else
        {    
          m_cover->set (Gdk::Pixbuf::create_from_file (build_filename (BMP_IMAGE_DIR_SOURCES, "lastfm.png"))
            ->scale_simple (64, 64, Gdk::INTERP_BILINEAR));
        }

        e_artist->set_text  (item.creator);
        e_album->set_text   (item.album);
        e_title->set_text   (item.title);

        show ();
        m_tag_cbox->set_active (2);

        update_view ();

        if (Dialog::run() == Gtk::RESPONSE_OK)
        {
          try {
            tag ();
            }
          catch (LastFM::Exception& cxe)
          {
            MessageDialog dialog (cxe.what(), false, MESSAGE_ERROR, BUTTONS_OK, true);
            dialog.run ();
          }
        }

        hide ();
      }

      void
      TagDialog::update_view ()
      {
        notebook->set_current_page (1);

        tag_store = ListStore::create (cr);
        tag_store->set_default_sort_func (sigc::mem_fun (*this, &LastFM::TagDialog::sort_func));
        tag_store->set_sort_column (Gtk::TreeSortable::DEFAULT_SORT_COLUMN_ID, Gtk::SORT_ASCENDING);

        using namespace LastFM::WS;
  
        TagV v, vdest;
        tags_glob (TAGS_GLOBAL_ARTIST, v, m_item.creator);
        tags_glob (TAGS_GLOBAL_TRACK,  v, m_item.creator, m_item.title);

        typedef std::set<ustring> Keeper;
        Keeper k;

        for (TagV::const_iterator i = v.begin(); i != v.end(); ++i)
        {
          if (k.find (i->name) == k.end())
          {
            vdest.push_back (*i);  
            k.insert (i->name);
          }
        }

        std::sort (vdest.begin(), vdest.end());

        for (TagV::const_iterator i = v.begin(); i != v.end(); ++i)
        {
          TreeIter iter = tag_store->append ();
          (*iter)[cr.name] = i->name;
        }

        tag_view->set_model (tag_store);
        m_tag_cbox->set_active (0);
        notebook->set_current_page (0);
      }

      int
      TagDialog::sort_func (Gtk::TreeModel::iterator const& m_iter1,
                            Gtk::TreeModel::iterator const& m_iter2)
      {
        ustring name1 ((*m_iter1)[cr.name]);
        ustring name2 ((*m_iter2)[cr.name]);

        return name1.compare (name2);
      }

      void
      TagDialog::on_tag_cbox_changed ()
      {
        int current = m_tag_cbox->get_active_row_number();

        // check whether album is empty; it's a possible case
        if (current == 1)
        {
          set_response_sensitive (Gtk::RESPONSE_OK, !m_item.album.empty()); 
        }

        e_artist->set_sensitive (false);
        e_album->set_sensitive (false);
        e_title->set_sensitive (false);

        l_artist->set_sensitive (false);
        l_album->set_sensitive (false);
        l_title->set_sensitive (false);

        switch (current)
        {
          case 0:
          {
            e_artist->set_sensitive (true);
            l_artist->set_sensitive (true);
            break;
          }

          case 1:
          {
            e_artist->set_sensitive (true);
            l_artist->set_sensitive (true);
            e_album->set_sensitive (true);
            l_album->set_sensitive (true);
            break;
          }

          case 2:
          {
            e_artist->set_sensitive (true);
            l_artist->set_sensitive (true);
            e_title->set_sensitive (true);
            l_title->set_sensitive (true);
            break;
          }
        }
      }

      void
      TagDialog::on_tag_selected ()
      {
        if (tag_view->get_selection()->count_selected_rows())
        {
          m_selection_clear_block = 1;
          TreeIter iter = tag_view->get_selection()->get_selected();
          m_tag_name->set_text (ustring ((*iter)[cr.name]));
        }
      }

      void
      TagDialog::on_tag_name_modified ()
      {
        if (!m_selection_clear_block)
        {
          tag_view->get_selection()->unselect_all ();
        }
        else
        {
          m_selection_clear_block = 0;
        }
      }

      void
      TagDialog::tag ()
      {
        using namespace Bmp::LastFM::XMLRPC;

        if (!m_tag_name->get_text().size())
          return;

        ustring method;
        switch (m_tag_cbox->get_active_row_number())
        {
          case 0:
            method = "tagArtist";
            break;

          case 1:
            method = "tagAlbum";
            break;

          case 2:
            method = "tagTrack";
            break;
        }

        UStrV v;
        v.push_back (m_tag_name->get_text());
        TagAction action (method, m_item, v); 
        action.run ();
      }
  }
}
