//  BMPx - The Dumb Music Player
//  Copyright (C) 2005-2007 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <glib/gstdio.h>
#include "minisoup.hh"

namespace Bmp
{
  namespace Soup
  {
    // Synchronous Request 

    Glib::RefPtr<RequestSync>
    RequestSync::create (std::string const& url, bool post)
    {
      return Glib::RefPtr<RequestSync>(new RequestSync (url, post));  
    }

    RequestSync::RequestSync (std::string const& url, bool post)
    : m_url   (url)
    , m_post  (post)
    {
      m_session = soup_session_sync_new ();
      m_message = soup_message_new (m_post ? "POST" : "GET", m_url.c_str());
    }
    
    guint
    RequestSync::run ()
    {
      g_signal_connect (G_OBJECT (m_message), "got-chunk", G_CALLBACK (got_chunk), this);
      soup_message_add_header_handler (m_message, "content-length", SOUP_HANDLER_PRE_BODY,
                                       SoupMessageCallbackFn (got_content_length), this);
      return soup_session_send_message (m_session, m_message);
    }
  
    std::string
    RequestSync::get_data ()
    {
      std::string buffer;
      buffer.append (m_message->response.body, m_message->response.length);
      return buffer;
    }

    void
    RequestSync::get_data (std::string & data)
    {
      data.append (m_message->response.body, m_message->response.length);
    }

    char const*
    RequestSync::get_data_raw ()
    {
      return m_message->response.body;
    }

    guint
    RequestSync::get_data_size ()
    {
      return m_message->response.length;
    }

    RequestSync::~RequestSync ()
    {
      g_object_unref (m_session);
    }

    void
    RequestSync::add_header (std::string const& name,
                             std::string const& value) 
    {
      soup_message_add_header (m_message->request_headers, name.c_str(), value.c_str());   
    }

    void
    RequestSync::add_request (std::string const& type,
                              std::string const& request)
    {
      soup_message_set_request (m_message, type.c_str(), SOUP_BUFFER_SYSTEM_OWNED /* we can't rely on the std::string not being destroyed */,
        g_strdup (const_cast<char*>(request.c_str())), strlen (request.c_str()));
      m_post = true;
    }

    void
    RequestSync::got_chunk (SoupMessage* message, gpointer data)
    {
      RequestSync & request = (*(reinterpret_cast<RequestSync*>(data)));

      request.m_read += message->response.length;
      double percent = (double (request.m_read) / double (request.m_size));
      request.s_progress_.emit (percent);
    }

    void
    RequestSync::got_content_length (SoupMessage* message, gpointer data)
    {
      RequestSync & request = (*(reinterpret_cast<RequestSync*>(data)));
      request.m_size = g_ascii_strtoull (soup_message_get_header (message->response_headers, "content-length"), NULL, 10);
    }

    // Default Request 
    Glib::RefPtr<Request>
    Request::create (std::string const& url, bool post)
    {
      return Glib::RefPtr<Request>(new Request (url, post));  
    }

    Request::Request (std::string const& url, bool post)
    : m_url         (url)
    , m_post        (post)
    , m_block_done  (0)
    {
      m_session = soup_session_async_new ();
      m_message = soup_message_new (m_post ? "POST" : "GET", m_url.c_str());
    }

    void
    Request::cancel ()
    {
      m_block_done = 1;
      soup_session_abort (m_session);
    }
    
    void
    Request::run ()
    {
      soup_session_queue_message (m_session, m_message, SoupMessageCallbackFn (got_answer), this);
    }

    Request::~Request ()
    {
      m_block_done = 1;

      if (m_message->status == SOUP_MESSAGE_STATUS_RUNNING)
      {
        soup_session_abort (m_session);
      }

      g_object_unref (m_session);
    }

    void
    Request::add_header (std::string const& name,
                         std::string const& value) 
    {
      soup_message_add_header (m_message->request_headers, name.c_str(), value.c_str());   
    }

    void
    Request::add_request (std::string const& type,
                          std::string const& request)
    {
      soup_message_set_request (m_message, type.c_str(), SOUP_BUFFER_SYSTEM_OWNED /* we can't rely on the std::string not being destroyed */,
        g_strdup (const_cast<char*>(request.c_str())), strlen (request.c_str()));
      m_post = true;
    }

    void
    Request::got_answer (SoupMessage* message, gpointer data)
    {
      Request & request = (*(reinterpret_cast<Request*>(data)));

      if( request.m_message->status != SOUP_MESSAGE_STATUS_FINISHED )
        return;
  
      if( request.m_block_done )
        return;

      request.m_block_done = 1;
      request.request_callback_.emit(
                                      request.m_message->response.body, 
                                      request.m_message->response.length,
                                      request.m_message->status_code
                                    );
    }

    // File Request
    Glib::RefPtr<RequestFile>
    RequestFile::create (std::string const& url, std::string const& filename)
    {
      return Glib::RefPtr<RequestFile>(new RequestFile (url, filename));  
    }

    RequestFile::RequestFile (std::string const& url, std::string const& filename) 
    : m_url           (url)
    , m_filename      (filename)
    , m_size          (0)
    , m_read          (0)
    , m_block_done    (0)
    {
      m_session = soup_session_async_new ();
      m_message = soup_message_new ("GET", m_url.c_str());
    }

    void
    RequestFile::cancel ()
    {
      m_block_done = 1;

      if (m_message->status == SOUP_MESSAGE_STATUS_RUNNING)
      {
        soup_session_abort (m_session);
        m_file.close ();
        g_unlink (m_filename.c_str());
      }
    }

    void
    RequestFile::run ()
    {
      g_signal_connect (G_OBJECT (m_message), "got-chunk", G_CALLBACK (got_chunk), this);
      soup_message_add_header_handler (m_message, "content-length", SOUP_HANDLER_PRE_BODY,
                                       SoupMessageCallbackFn (got_content_length), this);
      soup_session_queue_message (m_session, m_message, SoupMessageCallbackFn (got_answer), this);
      m_file.open (m_filename.c_str());
    }

    RequestFile::~RequestFile ()
    {
      m_block_done = 1;

      if (m_message->status == SOUP_MESSAGE_STATUS_RUNNING)
      {
        soup_session_abort (m_session);
        s_aborted_.emit ();

        m_file.close ();
        g_unlink (m_filename.c_str());
      }

      g_object_unref (m_session);
    }

    void
    RequestFile::add_header (std::string const& name,
                             std::string const& value) 
    {
      soup_message_add_header (m_message->request_headers, name.c_str(), value.c_str());   
    }

    void
    RequestFile::got_answer (SoupMessage* message, gpointer data)
    {
      RequestFile & request = (*(reinterpret_cast<RequestFile*>(data)));

      if( request.m_message->status != SOUP_MESSAGE_STATUS_FINISHED )
        return;

      if( request.m_block_done )
        return;
    
      request.m_file.close ();

      request.m_block_done = 1;
      request.s_done_.emit (request.m_filename);
    }

    void
    RequestFile::got_chunk (SoupMessage* message, gpointer data)
    {
      RequestFile & request = (*(reinterpret_cast<RequestFile*>(data)));

      request.m_file.write (message->response.body, message->response.length);
      request.m_read += message->response.length;

      double percent = (double (request.m_read) / double (request.m_size));
      request.s_progress_.emit (percent);
    }

    void
    RequestFile::got_content_length (SoupMessage* message, gpointer data)
    {
      RequestFile & request = (*(reinterpret_cast<RequestFile*>(data)));
      request.m_size = g_ascii_strtoull (soup_message_get_header (message->response_headers, "content-length"), NULL, 10);
    }
  }
}
