//  BMPx - The Dumb Music Player
//  Copyright (C) 2005-2007 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non GPL-compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/fcntl.h>
#include <errno.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <stdio.h>
#include <unistd.h>
#include <cstring>

#include <iostream>
#include <sstream>

#include <build.h>
#include <revision.h>

#include "util.hh"
#include "uri++.hh"
#include "network.hh"
#include "minisoup.hh"

#define SERVER_TEST_NAME  "google.com"
#define SERVER_NAME       "beep-media-player.org"

namespace
{
  bool connected = false;
  bool attempted_connect = false;
}

namespace Bmp
{
  namespace Network
  {
    void
    disable ()
    {
      attempted_connect = true;
      connected = false;
    }

    void
    send_statistic (StatisticAction action)
    {
      std::string action_s = (action == ACTION_QUIT) ? "quit":"start";
      std::string arch = URI::escape_string (BUILD_ARCH);

      std::stringstream data;
      data  << "http://"
            << SERVER_NAME
            << "/stats.php"
            << "?version="
            << VERSION
            << "&svnrev="
            << RV_REVISION
            << "&system="
            << arch 
            << "&action="
            << action_s;

      Soup::RequestSyncRefP request = Soup::RequestSync::create (data.str());
      guint code = request->run ();
    }

    namespace
    {
      bool
      check_host (std::string const& hostname,
                  uint16_t           port,
                  bool               write = false,
                  unsigned int       n_tries = 4,
                  double             initial_timeout = 2.0,
                  double             timeout_step = 0.5)
      {
        struct hostent* host = gethostbyname (hostname.c_str ());

        if (!host)
          {
            return false;
          }

        struct sockaddr_in serv_addr;

        serv_addr.sin_family = host->h_addrtype;
        std::memcpy (&serv_addr.sin_addr.s_addr, host->h_addr_list[0], host->h_length);
        serv_addr.sin_port = htons (port);

        // Create socket
        int sd = socket (AF_INET, SOCK_STREAM, 0);
        if (sd < 0)
        {
          return false;
        }

        struct sockaddr_in local_addr;

        // Bind Any Port Number
        local_addr.sin_family = AF_INET;
        local_addr.sin_addr.s_addr = htonl (INADDR_ANY);
        local_addr.sin_port = htons(0);

        int rc = bind (sd, reinterpret_cast<struct sockaddr const*> (&local_addr), sizeof (local_addr));
        if ( rc < 0 )
        {
          return false;
        }

        int flags = fcntl (rc, F_GETFL, 0);
        fcntl (rc, F_SETFL, flags | O_NONBLOCK);

        // Connect To Server
        rc = connect (sd, reinterpret_cast<struct sockaddr const*> (&serv_addr), sizeof (serv_addr));

        int errsv = errno;
        if (rc < 0)
        {
          if (!(rc == -1 && errsv == EINPROGRESS))
          {
            return false;
          }
        }

        double timeout = initial_timeout;
        struct timeval tv;
        fd_set rfds;
        int retval;

        for (unsigned int i = 0; i < n_tries; ++i)
        {
            FD_ZERO (&rfds);
            FD_SET (rc, &rfds);

            tv.tv_sec  = time_t (timeout);
            tv.tv_usec = suseconds_t ((timeout - tv.tv_sec) * 1000000);

            if (write)
              retval = select (rc+1, &rfds, NULL, NULL, &tv);
            else
              retval = select (rc+1, NULL, &rfds, NULL, &tv);

            if (retval >= 0)
            {
                connect (sd, reinterpret_cast<struct sockaddr const*> (&serv_addr), sizeof (serv_addr));
                shutdown (sd, SHUT_RDWR);
                return true;
            }
            timeout += timeout_step;
        }

        close (sd);
        shutdown (sd, SHUT_RDWR);
        return false;
      }
    }

    bool
    check_connected (bool force)
    {
      if (attempted_connect && !force)
        return connected;
      else if (force)
        connected = false;

      connected = check_host (SERVER_TEST_NAME, 80);
      attempted_connect = true;
      return connected;
    }
  }
}

