//  BMPx - The Dumb Music Player
//  Copyright (C) 2005-2007 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License Version 2
//  as published by the Free Software Foundation.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifndef BMP_UI_PART_LASTFM_HH
#define BMP_UI_PART_LASTFM_HH

#include <glibmm/ustring.h>
#include <gdkmm/pixbuf.h>
#include <gtkmm.h>
#include <libglademm/xml.h>
#include <sigc++/signal.h>

#include <boost/optional.hpp>
#include <boost/format.hpp>

#include "playbacksource.hh"
#include "ui-part-base.hh"

#include "xspf-types.hh"
#include "lastfm.hh"

using namespace Gtk;
using namespace Glib;
using namespace Bmp::LastFM;

namespace Bmp
{
  typedef RefPtr<TextBuffer> RefTextBuffer;

  class LastFmLinkTag
    : public TextTag
  {
    private:

        LastFmLinkTag (ustring const& url, uint64_t rank)
        : ObjectBase (typeid(LastFmLinkTag))
        , TextTag ()
        , m_url  (url)
        {
          property_foreground() = "#0000FF"; 
          property_scale() = (double (rank) / 50.) + 1;
        }

    public:

        static RefPtr<LastFmLinkTag> create (ustring const& url, uint64_t rank)
        {
          return RefPtr<LastFmLinkTag> (new LastFmLinkTag (url, rank)); 
        }

        virtual ~LastFmLinkTag () {}

    public:

      typedef sigc::signal<void, ustring const&> SignalUrlActivated;
    
    private:

      SignalUrlActivated signal_url_activated_;

    public:

      SignalUrlActivated&
      signal_url_activated ()
      {
        return signal_url_activated_;
      }

    protected:

      virtual bool
      on_event (RefPtr<Object> const& event_object, GdkEvent* event, TextIter const& iter) 
      {
        if (event->type == GDK_BUTTON_PRESS)
        {
              GdkEventButton * ev = (GdkEventButton*)(event);
              if (ev->button == 1)
              {
                    return true;
              }
        }
        if (event->type == GDK_BUTTON_RELEASE)
        {
              GdkEventButton * ev = (GdkEventButton*)(event);
              if (ev->button == 1)
              { 
                    signal_url_activated_.emit (m_url);
                    return true;
              }
        }
        return false;
      }

    private:
    
      ustring m_name;
      ustring m_url;
  };

  class LastFmTagView
    : public TextView
  {
      typedef RefPtr<TextTag>  RefTextTag;
      typedef std::vector<RefTextTag> TagV;
      RefTextTag m_current_tag;

    public:

      LastFmTagView (BaseObjectType*                  obj,
                     RefPtr<Gnome::Glade::Xml> const& xml)
      : TextView (obj) 
      { 
        gtk_widget_add_events (GTK_WIDGET (gobj()), 
                               GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK);
      }

      virtual ~LastFmTagView ()
      {}

    protected:

      virtual bool on_motion_notify_event (GdkEventMotion * event)
      {
        int x, y;
        int x_orig, y_orig;
        GdkModifierType state;

        if (event->is_hint)
          gdk_window_get_pointer (event->window, &x_orig, &y_orig, &state);
        else
          {
            x_orig = event->x;
            y_orig = event->y;
            state = GdkModifierType (event->state);
          }

        window_to_buffer_coords (TEXT_WINDOW_WIDGET, x_orig, y_orig, x, y);

        RefPtr<TextBuffer> buf = get_buffer();
        TextBuffer::iterator iter;
        get_iter_at_location (iter, x, y);      
      
        TagV tags = iter.get_tags();
    
        for (TagV::const_iterator i = tags.begin(); i != tags.end(); ++i) 
        {
          RefPtr<LastFmLinkTag> tag = RefPtr<LastFmLinkTag>::cast_dynamic (*i);
          if (tag)
          {
            GdkWindow * window = GDK_WINDOW (gtk_text_view_get_window (GTK_TEXT_VIEW (gobj()), GTK_TEXT_WINDOW_TEXT));
            GdkCursor * cursor = gdk_cursor_new_from_name (gdk_display_get_default (), "hand2");
            if (!cursor)
              cursor = gdk_cursor_new_for_display (gdk_display_get_default (), GDK_HAND2);
            gdk_window_set_cursor (window, cursor);

            if ((m_current_tag && m_current_tag->gobj()) && (m_current_tag->gobj() != tag->gobj()))
            {
              m_current_tag->property_foreground() = "#0000FF"; 
              m_current_tag->property_background_gdk() = get_style()->get_base (get_state()); 
            }

            m_current_tag = tag;
            m_current_tag->property_foreground_gdk() = get_style()->get_base (get_state()); 
            m_current_tag->property_background() = "#0000FF";
            return false;
          }
        }
 
        if (m_current_tag) 
        {
          GdkWindow * window = GDK_WINDOW (gtk_text_view_get_window (GTK_TEXT_VIEW (gobj()), GTK_TEXT_WINDOW_TEXT));
          GdkCursor * cursor = gdk_cursor_new_from_name (gdk_display_get_default (), "xterm");
          if (!cursor)
            cursor = gdk_cursor_new_for_display (gdk_display_get_default (), GDK_XTERM);
          gdk_window_set_cursor (window, cursor);
          m_current_tag->property_foreground() = "#0000FF"; 
          m_current_tag->property_background_gdk() = get_style()->get_base (get_state()); 
          m_current_tag = RefTextTag (0);
        }
    
        return false;
      }
  };


  namespace UiPart
  {
    class LASTFM
      : public  PlaybackSource,
        public  Base
    {

      public:

        LASTFM (RefPtr<Gnome::Glade::Xml> const& xml, RefPtr<UIManager> ui_manager);
        virtual ~LASTFM ();

        void set_uri (ustring const& uri);

      protected:

        virtual guint
        add_ui ();

        virtual guint
        add_context_ui ();

        virtual guint
        add_tray_ui ();

        virtual ustring
        get_uri ();

        virtual ustring
        get_type ();

        virtual bool
        go_next ();

        virtual bool
        go_prev ();

        virtual void
        stop ();

        virtual void
        play ();
      
        virtual void
        play_post ();

        virtual void
        next_post ();

        virtual void
        prev_post ();

        virtual void
        restore_context ();
    
        virtual void
        skipped ();

        virtual void
        segment ();

     private:

        class TreeViewTags
          : public TreeView
        {
          public:
            typedef sigc::signal<void, ustring const&> StationActivated;
          private:
            StationActivated
            station_activated_;

          public:

            TreeViewTags (BaseObjectType*                  obj,
                          RefPtr<Gnome::Glade::Xml> const& xml);
            virtual ~TreeViewTags ();

            StationActivated&
            station_activated ()
            {
              return station_activated_;
            }

            void set_user (ustring const& user, bool subscriber, bool match_tags);
            void clear_user ();

          protected:

            virtual void on_row_activated (TreePath const& path,
                                           TreeViewColumn* column);

          private:

            void  get_user_match_tags (std::string const& username);
            void  set_user_match_tags ();
            void  cell_data_func (CellRenderer* cell, TreeIter const& m_iter);

            enum RowType
            {
              ROW_MAINUSER,
              ROW_SYSREC_ROOT,
              ROW_ARTIST,
              ROW_URL,
              ROW_NEIGHBOUR_ROOT,
              ROW_NEIGHBOUR,
              ROW_FRIEND_ROOT,
              ROW_FRIEND,
              ROW_TAG_ROOT,
              ROW_TAG,
            };

            class Columns
              : public TreeModel::ColumnRecord
            {
              public:

                TreeModelColumn<ustring>  name;
                TreeModelColumn<ustring>  station;
                TreeModelColumn<RowType>  type;
                TreeModelColumn<ustring>  score;

                Columns ()
                {
                  add (name);
                  add (station);
                  add (type);
                  add (score);
                }
            };
            Columns columns;

            Notebook              * m_notebook;
            RefPtr<Gdk::Pixbuf>     m_pb_artist;
            RefPtr<Gdk::Pixbuf>     m_pb_rec;
            RefPtr<Gdk::Pixbuf>     m_pb_url;
            RefPtr<Gdk::Pixbuf>     m_pb_neighbour;
            RefPtr<Gdk::Pixbuf>     m_pb_friend;
            RefPtr<Gdk::Pixbuf>     m_pb_tag;
            RefPtr<Gdk::Pixbuf>     m_pb_user;
            RefPtr<Gdk::Pixbuf>     m_pb_mainuser;

            RefPtr<TreeStore> m_store;
            TreeIter i_root_stations;
            TreeIter i_root_neighbours;
            TreeIter i_root_friends;
            TreeIter i_root_matchtags;
            TreeIter i_root_recommended;
            TreeIter i_root_usertags;

            RankedTagV m_match_tags;
            Dispatcher disp;
        };

        TreeViewTags * m_treeview_tags;
        void  on_update_list (bool match_tags);

        void  playurl ();
        void  tune ();

        void  clear_metadata_info ();

        void  on_radio_connected ();
        void  on_radio_disconnected ();
        void  on_play_http_status (int status);

        void  on_love ();
        void  on_ban  ();
        void  on_tag  ();

        void  get_track_tags ();
        void  got_track_tags (TagV const& tags);

        void  display_tags_limited ();

        TrackMetadata   m_metadata;
        TagV            m_current_tags;
        ustring         m_current_uri;

        XSPF::Playlist  m_playlist;
        XSPF::ItemIter  m_playlist_iter;

        boost::optional<ustring> m_artist;
        boost::optional<ustring> m_album;
        boost::optional<ustring> m_track;

        RefTextBuffer     m_tag_buffer;
        LastFmTagView   * m_tag_view;
        HScale          * m_tag_limiter;
        Label           * m_lastfm_radio_station;
        Entry           * m_station_url_entry;
        ComboBox        * m_station_choice;

        sigc::connection  m_conn;
        WS::TagsGlobReqRefP m_tags_request;

    }; // class LASTFM
  } // namespace UiPart
} // namespace Bmp

#endif // !BMP_UI_PART_LASTFM_HH

