//  BMPx - The Dumb Music Player
//  Copyright (C) 2005-2007 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <glibmm/i18n.h>
#include <glib/gstdio.h>
#include <gtkmm.h>
#include <gdk/gdkkeysyms.h>
#include <cstdlib>
#include <cstring>
#include <iostream>
#include <sstream>
#include <fstream>

#include <boost/algorithm/string.hpp>
#include <boost/format.hpp>
#include <boost/shared_ptr.hpp>

#include <mcs/mcs.h>

// BMP Musicbrainz
#include "musicbrainz/mbxml-v2.hh"
#include "mb-tagger.hh"

// BMP Audio
#include "audio/audio.hh"

// BMP Widgets
#include "widgets/taskdialog.hh"

// BMP Misc
#include "dialog-simple-progress.hh"
#include "dialog-simple-entry.hh"
#include "dialog-export.hh"
#include "dialog-progress.hh"

#include "ui-tools.hh"

#include "debug.hh"
#include "lastfm.hh"
#include "main.hh"
#include "network.hh"
#include "paths.hh"
#include "stock.hh"

#include "uri++.hh"
#include "util.hh"
#include "util-file.hh"

#include "x_amazon.hh"
#ifdef HAVE_HAL
#  include "x_hal.hh"
#endif //HAVE_HAL
#include "x_library.hh"
#include "x_mcsbind.hh"
#include "x_play.hh"
#include "x_vfs.hh"

// UiPart Playlist
#include "ui-part-playlist.hh"

using namespace std;
using namespace boost;
using namespace Gtk;
using namespace Glib;

using namespace Bmp::DB;
using namespace Bmp::Util;
using namespace Bmp::VFS;
using namespace Bmp::Audio;
using namespace Bmp::MusicBrainzXml;

#define ACTION_PLAYLIST_PLAYLIST_CLEAR                    "playlist-action-playlist-clear"
#define ACTION_PLAYLIST_PLAYLIST_EXPORT                   "playlist-action-playlist-export"
#define ACTION_PLAYLIST_REMOVE_PLAYLIST_ITEMS             "playlist-action-remove-items"
#define ACTION_PLAYLIST_PLAYLIST_HISTORY_CLEAR            "playlist-action-history-clear"
#define ACTION_PLAYLIST_PLAYLIST_CREATE_FROM_LASTFM_TAG   "playlist-action-playlist-create-from-lastfm-tag"

namespace
{
  const char * library_menu_playlist =
  "<ui>"
  ""
  "<menubar name='popup-playlist-list'>"
  ""
  "   <menu action='dummy' name='menu-playlist-list'>"
  "     <menuitem action='" ACTION_PLAYLIST_REMOVE_PLAYLIST_ITEMS "'/>"
  "       <separator/>"
  "     <menuitem action='" ACTION_PLAYLIST_PLAYLIST_HISTORY_CLEAR "'/>"
  "     <menuitem action='" ACTION_PLAYLIST_PLAYLIST_CLEAR "'/>"
  "       <separator/>"
  "     <menuitem action='" ACTION_PLAYLIST_PLAYLIST_CREATE_FROM_LASTFM_TAG "'/>"
  "       <separator/>"
  "     <menuitem action='" ACTION_PLAYLIST_PLAYLIST_EXPORT "'/>"
  "   </menu>"
  ""
  "</menubar>"
  ""
  "</ui>";

  typedef std::vector < TargetEntry > DNDEntries;

  static boost::format int_f ("%d");
  static boost::format uint64_f ("%llu");
  static boost::format progress_f ("%d / %d");

  void
  menu_item_set_markup (RefPtr<UIManager> uimanager,
                        ustring const&    menupath,
                        ustring const&    markup)
  {
    Bin * bin = 0;
    bin = dynamic_cast <Bin*> (uimanager->get_widget (menupath));

    if( bin )
      reinterpret_cast <Label*> (bin->get_child())->set_markup (markup);
    else
      g_warning ("%s: Widget with path '%s' not found or not a Gtk::Bin", G_STRLOC, menupath.c_str());
  }

  enum DnDTypes
  {
    DND_URI_TARGETS = 0,
    DND_ALBUM       = 1,
    DND_TRACK       = 2,
  };

  //// Various functors for for_each

  template <class T>
  class TrackApply
  {
    public:

      TrackApply (T const& column, Bmp::Track const& track) : m_track (track), m_column (column) {} 

      void
      operator()(TreeIter const& i)
      {
        (*i)[m_column] = m_track;
      }

    private:
    
      Bmp::Track const& m_track;
      T const& m_column;
  };  

  template <class T>
  class PathCollect
  {
    public:

      PathCollect ( Glib::RefPtr<Gtk::TreeModel> const& model,
                    T const& column,
                    Bmp::TrackV & tracks)
      : m_model   (model)
      , m_tracks  (tracks)
      , m_column  (column)
      {} 

      void
      operator()(Gtk::TreePath const& p)
      {
        m_tracks.push_back ((*m_model->get_iter (p))[m_column]);
      }

    private:
        
      Glib::RefPtr<Gtk::TreeModel> const& m_model; 
      Bmp::TrackV & m_tracks;
      T const& m_column;
  };  

  class ReferenceCollect
  {
    public:

      ReferenceCollect (Glib::RefPtr<Gtk::TreeModel> const& model,
                        Bmp::ReferenceV & references)
      : m_model       (model)
      , m_references  (references)
      {} 

      void
      operator()(Gtk::TreePath const& p)
      {
        m_references.push_back (TreeRowReference (m_model, p));
      }

    private:
        
      Glib::RefPtr<Gtk::TreeModel> const& m_model; 
      Bmp::ReferenceV & m_references;
  };  

  std::string
  get_date_string_markup (std::string const& in)
  {
    int y = 0, m = 0, d = 0;

    const char * z (in.c_str());

    if( strlen (z) == 4 )
      sscanf (z, "%04d", &y);
    else
#if 0
    if( strlen (z) == 6 )
      sscanf (z, "%04d%02d", &y, &m);
    else
    if( strlen (z) == 7 )
      sscanf (z, "%04d-%02d", &y, &m);
    else
#endif
    if( strlen (z) == 8 )
      sscanf (z, "%04d%02d%02d", &y, &m, &d);
    else
    if( (strlen (z) == 10) ||  (strlen (z) == 19) )
      sscanf (z, "%04d-%02d-%02d", &y, &m, &d);

    char result[1024];

    struct tm * _tm = g_new0 (struct tm, 1);

    if (y) _tm->tm_year = (y - 1900);
    if (m) _tm->tm_mon  = m - 1;
    if (d) _tm->tm_mday = d;

    /*
    if( y && m && d )
    {
      char ys[256];
      char ms[256];
      char ds[256];
      strftime (ys, 255, "%Y", _tm);
      strftime (ms, 255, "%B", _tm);
      strftime (ds, 255, "%d", _tm);
      g_free (_tm);
      std::string r = (boost::format ("%s <small>(%s %s)</small>")
                          % std::string (ys)
                          % std::string (ds)
                          % Markup::escape_text (locale_to_utf8 (ustring (ms))).c_str()).str();
      return r;
    }
    else
    if( y && m )
    {
      char ys[256];
      char ms[256];
      strftime (ys, 255, "%Y", _tm);
      strftime (ms, 255, "%B", _tm);
      g_free (_tm);
      std::string r = (boost::format ("%s <small>(%s)</small>")
                          % std::string (ys)
                          % Markup::escape_text (locale_to_utf8 (ustring (ms))).c_str()).str();
      return r;
    }
    else
    */
    if( y )
    {
      char ys[256];
      strftime (ys, 255, "(%Y)", _tm);
      g_free (_tm);
      return std::string (ys); 
    }

    return std::string();
  }

  template <class T>
  inline T
  clamp (T min, T max, T val)
  {
    if( (val >= min) && (val <= max) )
    {
      return val;
    }
    else if( val < min )
    {
      return min;
    }
    else
    {
      return max;
    }
  }

  enum Renderer
  {
    R_TEXT,
    R_PIXBUF,
    R_TOGGLE,
    R_SURFACE
  };

  enum Column
  {
    COLUMN_PLAYING,
    COLUMN_NEW_ITEM,
    COLUMN_TRACK,
    COLUMN_TITLE,
    COLUMN_TIME,
    COLUMN_ALBUM,
    COLUMN_ARTIST,
    COLUMN_GENRE,
    COLUMN_BITRATE,
    COLUMN_SAMPLERATE,
    COLUMN_RATING,
    COLUMN_COUNT,
  };
}

namespace Bmp
{
    PlaylistList::PlaylistList (BaseObjectType                 *  obj,
                                RefPtr<Gnome::Glade::Xml> const&  xml)

      : TreeView            (obj)
      , m_ref_xml           (xml)
      , m_button_depressed  (0)
      , m_localUid          (0)
      , m_playing           (0)
      , m_bmpx_track_id     (0)
    {
      m_pb_playing = Gdk::Pixbuf::create_from_file (build_filename (BMP_IMAGE_DIR_STOCK, "play.png"));

      struct {
          char const* title;
          double      xalign;
          Renderer    r;
          int         column;
      } cells[] = {
          { " ",              0.5,   R_PIXBUF,   COLUMN_PLAYING        },
          { " ",              0.5,   R_TOGGLE,   0                     },
          { N_("Title"),      0.0,   R_TEXT,     COLUMN_TITLE          },
          { N_("Artist"),     0.0,   R_TEXT,     COLUMN_ARTIST         },
          { N_("Length"),     1.0,   R_TEXT,     COLUMN_TIME           },
          { N_("Album"),      0.0,   R_TEXT,     COLUMN_ALBUM          },
          { N_("Track"),      1.0,   R_TEXT,     COLUMN_TRACK          },
          { N_("Genre"),      0.0,   R_TEXT,     COLUMN_GENRE          },
      };

      for (unsigned int n = 0 ; n < G_N_ELEMENTS (cells); ++n)
      {
        TreeView::Column * c = manage (new TreeView::Column (_(cells[n].title)));
        CellRenderer     * r = 0;

        switch (cells[n].r)
        {
          case R_SURFACE:
            r = manage (new CellRendererCairoSurface());
            r->property_xpad() = 0; 
            c->set_fixed_width (64);
            c->set_sizing (Gtk::TREE_VIEW_COLUMN_FIXED);
            c->pack_end (*r);
            break;

          case R_PIXBUF:
            r = manage (new CellRendererPixbuf());
            r->property_xalign() = cells[n].xalign;
            c->pack_end (*r);
            c->set_min_width (24);
            break;

          case R_TEXT:
            r = manage (new CellRendererText());
            r->property_xalign() = cells[n].xalign;
            c->pack_end (*r);
            c->set_resizable (true);
            break;

          case R_TOGGLE:
            CellRendererToggle * _r = manage (new CellRendererToggle());
            c->pack_end (*_r, false);
            c->set_resizable (false);
            c->add_attribute (_r->property_active(), m_track_cr.playTrack);
            _r->signal_toggled().connect (sigc::mem_fun (*this, &Bmp::PlaylistList::cell_play_track_toggled));
            r = _r;
            break;
        }

        append_column (*c);

        if( cells[n].r != R_TOGGLE )
        {
          c->set_cell_data_func (*r, (sigc::bind (sigc::mem_fun (*this, &Bmp::PlaylistList::cell_data_func), cells[n].column, cells[n].r)));
        }
      }

#ifdef HAVE_HAL
      // Connect Bmp::HAL
      if( hal->is_initialized() )
      {
        hal->signal_volume_added().connect
          (sigc::mem_fun (*this, &Bmp::PlaylistList::hal_volume_add));
        hal->signal_volume_removed().connect
          (sigc::mem_fun (*this, &Bmp::PlaylistList::hal_volume_del));
      }
#endif //HAVE_HAL

      get_selection()->set_select_function
        (sigc::mem_fun (*this, &Bmp::PlaylistList::slot_select));

      ::library->signal_track_modified().connect
        (sigc::mem_fun (*this, &Bmp::PlaylistList::on_library_track_modified));

      m_store = Gtk::ListStore::create (m_track_cr);
      clear_current_iter ();
      set_model (m_store);
      get_selection()->set_mode (SELECTION_MULTIPLE);
      set_search_column (m_track_cr.searchKey);
      enable_drag_dest ();
    }

    void
    PlaylistList::on_row_activated (Gtk::TreeModel::Path const& path, Gtk::TreeViewColumn* G_GNUC_UNUSED )
    {
      TreeIter iter (m_store->get_iter(path));

      if ((*iter)[m_track_cr.present]) 
      {
        signal_activated_.emit ();
      }
    }

    void
    PlaylistList::enable_drag_dest ()
    {
      disable_drag_dest ();
      DNDEntries target_entries;
      target_entries.push_back (TargetEntry ("bmpx-dnd-album", Gtk::TARGET_SAME_APP, DND_ALBUM));
      target_entries.push_back (TargetEntry ("bmpx-dnd-track", Gtk::TARGET_SAME_APP, DND_TRACK));
      drag_dest_set (target_entries, Gtk::DEST_DEFAULT_MOTION);
      drag_dest_add_uri_targets ();
    }

    void
    PlaylistList::disable_drag_dest ()
    {
      drag_dest_unset ();  
    }

    void
    PlaylistList::put_track_at_iter (Bmp::Track const& track)
    {
      TreeIter i = m_store->append();
      put_track_at_iter (track, i);
    }
 
    void
    PlaylistList::put_track_at_iter (Bmp::Track const& track, Gtk::TreeModel::iterator & iter)
    {
      (*iter)[m_track_cr.track] = track;
      (*iter)[m_track_cr.uid] = track.bmpx_track_id;
      (*iter)[m_track_cr.localUid] = m_localUid;
      (*iter)[m_track_cr.searchKey] = track.title ? track.title.get() : std::string();
      (*iter)[m_track_cr.playTrack] = true; 

      bool present = file_test (filename_from_uri (track.location.get()), FILE_TEST_EXISTS);
#ifdef HAVE_HAL
      (*iter)[m_track_cr.present] = present && (hal->volume_is_mounted (HAL::VolumeKey (track.volume_udi.get(), track.device_udi.get())));
#else
      (*iter)[m_track_cr.present] = present; 
#endif //HAVE_HAL

      if( m_playing && m_bmpx_track_id == track.bmpx_track_id )
      {
        assign_current_iter (iter);
      }

      m_localUidIterMap.insert (std::make_pair (m_localUid, iter));

      UidIterSetMap::iterator i (m_UidIterMap.find (track.bmpx_track_id));
      if( i == m_UidIterMap.end() )
      {
        IterSet x;
        x.insert (iter);
        m_UidIterMap.insert (std::make_pair (track.bmpx_track_id, x));
      }
      else
      {
        IterSet & x (i->second);
        x.insert (iter);
      }
      m_localUid++;
    }

    bool
    PlaylistList::on_drag_motion (RefPtr<Gdk::DragContext> const& context, int x, int y, guint time)
    {
      TreeModel::Path path;
      TreeViewDropPosition pos;

      if( get_dest_row_at_pos (x, y, path, pos) )
      {
        switch (pos)
        {
            case TREE_VIEW_DROP_INTO_OR_BEFORE:
            case TREE_VIEW_DROP_BEFORE:
              set_drag_dest_row (path, TREE_VIEW_DROP_BEFORE);
              break;

            case TREE_VIEW_DROP_INTO_OR_AFTER:
            case TREE_VIEW_DROP_AFTER:
              set_drag_dest_row (path, TREE_VIEW_DROP_AFTER);
              break;
        }
      }
      return true;
    }

    void
    PlaylistList::on_drag_leave (RefPtr<Gdk::DragContext> const& context, guint time)
    {
      gtk_tree_view_set_drag_dest_row (gobj(), NULL, GTK_TREE_VIEW_DROP_BEFORE);
    }

    bool
    PlaylistList::on_drag_drop (RefPtr<Gdk::DragContext> const& context, int x, int y, guint time)
    {
      ustring target (drag_dest_find_target (context));

      if( !target.empty() )
      {
        drag_get_data (context, target, time);
        context->drag_finish  (true, false, time);
        return true;
      }
      else
      {
        context->drag_finish  (false, false, time);
        return false;
      }
    }

    void
    PlaylistList::on_drag_data_received (RefPtr<Gdk::DragContext> const& context, int x, int y,
                                          Gtk::SelectionData const& selection_data, guint info, guint time)
    {
      switch (info)
      {
        case DND_TRACK:
        {
          gconstpointer * data = (gconstpointer*)(selection_data.get_data());
          Bmp::Track track (*static_cast<Bmp::Track const*> (*data));

          TreeModel::Path       path;
          TreeModel::iterator   iter;
          TreeViewColumn      * c;
          int                   cell_x,
                                cell_y;
          TreeViewDropPosition  pos;

          if( !get_dest_row_at_pos (x, y, path, pos) )
          {
            iter = m_store->append ();
          }
          else
          {
            switch (pos)
            {
              case TREE_VIEW_DROP_BEFORE:
              case TREE_VIEW_DROP_INTO_OR_BEFORE:
                iter = m_store->insert (m_store->get_iter (path));
                break;

              case TREE_VIEW_DROP_AFTER:
              case TREE_VIEW_DROP_INTO_OR_AFTER:
                iter = m_store->insert_after (m_store->get_iter (path));
                break;
            }
          }

          put_track_at_iter (track, iter);
          break;
        }

        case DND_ALBUM:
        {
          gconstpointer* data = (gconstpointer*)(selection_data.get_data());

          Bmp::Album album (*(static_cast<Bmp::Album const*> (*data)));
          RowV rows (library->get_album_tracks (album.bmpx_album_id));

          TreeModel::Path       path;
          TreeViewColumn      * c;
          int                   cell_x,
                                cell_y;
          TreeViewDropPosition  pos;

          if( !get_dest_row_at_pos (x, y, path, pos) )
          {
            path = Gtk::TreeModel::Path();
            path.append_index (m_store->children().size());
          }
          else
          {
            switch (pos)
            {
              case TREE_VIEW_DROP_BEFORE:
              case TREE_VIEW_DROP_INTO_OR_BEFORE:
                break;

              case TREE_VIEW_DROP_AFTER:
              case TREE_VIEW_DROP_INTO_OR_AFTER:
                path.next ();
                break;
            }
          }

          TreeModel::iterator iter (m_store->get_iter (path));

          if( !iter )
          {
            iter = m_store->append ();
            for (RowV::size_type n = 0; n < rows.size()-1; ++n)
            {
              m_store->insert (iter);
            }
          }
          else
          {
            for (RowV::size_type n = 0; n < rows.size(); ++n)
            {
              m_store->insert (iter);
            }
          }

          for (RowV::const_iterator i = rows.begin(); i != rows.end(); ++i)
          {
            TreeModel::iterator iter = m_store->get_iter (path);
            put_track_at_iter (*i, iter);
            path.next ();
          }
          break;
        }

        case DND_URI_TARGETS:
        {
          VUri uris (selection_data.get_uris());
          VUri v;

          for (VUri::const_iterator u = uris.begin() ; u != uris.end() ; ++u)
          {
            if( !vfs->has_container (*u) )
            {
              v.push_back (*u);
            }
            else
            {
              VFS::Handle h (*u);
              VUri list;

              try{
                if( file_test (filename_from_uri (*u), FILE_TEST_IS_DIR) )
                    vfs->read (h, list, VFS::CONTAINER);
                else
                    vfs->read (h, list, VFS::ProcessingFlags (VFS::TRANSPORT | VFS::CONTAINER));

                for (VUri::const_iterator u = list.begin(); u != list.end() ; ++u)
                  {
                    Bmp::URI uri (*u);
                    Bmp::URI::Protocol p (uri.get_protocol());

                    switch (p)
                    {
                      case URI::PROTOCOL_UNKNOWN:
                      case URI::PROTOCOL_CDDA:
                      case URI::PROTOCOL_LASTFM:
                      case URI::PROTOCOL_FTP:
                      case URI::PROTOCOL_QUERY:
                      case URI::PROTOCOL_TRACK:
                              break;

                      case URI::PROTOCOL_FILE:
                              if( file_test (filename_from_uri (*u), FILE_TEST_EXISTS) )
                              {
                                    v.push_back (*u);
                              }
                              break;

                      case URI::PROTOCOL_HTTP:
                      case URI::PROTOCOL_MMS:
                      case URI::PROTOCOL_MMSU:
                      case URI::PROTOCOL_MMST:
                              break;
                    }
                  }
                }
              catch (...) {} //FIXME: Catch conversion error
            }
          }

          SimpleProgress * d = SimpleProgress::create ();
          d->set_title (_("Adding Files - BMP"));
          d->present ();
          d->enable_cancel ();

#define INSERT_STEP \
      d->set_label (Markup::escape_text (path_get_basename (filename_to_utf8 (filename_from_uri (*i))))); \
      d->step (v.size(), n);

          VUri::size_type n = 0;
          VUri::const_iterator i = v.begin ();

          TreeModel::Path       path;
          int                   cell_x, 
                                cell_y;
          TreeViewDropPosition  pos;

          if( !get_dest_row_at_pos (x, y, path, pos) )
          {
                path = TreePath(); 
                path.append_index (m_store->children().size());

                for (VUri::size_type n = 0; n < v.size(); ++n)
                {
                    TreeIter iter = m_store->append ();
                    INSERT_STEP
                    put_track_at_iter (library->get_track (*i), iter);
                    if (d->canceled()) break;
                    ++i;
                }
          }
          else
          {
            switch (pos)
            {
              case TREE_VIEW_DROP_BEFORE:
              case TREE_VIEW_DROP_INTO_OR_BEFORE:
                break;

              case TREE_VIEW_DROP_AFTER:
              case TREE_VIEW_DROP_INTO_OR_AFTER:
                path.next ();
                break;
            }

            TreeIter iter = m_store->get_iter (path);

            if( !iter )
            {
                  iter = m_store->append ();
                  INSERT_STEP
                  put_track_at_iter (library->get_track (*i), iter);

                  for (VUri::size_type n = 0; n < v.size()-1; ++n)
                  {
                        iter = m_store->insert (iter);
                        INSERT_STEP
                        put_track_at_iter (library->get_track (*i), iter);
                        if (d->canceled()) break;
                        ++i;
                  }
            }
            else
            {
                  for (VUri::size_type n = 0; n < v.size(); ++n)
                  {
                        iter = m_store->insert (iter);
                        INSERT_STEP
                        put_track_at_iter (library->get_track (*i), iter);
                        if (d->canceled()) break;
                        ++i;
                  }
            }
          }

          delete d; break;
        }
      }
      signal_recheck_caps_.emit ();
    }

    bool
    PlaylistList::slot_select (RefPtr < Gtk::TreeModel > const& model, Gtk::TreeModel::Path const& path, bool was_selected)
    {
      return bool ((*m_store->get_iter (path))[m_track_cr.present]);
    }

    void
    PlaylistList::cell_play_track_toggled (Glib::ustring const& path)
    {
      if( bool ((*m_store->get_iter (path))[m_track_cr.present]) )
      {
            TreeIter i (m_store->get_iter (path));
            (*i)[m_track_cr.playTrack] = !((*i)[m_track_cr.playTrack]);
      }
    }

    void
    PlaylistList::cell_data_func (CellRenderer * basecell, TreeModel::iterator const& iter, int column, int renderer)
    {
      basecell->property_sensitive() = bool ((*iter)[m_track_cr.present]);

      CellRendererText          * cell_t = 0;
      CellRendererPixbuf        * cell_p = 0;
      CellRendererCairoSurface  * cell_s = 0;

      Bmp::Track const& track = (*iter)[m_track_cr.track];

      UID uid       = (*iter)[m_track_cr.uid];
      UID localUid = (*iter)[m_track_cr.localUid];

      switch (renderer)
      {
        case R_TEXT:
          cell_t = dynamic_cast <CellRendererText *>    (basecell);
          break;

        case R_PIXBUF:
          cell_p = dynamic_cast <CellRendererPixbuf *>  (basecell);
          break;

        case R_SURFACE:
          cell_s = dynamic_cast <CellRendererCairoSurface *>  (basecell);
          break;
      }

      switch (column)
      {
        case COLUMN_PLAYING:
        {
          if( m_current_iter && (m_current_iter.get() == iter) )
            cell_p->property_pixbuf() = m_pb_playing;
          else
            cell_p->property_pixbuf() = RefPtr <Gdk::Pixbuf> (0);

          break;
        }

        case COLUMN_TRACK:
        {
          if( track.tracknumber && (track.tracknumber.get() != 0) )
            cell_t->property_text() = (uint64_f % track.tracknumber.get()).str();
          else
            cell_t->property_text() = "";
          break;
        }

#if 0
        case COLUMN_RATING:
        {
          int rating = int (track.rating ? track.rating.get() : 0);

          ::Cairo::RefPtr< ::Cairo::ImageSurface> surface = ::Cairo::ImageSurface::create (::Cairo::FORMAT_ARGB32, 64, 16);
          ::Cairo::RefPtr< ::Cairo::Context> cr = ::Cairo::Context::create (surface); 
          cr->set_operator (::Cairo::OPERATOR_CLEAR);
          cr->paint ();

          if ((rating > 0) && (rating <= 3)) // don't overflow the gradient array
          {
            Gdk::Color color (gradient[rating]);
            cr->set_operator (::Cairo::OPERATOR_SOURCE);
            cr->set_source_rgba (color.get_red_p(), color.get_green_p(), color.get_blue_p(), 1.);
            cr->rectangle (0, 0, 64, 16);
            cr->fill ();
          }
          cell_s->property_surface() = surface;
          break;
        }
#endif

        /*
        case COLUMN_LIBRARY_TRACK:
        {
          if( track.new_item )
            cell_p->property_pixbuf() = render_icon (Gtk::StockID (BMP_STOCK_LIBRARY_TRACK), Gtk::ICON_SIZE_MENU);
          else
            cell_p->property_pixbuf() = RefPtr <Gdk::Pixbuf> (0);
          break;
        }
        */

        case COLUMN_TITLE:
        {
          cell_t->property_text() = track.title
                                    ? track.title.get().c_str()           
                                    : "";
          break;
        }

        case COLUMN_ARTIST:
        {
          cell_t->property_text() = track.artist
                                    ? track.artist.get().c_str()
                                    : ""; 
          break;
        }

        case COLUMN_ALBUM:
        {
          cell_t->property_text() = track.album             ? track.album.get().c_str()           
                                                            : "";
          break;
        }

        case COLUMN_TIME:
        {
          if( track.duration )
          {
            uint64_t duration (track.duration.get());
            cell_t->property_text() = (boost::format ("%d:%02d") % (duration / 60) % (duration % 60)).str();
          }
          else
            cell_t->property_text() = "";
          break;
        }

        case COLUMN_GENRE:
        {
          cell_t->property_text() = track.genre       ? track.genre.get().c_str() 
                                                      : "";
          break;
        }

        case COLUMN_BITRATE:
        {
          cell_t->property_text() = track.bitrate     ? (uint64_f % track.bitrate.get()).str()    
                                                      : "";
          break;
        }

        case COLUMN_SAMPLERATE:
        {
          cell_t->property_text() = track.samplerate  ? (uint64_f % track.samplerate.get()).str() 
                                                      : "";
          break;
        }

        case COLUMN_COUNT:
        {
          cell_t->property_text() = track.count   ? (uint64_f % track.count.get()).str() 
                                                  : "";
          break;
        }
      }
    }

    // PlaybackSource
    bool
    PlaylistList::has_playing ()
    {
      return bool (m_current_iter);
    }

    void
    PlaylistList::set_first_iter ()
    {
      assign_current_iter (m_store->get_iter (TreePath (TreePath::size_type (1), TreePath::value_type (0))));
    }

    Track
    PlaylistList::get_track ()
    {
      return Bmp::Track ((*m_current_iter.get())[m_track_cr.track]);
    }

    ustring
    PlaylistList::get_uri ()
    {
      return Bmp::Track ((*m_current_iter.get())[m_track_cr.track]).location.get();
    }

    void
    PlaylistList::notify_stop ()
    {
      clear_current_iter ();
    }

    bool
    PlaylistList::notify_play ()
    {
      PathV paths (get_selection()->get_selected_rows ());

      if( paths.size() > 1 )
      { 
        return false;
      }
      else
      if( paths.size() < 1 )
      {
        if( m_current_iter )
        {
          return true;
        }

        // Find the first playable row
        TreePath path = TreePath (TreePath::size_type (1), TreePath::value_type (0));
        TreeIter iter;

        bool has_track = false;       
        bool play = false;

        do {
          if( path.get_indices().data()[0] == (m_store->children().size()-1) )
            break;
          iter = m_store->get_iter (path); 
          bool q1 ((*iter)[m_track_cr.playTrack]);
          bool q2 ((*iter)[m_track_cr.present]);
          play = (q1 && q2); 
          has_track = (path.get_indices().data()[0] < (m_store->children().size()-1));
          path.next ();
          }
        while (!play && has_track);
  
        if( play && has_track )
        {
          m_history.set (UID ((*iter)[m_track_cr.localUid]));
          assign_current_iter (iter);
          return true;
        }
        else
          return false;
      }
      else
      {
        m_history.set (UID ((*m_store->get_iter (paths[0]))[m_track_cr.localUid]));
        assign_current_iter (m_store->get_iter (paths[0]));
        return true;
      }
    }

    bool
    PlaylistList::check_play ()
    {
      if( !m_store->children().size() )
        return false;

      if( get_selection()->count_selected_rows() == 1 )
      {
        TreeIter iter = m_store->get_iter (PathV (get_selection()->get_selected_rows())[0]);
        bool q1 ((*iter)[m_track_cr.playTrack]);
        bool q2 ((*iter)[m_track_cr.present]);
        return (q1 && q2);
      }
      else
      {
        // Find the first playable row
        TreePath path = TreePath (TreePath::size_type (1), TreePath::value_type (0));
        TreeIter iter;

        bool last = false;       
        bool play = false;

        do {
          if( path.get_indices().data()[0] == (m_store->children().size()-1) )
            break;
          iter = m_store->get_iter (path); 
          bool q1 ((*iter)[m_track_cr.playTrack]);
          bool q2 ((*iter)[m_track_cr.present]);
          play = (q1 && q2); 
          last = (path.get_indices().data()[0] == (m_store->children().size()-1));
          path.next ();
          }
        while (!play && !last);
        return play;
      }
    }

    void
    PlaylistList::assign_current_iter (Gtk::TreeModel::iterator const& iter)
    {
      m_current_iter  = iter;
      m_bmpx_track_id = Bmp::Track ((*iter)[m_track_cr.track]).bmpx_track_id;

      g_assert (bool (iter));

      TreeModel::Path path1, path2, path3 (iter);
      if( get_visible_range (path1, path2) )
      {
        if( (path3 < path1) || (path3 > path2) )
        {
          scroll_to_row (path3, 0.5);
        }
      }

      queue_draw ();
    }

    bool
    PlaylistList::notify_next ()
    {
      TreeModel::Children const& children (m_store->children());
      TreeModel::Children::size_type size (children.size());

      /* There is no next either way if the datastore is empty */
      if( !size )
      {
        return false;
      }

      /* First let's see if the history has a next item (this should not be the case if the store was cleared previously, which
       * is not unimportant, but implict here
       */
      if( m_history.have_next() )
      {
        UID uid (m_history.get_next());
        assign_current_iter (m_localUidIterMap.find (uid)->second);
        return true;
      }

      if( mcs->key_get <bool> ("bmp", "shuffle") )
      {
        TreeNodeChildren::size_type n1 = 0; 
        TreeNodeChildren::size_type n2 = (m_store->children().size()-1);

        // Find a random iter until history doesn't have it
        Glib::Rand rand;
        std::set<UID> UidKeeper;
        UID uid = 0;
        while (n1 != n2)
        {
          guint32 index = rand.get_int_range (0, n2+1); 
          TreeIter iter = m_store->get_iter (TreePath (TreePath::size_type (1), TreePath::value_type (index)));
          uid = ((*iter)[m_track_cr.localUid]);

          if (UidKeeper.find (uid) != UidKeeper.end()) 
          {
            continue;
          }
          else
          {
            UidKeeper.insert (uid);

            bool q1 ((*iter)[m_track_cr.playTrack]);
            bool q2 ((*iter)[m_track_cr.present]);
  
            if( !(q1 && q2) )
            {
              ++n1;
              continue;
            }

            if( m_history.has_uid (uid) )
            {
              ++n1; 
              continue;
            }
          }
          break;
        }

        if( (uid == 0) || (n1 == n2) )
        {
          // History is exhausted
          if( mcs->key_get <bool> ("bmp", "repeat") )
          {
            if( !m_history.empty() )
            {
              m_history.rewind ();
              UID uid;
              if( m_history.get (uid) )
              {
                UidIterMap::iterator i = m_localUidIterMap.find (uid);
                if( i != m_localUidIterMap.end() )
                {
                  assign_current_iter (i->second);
                  return true;
                }
                else
                  g_warning ("%s: History is not empty but uid %llu is not present in current view", G_STRLOC, uid);
              }
              else
                g_warning ("%s: History claims to be not empty, but can not deliver an uid (this is EXTREMELY strange!)", G_STRLOC);
            }
          }
          m_history.clear();
          return false;
        }
        else
        {
          UidIterMap::iterator i = m_localUidIterMap.find (uid);
          if( i != m_localUidIterMap.end() )
          {
            m_history.append (uid);
            assign_current_iter (i->second);
            return true;
          }
          else
          {
            g_warning ("%s: History is not empty but uid %llu is not present in current view", G_STRLOC, uid);
            return false;
          }
        }
      }

      TreeIter  iter;
      TreePath  path;
      bool      advance = 0;

      /* There is no current iter set, which should be only the case if the store is empty,
       * or it's not, but the current track is not present in the datastore 
       */
      if( !m_current_iter )
      {
        if( mcs->key_get <bool> ("bmp", "repeat") )
        {
          m_history.clear();
          path = TreePath (TreePath::size_type (1), TreePath::value_type (0));
          iter = m_store->get_iter (path);
          advance = 0;
        }
        else
        {
          /* No current iter, no repeat = nada */
          return false;
        }
      }
      else
      {
        /* I can has current iter? KTHX */
        iter = m_current_iter.get();
        path = m_store->get_path (iter);
        advance = 1;

        if( (path.get_indices().data()[0] == (m_store->children().size()-1) )
            &&  mcs->key_get<bool>("bmp","repeat"))
        {
          if( !m_history.empty() )
          {
            m_history.rewind ();
            UID uid;
            if( m_history.get (uid) )
            {
              UidIterMap::iterator i = m_localUidIterMap.find (uid);
              if( i != m_localUidIterMap.end() )
              {
                assign_current_iter (i->second);
                return true;
              }
              else
                g_warning ("%s: History is not empty but uid %llu is not present in current view", G_STRLOC, uid);
            }
            else
              g_warning ("%s: History claims to be not empty, but can not deliver an uid (this is EXTREMELY strange!)", G_STRLOC);
          }

          path = TreePath (TreePath::size_type (1), TreePath::value_type (0));
          iter = m_store->get_iter (path);
          advance = 0;
        }
      }

      do{
          if( advance )
            path.next ();
          else
            advance = 1;
  
          if (path.get_indices().data()[0] < m_store->children().size()) 
          {
            iter = m_store->get_iter (path);
            bool q1 ((*iter)[m_track_cr.playTrack]);
            bool q2 ((*iter)[m_track_cr.present]);
            if( q1 && q2 )
            {
              m_history.append (UID ((*iter)[m_track_cr.localUid]));
              assign_current_iter (iter);
              return true;
            }
          }
          else
            break;
        }
      while (1);

      clear_current_iter ();
      return false;

    }

    bool
    PlaylistList::notify_prev ()
    {
      TreeModel::Children const& children (m_store->children());
      TreeModel::Children::size_type size (children.size());

      if( !size )
      {
        return false;
      }

      if( m_history.have_prev() )
      {
        UID uid (m_history.get_prev());
        assign_current_iter (m_localUidIterMap.find (uid)->second);
        return true;
      }

      TreeIter iter = m_current_iter.get();
      TreePath path = m_store->get_path (iter);

      do{
          path.prev ();
          if( path.get_indices().data()[0] >= 0 )
          {
            iter = m_store->get_iter (path);
            bool q1 ((*iter)[m_track_cr.playTrack]);
            bool q2 ((*iter)[m_track_cr.present]);
            if( q1 && q2 )
            {
              m_history.prepend (UID ((*iter)[m_track_cr.localUid]));
              assign_current_iter (iter);
              return true;
            }
          }
          else
            break;
        }
      while (1);

      clear_current_iter ();
      return false;
    }

    void
    PlaylistList::has_next_prev (bool & next, bool & prev)
    {
      TreeModel::Children const& children (m_store->children());
      TreeModel::Children::size_type size = children.size();

      next = false;
      prev = false;

      if( !size )
        return;

      if( !m_history.boundary() )
      {
        next = m_history.have_next();
        prev = m_history.have_prev();
        return;
      }

      TreePath path;
      TreeIter iter;

      if( m_current_iter && m_current_iter.get() )
      {
        path = TreePath (m_store->get_path (m_current_iter.get()));
        if (path.get_indices().data()[0] > 0) 
        do{
            path.prev ();
            if (path.get_indices().data()[0] >= 0) 
            {
              iter = m_store->get_iter (path); 
              bool q1 ((*iter)[m_track_cr.playTrack]);
              bool q2 ((*iter)[m_track_cr.present]);
              prev = (q1 && q2); 
            }
          }
        while (!prev && (path.get_indices().data()[0] > 0));
      }

      if( mcs->key_get<bool>("bmp","repeat") || mcs->key_get<bool>("bmp","shuffle") )
      {
        next = true;
      }
      else if( m_current_iter )
      {
        path = TreePath (m_store->get_path (m_current_iter.get()));
        if( path.get_indices().data()[0] < (m_store->children().size()-1) )
        do{
            path.next ();
            if( path.get_indices().data()[0] < m_store->children().size() )
            {
              iter = m_store->get_iter (path); 
              bool q1 ((*iter)[m_track_cr.playTrack]);
              bool q2 ((*iter)[m_track_cr.present]);
              next = (q1 && q2); 
            }
            else
              break;
          }
        while (!next);
      }
    }

    void
    PlaylistList::clear_current_iter ()
    {
      if( m_current_iter && m_store->children().size() )
      {
        // Weird construct to notify treeview that the row has changed
        TreeIter iter = m_current_iter.get();
        m_current_iter.reset ();
        if( iter )
        {
          m_store->row_changed (m_store->get_path (iter), iter);
        }
      }
      else
      {
        m_current_iter.reset ();
      }
      signal_recheck_caps_.emit ();
    }

    void
    PlaylistList::on_library_track_modified (Bmp::Track const& track)
    {
      UidIterSetMap::iterator m (m_UidIterMap.find (track.bmpx_track_id));
      if( m != m_UidIterMap.end() )
      {
        IterSet& x = m->second;
        std::for_each (x.begin(), x.end(), TrackApply<ColumnTrack> (m_track_cr.track, track));
      }
    }

    void
    PlaylistList::clear ()
    {
      m_UidIterMap.clear();
      m_localUidIterMap.clear();
      m_store->clear();
      m_history.clear ();
      clear_current_iter ();
    }

    void
    PlaylistList::on_playlist_export ()
    {
      TrackV v;
      PathV p = get_selection()->get_selected_rows ();
      std::for_each (p.begin(), p.end(), PathCollect<ColumnTrack> (m_store, m_track_cr.track, v));
      ExportDialog * dialog = ExportDialog::create ();
      dialog->run (v);
      delete dialog;
    }

    void
    PlaylistList::add_tracks (TrackV const& tracks)
    {
      TreeModel::Path last;
      last.append_index (m_store->children().size());
      for (TrackV::const_iterator i = tracks.begin(); i != tracks.end(); ++i)
      {
        put_track_at_iter (*i);
      }
      scroll_to_row (last, 0.5);
      signal_recheck_caps_.emit ();
    }

    void
    PlaylistList::add_uris (VUri const& uris, bool playback)
    {
      VUri v;
      for (VUri::const_iterator u = uris.begin(); u != uris.end() ; ++u)
      {
        Bmp::URI uri (*u);
        Bmp::URI::Protocol p (uri.get_protocol());

        if( (p == URI::PROTOCOL_FILE) && !file_test (filename_from_uri (*u), FILE_TEST_EXISTS) )
        {
          continue; 
        }

        if( !vfs->has_container (*u) )
        {
          v.push_back (*u);
        }
        else
        {
          VFS::Handle h (*u);
          VUri list;

          if( file_test (filename_from_uri (*u), FILE_TEST_IS_DIR) )
              vfs->read (h, list, VFS::CONTAINER);
          else
              vfs->read (h, list, VFS::ProcessingFlags (VFS::TRANSPORT | VFS::CONTAINER));

          for (VUri::const_iterator u = list.begin(); u != list.end() ; ++u)
          {
            Bmp::URI uri (*u);
            Bmp::URI::Protocol p (uri.get_protocol());

            switch (p)
            {
              case URI::PROTOCOL_UNKNOWN:
              case URI::PROTOCOL_CDDA:
              case URI::PROTOCOL_LASTFM:
              case URI::PROTOCOL_FTP:
              case URI::PROTOCOL_QUERY:
              case URI::PROTOCOL_TRACK:
                  break;

              case URI::PROTOCOL_FILE:
                  v.push_back (*u);
                  break;

              case URI::PROTOCOL_HTTP:
              case URI::PROTOCOL_MMS:
              case URI::PROTOCOL_MMSU:
              case URI::PROTOCOL_MMST:
                  break;
            }
          }
        }
      }

      TreeModel::Path path_o (TreePath::size_type (1), TreePath::value_type (m_store->children().size()));

      for (VUri::const_iterator i = v.begin(); i != v.end(); ++i)
      {
        Track track = library->get_track (*i);
        if( file_test (filename_from_uri (track.location.get()), FILE_TEST_EXISTS) )
        {
          put_track_at_iter (track);
        }
      }

      if( playback )
      {
        assign_current_iter (m_store->get_iter (path_o));
      }

      signal_recheck_caps_.emit ();
    }

    void
    PlaylistList::set_ui_manager (RefPtr<Gtk::UIManager> const& ui_manager)
    {
      m_ui_manager = ui_manager;

      m_actions = Gtk::ActionGroup::create ("Actions_UiPartPlaylist-PlaylistList");

      m_actions->add  (Gtk::Action::create (ACTION_PLAYLIST_PLAYLIST_EXPORT,
                                            Gtk::Stock::SAVE_AS,
                                            _("Export Selected")),
                                            sigc::mem_fun (*this, &PlaylistList::on_playlist_export));

      m_actions->add  (Gtk::Action::create (ACTION_PLAYLIST_PLAYLIST_HISTORY_CLEAR,
                                            Gtk::StockID (GTK_STOCK_CLEAR),
                                            _("Clear Playback History")),
                                            sigc::mem_fun (m_history, &PlaylistList::History::clear));

      m_actions->add  (Gtk::Action::create (ACTION_PLAYLIST_PLAYLIST_CLEAR,
                                            Gtk::StockID (GTK_STOCK_CLEAR),
                                            _("Clear Playlist")),
                                            sigc::mem_fun (*this, &PlaylistList::clear));

      m_actions->add  (Gtk::Action::create (ACTION_PLAYLIST_REMOVE_PLAYLIST_ITEMS,
                                            Gtk::StockID (GTK_STOCK_REMOVE),
                                            _("Remove Selected Tracks")),
                                            sigc::mem_fun (*this, &PlaylistList::on_playlist_remove_items));

      m_actions->add  (Gtk::Action::create (ACTION_PLAYLIST_PLAYLIST_CREATE_FROM_LASTFM_TAG,
                                            Gtk::StockID (BMP_STOCK_LASTFM),
                                            _("Create Playlist from Last.fm Tag")),
                                            sigc::mem_fun (*this, &PlaylistList::on_playlist_create_from_lastfm_tag));

      if (!Network::check_connected())
      {
        m_actions->get_action (ACTION_PLAYLIST_PLAYLIST_CREATE_FROM_LASTFM_TAG)->set_sensitive (0);
      }

      m_ui_manager->insert_action_group (m_actions);
      m_ui_manager->add_ui_from_string (library_menu_playlist);
    }

    void
    PlaylistList::on_playlist_create_from_lastfm_tag ()
    {
      DialogSimpleEntry * d = DialogSimpleEntry::create ();
      d->set_title (_("Please enter a Last.fm tag to create the Playlist from"));
      ustring tag;
      int response = d->run (tag);
      delete d;

      if (response == GTK_RESPONSE_OK)
      {
        clear ();

        using namespace LastFM;
        using namespace WS;

        SimpleProgress * d = SimpleProgress::create ();
        d->set_title (_("Creating Last.fm Tag Playlist - BMP"));
        d->present ();

        d->step (5, 1);
        d->set_label (_("Fetching Last.fm Information..."));

        LastFMArtistV v;
        artists (AT_TAG_TOP, tag, v);

        if (v.empty())
        {
          delete d;
          MessageDialog dialog ((boost::format (_("There is not enough Last.fm data available to create a Playlist based on the Tag '%s'))")) 
                                  % tag.c_str()).str().c_str(),
                                false, MESSAGE_QUESTION, BUTTONS_OK, true);
          dialog.set_title (_("Creating Last.fm Tag Playlist: No Last.fm Data - BMP"));
          dialog.run ();
          return;
        }

        d->step (5, 2);
        d->set_label (_("Accessing Library..."));

        RowV rows;
        for (LastFMArtistV::const_iterator i = v.begin(); i != v.end(); ++i)
        {
          AttributeV  a;
          Query       q;

          a.push_back  (Attribute (FUZZY, get_attribute_info (ATTRIBUTE_ARTIST).id, std::string (i->name)));
          q.add_attr_v (a, CHAIN_OR);
          q.seq_chain_append (CHAIN_AND);
          q.set_suffix  (" ORDER BY ifnull(mb_album_artist_sort_name, mb_album_artist), ifnull(mb_release_date, album), tracknumber ");
          q.set_columns (" * ");
          ::library->query (q, rows, true); 
        }

        d->step (5, 3);
        d->set_label (_("Reticulating Splines..."));

        std::random_shuffle (rows.begin(), rows.end());

        d->step (5, 4);
        d->set_label (_("Creating Playlist..."));

        RowV::size_type n = 0;
        for (RowV::const_iterator r = rows.begin(); r != rows.end(); ++r)
        {
          put_track_at_iter (Track (*r));
          d->step (rows.size(), ++n);
        }
    
        d->step (5, 5);
        d->set_label (_("Starting Playback"));

        delete d;

        if (rows.size())
        {
          get_selection()->select (m_store->get_iter (TreePath (TreePath::size_type (1), TreePath::value_type (0))));
          signal_activated_.emit ();
        }
      }
    }

    void
    PlaylistList::on_playlist_remove_items ()
    {
      PathV p = get_selection()->get_selected_rows();
      ReferenceV r;
      std::for_each (p.begin(), p.end(), ReferenceCollect (m_store, r));
      ReferenceVIter i;

      for (i = r.begin() ; !r.empty() ; )
      {
        TreeIter iter = m_store->get_iter (i->get_path());
        UID localUid ((*iter)[m_track_cr.localUid]);
        m_localUidIterMap.erase (m_localUidIterMap.find (localUid));
        m_store->erase (iter);
        i = r.erase (i);
      }
    }

    bool
    PlaylistList::on_event (GdkEvent * ev)
    {
      if ((ev->type == GDK_BUTTON_RELEASE) ||
          (ev->type == GDK_2BUTTON_PRESS) ||
          (ev->type == GDK_3BUTTON_PRESS))
      {
        m_button_depressed = false;
        return false;
      }
      else
      if( ev->type == GDK_BUTTON_PRESS )
      {
        GdkEventButton * event = reinterpret_cast <GdkEventButton *> (ev);

        if( event->button == 1)
        {
          TreeViewColumn      * column;
          int                   cell_x,
                                cell_y;
          int                   tree_x,
                                tree_y;
          TreePath              path;

          widget_to_tree_coords (int (event->x), int (event->y), tree_x, tree_y);
          bool have_path = get_path_at_pos (tree_x, tree_y, path, column, cell_x, cell_y) ;

          m_button_depressed  = have_path;
          m_path_button_press = path;  
        }
        else
        if( event->button == 3 )
        {
          m_actions->get_action (ACTION_PLAYLIST_PLAYLIST_HISTORY_CLEAR)->set_sensitive
            (!m_history.empty());

          m_actions->get_action (ACTION_PLAYLIST_PLAYLIST_CLEAR)->set_sensitive
            (m_store->children().size());

          m_actions->get_action (ACTION_PLAYLIST_REMOVE_PLAYLIST_ITEMS)->set_sensitive
            (get_selection()->count_selected_rows());

          m_actions->get_action (ACTION_PLAYLIST_PLAYLIST_EXPORT)->set_sensitive
            (get_selection()->count_selected_rows());

          Gtk::Menu * menu = dynamic_cast < Gtk::Menu* >
                                (Util::get_popup (m_ui_manager, "/popup-playlist-list/menu-playlist-list"));

          if (menu) // better safe than screwed
          {
            menu->popup (event->button, event->time);
          }
          return true;
        }
      }
      return false;
    }

    bool
    PlaylistList::on_motion_notify_event (GdkEventMotion * event)
    {
      TreeView::on_motion_notify_event (event);
      if( m_button_depressed )
      {
        TreeViewColumn    * column;
        TreeModel::Path     path;

        int                 cell_x,
                            cell_y;

        int                 tree_x,
                            tree_y;

        widget_to_tree_coords (int (event->x), int (event->y), tree_x, tree_y);
        if( get_path_at_pos (tree_x, tree_y, path, column, cell_x, cell_y) )
        {
          if( path != m_path_button_press )
          {
            if( path < m_path_button_press )
              move_selected_rows_up ();
            else
            if( path > m_path_button_press )
              move_selected_rows_down ();

            m_path_button_press = path;
          }
        }
      }
      return false;
    }

    void
    PlaylistList::move_selected_rows_up ()
    {
      PathV p = get_selection()->get_selected_rows();

      if( p[0].get_indices().data()[0] < 1 )
        return; // we don't move rows if the uppermost would be pushed out of the list

      ReferenceV r;
      std::for_each (p.begin(), p.end(), ReferenceCollect (m_store, r));
      ReferenceVIter i = r.begin();

      for ( ; !r.empty() ; )
      {
        TreeModel::Path path_a (i->get_path());
        TreeModel::Path path_b (i->get_path());

        path_a.prev ();
        path_b.next ();

        TreeModel::iterator iter_a (m_store->get_iter (path_a));
        TreeModel::iterator iter_b (m_store->get_iter (path_b));

        if( G_UNLIKELY (path_b.get_indices().data()[0] == m_store->children().size()) )
        {
          m_store->iter_swap (iter_a, m_store->get_iter (i->get_path()));
        }
        else
        {
          m_store->move (iter_a, iter_b);
        }

        i = r.erase (i);
      }

      signal_recheck_caps_.emit ();
    }

    void
    PlaylistList::move_selected_rows_down ()
    {
      PathV p = get_selection()->get_selected_rows();

      if( p[p.size()-1].get_indices().data()[0] >= (m_store->children().size()-1) )
        return; // we don't move rows if the LOWER-most would be pushed out of the list, either

      ReferenceV r;
      std::for_each (p.begin(), p.end(), ReferenceCollect (m_store, r));
      ReferenceVIter i = r.begin();

      for ( ; !r.empty() ; )
      {
        TreeModel::Path path_a (i->get_path());
        TreeModel::Path path_b (i->get_path());

        path_b.next ();

        TreeModel::iterator iter_a (m_store->get_iter (path_a));
        TreeModel::iterator iter_b (m_store->get_iter (path_b));

        m_store->move (iter_b, iter_a);
        i = r.erase (i);
      }
      signal_recheck_caps_.emit ();
    }

    PlaylistList::~PlaylistList ()
    {}

#ifdef HAVE_HAL
    void
    PlaylistList::hal_volume_del  (HAL::Volume const& volume)
    {
      unselect_missing ();
      TreeNodeChildren nodes = m_store->children();
      for (TreeNodeChildren::iterator i = nodes.begin(); i != nodes.end(); ++i)
      {
        Bmp::Track const& track = (*i)[m_track_cr.track];
        (*i)[m_track_cr.present] = file_test (filename_from_uri (track.location.get()), FILE_TEST_EXISTS); 
      }
    }

    void
    PlaylistList::hal_volume_add  (HAL::Volume const& volume)
    {
      TreeNodeChildren nodes = m_store->children();
      for (TreeNodeChildren::iterator i = nodes.begin(); i != nodes.end(); ++i)
      {
        Bmp::Track const& track = (*i)[m_track_cr.track];
        (*i)[m_track_cr.present] = file_test (filename_from_uri (track.location.get()), FILE_TEST_EXISTS); 
      }
    }

    void
    PlaylistList::unselect_missing ()
    {
      PathV list (get_selection()->get_selected_rows());
      for (PathV::const_iterator i = list.begin(); i != list.end(); ++i)
      {
        Bmp::Track const& track = (*m_store->get_iter (*i))[m_track_cr.track];
        if( !hal->volume_is_mounted (HAL::VolumeKey (track.volume_udi.get(), track.device_udi.get())) )
        {
          get_selection()->unselect (*i);
        }
      }
    }
#endif //HAVE_HAL
}

namespace
{
  using namespace Bmp;
  const PlaybackSource::Flags flags =
    PlaybackSource::Flags (PlaybackSource::F_ALWAYS_IMAGE_FRAME | PlaybackSource::F_HANDLE_LASTFM |
                           PlaybackSource::F_USES_REPEAT | PlaybackSource::F_USES_SHUFFLE);
}

namespace Bmp
{
  namespace UiPart
  {
      Playlist::Playlist (RefPtr<Gnome::Glade::Xml> const& xml,RefPtr<UIManager> ui_manager)

      : PlaybackSource          (_("Playlist"), PlaybackSource::CAN_SEEK, flags)
      , Base                    (xml, ui_manager)
      , m_playing               (0)
      {
        m_ref_xml->get_widget_derived ("playlist-view-playlist", m_playlist);

        m_playlist->set_ui_manager (m_ui_manager);
        m_playlist->get_selection()->signal_changed()
          .connect (sigc::mem_fun (*this, &Bmp::UiPart::Playlist::on_playlist_selection_changed)); // could be merged with sig caps recheck ?
        m_playlist->signal_activated()
          .connect (sigc::mem_fun (*this, &Bmp::UiPart::Playlist::on_playlist_activated));
        m_playlist->signal_recheck_caps()
          .connect (sigc::mem_fun (*this, &Bmp::UiPart::Playlist::query_playlist_caps)); // something changed

        m_actions = Gtk::ActionGroup::create ("Actions_UiPartPlaylist");
        m_ui_manager->insert_action_group (m_actions);

        mcs->subscribe ("UiPart::Playlist", "bmp", "shuffle",
          sigc::mem_fun (*this, &Bmp::UiPart::Playlist::on_shuffle_repeat_toggled));

        mcs->subscribe ("UiPart::Playlist", "bmp", "repeat",
          sigc::mem_fun (*this, &Bmp::UiPart::Playlist::on_shuffle_repeat_toggled));

      }

      Playlist::~Playlist () {}
      guint Playlist::add_ui () { return 0; }

      void
      Playlist::on_playlist_activated ()
      {
        s_playback_request_.emit();
      }

      void
      Playlist::query_playlist_caps ()
      {
        if( m_playing )
        {
          bool next, prev;
          m_playlist->has_next_prev (next, prev);

          if( next )
            m_caps = Caps (m_caps |  PlaybackSource::CAN_GO_NEXT);
          else
            m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_NEXT);

          if( prev )
            m_caps = Caps (m_caps |  PlaybackSource::CAN_GO_PREV);
          else
            m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_PREV);
        }
        else
        {
            m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_NEXT);
            m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_PREV);
        }
        s_caps_.emit (m_caps);
      }

      void
      Playlist::on_playlist_selection_changed ()
      {
        if( m_playlist->check_play() )
          m_caps = Caps (m_caps | PlaybackSource::CAN_PLAY);
        else
          m_caps = Caps (m_caps & ~PlaybackSource::CAN_PLAY);

        s_caps_.emit (m_caps);
      }

      void
      Playlist::on_shuffle_repeat_toggled (MCS_CB_DEFAULT_SIGNATURE)
      {
        query_playlist_caps ();
      }

      ustring
      Playlist::get_uri ()
      {
        return m_playlist->get_uri ();
      }

      GHashTable*
      Playlist::get_metadata ()
      {
        return library->get_metadata_hash_table_for_uri (m_playlist->get_uri());
      }

      void
      Playlist::send_metadata ()
      {
        if( m_playlist->has_playing() )
        {
          s_track_metadata_.emit (m_playlist->get_track ());
        }
      }

      bool
      Playlist::go_next ()
      {
        if( !m_playlist->notify_next () )
          return false;

        query_playlist_caps ();
        send_metadata ();
        return true;
      }

      bool
      Playlist::go_prev ()
      {
        if( !m_playlist->notify_prev () )
          return false;

        query_playlist_caps ();
        send_metadata ();
        return true;
      }

      void
      Playlist::stop ()
      {
        m_caps = Caps (m_caps & ~PlaybackSource::CAN_PAUSE);
        m_caps = Caps (m_caps & ~PlaybackSource::CAN_PROVIDE_METADATA);
        m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_NEXT);
        m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_PREV);

        m_playing = 0;
        m_playlist->m_playing = 0;

        m_playlist->notify_stop ();
      }

      void
      Playlist::play ()
      {
        if( !m_playlist->notify_play () )
        {
          throw UnableToInitiatePlaybackError();
        }
      }

      void
      Playlist::play_post ()
      {
        m_caps = Caps (m_caps | PlaybackSource::CAN_PAUSE);
        m_caps = Caps (m_caps | PlaybackSource::CAN_PROVIDE_METADATA);
        m_playing = 1;
        m_playlist->m_playing = 1;
        query_playlist_caps ();
        send_metadata ();
      }
  }
}
