//  BMPx - The Dumb Music Player
//  Copyright (C) 2005-2007 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifndef BMP_UI_PART_PLAYLIST_HH
#define BMP_UI_PART_PLAYLIST_HH

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#ifdef HAVE_TR1
#include<tr1/unordered_map>
#include<string>
#include<iostream>
using namespace std;
using namespace std::tr1;
#endif

#include <list>
#include <set>
#include <string>
#include <vector>
#include <queue>
#include <boost/optional.hpp>
#include <boost/shared_ptr.hpp>
#include <gdkmm/pixbuf.h>
#include <glibmm/ustring.h>
#include <gtkmm.h>
#include <libglademm/xml.h>
#include <libsexymm/icon-entry.h>

#include "mcs/mcs.h"
#include "bmp/base-types.hh"
#include "bmp/library-types.hh"
#include "widgets/cell-renderer-cairo-surface.hh"

// BMP Audio
#include "audio/audio.hh"

// BMP Misc
#ifdef HAVE_HAL
#  include "hal.hh"
#endif //HAVE_HAL

#include "dialog-simple-progress.hh"
#include "util.hh"
#include "util-file.hh"

#include "playbacksource.hh"
#include "ui-part-base.hh"

using namespace Bmp;
using namespace DB;
using namespace Gtk;

namespace Bmp
{              
  typedef sigc::signal<void>  SVoidT;
  typedef std::map<UID, RowV::size_type> UidIndexMap;
  typedef UidIndexMap::value_type UidIndexPair;
  typedef UidIndexMap::iterator UidIndexMapIter;

  class PlaylistList
    : public Gtk::TreeView
  {
    private:

      Glib::RefPtr <Gdk::Pixbuf> m_pb_playing;
      Glib::RefPtr <Gnome::Glade::Xml> m_ref_xml;
      Glib::RefPtr <Gtk::UIManager> m_ui_manager;
      Glib::RefPtr <Gtk::ActionGroup> m_actions;

      // History
      class History
      {
        public:

          History () : m_position (m_queue.begin()) {}
          ~History () {}

          bool
          boundary ()
          {
            return ( (std::distance (m_queue.begin(), m_position) == 0) ||
                     (std::distance (m_position, (m_queue.end()-1)) == 0));
          }
    
          bool         
          have_prev ()
          {
            return ((m_queue.size() > 1) && (std::distance (m_queue.begin(), m_position) > 0));
          }

          bool
          have_next ()
          {
            return ((m_queue.size() > 1) && (std::distance (m_position, (m_queue.end()-1)) > 0));
          }

          UID
          get_prev ()
          {
            --m_position;
            return UID (*m_position);
          }

          UID
          get_next ()
          {
            ++m_position;
            return UID (*m_position);
          }

          void
          append (UID uid)
          {
            if (m_queue.size() == 1024)      
            {
              Position p (m_queue.begin());
              m_queue.erase (p);
            }
            m_queue.push_back (uid);
            m_position = m_queue.end() - 1; 
          }

          void
          prepend (UID uid)
          {
            if (m_queue.size() == 1024)      
            {
              Position p (m_queue.end() - 1);
              m_queue.erase (p);
            }
            m_queue.push_front (uid);
            m_position = m_queue.begin ();
          }

          void
          erase  (UID uid)
          {
            Position p = m_queue.begin();
            for ( ; p != m_queue.end() ; )
            {
              if (*p == uid)
                p = m_queue.erase (p);
              else
                ++p;
            }
          } 

          bool
          has_uid (UID uid)
          {
            for (Position p = m_queue.begin() ; p != m_queue.end() ; ++p)
            {
              if (*p == uid)
                return true;
            }
            return false;
          }

          void
          clear ()
          {
            m_queue.clear ();
            m_position = m_queue.begin();
          }

          void
          set (UID uid)
          {
            if (!boundary())
              m_queue.erase (m_position+1, m_queue.end());
            append (uid);
          }

          void
          rewind ()
          {
            m_position = m_queue.begin();
          }

          bool
          get (UID & uid)
          {
            if (m_queue.size())
            {
              uid = *m_position;
              return true;
            }
            return false;
          }

          bool
          empty ()
          {
            return m_queue.empty();
          }

          void
          print (const char* func)
          {
            printf ("Func: %s\nSize: %llu\n", func, uint64_t (m_queue.size()));
            for (Queue::const_iterator i = m_queue.begin(); i != m_queue.end(); ++i)
            {
              if (i == m_position)
              {
                printf ("%llu\t*\n", *i);
              }
              else
              {
                printf ("%llu\t-\n", *i);
              }
            }
            printf ("\n");
          }

        private:

          typedef std::deque <UID>  Queue;
          typedef Queue::iterator   Position;

          Queue     m_queue; 
          Position  m_position;
      };

      UidIterSetMap                   m_UidIterMap;
      UidIterMap                      m_localUidIterMap;
      UID                             m_localUid;
      History                         m_history;

      // Main Datastore
      class ColumnsTrack
        : public Gtk::TreeModel::ColumnRecord
      {
        public:
        
          ColumnTrack   track;
          ColumnUID     uid;
          ColumnUID     localUid;
          ColumnString  searchKey;
          ColumnBool    playTrack;
          ColumnBool    present;

          ColumnsTrack ()
          {
            add (track);
            add (uid);
            add (localUid);
            add (searchKey);
            add (playTrack);
            add (present);
          }
      };

      ColumnsTrack
      m_track_cr;

      Glib::RefPtr <Gtk::ListStore>
      m_store;

      boost::optional <Gtk::TreeIter>
      m_current_iter;

      uint64_t
      m_bmpx_track_id;

      void  put_track_at_iter (Track const& track, Gtk::TreeIter & iter);
      void  put_track_at_iter (Track const& track); // this will append the track

      void  cell_data_func (Gtk::CellRenderer * basecell, Gtk::TreeIter const& iter, int column, int renderer);
      void  cell_play_track_toggled (Glib::ustring const& path);
      bool  slot_select (Glib::RefPtr <Gtk::TreeModel> const& model, Gtk::TreePath const& path, bool was_selected);

      void  assign_current_iter (Gtk::TreeIter const& iter);
      void  clear_current_iter ();

      void  move_selected_rows_up ();
      void  move_selected_rows_down ();

      TreePath  m_path_button_press;
      bool      m_button_depressed;

      void  on_playlist_export ();
      void  on_playlist_create_from_lastfm_tag ();
      void  on_playlist_remove_items ();

#ifdef HAVE_HAL
      void  hal_volume_del (HAL::Volume const& volume);
      void  hal_volume_add (HAL::Volume const& volume);
      void  unselect_missing ();
#endif //HAVE_HAL

      void  on_library_track_modified (Track const& track);

    protected:

      virtual bool on_event (GdkEvent * event);
      virtual bool on_motion_notify_event (GdkEventMotion * event);

      virtual bool on_drag_motion (Glib::RefPtr<Gdk::DragContext> const& context, int x, int y, guint time);
      virtual bool on_drag_drop (Glib::RefPtr<Gdk::DragContext> const& context, int x, int y, guint time);
      virtual void on_drag_data_received (Glib::RefPtr<Gdk::DragContext> const& context, int x, int y,
                                            Gtk::SelectionData const& selection_data, guint info, guint time);
      virtual void on_drag_leave (Glib::RefPtr<Gdk::DragContext> const& context, guint time);
      virtual void on_row_activated (Gtk::TreeModel::Path const&, Gtk::TreeViewColumn*);

    public:

      Glib::ustring get_uri ();
      Track get_track ();

      bool  notify_next ();
      bool  notify_prev ();
      bool  notify_play ();
      void  notify_stop ();
      bool  check_play ();

      void  has_next_prev (bool & next, bool & prev);
      bool  has_playing ();
      void  set_first_iter ();

      void  enable_drag_dest ();
      void  disable_drag_dest ();

      bool  m_playing;

      void  clear ();

      typedef sigc::signal <void> SignalRecheckCaps;
      typedef sigc::signal <void> Signal;
      
    private:

      Signal            signal_activated_;
      SignalRecheckCaps signal_recheck_caps_;

    public:

      Signal&
      signal_activated()
      {
        return signal_activated_;
      }

      SignalRecheckCaps&
      signal_recheck_caps ()
      {
        return signal_recheck_caps_;
      }

      PlaylistList (BaseObjectType                       * obj,
                    Glib::RefPtr<Gnome::Glade::Xml> const& xml);
      virtual ~PlaylistList ();

      void  set_ui_manager (Glib::RefPtr<Gtk::UIManager> const& ui_manager);
      void  add_uris (VUri const& uris, bool playback);
      void  add_tracks (TrackV const& tracks);
  };

  namespace UiPart
  {
    class Playlist
      : public  PlaybackSource,
        public  Base
    {
      public:

        Playlist (Glib::RefPtr<Gnome::Glade::Xml> const& xml, Glib::RefPtr<Gtk::UIManager> ui_manager);
        virtual ~Playlist ();

      private:

        PlaylistList    * m_playlist;

        void  on_playlist_selection_changed ();
        void  on_playlist_activated ();
        void  on_shuffle_repeat_toggled (MCS_CB_DEFAULT_SIGNATURE);

        void  send_metadata ();
        void  query_playlist_caps ();

        bool  m_playing;

      protected:

        virtual guint
        add_ui ();

        virtual Glib::ustring
        get_uri ();

        virtual Glib::ustring
        get_type () { return Glib::ustring(); }

        virtual bool
        go_next ();

        virtual bool
        go_prev ();

        virtual void
        stop ();

        virtual void
        play ();

        virtual void
        play_post ();

        virtual void
        restore_context () {}

        virtual GHashTable*
        get_metadata ();
    };
  }
}
#endif //!BMP_UI_PART_PLAYLIST_HH
