//  BMP
//  Copyright (C) 2005-2007 BMP development.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#include <gtkmm.h>
#include <glibmm/i18n.h>
#include <libglademm.h>

#include "bmp/types/types-basic.hh"
#include "widgets/taskdialog.hh"

#include "paths.hh"
#include "ui-tools.hh"

#include "x_hal.hh"
#include "x_library.hh"

#include "dialog-manage-volume.hh"

using namespace Gtk;
using namespace Glib;
using namespace std;

namespace Bmp
{
      VolumeManageDialog*
      VolumeManageDialog::create (Glib::RefPtr <Gnome::Glade::Xml> const& xml)
      {
        VolumeManageDialog *p = 0;
        xml->get_widget_derived ("preferences-manage-volume", p);
        return p;
      }

      VolumeManageDialog::VolumeManageDialog (BaseObjectType                       *  obj,
                                              Glib::RefPtr<Gnome::Glade::Xml> const&  xml)
      : Gtk::Dialog (obj),
        m_ref_xml   (xml)
      {
        Util::window_set_icon_list (*this, "player");

        m_ref_xml->get_widget ("preferences-volumes-music-paths-view", m_view_music_paths);

        m_store = Gtk::ListStore::create (columns);
        m_view_music_paths->set_model (m_store);

        set_response_sensitive (0, false);
        set_response_sensitive (1, false);

        m_view_music_paths->get_selection()->set_mode (Gtk::SELECTION_SINGLE);
        m_view_music_paths->get_selection()->signal_changed().connect
          (sigc::mem_fun (this, &Bmp::VolumeManageDialog::on_selection_changed));

        CellRendererText * cell = manage (new CellRendererText());
        m_view_music_paths->append_column (_("Path"), *cell);
        m_view_music_paths->get_column (0)->set_cell_data_func
          (*cell, sigc::mem_fun (this, &Bmp::VolumeManageDialog::cell_data_func));
      }

      void
      VolumeManageDialog::cell_data_func (Gtk::CellRenderer*              basecell,
                                          Gtk::TreeModel::iterator const& iter)
      {
        CellRendererText* cell = dynamic_cast<CellRendererText *> (basecell);

        cell->property_text () =
          filename_to_utf8 (build_filename (m_volume.mount_point, std::string ((*iter)[columns.path])));
      }

      void
      VolumeManageDialog::on_selection_changed ()
      {
        bool selection (m_view_music_paths->get_selection()->count_selected_rows());
        set_response_sensitive (0, selection);
        set_response_sensitive (1, selection);
      }

      void
      VolumeManageDialog::go (HAL::Volume const& volume)
      {
        m_volume = volume;

        m_view_music_paths->unset_model ();
        m_store->clear ();

        StrV insert_paths; 
        Library::Obj()->volume_insert_paths (volume.volume_udi,
                                      volume.device_udi,
                                      insert_paths);

        for (StrV::const_iterator i = insert_paths.begin(); i != insert_paths.end(); ++i)
        {
          TreeModel::iterator iter (m_store->append());
          (*iter)[columns.path] = *i;
        }

        m_view_music_paths->set_model (m_store);

        int response = 0;

        enum DialogResponses
        {
          RESPONSE_REMOVE_PATH,
          RESPONSE_RESCAN_PATH,
          RESPONSE_FORGET_VOLUME
        };

        do {
          response = Gtk::Dialog::run ();
          switch (response)
          {
            case RESPONSE_REMOVE_PATH:
            {
              TaskDialog dialog ( this,
                                  _("Remove Path - BMP"),
                                  _("Remove the given path (remove all files on this path from the library)"),
                                  MESSAGE_QUESTION);
              dialog.add_button ( _("Remove"),
                                  _("Remove the path right now"),
                                  Gtk::Stock::APPLY,
                                  Gtk::RESPONSE_OK);
              dialog.add_button ( _("Do not Remove"),
                                  _("Do not Remove this path at this time"),
                                  Gtk::Stock::CANCEL,
                                  Gtk::RESPONSE_CANCEL);
              dialog.set_default_response (Gtk::RESPONSE_OK);

              set_modal (false);
              int response = dialog.run();
              set_modal ();

              if (response == Gtk::RESPONSE_OK)
              {
                TreeModel::iterator const& iter = m_view_music_paths->get_selection()->get_selected ();
                std::string path ((*iter)[columns.path]);

                Library::Obj()->remove_path ( m_volume.volume_udi,
                                       m_volume.device_udi,
                                       path);
                m_store->erase (iter);
              }
              break;
            }

            case RESPONSE_RESCAN_PATH:
            {
              TreeModel::iterator const& iter = m_view_music_paths->get_selection()->get_selected ();
              std::string path ((*iter)[columns.path]);

              Library::Obj()->vacuum_nonexistent ( m_volume.volume_udi,
                                            m_volume.device_udi,
                                            path);
              signal_scan_path_.emit (build_filename (m_volume.mount_point, path));
              break;
            }

            case RESPONSE_FORGET_VOLUME:
            {
              TaskDialog dialog ( this,
                                  _("Forget Volume - BMP"),
                                  _("Do you really want make BMP forget all music from this volume?"),
                                  MESSAGE_QUESTION);
              dialog.add_button ( _("Forget Volume"),
                                  _("This will make BMP forget all Music from this Volume"),
                                  Gtk::Stock::GO_FORWARD,
                                  Gtk::RESPONSE_OK);
              dialog.add_button ( _("Don't Forget Volume"),
                                  _("Do not forget the Volume at this time"),
                                  Gtk::Stock::CANCEL,
                                  Gtk::RESPONSE_CANCEL);
              dialog.set_default_response (Gtk::RESPONSE_OK);

              set_modal (false);
              int response = dialog.run();
              set_modal ();

              if (response == Gtk::RESPONSE_OK)
              {
                Library::Obj()->remove_volume (m_volume.volume_udi, m_volume.device_udi);
              }
              break;
            }

            default: break;
          }

        } while (response != Gtk::RESPONSE_CLOSE);

      }
}
