{-
    BNF Converter: Abstract syntax
    Copyright (C) 2004  Author: Markus Forsberg, Aarne Ranta

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
-}


module GetCF where

import Directory	( doesFileExist, renameFile )
import Monad		( when )

import CF
import Utils
import ParBNF
import List(nub,partition)
import qualified AbsBNF as Abs
-- import LexBNF
import ErrM
import Char

cfFile        name = name ++ ".cf"

readCF :: FilePath -> IO CF
readCF f = tryReadCF f >>= return . fst

tryReadCF :: FilePath -> IO (CF,Bool)
tryReadCF name = do
  s <- readFile $ cfFile name
  putStrLn $ "\nReading grammar from " ++ name
  let (cf,msg) = getCF s
  if not (null msg) then do
    putStrLn $ unlines msg
    return (cf,False)
   else do
    putStrLn $ show (length (rulesOfCF cf)) +++ "rules accepted\n"
    case (notUniqueFuns cf) of
     [] -> case (badInheritence cf) of
       [] -> return (cf,True)
       xs -> do
        putStrLn "Warning :"
        putStrLn $ "  Bad Label name in Category(s) :" ++ unwords xs
        putStrLn $ "  These categories have more than one Label, yet one of these"
        putStrLn $ "  Labels has the same name as the Category. This will almost"
        putStrLn $ "  certainly cause problems in languages other than Haskell.\n"
        return (cf,True)
     xs -> do  
       putStrLn $ "Warning :" 
       putStrLn $ "  Non-unique label name(s) : " ++ unwords xs
       putStrLn $ "  There may be problems with the pretty-printer.\n"
       case (badInheritence cf) of
         [] -> return (cf,True)
         xs -> do
          putStrLn $ "Warning :"
          putStrLn $ "  Bad Label name in Category(s) :" ++ unwords xs
          putStrLn $ "  These categories have more than one Label, yet one of these"
          putStrLn $ "  Labels has the same name as the Category. This will almost"
          putStrLn $ "  certainly cause problems in languages other than Haskell.\n"
          return (cf,True)

-- peteg: FIXME this is racey.
-- want to be a bit smarter about whether we actually generate the file
-- or save it... e.g. ErrM.hs need not be regenerated if it exists.
writeFileRep :: FilePath -> String -> IO ()
writeFileRep f s =
    do exists <- doesFileExist f
       backedUp <- if exists
		     then do let fbak = f ++ ".bak"
		             renameFile f fbak
			     return $ " (saved old file as " ++ fbak ++ ")"
		     else return ""
       writeFile f s
       putStrLn $ "wrote file " ++ f ++ backedUp

getCF :: String -> (CF, [String])
getCF s = (cf,msgs ++ msgs1) where
  (cf,msgs1) = ((exts,ruls2),msgs2)
  (ruls2,msgs2) = untag $ partition (isRule) $ map (checkRule cf0) $ rulesOfCF cf0
  untag (ls,rs) = ([c | Left c <- ls], [c | Right c <- rs])
  isRule = either (const True) (const False)
  (cf0@(exts,_),msgs) = (revs . srt . conv . pGrammar . myLexer) s
  srt rs = let rules              = [r | Left (Right r) <- rs]
	       literals           = nub  [lit | xs <- map rhsRule rules,
					        (Left lit) <- xs,
					        elem lit specialCatsP]
	       pragma             = [r | Left (Left r) <- rs]
	       errors             = [s | Right s <- rs, not (null s)]
	       (symbols,keywords) = partition (any isSpec) reservedWords
	       isSpec             = flip elem "$+-*=<>[](){}!?.,;:^~|&%#/\\$_@\"\'"
	       reservedWords      = nub [t | (_,(_,its)) <- rules, Right t <- its]
               cats               = []
	    in (((pragma,(literals,symbols,keywords,cats)),rules),errors)
  revs (cf@((pragma,(literals,symbols,keywords,_)),rules),errors) =
    (((pragma,(literals,symbols,keywords,findAllReversibleCats cf)),rules),errors)

conv :: Err Abs.Grammar -> [Either (Either Pragma Rule) String]
conv (Bad s)                 = [Right s]
conv (Ok (Abs.Grammar defs)) = map Left $ concatMap transDef defs

transDef :: Abs.Def -> [Either Pragma Rule]
transDef x = case x of
 Abs.Rule label cat items -> 
   [Right (transLabel label,(transCat cat,map transItem items))]
 Abs.Comment str               -> [Left $ CommentS str]
 Abs.Comments str0 str         -> [Left $ CommentM (str0,str)]
 Abs.Token ident reg           -> [Left $ TokenReg (transIdent ident) reg]
 Abs.Entryp idents             -> [Left $ EntryPoints (map transIdent idents)]
 Abs.Internal label cat items  -> 
   [Right (transLabel label,(transCat cat,(Left "#":(map transItem items))))]
 Abs.Separator size ident str -> map Right $ separatorRules size ident str
 Abs.Terminator size ident str -> map Right $ terminatorRules size ident str
 Abs.Coercions ident int -> map Right $ coercionRules ident int
 Abs.Rules ident strs -> map Right $ ebnfRules ident strs
 Abs.Layout ss      -> [Left $ Layout ss]
 Abs.LayoutStop ss  -> [Left $ LayoutStop ss]
 Abs.LayoutTop      -> [Left $ LayoutTop]

separatorRules :: Abs.MinimumSize -> Abs.Cat -> String -> [Rule]
separatorRules size c s = if null s then terminatorRules size c s else ifEmpty [
  ("(:[])", (cs,[Left c'])),
  ("(:)",   (cs,[Left c', Right s, Left cs]))
  ]
 where 
   c' = transCat c
   cs = "[" ++ c' ++ "]"
   ifEmpty rs = if (size == Abs.MNonempty) then rs else (("[]", (cs,[])) : rs)

terminatorRules :: Abs.MinimumSize -> Abs.Cat -> String -> [Rule]
terminatorRules size c s = [
  ifEmpty,
  ("(:)",   (cs,Left c' : s' [Left cs]))
  ]
 where 
   c' = transCat c
   cs = "[" ++ c' ++ "]"
   s' its = if null s then its else (Right s : its)
   ifEmpty = if (size == Abs.MNonempty) 
                then ("(:[])",(cs,[Left c'] ++ if null s then [] else [Right s]))
                else ("[]",   (cs,[]))

coercionRules :: Abs.Ident -> Integer -> [Rule]
coercionRules (Abs.Ident c) n = 
   ("_", (c,               [Left (c ++ "1")])) :
  [("_", (c ++ show (i-1), [Left (c ++ show i)])) | i <- [2..n]] ++
  [("_", (c ++ show n,     [Right "(", Left c, Right ")"]))]

ebnfRules :: Abs.Ident -> [Abs.RHS] -> [Rule]
ebnfRules (Abs.Ident c) rhss = 
  [(mkFun k c its, (c, map transItem its)) | (k, Abs.RHS its) <- zip [1 :: Int ..] rhss]
 where
   mkFun k c i = case i of
     [Abs.Terminal s]  -> c' ++ "_" ++ mkName k s
     [Abs.NTerminal n] -> c' ++ identCat (transCat n)
     _ -> c' ++ "_" ++ show k
   c' = c --- normCat c
   mkName k s = if all (\c -> isAlphaNum c || elem c "_'") s 
                   then s else show k

transItem :: Abs.Item -> Either Cat String
transItem x = case x of
 Abs.Terminal str   -> Right str
 Abs.NTerminal cat  -> Left (transCat cat)

transCat :: Abs.Cat -> Cat
transCat x = case x of
 Abs.ListCat cat  -> "[" ++ (transCat cat) ++ "]"
 Abs.IdCat id     -> transIdent id

transLabel :: Abs.Label -> Fun
transLabel x = case x of
 Abs.Id id     -> transIdent id
 Abs.Wild      -> "_"
 Abs.ListE     -> "[]"
 Abs.ListCons  -> "(:)"
 Abs.ListOne   -> "(:[])"

transIdent :: Abs.Ident -> String
transIdent x = case x of
 Abs.Ident str  -> str
