/***************************************************************************
 *            cdrecord.c
 *
 *  dim jan 22 15:22:52 2006
 *  Copyright  2006  Rouquier Philippe
 *  bonfire-app@wanadoo.fr
 ***************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <math.h>

#include <glib.h>
#include <glib-object.h>
#include <glib/gi18n-lib.h>
#include <glib/gstdio.h>

#include <nautilus-burn-drive.h>

#include "burn-basics.h"
#include "burn-common.h"
#include "burn-cdrecord.h"
#include "burn-process.h"
#include "burn-recorder.h"
#include "burn-imager.h"

static void bonfire_cdrecord_class_init (BonfireCDRecordClass *klass);
static void bonfire_cdrecord_init (BonfireCDRecord *sp);
static void bonfire_cdrecord_finalize (GObject *object);
static void bonfire_cdrecord_iface_init_record (BonfireRecorderIFace *iface);

static BonfireBurnResult
bonfire_cdrecord_stderr_read (BonfireProcess *process,
			      const char *line);
static BonfireBurnResult
bonfire_cdrecord_stdout_read (BonfireProcess *process,
			      const char *line);
static BonfireBurnResult
bonfire_cdrecord_set_argv (BonfireProcess *process,
			   GPtrArray *argv,
			   gboolean has_master, 
			   GError **error);

static BonfireBurnResult
bonfire_cdrecord_get_status (BonfireRecorder *recorder,
			     int *speed,
			     int *fifo,
			     gint64 *bytes_written);

static BonfireBurnResult
bonfire_cdrecord_set_drive (BonfireRecorder *recorder,
			    NautilusBurnDrive *drive,
			    GError **error);
static BonfireBurnResult
bonfire_cdrecord_set_source (BonfireJob *job,
			     const BonfireTrackSource *track,
			     GError **error);
static BonfireBurnResult
bonfire_cdrecord_set_flags (BonfireRecorder *recorder,
			    BonfireRecorderFlag flags,
			    GError **error);
static BonfireBurnResult
bonfire_cdrecord_set_speed (BonfireJob *job,
			    gint64 speed);

static BonfireBurnResult
bonfire_cdrecord_record (BonfireRecorder *recorder,
			 GError **error);
static BonfireBurnResult
bonfire_cdrecord_blank (BonfireRecorder *recorder,
			GError **error);

static BonfireBurnResult
bonfire_cdrecord_get_action_string (BonfireJob *job,
				    BonfireBurnAction action,
				    char **string);

typedef enum {
	BONFIRE_CD_RECORD_ACTION_NONE,
	BONFIRE_CD_RECORD_ACTION_BLANK,
	BONFIRE_CD_RECORD_ACTION_RECORD
} BonfireCDRecordAction;

struct BonfireCDRecordPrivate {
	BonfireCDRecordAction action;

	NautilusBurnDrive *drive;
	BonfireTrackSource *track;
	BonfireTrackSource *inf;
	int speed;

	gint64 current_track_end_pos;
	gint64 current_track_written;
	gint64 tracks_total_bytes;
	gint64 b_written;

	int current_track_num;
	int track_count;

	int fifo;
	int cur_speed;

	int dao:1;
	int dummy:1;
	int multi:1;
	int nograce:1;
	int overburn:1;
	int burnproof:1;

	int blank_fast:1;
};

static GObjectClass *parent_class = NULL;

GType
bonfire_cdrecord_get_type ()
{
	static GType type = 0;

	if(type == 0) {
		static const GTypeInfo our_info = {
			sizeof (BonfireCDRecordClass),
			NULL,
			NULL,
			(GClassInitFunc)bonfire_cdrecord_class_init,
			NULL,
			NULL,
			sizeof (BonfireCDRecord),
			0,
			(GInstanceInitFunc)bonfire_cdrecord_init,
		};

		static const GInterfaceInfo recorder_info =
		{
			(GInterfaceInitFunc) bonfire_cdrecord_iface_init_record,
			NULL,
			NULL
		};

		type = g_type_register_static (BONFIRE_TYPE_PROCESS, 
					       "BonfireCDRecord", 
					       &our_info,
					       0);
		g_type_add_interface_static (type,
					     BONFIRE_TYPE_RECORDER,
					     &recorder_info);
	}

	return type;
}

static void
bonfire_cdrecord_class_init (BonfireCDRecordClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	BonfireJobClass *job_class = BONFIRE_JOB_CLASS (klass);
	BonfireProcessClass *process_class = BONFIRE_PROCESS_CLASS (klass);

	parent_class = g_type_class_peek_parent(klass);
	object_class->finalize = bonfire_cdrecord_finalize;

	job_class->get_action_string = bonfire_cdrecord_get_action_string;
	job_class->set_source = bonfire_cdrecord_set_source;
	job_class->set_rate = bonfire_cdrecord_set_speed;

	process_class->stderr_func = bonfire_cdrecord_stderr_read;
	process_class->stdout_func = bonfire_cdrecord_stdout_read;
	process_class->set_argv = bonfire_cdrecord_set_argv;
}

static void
bonfire_cdrecord_iface_init_record (BonfireRecorderIFace *iface)
{
	iface->blank = bonfire_cdrecord_blank;
	iface->record = bonfire_cdrecord_record;
	iface->set_drive = bonfire_cdrecord_set_drive;
	iface->set_flags = bonfire_cdrecord_set_flags;
	iface->get_status = bonfire_cdrecord_get_status;
}

static void
bonfire_cdrecord_init (BonfireCDRecord *obj)
{
	obj->priv = g_new0 (BonfireCDRecordPrivate, 1);
}

static void
bonfire_cdrecord_finalize (GObject *object)
{
	BonfireCDRecord *cobj;
	cobj = BONFIRE_CD_RECORD(object);

	if (cobj->priv->drive) {
		nautilus_burn_drive_unref (cobj->priv->drive);
		cobj->priv->drive = NULL;
	}

	if (cobj->priv->track) {
		bonfire_track_source_free (cobj->priv->track);
		cobj->priv->track = NULL;
	}

	if (cobj->priv->inf) {
		bonfire_track_source_free (cobj->priv->inf);
		cobj->priv->inf = NULL;
	}

	g_free(cobj->priv);
	G_OBJECT_CLASS(parent_class)->finalize(object);
}

BonfireCDRecord *
bonfire_cdrecord_new ()
{
	BonfireCDRecord *obj;
	
	obj = BONFIRE_CD_RECORD(g_object_new(BONFIRE_TYPE_CD_RECORD, NULL));
	
	return obj;
}

static BonfireBurnResult
bonfire_cdrecord_stderr_read (BonfireProcess *process, const char *line)
{
	if (strstr (line, "No disk / Wrong disk!") != NULL) {
		/* This is not supposed to happen since we checked for the cd
		   before starting, but we try to handle it anyway, since mmc
		   profiling can fail. */
		bonfire_job_error (BONFIRE_JOB (process),
				   g_error_new (BONFIRE_BURN_ERROR,
						BONFIRE_BURN_ERROR_MEDIA_NONE,
						_("There doesn't seem to be a disc in the drive")));
	}
	else if (strstr (line, "This means that we are checking recorded media.") != NULL) {
		bonfire_job_error (BONFIRE_JOB (process),
				   g_error_new (BONFIRE_BURN_ERROR,
						BONFIRE_BURN_ERROR_MEDIA_NOT_WRITABLE,
						_("The CD has already been recorded")));
	}
	else if (strstr (line, "Cannot blank disk, aborting.") != NULL) {
		/* This is not supposed to happen since we checked for the cd
		   type before starting, but we try to handle it anyway, since
		   mmc profiling can fail. */
		bonfire_job_error (BONFIRE_JOB (process),
				   g_error_new (BONFIRE_BURN_ERROR,
						BONFIRE_BURN_ERROR_MEDIA_NOT_REWRITABLE,
						_("The CD cannot be blanked")));
	}
	else if (strstr (line, "Data may not fit on current disk")) {
		bonfire_job_error (BONFIRE_JOB (process),
				   g_error_new (BONFIRE_BURN_ERROR,
						BONFIRE_BURN_ERROR_MEDIA_SPACE,
						_("The files selected did not fit on the CD")));
		/* FIXME should we really error out in that case? */
	}
	else if (strstr (line, "Inappropriate audio coding")) {
		bonfire_job_error (BONFIRE_JOB (process),
				   g_error_new (BONFIRE_BURN_ERROR,
						BONFIRE_BURN_ERROR_INCOMPATIBLE_FORMAT,
						_("All audio files must be stereo, 16-bit digital audio with 44100Hz samples")));
	}
	else if (strstr (line, "cannot write medium - incompatible format") != NULL) {
		/* This is not supposed to happen since we checked for the cd
		   type before starting, but we try to handle it anyway, since
		   mmc profiling can fail. */
		bonfire_job_error (BONFIRE_JOB (process),
				   g_error_new (BONFIRE_BURN_ERROR,
						BONFIRE_BURN_ERROR_INCOMPATIBLE_FORMAT,
						_("The image does not seem to be a proper iso9660 file system")));
	}
	else if (strstr (line, "DMA speed too slow") != NULL) {
		bonfire_job_error (BONFIRE_JOB (process),
				   g_error_new (BONFIRE_BURN_ERROR,
						BONFIRE_BURN_ERROR_SLOW_DMA,
						_("The system is too slow to write the CD at this speed. Try a lower speed")));
	}
	else if (strstr (line, "Operation not permitted. Cannot send SCSI cmd via ioctl")) {
		bonfire_job_error (BONFIRE_JOB (process),
				   g_error_new (BONFIRE_BURN_ERROR,
						BONFIRE_BURN_ERROR_SCSI_IOCTL,
						_("You don't seem to have the required permission to use this drive")));
	}
	else if (strstr (line, "Device or resource busy")) {
		bonfire_job_error (BONFIRE_JOB (process),
				   g_error_new (BONFIRE_BURN_ERROR,
						BONFIRE_BURN_ERROR_BUSY_DRIVE,
						_("The drive seems to be busy (maybe check you have proper permissions to use it)")));
	}
	else if (strstr (line, "Illegal write mode for this drive")) {
		/* NOTE : when it happened I had to unlock the
		 * drive with cdrdao and eject it. Should we ? */
		bonfire_job_error (BONFIRE_JOB (process),
				   g_error_new (BONFIRE_BURN_ERROR,
						BONFIRE_BURN_ERROR_BUSY_DRIVE,
						_("The drive seems to be busy (maybe you should reload the media)")));
	}
	else if (strstr (line, "cdrecord: No such file or directory. Cannot open")) {
		bonfire_job_error (BONFIRE_JOB (process),
				   g_error_new (BONFIRE_BURN_ERROR,
						BONFIRE_BURN_ERROR_GENERAL,
						_("the image file cannot be found")));
	}
	else if (strstr (line, "Bad file descriptor. read error on input file")) {
		bonfire_job_error (BONFIRE_JOB (process),
				   g_error_new (BONFIRE_BURN_ERROR,
						BONFIRE_BURN_ERROR_GENERAL,
						_("internal error")));
	}
	else if (strstr (line, "Could not write Lead-in")) {
		bonfire_job_error (BONFIRE_JOB (process),
				   g_error_new (BONFIRE_BURN_ERROR,
						BONFIRE_BURN_ERROR_GENERAL,
						_("the cd information could not be written")));
	}
	return BONFIRE_BURN_OK;
}

static void
bonfire_cdrecord_check_fifo (BonfireCDRecord *cdrecord, int buf)
{
	BonfireJob *slave;
	gint64 rate = (cdrecord->priv->speed + 1) * CDR_SPEED;

	slave = bonfire_job_get_slave (BONFIRE_JOB (cdrecord));
	if (!slave)
		return;

	if (cdrecord->priv->fifo < 50) {
		/* we try to raise the rate */
		rate = (cdrecord->priv->speed * 2) * CDR_SPEED;
	}
	else if (cdrecord->priv->fifo > 95) {
		/* we try to lower the rate */
		if (buf < 90)
			rate = (cdrecord->priv->speed * 1.5) * CDR_SPEED;
	}

	bonfire_job_debug_message (BONFIRE_JOB (cdrecord),
				   "setting rate to %" G_GINT64_FORMAT,
				   rate);
	bonfire_job_set_rate (slave, rate);
}

static void
bonfire_cdrecord_compute (BonfireCDRecord *cdrecord,
			  int mb_written,
			  int mb_total,
			  gdouble speed,
			  int track_num)
{
	double percent;
	gint64 bytes;
	gint64 this_remain;
	gint64 total;
	long secs;

	if (cdrecord->priv->tracks_total_bytes > 0)
		total = cdrecord->priv->tracks_total_bytes;
	else
		total = mb_total * 1048576;

	if (track_num > cdrecord->priv->current_track_num) {
		cdrecord->priv->current_track_num = track_num;
		cdrecord->priv->current_track_end_pos += mb_total * 1048576;
	}

	this_remain = (mb_total - mb_written) * 1048576;
	bytes = (total - cdrecord->priv->current_track_end_pos) + this_remain;
	cdrecord->priv->b_written = total - bytes;

	secs = bonfire_burn_common_compute_time_remaining (bytes, (double) speed * CDR_SPEED);

	bonfire_job_action_changed (BONFIRE_JOB (cdrecord),
				    BONFIRE_BURN_ACTION_WRITING,
				    FALSE);

	if (cdrecord->priv->tracks_total_bytes > 0) {
		percent = 0.98 * (1.0 - (double) bytes / (double) cdrecord->priv->tracks_total_bytes);
	}
	else {
		percent = 0.98 * ((double) ((track_num - 1) / (double)cdrecord->priv->track_count)
			  + ((double)mb_written / mb_total) / (double)cdrecord->priv->track_count);
	}

	bonfire_job_progress_changed (BONFIRE_JOB (cdrecord), percent, secs);
}

static BonfireBurnResult
bonfire_cdrecord_stdout_read (BonfireProcess *process, const char *line)
{
	guint track;
	gdouble speed;
	BonfireCDRecord *cdrecord;
	int mb_written = 0, mb_total = 0, fifo = 0, buf = 0;

	cdrecord = BONFIRE_CD_RECORD (process);

	if (sscanf (line, "Track %2u: %d of %d MB written (fifo %d%%) [buf %d%%] %lfx.",
		    &track, &mb_written, &mb_total, &fifo, &buf, &speed) == 6) {
		cdrecord->priv->fifo = fifo;
		cdrecord->priv->cur_speed = (int) ceil (speed);
		cdrecord->priv->current_track_written = mb_written * 1048576;
		bonfire_cdrecord_compute (cdrecord,
					  mb_written,
					  mb_total,
					  speed,
					  track);

		bonfire_cdrecord_check_fifo (cdrecord, buf);
	} 
	else if (sscanf (line, "Track %2u:    %d MB written (fifo %d%%) [buf  %d%%]  %lfx.",
			 &track, &mb_written, &fifo, &buf, &speed) == 5) {
		int mb_total;

		/* this line is printed when cdrecord writes on the fly */
		cdrecord->priv->fifo = fifo;
		cdrecord->priv->cur_speed = (int) ceil (speed);
		cdrecord->priv->current_track_written = mb_written * 1048576;

		if (cdrecord->priv->track->type == BONFIRE_TRACK_SOURCE_IMAGER) {
			/* we must ask the imager what is the total size */
			bonfire_imager_get_size (BONFIRE_IMAGER (cdrecord->priv->track->contents.imager.obj),
						 &cdrecord->priv->tracks_total_bytes,
						 FALSE, 
						 NULL);
			mb_total = cdrecord->priv->tracks_total_bytes / 1048576;

			bonfire_cdrecord_compute (cdrecord,
						  mb_written,
						  mb_total,
						  speed,
						  track);
		}
		else
			bonfire_job_action_changed (BONFIRE_JOB (cdrecord),
						    BONFIRE_BURN_ACTION_WRITING,
						    FALSE);

		bonfire_cdrecord_check_fifo (cdrecord, buf);
	}
	else if (sscanf (line, "Track %*d: %*s %d MB ", &mb_total) == 1) {
		if (mb_total > 0) {
			cdrecord->priv->tracks_total_bytes += mb_total * 1048576;
		}
	}
	else if (strstr (line, "Sending CUE sheet")) {
		bonfire_job_action_changed (BONFIRE_JOB (process),
					    BONFIRE_BURN_ACTION_WRITING_CD_TEXT,
					    FALSE);
	}
	else if (g_str_has_prefix (line, "Re-load disk and hit <CR>")
	     ||  g_str_has_prefix (line, "send SIGUSR1 to continue")) {
		/* NOTE: There seems to be a BUG somewhere when writing raw images
		 * with clone mode. After disc has been written and fixated cdrecord
		 * asks the media to be reloaded. So we simply ignore this message
		 * and returns that everything went well. Which is indeed the case */
		 if (bonfire_job_get_current_action (BONFIRE_JOB (process)) == BONFIRE_BURN_ACTION_FIXATING) {
			bonfire_job_finished (BONFIRE_JOB (process));
			return BONFIRE_BURN_OK;
		 }

		/* This is not supposed to happen since we checked for the cd
		   before starting, but we try to handle it anyway, since mmc
		   profiling can fail. */
		/* NOTE : nautilus_burn_recorder used to send sigusr1 or return */
		bonfire_job_error (BONFIRE_JOB (process),
				   g_error_new (BONFIRE_BURN_ERROR,
						BONFIRE_BURN_ERROR_RELOAD_MEDIA,
						_("The media needs to be reloaded before being recorded")));
	}
	else if (g_str_has_prefix (line, "Fixating...")) {
		bonfire_job_progress_changed (BONFIRE_JOB (process), 
					      0.98,
					      -1);
		bonfire_job_action_changed (BONFIRE_JOB (process),
					    BONFIRE_BURN_ACTION_FIXATING,
					    FALSE);
	}
	else if (g_str_has_prefix (line, "Last chance to quit, ")) {
		bonfire_job_set_dangerous (BONFIRE_JOB (process), TRUE);
	}
	else if (g_str_has_prefix (line, "Blanking PMA, TOC, pregap")) {
		bonfire_job_action_changed (BONFIRE_JOB (process),
					    BONFIRE_BURN_ACTION_BLANKING,
					    FALSE);
	}
	else if (strstr (line, "Use tsize= option in SAO mode to specify track size")) {
		bonfire_job_error (BONFIRE_JOB (process),
				   g_error_new (BONFIRE_BURN_ERROR,
						BONFIRE_BURN_ERROR_GENERAL,
						_("internal error")));
	}

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_cdrecord_set_argv_record (BonfireCDRecord *cdrecord,
				  GPtrArray *argv, 
				  GError **error)
{
	if (!cdrecord->priv->track)
		return BONFIRE_BURN_NOT_READY;

        if (cdrecord->priv->speed > 0) {
		char *speed_str;

		speed_str = g_strdup_printf ("speed=%d", cdrecord->priv->speed);
		g_ptr_array_add (argv, speed_str);
	}

	if (cdrecord->priv->dao)
		g_ptr_array_add (argv, g_strdup ("-dao"));
	if (cdrecord->priv->overburn)
		g_ptr_array_add (argv, g_strdup ("-overburn"));
	if (cdrecord->priv->burnproof)
		g_ptr_array_add (argv, g_strdup ("driveropts=burnfree"));
	if (cdrecord->priv->multi)
		g_ptr_array_add (argv, g_strdup ("-multi"));

	if (cdrecord->priv->track->type == BONFIRE_TRACK_SOURCE_IMAGER) {
		BonfireTrackSourceType track_type;
		BonfireBurnResult result;
		BonfireImager *imager;
		int buffer_size;
		gint64 sectors;
		
		imager = cdrecord->priv->track->contents.imager.obj;

		/* we need to know what is the type of the track (audio / data) */
		result = bonfire_imager_get_track_type (imager, &track_type);
		if (result != BONFIRE_BURN_OK) {
			g_set_error (error,
				     BONFIRE_BURN_ERROR,
				     BONFIRE_BURN_ERROR_GENERAL,
				     _("imager doesn't seem to be ready"));
			return BONFIRE_BURN_ERR;
		}
		
		/* ask the size */
		result = bonfire_imager_get_size (imager, &sectors, TRUE, error);
		if (result != BONFIRE_BURN_OK) {
			if (!error)
				g_set_error (error,
					     BONFIRE_BURN_ERROR,
					     BONFIRE_BURN_ERROR_GENERAL,
					     _("imager doesn't seem to be ready"));
			return BONFIRE_BURN_ERR;
		}

		/* we create a buffer depending on the size 
		 * buffer 4m> < 64m and is 1/25th of size otherwise */
		buffer_size = sectors * 2352 / 1024 / 1024 / 25;
		if (buffer_size > 32)
			buffer_size = 32;
		else if (buffer_size < 4)
			buffer_size = 4;

		g_ptr_array_add (argv, g_strdup_printf ("fs=%im", buffer_size));

		if (track_type == BONFIRE_TRACK_SOURCE_ISO
		||  track_type == BONFIRE_TRACK_SOURCE_ISO_JOLIET) {
			g_ptr_array_add (argv, g_strdup_printf ("tsize=%Lis", sectors));

			g_ptr_array_add (argv, g_strdup ("-data"));
			g_ptr_array_add (argv, g_strdup ("-nopad"));
			g_ptr_array_add (argv, g_strdup ("-"));

			bonfire_job_set_slave (BONFIRE_JOB (cdrecord), BONFIRE_JOB (imager));
			bonfire_job_set_relay_slave_signals (BONFIRE_JOB (cdrecord), FALSE);
		}
		else if (track_type == BONFIRE_TRACK_SOURCE_RAW) {
			g_ptr_array_add (argv, g_strdup ("fs=16m"));
			g_ptr_array_add (argv, g_strdup ("-raw96r"));
			g_ptr_array_add (argv, g_strdup ("-clone"));

			/* we need to generate the toc first */

			if (result != BONFIRE_BURN_OK)
				return result;

			bonfire_job_set_slave (BONFIRE_JOB (cdrecord), BONFIRE_JOB (imager));
			bonfire_job_set_relay_slave_signals (BONFIRE_JOB (cdrecord), FALSE);
		}
		else if (track_type == BONFIRE_TRACK_SOURCE_AUDIO) {
			GSList *iter;

			/* we need to get the inf first */
			result = bonfire_imager_set_output_type (imager,
								 BONFIRE_TRACK_SOURCE_INF,
								 error);
			if (result != BONFIRE_BURN_OK)
				return result;

			bonfire_job_set_slave (BONFIRE_JOB (cdrecord), BONFIRE_JOB (imager));
			bonfire_job_set_relay_slave_signals (BONFIRE_JOB (cdrecord), TRUE);

			result = bonfire_imager_get_track (imager,
							   &cdrecord->priv->inf,
							   error);

			bonfire_job_set_relay_slave_signals (BONFIRE_JOB (cdrecord), FALSE);

			if (result != BONFIRE_BURN_OK)
				return result;
	
			result = bonfire_imager_set_output_type (imager,
								 BONFIRE_TRACK_SOURCE_AUDIO,
								 error);
			if (result != BONFIRE_BURN_OK)
				return result;

			/* now we set the rate of the slave slightly above the speed */
			bonfire_job_set_rate (BONFIRE_JOB (imager),
					      (cdrecord->priv->speed + 1) * CDR_SPEED);

			/* now set the rest of the arguments */
			g_ptr_array_add (argv, g_strdup ("-dao"));
			g_ptr_array_add (argv, g_strdup ("-audio"));
			g_ptr_array_add (argv, g_strdup ("-useinfo"));
			g_ptr_array_add (argv, g_strdup ("-text"));

			for (iter = cdrecord->priv->inf->contents.inf.files; iter; iter = iter->next) {
				char *arg;
	
				arg = g_strdup (iter->data);
				g_ptr_array_add (argv, arg);
			}
		}
		else
			return BONFIRE_BURN_NOT_SUPPORTED;

		bonfire_job_set_run_slave (BONFIRE_JOB (cdrecord), TRUE);
	}
	else if (cdrecord->priv->track->type == BONFIRE_TRACK_SOURCE_AUDIO) {
		GSList *iter;

		/* CD-text cannot be written in tao mode (which is the default) */
		g_ptr_array_add (argv, g_strdup ("-dao"));

		g_ptr_array_add (argv, g_strdup ("fs=16m"));
		g_ptr_array_add (argv, g_strdup ("-audio"));
		g_ptr_array_add (argv, g_strdup ("-pad"));
		g_ptr_array_add (argv, g_strdup ("-useinfo"));
		g_ptr_array_add (argv, g_strdup ("-text"));

		for (iter = cdrecord->priv->track->contents.audio.files; iter; iter = iter->next) {
			char *arg;

			arg = g_strdup (iter->data);
			g_ptr_array_add (argv, arg);
		}

		bonfire_job_set_run_slave (BONFIRE_JOB (cdrecord), FALSE);
	}
	else if (cdrecord->priv->track->type == BONFIRE_TRACK_SOURCE_ISO
	      ||  cdrecord->priv->track->type == BONFIRE_TRACK_SOURCE_ISO_JOLIET) {

		g_ptr_array_add (argv, g_strdup ("fs=16m"));
		g_ptr_array_add (argv, g_strdup ("-data"));
		g_ptr_array_add (argv, g_strdup ("-nopad"));
		g_ptr_array_add (argv, g_strdup (cdrecord->priv->track->contents.iso.image));

		bonfire_job_set_run_slave (BONFIRE_JOB (cdrecord), FALSE);
	}
	else if (cdrecord->priv->track->type == BONFIRE_TRACK_SOURCE_RAW) {
		g_ptr_array_add (argv, g_strdup ("fs=16m"));
		g_ptr_array_add (argv, g_strdup ("-raw96r"));
		g_ptr_array_add (argv, g_strdup ("-clone"));

		if (cdrecord->priv->track->contents.raw.image)
			g_ptr_array_add (argv, g_strdup (cdrecord->priv->track->contents.raw.image));

		bonfire_job_set_run_slave (BONFIRE_JOB (cdrecord), FALSE);
	}
	else if (cdrecord->priv->track->type == BONFIRE_TRACK_SOURCE_CUE) {
		char *cue_str;

		g_ptr_array_add (argv, g_strdup ("fs=16m"));
		g_ptr_array_add (argv, g_strdup ("-dao"));

		cue_str = g_strdup_printf ("cuefile=%s", cdrecord->priv->track->contents.cue.toc);
		g_ptr_array_add (argv, cue_str);

		bonfire_job_set_run_slave (BONFIRE_JOB (cdrecord), FALSE);
	}
	else
		return BONFIRE_BURN_NOT_SUPPORTED;

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_cdrecord_set_argv_blank (BonfireCDRecord *cdrecord, GPtrArray *argv)
{
	char *blank_str;

	blank_str = g_strdup_printf ("blank=%s", cdrecord->priv->blank_fast ? "fast" : "all");
	g_ptr_array_add (argv, blank_str);

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_cdrecord_set_argv (BonfireProcess *process,
			   GPtrArray *argv,
			   gboolean has_master,
			   GError **error)
{
	BonfireCDRecord *cdrecord;
	BonfireBurnResult result;
	char *dev_str;

	cdrecord = BONFIRE_CD_RECORD (process);

	if (has_master)
		return BONFIRE_BURN_NOT_SUPPORTED;

	if (!cdrecord->priv->drive)
		return BONFIRE_BURN_NOT_READY;

	g_ptr_array_add (argv, g_strdup ("cdrecord"));
	g_ptr_array_add (argv, g_strdup ("-v"));

	if (cdrecord->priv->drive->cdrecord_id)
		dev_str = g_strdup_printf ("dev=%s", cdrecord->priv->drive->cdrecord_id);
	else
		dev_str = g_strdup_printf ("dev=%s", cdrecord->priv->drive->device);

	g_ptr_array_add (argv, dev_str);

        if (cdrecord->priv->dummy)
		g_ptr_array_add (argv, g_strdup ("-dummy"));

	if (cdrecord->priv->nograce)
		g_ptr_array_add (argv, g_strdup ("gracetime=0"));

	if (cdrecord->priv->action == BONFIRE_CD_RECORD_ACTION_RECORD)
		result = bonfire_cdrecord_set_argv_record (cdrecord, argv, error);
	else if (cdrecord->priv->action == BONFIRE_CD_RECORD_ACTION_BLANK)
		result = bonfire_cdrecord_set_argv_blank (cdrecord, argv);
	else
		return BONFIRE_BURN_NOT_READY;

	if (result == BONFIRE_BURN_OK || result == BONFIRE_BURN_CANCEL) {
		bonfire_job_action_changed (BONFIRE_JOB (cdrecord),
					    BONFIRE_BURN_ACTION_PREPARING,
					    FALSE);
		bonfire_job_progress_changed (BONFIRE_JOB (cdrecord), 0, -1);
	}

	return result;	
}

static BonfireBurnResult
bonfire_cdrecord_set_drive (BonfireRecorder *recorder,
			    NautilusBurnDrive *drive,
			    GError **error)
{
	BonfireCDRecord *cdrecord;
	NautilusBurnMediaType media;

	media = nautilus_burn_drive_get_media_type (drive);
	if (media > NAUTILUS_BURN_MEDIA_TYPE_CDRW)
		return BONFIRE_BURN_NOT_SUPPORTED;

	cdrecord = BONFIRE_CD_RECORD (recorder);

	if (cdrecord->priv->drive) {
		nautilus_burn_drive_unref (cdrecord->priv->drive);
		cdrecord->priv->drive = NULL;
	}

	cdrecord->priv->drive = drive;
	nautilus_burn_drive_ref (drive);

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_cdrecord_set_source (BonfireJob *job,
			     const BonfireTrackSource *track,
			     GError **error)
{
	BonfireCDRecord *cdrecord;

	cdrecord = BONFIRE_CD_RECORD (job);

	if (track->type != BONFIRE_TRACK_SOURCE_AUDIO
	&&  track->type != BONFIRE_TRACK_SOURCE_CUE
	&&  track->type != BONFIRE_TRACK_SOURCE_RAW
	&&  track->type != BONFIRE_TRACK_SOURCE_ISO
	&&  track->type != BONFIRE_TRACK_SOURCE_ISO_JOLIET
	&&  track->type != BONFIRE_TRACK_SOURCE_IMAGER)
		return BONFIRE_BURN_NOT_SUPPORTED;

	if (cdrecord->priv->inf) {
		bonfire_track_source_free (cdrecord->priv->inf);
		cdrecord->priv->inf = NULL;
	}

	if (cdrecord->priv->track)
		bonfire_track_source_free (cdrecord->priv->track);

	cdrecord->priv->track = bonfire_track_source_copy (track);

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_cdrecord_set_flags (BonfireRecorder *recorder,
			    BonfireRecorderFlag flags,
			    GError **error)
{
	BonfireCDRecord *cdrecord;

	cdrecord = BONFIRE_CD_RECORD (recorder);

	cdrecord->priv->dummy = (flags & BONFIRE_RECORDER_FLAG_DUMMY) != 0;
	cdrecord->priv->dao = (flags & BONFIRE_RECORDER_FLAG_DAO) != 0;
	cdrecord->priv->nograce = (flags & BONFIRE_RECORDER_FLAG_NOGRACE) != 0;
	cdrecord->priv->burnproof = (flags & BONFIRE_RECORDER_FLAG_BURNPROOF) != 0;
	cdrecord->priv->overburn = (flags & BONFIRE_RECORDER_FLAG_OVERBURN) != 0;
	cdrecord->priv->blank_fast = (flags & BONFIRE_RECORDER_FLAG_FAST_BLANK) != 0;
	cdrecord->priv->multi = (flags & BONFIRE_RECORDER_FLAG_MULTI) != 0;

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_cdrecord_set_speed (BonfireJob *job,
			    gint64 speed)
{
	BonfireCDRecord *cdrecord;

	if (bonfire_job_is_running (job))
		return BONFIRE_BURN_RUNNING;

	cdrecord = BONFIRE_CD_RECORD (job);
	cdrecord->priv->speed = speed / CDR_SPEED;
	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_cdrecord_blank (BonfireRecorder *recorder,
			GError **error)
{
	BonfireCDRecord *cdrecord;
	BonfireBurnResult result;

	cdrecord = BONFIRE_CD_RECORD (recorder);

	if (!nautilus_burn_drive_can_rewrite (cdrecord->priv->drive)) {
		g_set_error (error,
			     BONFIRE_BURN_ERROR,
			     BONFIRE_BURN_ERROR_GENERAL,
			     _("the drive cannot rewrite CDs or DVDs"));
		return BONFIRE_BURN_ERR;
	}

	cdrecord->priv->action = BONFIRE_CD_RECORD_ACTION_BLANK;
	result = bonfire_job_run (BONFIRE_JOB (cdrecord), error);
	cdrecord->priv->action = BONFIRE_CD_RECORD_ACTION_NONE;

	bonfire_job_action_changed (BONFIRE_JOB (cdrecord),
				    BONFIRE_BURN_ACTION_NONE,
				    FALSE);

	return result;	
}

static BonfireBurnResult
bonfire_cdrecord_record (BonfireRecorder *recorder,
			 GError **error)
{
	BonfireCDRecord *cdrecord;
	BonfireBurnResult result;

	cdrecord = BONFIRE_CD_RECORD (recorder);

	if (!cdrecord->priv->track)
		return BONFIRE_BURN_NOT_READY;

	cdrecord->priv->action = BONFIRE_CD_RECORD_ACTION_RECORD;

	/* set as slave if track is an imager (on the fly burning) */
	if (cdrecord->priv->track->type == BONFIRE_TRACK_SOURCE_IMAGER) {
		BonfireJob *slave;

		slave = BONFIRE_JOB (cdrecord->priv->track->contents.imager.obj);
		bonfire_job_set_slave (BONFIRE_JOB (cdrecord), slave);
		bonfire_job_set_relay_slave_signals (BONFIRE_JOB (cdrecord), FALSE);
		bonfire_job_set_run_slave (BONFIRE_JOB (cdrecord), TRUE);
	}
	else
		bonfire_job_set_run_slave (BONFIRE_JOB (cdrecord), FALSE);

	result = bonfire_job_run (BONFIRE_JOB (cdrecord), error);
	bonfire_job_set_slave (BONFIRE_JOB (cdrecord), NULL);
	cdrecord->priv->action = BONFIRE_CD_RECORD_ACTION_NONE;
	return result;				
}

static BonfireBurnResult
bonfire_cdrecord_get_status (BonfireRecorder *recorder,
			     int *speed,
			     int *fifo,
			     gint64 *bytes_written)
{
	BonfireCDRecord *cdrecord;

	cdrecord = BONFIRE_CD_RECORD (recorder);

	if (speed)
		*speed = cdrecord->priv->cur_speed;

	if (fifo)
		*fifo = cdrecord->priv->fifo;

	if (bytes_written)
		*bytes_written = cdrecord->priv->b_written;

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_cdrecord_get_action_string (BonfireJob *job,
				    BonfireBurnAction action,
				    char **string)
{
	job = bonfire_job_get_slave (job);
	if (!job)
		return BONFIRE_BURN_NOT_SUPPORTED;

	return bonfire_job_get_action_string (job, action, string);
}
